import React, { useState } from "react";
import { Head, Link, router, usePage } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Plus,
    Edit,
    Trash2,
    ToggleLeft,
    ToggleRight,
    DollarSign,
    CheckCircle2,
    XCircle,
    AlertTriangle,
} from "lucide-react";

export default function Index({ currencies }) {
    const { flash } = usePage().props;
    const [deleting, setDeleting] = useState(null);

    const handleDelete = (currency) => {
        if (confirm(`Are you sure you want to delete ${currency.name}?`)) {
            setDeleting(currency.id);
            router.delete(route("currencies.destroy", currency.id), {
                onFinish: () => setDeleting(null),
            });
        }
    };

    const handleToggleStatus = (currency) => {
        router.post(route("currencies.toggle-status", currency.id));
    };

    const getCurrencySymbol = (code) => {
        const symbols = {
            USD: "$",
            EUR: "€",
            GBP: "£",
            JPY: "¥",
            CAD: "C$",
            AUD: "A$",
            CHF: "CHF",
            CNY: "¥",
            INR: "₹",
        };
        return symbols[code] || code;
    };

    return (
        <AdminLayout>
            <Head title="Currency Management" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto">
                    {/* Header */}
                    <div className="flex justify-between items-center mb-6">
                        <div>
                            <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                                Currency Management
                            </h1>
                            <p className="text-gray-600 dark:text-gray-400 mt-1">
                                Manage currencies displayed on the order form
                            </p>
                        </div>
                        <Link
                            href={route("currencies.create")}
                            className="flex items-center gap-2 px-4 py-2 bg-indigo-600 hover:bg-indigo-700 text-white font-medium rounded-lg transition-colors"
                        >
                            <Plus size={20} />
                            Add Currency
                        </Link>
                    </div>

                    {/* Flash Messages */}
                    {flash?.message && (
                        <div className="mb-6 p-4 bg-green-100 dark:bg-green-900/30 border border-green-200 dark:border-green-800 text-green-800 dark:text-green-200 rounded-lg flex items-center gap-2">
                            <CheckCircle2 size={20} />
                            <span>{flash.message}</span>
                        </div>
                    )}

                    {flash?.error && (
                        <div className="mb-6 p-4 bg-red-100 dark:bg-red-900/30 border border-red-200 dark:border-red-800 text-red-800 dark:text-red-200 rounded-lg flex items-center gap-2">
                            <AlertTriangle size={20} />
                            <span>{flash.error}</span>
                        </div>
                    )}

                    {/* Info Card */}
                    <div className="mb-6 p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
                        <p className="text-sm text-blue-800 dark:text-blue-200">
                            💡 <strong>Note:</strong> Only active currencies will appear on the order form. USD is the base currency and cannot be deactivated or deleted.
                        </p>
                    </div>

                    {/* Currencies Grid */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6">
                        {currencies.map((currency) => (
                            <div
                                key={currency.id}
                                className={`bg-white dark:bg-gray-800 rounded-lg shadow-sm border-2 transition-all ${
                                    currency.is_active
                                        ? "border-green-200 dark:border-green-800"
                                        : "border-gray-200 dark:border-gray-700 opacity-60"
                                }`}
                            >
                                <div className="p-6">
                                    {/* Header */}
                                    <div className="flex items-start justify-between mb-4">
                                        <div className="flex items-center gap-3">
                                            <div
                                                className={`w-12 h-12 rounded-full flex items-center justify-center text-2xl font-bold ${
                                                    currency.is_active
                                                        ? "bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300"
                                                        : "bg-gray-100 dark:bg-gray-700 text-gray-500 dark:text-gray-400"
                                                }`}
                                            >
                                                {getCurrencySymbol(currency.currency_code)}
                                            </div>
                                            <div>
                                                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                                    {currency.name}
                                                </h3>
                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                    {currency.currency_code}
                                                </p>
                                            </div>
                                        </div>
                                        <div className="flex items-center">
                                            {currency.is_active ? (
                                                <span className="flex items-center gap-1 text-green-600 dark:text-green-400 text-sm font-medium">
                                                    <CheckCircle2 size={16} />
                                                    Active
                                                </span>
                                            ) : (
                                                <span className="flex items-center gap-1 text-gray-500 dark:text-gray-400 text-sm font-medium">
                                                    <XCircle size={16} />
                                                    Inactive
                                                </span>
                                            )}
                                        </div>
                                    </div>

                                    {/* Exchange Rate */}
                                    <div className="mb-4 p-3 bg-gray-50 dark:bg-gray-700/50 rounded-lg">
                                        <p className="text-xs text-gray-600 dark:text-gray-400 mb-1">
                                            Exchange Rate (vs USD)
                                        </p>
                                        <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                            {parseFloat(currency.value).toFixed(6)}
                                        </p>
                                        <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                            1 USD = {parseFloat(currency.value).toFixed(4)} {currency.currency_code}
                                        </p>
                                    </div>

                                    {/* Actions */}
                                    <div className="flex gap-2">
                                        <Link
                                            href={route("currencies.edit", currency.id)}
                                            className="flex-1 flex items-center justify-center gap-2 px-3 py-2 bg-blue-600 hover:bg-blue-700 text-white text-sm font-medium rounded-md transition-colors"
                                        >
                                            <Edit size={16} />
                                            Edit
                                        </Link>
                                        <button
                                            onClick={() => handleToggleStatus(currency)}
                                            disabled={currency.currency_code === "USD" && currency.is_active}
                                            className={`flex-1 flex items-center justify-center gap-2 px-3 py-2 text-sm font-medium rounded-md transition-colors ${
                                                currency.is_active
                                                    ? "bg-amber-600 hover:bg-amber-700 text-white disabled:opacity-50 disabled:cursor-not-allowed"
                                                    : "bg-green-600 hover:bg-green-700 text-white"
                                            }`}
                                        >
                                            {currency.is_active ? (
                                                <>
                                                    <ToggleLeft size={16} />
                                                    Deactivate
                                                </>
                                            ) : (
                                                <>
                                                    <ToggleRight size={16} />
                                                    Activate
                                                </>
                                            )}
                                        </button>
                                        <button
                                            onClick={() => handleDelete(currency)}
                                            disabled={currency.currency_code === "USD" || deleting === currency.id}
                                            className="px-3 py-2 bg-red-600 hover:bg-red-700 text-white text-sm font-medium rounded-md transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                                            title={currency.currency_code === "USD" ? "Cannot delete base currency" : "Delete currency"}
                                        >
                                            <Trash2 size={16} />
                                        </button>
                                    </div>

                                    {currency.currency_code === "USD" && (
                                        <p className="mt-3 text-xs text-center text-gray-500 dark:text-gray-400">
                                            🔒 Base currency - protected
                                        </p>
                                    )}
                                </div>
                            </div>
                        ))}
                    </div>

                    {/* Empty State */}
                    {currencies.length === 0 && (
                        <div className="text-center py-12">
                            <DollarSign
                                size={48}
                                className="mx-auto text-gray-400 dark:text-gray-600 mb-4"
                            />
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                                No currencies found
                            </h3>
                            <p className="text-gray-600 dark:text-gray-400 mb-4">
                                Get started by adding your first currency
                            </p>
                            <Link
                                href={route("currencies.create")}
                                className="inline-flex items-center gap-2 px-4 py-2 bg-indigo-600 hover:bg-indigo-700 text-white font-medium rounded-lg transition-colors"
                            >
                                <Plus size={20} />
                                Add Currency
                            </Link>
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
}
