import { Head, useForm, Link } from "@inertiajs/react";
import React, { useState } from "react";
import { ArrowLeft, Save, RefreshCw } from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";
import { Button } from "@/Components/ui/Button";
import ValidationErrors from "@/Components/OrderForm/ValidationErrors";

export default function Create() {
    const { data, setData, post, processing, errors } = useForm({
        coupon_code: "DEFAULT10",
        discount_type: "percentage",
        discount_amount: 10,
        per_user_limit: "",
        max_uses: "",
    });

    const [isGeneratingCode, setIsGeneratingCode] = useState(false);

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.default-coupons.store"));
    };

    const generateCode = async () => {
        setIsGeneratingCode(true);
        try {
            const response = await fetch(route("admin.coupons.generate-code"));
            const data = await response.json();
            if (data.success) {
                setData("coupon_code", data.code);
            }
        } catch (error) {
            console.error("Error generating code:", error);
        } finally {
            setIsGeneratingCode(false);
        }
    };

    return (
        <AdminLayout>
            <Head title="Create Default Coupon" />

            <div className="py-6">
                <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div>
                                <div className="flex items-center">
                                    <Link
                                        href={route("admin.default-coupons.index")}
                                        className="mr-4 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
                                    >
                                        <ArrowLeft className="w-5 h-5" />
                                    </Link>
                                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                        Create Default Coupon
                                    </h1>
                                </div>
                                <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                                    Create a default discount coupon that will be available to all users
                                </p>
                            </div>
                        </div>
                    </div>

                    {/* Form */}
                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg">
                        <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                Coupon Details
                            </h3>
                        </div>

                        <form onSubmit={handleSubmit} className="p-6 space-y-6">
                            <ValidationErrors errors={errors} />

                            {/* Coupon Code */}
                            <div>
                                <label htmlFor="coupon_code" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Coupon Code
                                </label>
                                <div className="flex gap-2">
                                    <input
                                        type="text"
                                        id="coupon_code"
                                        value={data.coupon_code}
                                        onChange={(e) => setData("coupon_code", e.target.value)}
                                        className="flex-1 px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                        placeholder="Enter coupon code"
                                    />
                                    <Button
                                        type="button"
                                        onClick={generateCode}
                                        disabled={isGeneratingCode}
                                        variant="secondary"
                                    >
                                        {isGeneratingCode ? (
                                            <RefreshCw className="w-4 h-4 animate-spin" />
                                        ) : (
                                            <RefreshCw className="w-4 h-4" />
                                        )}
                                    </Button>
                                </div>
                                {errors.coupon_code && (
                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.coupon_code}</p>
                                )}
                            </div>

                            {/* Discount Type */}
                            <div>
                                <label htmlFor="discount_type" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Discount Type
                                </label>
                                <select
                                    id="discount_type"
                                    value={data.discount_type}
                                    onChange={(e) => setData("discount_type", e.target.value)}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                >
                                    <option value="percentage">Percentage</option>
                                    <option value="fixed">Fixed Amount</option>
                                </select>
                                {errors.discount_type && (
                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.discount_type}</p>
                                )}
                            </div>

                            {/* Discount Amount */}
                            <div>
                                <label htmlFor="discount_amount" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Discount Amount
                                </label>
                                <input
                                    type="number"
                                    id="discount_amount"
                                    value={data.discount_amount}
                                    onChange={(e) => setData("discount_amount", parseFloat(e.target.value))}
                                    step="0.01"
                                    min="0"
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                    placeholder={data.discount_type === "percentage" ? "10" : "50.00"}
                                />
                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                    {data.discount_type === "percentage" 
                                        ? "Enter the percentage discount (e.g., 10 for 10%)"
                                        : "Enter the fixed amount discount (e.g., 50.00 for $50)"
                                    }
                                </p>
                                {errors.discount_amount && (
                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.discount_amount}</p>
                                )}
                            </div>

                            {/* Usage Limits Section */}
                            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                                <h4 className="text-sm font-medium text-blue-900 dark:text-blue-200 mb-4">
                                    Usage Limits (Optional)
                                </h4>
                                
                                {/* Per User Limit */}
                                <div className="mb-4">
                                    <label htmlFor="per_user_limit" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Uses Per User
                                    </label>
                                    <input
                                        type="number"
                                        id="per_user_limit"
                                        value={data.per_user_limit}
                                        onChange={(e) => setData("per_user_limit", e.target.value === "" ? "" : parseInt(e.target.value))}
                                        min="0"
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                        placeholder="Leave empty for unlimited"
                                    />
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        Maximum number of times a single user can use this coupon. Leave empty or enter 0 for unlimited uses per user.
                                    </p>
                                    {errors.per_user_limit && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.per_user_limit}</p>
                                    )}
                                </div>

                                {/* Max Uses */}
                                <div>
                                    <label htmlFor="max_uses" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Total Maximum Uses
                                    </label>
                                    <input
                                        type="number"
                                        id="max_uses"
                                        value={data.max_uses}
                                        onChange={(e) => setData("max_uses", e.target.value === "" ? "" : parseInt(e.target.value))}
                                        min="0"
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                        placeholder="Leave empty for unlimited"
                                    />
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        Maximum number of times this coupon can be used in total (by all users). Leave empty or enter 0 for unlimited total uses.
                                    </p>
                                    {errors.max_uses && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.max_uses}</p>
                                    )}
                                </div>
                            </div>

                            {/* Info Box */}
                            <div className="bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
                                <h4 className="text-sm font-medium text-green-800 dark:text-green-200 mb-2">
                                    Default Coupon Features
                                </h4>
                                <ul className="text-sm text-green-700 dark:text-green-300 space-y-1">
                                    <li>• No expiration date - never expires</li>
                                    <li>• Available to all users as a fallback option</li>
                                    <li>• Will be shown when users don't qualify for special discounts</li>
                                    <li>• Recommended: Leave usage limits empty for unlimited uses</li>
                                </ul>
                            </div>

                            {/* Actions */}
                            <div className="flex items-center justify-end gap-4 pt-6 border-t border-gray-200 dark:border-gray-700">
                                <Link
                                    href={route("admin.default-coupons.index")}
                                    className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                >
                                    Cancel
                                </Link>
                                <Button
                                    type="submit"
                                    disabled={processing}
                                    variant="primary"
                                >
                                    {processing ? (
                                        <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                    ) : (
                                        <Save className="w-4 h-4 mr-2" />
                                    )}
                                    Create Default Coupon
                                </Button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
} 