import { Head, useForm, Link, usePage } from "@inertiajs/react";
import React, { useEffect } from "react";
import { ArrowLeft, Save, RefreshCw, CheckCircle, XCircle } from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";
import { Button } from "@/Components/ui/Button";
import ValidationErrors from "@/Components/OrderForm/ValidationErrors";
import { toast } from "react-hot-toast";

export default function Edit({ defaultCoupon }) {
    const { flash } = usePage().props;
    
    const { data, setData, put, processing, errors } = useForm({
        coupon_code: defaultCoupon.coupon_code,
        discount_type: defaultCoupon.discount_type,
        discount_amount: defaultCoupon.discount_amount,
        is_active: defaultCoupon.is_active,
        per_user_limit: defaultCoupon.per_user_limit === null ? "" : defaultCoupon.per_user_limit,
        max_uses: defaultCoupon.max_uses === null ? "" : defaultCoupon.max_uses,
    });

    // Show toast notifications for flash messages
    useEffect(() => {
        if (flash.success || flash.message) {
            const message = flash.message || flash.success;
            toast.success(message, {
                duration: 5000,
                position: 'top-right',
            });
        }
        if (flash.error) {
            toast.error(flash.error, {
                duration: 5000,
                position: 'top-right',
            });
        }
    }, [flash]);

    const handleSubmit = (e) => {
        e.preventDefault();
        
        // Prepare data with explicit null values for empty fields
        const submitData = {
            ...data,
            per_user_limit: data.per_user_limit === "" || data.per_user_limit === 0 ? null : data.per_user_limit,
            max_uses: data.max_uses === "" || data.max_uses === 0 ? null : data.max_uses,
        };
        
        put(route("admin.default-coupons.update", defaultCoupon.id), {
            data: submitData,
            preserveScroll: false,
        });
    };

    return (
        <AdminLayout>
            <Head title="Edit Default Coupon" />

            <div className="py-6">
                <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
                    {/* Flash Messages */}
                    {flash?.success && (
                        <div className="mb-4 rounded-lg bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 p-4">
                            <div className="flex items-center">
                                <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400 mr-3" />
                                <div>
                                    <p className="text-sm font-medium text-green-800 dark:text-green-200">
                                        {flash.success}
                                    </p>
                                    {flash.message && (
                                        <p className="text-sm text-green-700 dark:text-green-300 mt-1">
                                            {flash.message}
                                        </p>
                                    )}
                                </div>
                            </div>
                        </div>
                    )}

                    {flash?.error && (
                        <div className="mb-4 rounded-lg bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 p-4">
                            <div className="flex items-center">
                                <XCircle className="w-5 h-5 text-red-600 dark:text-red-400 mr-3" />
                                <div>
                                    <p className="text-sm font-medium text-red-800 dark:text-red-200">
                                        {flash.error}
                                    </p>
                                    {flash.message && (
                                        <p className="text-sm text-red-700 dark:text-red-300 mt-1">
                                            {flash.message}
                                        </p>
                                    )}
                                </div>
                            </div>
                        </div>
                    )}

                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div>
                                <div className="flex items-center">
                                    <Link
                                        href={route("admin.default-coupons.index")}
                                        className="mr-4 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
                                    >
                                        <ArrowLeft className="w-5 h-5" />
                                    </Link>
                                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                        Edit Default Coupon
                                    </h1>
                                </div>
                                <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                                    Update the default discount coupon settings
                                </p>
                            </div>
                        </div>
                    </div>

                    {/* Form */}
                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg">
                        <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                Coupon Details
                            </h3>
                        </div>

                        <form onSubmit={handleSubmit} className="p-6 space-y-6">
                            <ValidationErrors errors={errors} />

                            {/* Coupon Code */}
                            <div>
                                <label htmlFor="coupon_code" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Coupon Code
                                </label>
                                <input
                                    type="text"
                                    id="coupon_code"
                                    value={data.coupon_code}
                                    onChange={(e) => setData("coupon_code", e.target.value.toUpperCase())}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                    placeholder="Enter coupon code"
                                    required
                                />
                                {errors.coupon_code && (
                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.coupon_code}</p>
                                )}
                            </div>

                            {/* Discount Type */}
                            <div>
                                <label htmlFor="discount_type" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Discount Type
                                </label>
                                <select
                                    id="discount_type"
                                    value={data.discount_type}
                                    onChange={(e) => setData("discount_type", e.target.value)}
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                >
                                    <option value="percentage">Percentage</option>
                                    <option value="fixed">Fixed Amount</option>
                                </select>
                                {errors.discount_type && (
                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.discount_type}</p>
                                )}
                            </div>

                            {/* Discount Amount */}
                            <div>
                                <label htmlFor="discount_amount" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Discount Amount
                                </label>
                                <input
                                    type="number"
                                    id="discount_amount"
                                    value={data.discount_amount}
                                    onChange={(e) => setData("discount_amount", parseFloat(e.target.value))}
                                    step="0.01"
                                    min="0"
                                    className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                    placeholder={data.discount_type === "percentage" ? "10" : "50.00"}
                                />
                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                    {data.discount_type === "percentage" 
                                        ? "Enter the percentage discount (e.g., 10 for 10%)"
                                        : "Enter the fixed amount discount (e.g., 50.00 for $50)"
                                    }
                                </p>
                                {errors.discount_amount && (
                                    <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.discount_amount}</p>
                                )}
                            </div>

                            {/* Usage Limits Section */}
                            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                                <h4 className="text-sm font-medium text-blue-900 dark:text-blue-200 mb-4">
                                    Usage Limits
                                </h4>
                                
                                {/* Per User Limit */}
                                <div className="mb-4">
                                    <label htmlFor="per_user_limit" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Uses Per User
                                    </label>
                                    <input
                                        type="number"
                                        id="per_user_limit"
                                        value={data.per_user_limit}
                                        onChange={(e) => {
                                            const value = e.target.value;
                                            setData("per_user_limit", value === "" ? "" : parseInt(value, 10));
                                        }}
                                        min="0"
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                        placeholder="Leave empty for unlimited"
                                    />
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        Maximum number of times a single user can use this coupon. Leave empty or enter 0 for unlimited uses per user.
                                        {data.per_user_limit === "" && (
                                            <span className="font-semibold text-green-600 dark:text-green-400"> (Currently: Unlimited)</span>
                                        )}
                                        {data.per_user_limit === 0 && (
                                            <span className="font-semibold text-green-600 dark:text-green-400"> (Will be saved as: Unlimited)</span>
                                        )}
                                    </p>
                                    {errors.per_user_limit && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.per_user_limit}</p>
                                    )}
                                </div>

                                {/* Max Uses */}
                                <div>
                                    <label htmlFor="max_uses" className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        Total Maximum Uses
                                    </label>
                                    <input
                                        type="number"
                                        id="max_uses"
                                        value={data.max_uses}
                                        onChange={(e) => {
                                            const value = e.target.value;
                                            setData("max_uses", value === "" ? "" : parseInt(value, 10));
                                        }}
                                        min="0"
                                        className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 dark:focus:ring-blue-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                                        placeholder="Leave empty for unlimited"
                                    />
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        Maximum number of times this coupon can be used in total (by all users). Leave empty or enter 0 for unlimited total uses.
                                        {data.max_uses === "" && (
                                            <span className="font-semibold text-green-600 dark:text-green-400"> (Currently: Unlimited)</span>
                                        )}
                                        {data.max_uses === 0 && (
                                            <span className="font-semibold text-green-600 dark:text-green-400"> (Will be saved as: Unlimited)</span>
                                        )}
                                    </p>
                                    {errors.max_uses && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">{errors.max_uses}</p>
                                    )}
                                </div>
                            </div>

                            {/* Active Status */}
                            <div>
                                <label className="flex items-center">
                                    <input
                                        type="checkbox"
                                        checked={data.is_active}
                                        onChange={(e) => setData("is_active", e.target.checked)}
                                        className="rounded border-gray-300 text-blue-600 focus:ring-blue-500"
                                    />
                                    <span className="ml-2 text-sm text-gray-700 dark:text-gray-300">
                                        Active
                                    </span>
                                </label>
                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                    Enable or disable this default coupon
                                </p>
                            </div>

                            {/* Usage Info */}
                            <div className="bg-gray-50 dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg p-4">
                                <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-2">
                                    Current Statistics
                                </h4>
                                <div className="text-sm text-gray-600 dark:text-gray-400 space-y-1">
                                    <p>• Total uses: {defaultCoupon.uses_count}</p>
                                    <p>• Per-user limit: {defaultCoupon.per_user_limit === null ? "Unlimited" : defaultCoupon.per_user_limit}</p>
                                    <p>• Max total uses: {defaultCoupon.max_uses === null ? "Unlimited" : defaultCoupon.max_uses}</p>
                                    <p>• Created: {new Date(defaultCoupon.created_at).toLocaleDateString()}</p>
                                    <p>• Last updated: {new Date(defaultCoupon.updated_at).toLocaleDateString()}</p>
                                </div>
                            </div>

                            {/* Actions */}
                            <div className="flex items-center justify-end gap-4 pt-6 border-t border-gray-200 dark:border-gray-700">
                                <Link
                                    href={route("admin.default-coupons.index")}
                                    className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                >
                                    Cancel
                                </Link>
                                <Button
                                    type="submit"
                                    disabled={processing}
                                    variant="primary"
                                >
                                    {processing ? (
                                        <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                    ) : (
                                        <Save className="w-4 h-4 mr-2" />
                                    )}
                                    Update Default Coupon
                                </Button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
} 