import { Head, Link, router } from "@inertiajs/react";
import React from "react";
import { ArrowLeft, Edit, Trash2, Power, PowerOff, Calendar, Tag, Percent, DollarSign } from "lucide-react";
import AdminLayout from "@/Layouts/AdminLayout";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";

export default function Show({ defaultCoupon }) {
    const formatCurrency = (amount, type) => {
        if (type === "percentage") {
            return `${amount}%`;
        }
        return `$${parseFloat(amount).toFixed(2)}`;
    };

    const getStatusBadge = (coupon) => {
        if (coupon.is_active) {
            return (
                <Badge variant="success" className="flex items-center gap-1">
                    <Power size={12} />
                    Active
                </Badge>
            );
        }
        return (
            <Badge variant="secondary" className="flex items-center gap-1">
                <PowerOff size={12} />
                Inactive
            </Badge>
        );
    };

    return (
        <AdminLayout>
            <Head title="Default Coupon Details" />

            <div className="py-6">
                <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center justify-between">
                            <div>
                                <div className="flex items-center">
                                    <Link
                                        href={route("admin.default-coupons.index")}
                                        className="mr-4 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
                                    >
                                        <ArrowLeft className="w-5 h-5" />
                                    </Link>
                                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                        Default Coupon Details
                                    </h1>
                                </div>
                                <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                                    View details for default discount coupon
                                </p>
                            </div>
                            <div className="flex gap-2">
                                <Link
                                    href={route("admin.default-coupons.edit", defaultCoupon.id)}
                                    className="inline-flex items-center px-4 py-2 bg-blue-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-blue-700 active:bg-blue-900 focus:outline-none focus:border-blue-900 focus:ring ring-blue-300 disabled:opacity-25 transition ease-in-out duration-150"
                                >
                                    <Edit className="w-4 h-4 mr-2" />
                                    Edit
                                </Link>
                                <button
                                    onClick={() => {
                                        if (confirm("Are you sure you want to delete this coupon?")) {
                                            router.delete(route("admin.default-coupons.destroy", defaultCoupon.id));
                                        }
                                    }}
                                    className="inline-flex items-center px-4 py-2 bg-red-600 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-red-700 active:bg-red-900 focus:outline-none focus:border-red-900 focus:ring ring-red-300 disabled:opacity-25 transition ease-in-out duration-150"
                                >
                                    <Trash2 className="w-4 h-4 mr-2" />
                                    Delete
                                </button>
                            </div>
                        </div>
                    </div>

                    {/* Coupon Details */}
                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg">
                        <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white">
                                Coupon Information
                            </h3>
                        </div>

                        <div className="p-6">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                {/* Basic Information */}
                                <div className="space-y-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Coupon Code
                                        </label>
                                        <div className="flex items-center gap-2">
                                            <Tag className="w-5 h-5 text-gray-400" />
                                            <span className="text-lg font-mono text-gray-900 dark:text-white">
                                                {defaultCoupon.coupon_code}
                                            </span>
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Discount
                                        </label>
                                        <div className="flex items-center gap-2">
                                            {defaultCoupon.discount_type === "percentage" ? (
                                                <Percent className="w-5 h-5 text-blue-500" />
                                            ) : (
                                                <DollarSign className="w-5 h-5 text-green-500" />
                                            )}
                                            <Badge variant="primary" className="text-lg">
                                                {formatCurrency(defaultCoupon.discount_amount, defaultCoupon.discount_type)}
                                            </Badge>
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Status
                                        </label>
                                        {getStatusBadge(defaultCoupon)}
                                    </div>

                                    {defaultCoupon.description && (
                                        <div>
                                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                Description
                                            </label>
                                            <p className="text-gray-900 dark:text-white">
                                                {defaultCoupon.description}
                                            </p>
                                        </div>
                                    )}
                                </div>

                                {/* Usage Information */}
                                <div className="space-y-4">
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Usage Statistics
                                        </label>
                                        <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4 space-y-2">
                                            <div className="flex justify-between">
                                                <span className="text-gray-600 dark:text-gray-400">Total Uses:</span>
                                                <span className="font-medium text-gray-900 dark:text-white">
                                                    {defaultCoupon.uses_count}
                                                </span>
                                            </div>
                                            <div className="flex justify-between">
                                                <span className="text-gray-600 dark:text-gray-400">Max Uses:</span>
                                                <span className="font-medium text-gray-900 dark:text-white">
                                                    Unlimited
                                                </span>
                                            </div>
                                            <div className="flex justify-between">
                                                <span className="text-gray-600 dark:text-gray-400">Per User Limit:</span>
                                                <span className="font-medium text-gray-900 dark:text-white">
                                                    Unlimited
                                                </span>
                                            </div>
                                        </div>
                                    </div>

                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Date Information
                                        </label>
                                        <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4 space-y-2">
                                            <div className="flex justify-between">
                                                <span className="text-gray-600 dark:text-gray-400">Created:</span>
                                                <span className="font-medium text-gray-900 dark:text-white">
                                                    {new Date(defaultCoupon.created_at).toLocaleDateString()}
                                                </span>
                                            </div>
                                            <div className="flex justify-between">
                                                <span className="text-gray-600 dark:text-gray-400">Last Updated:</span>
                                                <span className="font-medium text-gray-900 dark:text-white">
                                                    {new Date(defaultCoupon.updated_at).toLocaleDateString()}
                                                </span>
                                            </div>
                                            <div className="flex justify-between">
                                                <span className="text-gray-600 dark:text-gray-400">Expires:</span>
                                                <span className="font-medium text-gray-900 dark:text-white">
                                                    Never
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {/* Default Coupon Features */}
                            <div className="mt-6 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                                <h4 className="text-sm font-medium text-blue-800 dark:text-blue-200 mb-2">
                                    Default Coupon Features
                                </h4>
                                <ul className="text-sm text-blue-700 dark:text-blue-300 space-y-1">
                                    <li>• No usage limits - can be used unlimited times</li>
                                    <li>• No expiration date - never expires</li>
                                    <li>• Available to all users as a fallback option</li>
                                    <li>• Will be shown when users don't qualify for special discounts</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
} 