import { useState } from "react";
import { Head, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import Pagination from "@/Components/Pagination";
import {
    FiClock,
    FiSearch,
    FiFilter,
    FiChevronUp,
    FiChevronDown,
} from "react-icons/fi";
import {
    FaFilePdf,
    FaFileWord,
    FaFileAlt,
    FaFile,
    FaSort,
    FaEye,
} from "react-icons/fa";
import { FiUser } from "react-icons/fi";
import { BsCalendar } from "react-icons/bs";

export default function EssayIndex({ essays, filters }) {
    const [searchTerm, setSearchTerm] = useState(filters.search || "");

    const handleSearch = (e) => {
        e.preventDefault();
        const url = new URL(window.location.href);
        url.searchParams.set("search", searchTerm);
        window.location.href = url.toString();
    };

    // Get appropriate file icon based on file path
    const getFileIcon = (filePath) => {
        if (!filePath) return <FaFile className="h-4 w-4 text-gray-400" />;

        if (filePath.endsWith(".pdf")) {
            return <FaFilePdf className="h-4 w-4 text-red-500" />;
        } else if (filePath.endsWith(".docx") || filePath.endsWith(".doc")) {
            return <FaFileWord className="h-4 w-4 text-blue-500" />;
        } else {
            return <FaFileAlt className="h-4 w-4 text-gray-500" />;
        }
    };

    // Format date for display
    const formatDate = (dateString) => {
        if (!dateString) return "Not available";
        const date = new Date(dateString);
        return date.toLocaleDateString();
    };

    // Format time for display
    const formatTime = (dateString) => {
        if (!dateString) return "";
        const date = new Date(dateString);
        return date.toLocaleTimeString([], {
            hour: "2-digit",
            minute: "2-digit",
        });
    };

    return (
        <AdminLayout>
            <Head title="Writer Essay Submissions" />

            <div className="py-6 sm:py-12">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6 gap-4">
                        <h1 className="text-xl sm:text-2xl font-bold text-gray-800 dark:text-gray-100">
                            Writer Essay Submissions
                        </h1>

                        <div className="flex flex-col sm:flex-row items-stretch sm:items-center gap-2 w-full sm:w-auto">
                            <form
                                onSubmit={handleSearch}
                                className="flex flex-1 sm:flex-auto"
                            >
                                <div className="relative flex-1">
                                    <div className="absolute inset-y-0 left-0 flex items-center pl-3 pointer-events-none">
                                        <FiSearch className="w-4 h-4 text-gray-500 dark:text-gray-400" />
                                    </div>
                                    <input
                                        type="text"
                                        className="py-2 pl-10 pr-4 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:text-gray-200 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm"
                                        placeholder="Search by writer name..."
                                        value={searchTerm}
                                        onChange={(e) =>
                                            setSearchTerm(e.target.value)
                                        }
                                    />
                                </div>
                                <button
                                    type="submit"
                                    className="ml-2 inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-800"
                                >
                                    Search
                                </button>
                            </form>

                            <div className="relative w-full sm:w-auto mt-2 sm:mt-0">
                                <select
                                    className="block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-700 dark:bg-gray-800 dark:text-gray-200 focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 sm:text-sm rounded-md"
                                    defaultValue={filters.status || "all"}
                                    onChange={(e) => {
                                        const url = new URL(
                                            window.location.href
                                        );
                                        url.searchParams.set(
                                            "status",
                                            e.target.value
                                        );
                                        window.location.href = url.toString();
                                    }}
                                >
                                    <option value="all">All Submissions</option>
                                    <option value="pending">
                                        Pending Review
                                    </option>
                                    <option value="approved">Approved</option>
                                    <option value="rejected">Rejected</option>
                                </select>
                                <div className="absolute inset-y-0 right-0 flex items-center pr-2 pointer-events-none">
                                    <FiFilter className="h-4 w-4 text-gray-400" />
                                </div>
                            </div>
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        {essays.data.length > 0 ? (
                            <>
                                {/* Desktop Table View */}
                                <div className="hidden md:block overflow-x-auto">
                                    <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                        <thead className="bg-gray-50 dark:bg-gray-700">
                                            <tr>
                                                <th
                                                    scope="col"
                                                    className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                >
                                                    <div className="flex items-center cursor-pointer">
                                                        Writer
                                                        <FaSort className="ml-1 h-4 w-4" />
                                                    </div>
                                                </th>
                                                <th
                                                    scope="col"
                                                    className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                >
                                                    <div className="flex items-center cursor-pointer">
                                                        Submitted
                                                        <FaSort className="ml-1 h-4 w-4" />
                                                    </div>
                                                </th>
                                                <th
                                                    scope="col"
                                                    className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                >
                                                    Status
                                                </th>
                                                <th
                                                    scope="col"
                                                    className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                >
                                                    Topic
                                                </th>
                                                <th
                                                    scope="col"
                                                    className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                                >
                                                    Actions
                                                </th>
                                            </tr>
                                        </thead>
                                        <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                            {essays.data.map((essay) => (
                                                <tr key={essay.id}>
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        <div className="flex items-center">
                                                            <div className="flex-shrink-0 h-10 w-10 flex items-center justify-center bg-gray-100 dark:bg-gray-700 rounded-full">
                                                                <FiUser className="h-5 w-5 text-gray-500 dark:text-gray-400" />
                                                            </div>
                                                            <div className="ml-4">
                                                                <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                    {
                                                                        essay
                                                                            .writer
                                                                            .name
                                                                    }
                                                                </div>
                                                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                                                    {
                                                                        essay
                                                                            .writer
                                                                            .email
                                                                    }
                                                                </div>
                                                            </div>
                                                        </div>
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        {essay.submitted_at ? (
                                                            <>
                                                                <div className="flex items-center text-sm text-gray-500 dark:text-gray-400">
                                                                    <BsCalendar className="h-4 w-4 mr-1" />
                                                                    {formatDate(
                                                                        essay.submitted_at
                                                                    )}
                                                                </div>
                                                                <div className="text-sm text-gray-500 dark:text-gray-400">
                                                                    {formatTime(
                                                                        essay.submitted_at
                                                                    )}
                                                                </div>
                                                            </>
                                                        ) : (
                                                            <span className="text-sm text-gray-500 dark:text-gray-400">
                                                                Not available
                                                            </span>
                                                        )}
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        <span
                                                            className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                                                essay.status ===
                                                                "pending"
                                                                    ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-200"
                                                                    : essay.status ===
                                                                      "approved"
                                                                    ? "bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-200"
                                                                    : "bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-200"
                                                            }`}
                                                        >
                                                            {essay.status
                                                                .charAt(0)
                                                                .toUpperCase() +
                                                                essay.status.slice(
                                                                    1
                                                                )}
                                                        </span>
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap">
                                                        <div className="flex items-center text-sm text-gray-500 dark:text-gray-400">
                                                            {getFileIcon(
                                                                essay.file_path
                                                            )}
                                                            <span className="ml-1">
                                                                {
                                                                    essay.topic
                                                                        .title
                                                                }
                                                            </span>
                                                        </div>
                                                    </td>
                                                    <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                        <Link
                                                            href={route(
                                                                "admin.essays.show",
                                                                essay.id
                                                            )}
                                                            className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300 mr-3"
                                                        >
                                                            Review
                                                        </Link>
                                                    </td>
                                                </tr>
                                            ))}
                                        </tbody>
                                    </table>
                                </div>

                                {/* Mobile Card View */}
                                <div className="md:hidden">
                                    <div className="divide-y divide-gray-200 dark:divide-gray-700">
                                        {essays.data.map((essay) => (
                                            <div key={essay.id} className="p-4">
                                                <div className="flex items-center justify-between mb-2">
                                                    <div className="flex items-center">
                                                        <div className="flex-shrink-0 h-10 w-10 flex items-center justify-center bg-gray-100 dark:bg-gray-700 rounded-full">
                                                            <FiUser className="h-5 w-5 text-gray-500 dark:text-gray-400" />
                                                        </div>
                                                        <div className="ml-3">
                                                            <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                {
                                                                    essay.writer
                                                                        .name
                                                                }
                                                            </div>
                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                {
                                                                    essay.writer
                                                                        .email
                                                                }
                                                            </div>
                                                        </div>
                                                    </div>
                                                    <span
                                                        className={`px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                                            essay.status ===
                                                            "pending"
                                                                ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-200"
                                                                : essay.status ===
                                                                  "approved"
                                                                ? "bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-200"
                                                                : "bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-200"
                                                        }`}
                                                    >
                                                        {essay.status
                                                            .charAt(0)
                                                            .toUpperCase() +
                                                            essay.status.slice(
                                                                1
                                                            )}
                                                    </span>
                                                </div>

                                                <div className="mt-2 grid grid-cols-2 gap-2 text-sm">
                                                    <div>
                                                        <span className="text-gray-500 dark:text-gray-400 text-xs">
                                                            Submitted:
                                                        </span>
                                                        <div className="flex items-center text-gray-700 dark:text-gray-300">
                                                            <BsCalendar className="h-3 w-3 mr-1" />
                                                            {formatDate(
                                                                essay.submitted_at
                                                            )}
                                                        </div>
                                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                                            {formatTime(
                                                                essay.submitted_at
                                                            )}
                                                        </div>
                                                    </div>

                                                    <div>
                                                        <span className="text-gray-500 dark:text-gray-400 text-xs">
                                                            Topic:
                                                        </span>
                                                        <div className="flex items-center text-gray-700 dark:text-gray-300">
                                                            {getFileIcon(
                                                                essay.file_path
                                                            )}
                                                            <span className="ml-1 truncate">
                                                                {
                                                                    essay.topic
                                                                        .title
                                                                }
                                                            </span>
                                                        </div>
                                                    </div>
                                                </div>

                                                <div className="mt-3 flex justify-end">
                                                    <Link
                                                        href={route(
                                                            "admin.essays.show",
                                                            essay.id
                                                        )}
                                                        className="inline-flex items-center px-3 py-1.5 border border-transparent text-xs font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-800"
                                                    >
                                                        <FaEye className="h-3.5 w-3.5 mr-1" />
                                                        Review
                                                    </Link>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </div>
                            </>
                        ) : (
                            <div className="p-6 text-center text-gray-500 dark:text-gray-400">
                                <FaFileAlt className="mx-auto h-12 w-12 text-gray-400 dark:text-gray-500 mb-2" />
                                <p>
                                    No essay submissions found matching your
                                    criteria.
                                </p>
                            </div>
                        )}
                    </div>

                    <div className="mt-6">
                        <Pagination links={essays.links} />
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
