import { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import Modal from "@/Components/Modal";
import InputLabel from "@/Components/InputLabel";
import { Textarea } from "@/Components/ui/Textarea";
import PrimaryButton from "@/Components/PrimaryButton";
import SecondaryButton from "@/Components/SecondaryButton";
import {
    FiUser,
    FiArrowLeft,
    FiMessageSquare,
    FiClock,
    FiAlertCircle,
} from "react-icons/fi";
import {
    FaCalendarAlt,
    FaFileAlt,
    FaCheckCircle,
    FaTimesCircle,
    FaFilePdf,
    FaFileWord,
    FaDownload,
    FaFile,
    FaGraduationCap,
    FaBriefcase,
    FaGlobe,
    FaPercentage,
} from "react-icons/fa";

export default function EssayShow({ essay, writer }) {
    const [showRejectModal, setShowRejectModal] = useState(false);
    const [showApproveModal, setShowApproveModal] = useState(false);

    const { data, setData, post, processing, errors, reset } = useForm({
        feedback: "",
        status: "",
    });

    const handleApprove = () => {
        setData("status", "approved");
        setShowApproveModal(true);
    };

    const handleReject = () => {
        setData("status", "rejected");
        setShowRejectModal(true);
    };

    const submitReview = () => {
        post(route("admin.essays.review", essay.id), {
            onSuccess: () => {
                reset();
                setShowApproveModal(false);
                setShowRejectModal(false);
            },
        });
    };

    // Get appropriate file icon based on file info
    const getFileIcon = () => {
        if (!essay.file_info || !essay.file_info.exists) {
            return <FaFile className="h-5 w-5 text-gray-500" />;
        }

        if (essay.file_info.is_pdf) {
            return <FaFilePdf className="h-5 w-5 text-red-500" />;
        } else if (essay.file_info.is_docx) {
            return <FaFileWord className="h-5 w-5 text-blue-500" />;
        } else {
            return <FaFileAlt className="h-5 w-5 text-gray-500" />;
        }
    };

    // Format file size
    const formatFileSize = (bytes) => {
        if (!bytes) return "Unknown size";

        const units = ["B", "KB", "MB", "GB"];
        let size = bytes;
        let unitIndex = 0;

        while (size >= 1024 && unitIndex < units.length - 1) {
            size /= 1024;
            unitIndex++;
        }

        return `${size.toFixed(1)} ${units[unitIndex]}`;
    };

    // Format date for display
    const formatDate = (dateString) => {
        if (!dateString) return "Not available";
        const date = new Date(dateString);
        return date.toLocaleDateString();
    };

    // Format time for display
    const formatTime = (dateString) => {
        if (!dateString) return "";
        const date = new Date(dateString);
        return date.toLocaleTimeString([], {
            hour: "2-digit",
            minute: "2-digit",
        });
    };

    // Format datetime for display
    const formatDateTime = (dateString) => {
        if (!dateString) return "Not available";
        return `${formatDate(dateString)} at ${formatTime(dateString)}`;
    };

    return (
        <AdminLayout>
            <Head title={`Review Essay - ${writer.name}`} />

            <div className="py-6 sm:py-12">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Header with back button and action buttons */}
                    <div className="mb-6 flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4">
                        <div className="flex items-center">
                            <Link
                                href={route("admin.essays.index")}
                                className="mr-4 text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300 transition-colors"
                            >
                                <FiArrowLeft className="h-5 w-5" />
                            </Link>
                            <h1 className="text-xl sm:text-2xl font-bold text-gray-800 dark:text-gray-100">
                                Essay Review
                            </h1>
                        </div>

                        <div className="flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-3 w-full sm:w-auto">
                            <SecondaryButton
                                onClick={handleReject}
                                className="bg-red-50 text-red-700 border-red-300 hover:bg-red-100 dark:bg-red-900/20 dark:text-red-300 dark:border-red-700 dark:hover:bg-red-900/40 w-full sm:w-auto"
                            >
                                <FaTimesCircle className="h-4 w-4 mr-1" />
                                Reject Essay
                            </SecondaryButton>
                            <PrimaryButton
                                onClick={handleApprove}
                                className="bg-green-600 hover:bg-green-700 focus:bg-green-700 active:bg-green-900 focus:ring-green-500 w-full sm:w-auto"
                            >
                                <FaCheckCircle className="h-4 w-4 mr-1" />
                                Approve Writer
                            </PrimaryButton>
                        </div>
                    </div>

                    {/* Status badge - visible on mobile */}
                    <div className="mb-4 sm:hidden">
                        <span
                            className={`inline-flex items-center px-2.5 py-1 rounded-full text-sm font-medium ${
                                essay.status === "pending"
                                    ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-200"
                                    : essay.status === "approved"
                                    ? "bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-200"
                                    : "bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-200"
                            }`}
                        >
                            {essay.status === "pending" && (
                                <FiClock className="mr-1 h-4 w-4" />
                            )}
                            {essay.status === "approved" && (
                                <FaCheckCircle className="mr-1 h-4 w-4" />
                            )}
                            {essay.status === "rejected" && (
                                <FaTimesCircle className="mr-1 h-4 w-4" />
                            )}
                            {essay.status.charAt(0).toUpperCase() +
                                essay.status.slice(1)}
                        </span>
                    </div>

                    {/* Writer Information Card */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-4 sm:p-6 border-b border-gray-200 dark:border-gray-700">
                            <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-4">
                                <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100">
                                    Writer Information
                                </h2>

                                {/* Status badge - visible on desktop */}
                                <span
                                    className={`hidden sm:inline-flex items-center px-2.5 py-1 rounded-full text-sm font-medium ${
                                        essay.status === "pending"
                                            ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-900/50 dark:text-yellow-200"
                                            : essay.status === "approved"
                                            ? "bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-200"
                                            : "bg-red-100 text-red-800 dark:bg-red-900/50 dark:text-red-200"
                                    }`}
                                >
                                    {essay.status === "pending" && (
                                        <FiClock className="mr-1 h-4 w-4" />
                                    )}
                                    {essay.status === "approved" && (
                                        <FaCheckCircle className="mr-1 h-4 w-4" />
                                    )}
                                    {essay.status === "rejected" && (
                                        <FaTimesCircle className="mr-1 h-4 w-4" />
                                    )}
                                    {essay.status.charAt(0).toUpperCase() +
                                        essay.status.slice(1)}
                                </span>
                            </div>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <div className="flex items-center mb-4">
                                        <div className="flex-shrink-0 h-12 w-12 flex items-center justify-center bg-gray-100 dark:bg-gray-700 rounded-full">
                                            <FiUser className="h-6 w-6 text-gray-500 dark:text-gray-400" />
                                        </div>
                                        <div className="ml-4">
                                            <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100">
                                                {writer.name}
                                            </h3>
                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                {writer.email}
                                            </p>
                                        </div>
                                    </div>

                                    <div className="space-y-2">
                                        <div className="flex items-start">
                                            <div className="flex items-center w-32">
                                                <FaGlobe className="h-4 w-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Country:
                                                </span>
                                            </div>
                                            <span className="text-sm text-gray-900 dark:text-gray-100">
                                                {writer.writerProfile.country}
                                            </span>
                                        </div>
                                        <div className="flex items-start">
                                            <div className="flex items-center w-32">
                                                <FaGraduationCap className="h-4 w-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Education:
                                                </span>
                                            </div>
                                            <span className="text-sm text-gray-900 dark:text-gray-100">
                                                {
                                                    writer.writerProfile
                                                        .education_level
                                                }
                                            </span>
                                        </div>
                                        <div className="flex items-start">
                                            <div className="flex items-center w-32">
                                                <FaBriefcase className="h-4 w-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Experience:
                                                </span>
                                            </div>
                                            <span className="text-sm text-gray-900 dark:text-gray-100">
                                                {
                                                    writer.writerProfile
                                                        .experience_years
                                                }{" "}
                                                years
                                            </span>
                                        </div>
                                    </div>
                                </div>

                                <div>
                                    <div className="space-y-2">
                                        <div className="flex items-start">
                                            <div className="flex items-center w-32">
                                                <FaPercentage className="h-4 w-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Test Score:
                                                </span>
                                            </div>
                                            <span className="text-sm text-gray-900 dark:text-gray-100">
                                                {
                                                    writer.writerProfile
                                                        .questions_score
                                                }
                                                %
                                            </span>
                                        </div>
                                        <div className="flex items-start">
                                            <div className="flex items-center w-32">
                                                <FaCalendarAlt className="h-4 w-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Test Date:
                                                </span>
                                            </div>
                                            <span className="text-sm text-gray-900 dark:text-gray-100">
                                                {formatDate(
                                                    writer.writerProfile
                                                        .questions_completed_at
                                                )}
                                            </span>
                                        </div>
                                        <div className="flex items-start">
                                            <div className="flex items-center w-32">
                                                <FaCalendarAlt className="h-4 w-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Submitted:
                                                </span>
                                            </div>
                                            <span className="text-sm text-gray-900 dark:text-gray-100">
                                                {formatDateTime(
                                                    essay.submitted_at
                                                )}
                                            </span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Essay Topic */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-4 sm:p-6">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                {" "}
                                <FaFileAlt className="h-5 w-5 mr-2 text-indigo-500 dark:text-indigo-400" />
                                Essay Topic
                            </h2>
                            <div className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                                <h3 className="font-medium text-gray-900 dark:text-gray-100 mb-2">
                                    {essay.topic.title}
                                </h3>
                                <p className="text-gray-700 dark:text-gray-300 whitespace-pre-line">
                                    {essay.topic.description}
                                </p>
                            </div>
                        </div>
                    </div>

                    {/* Essay Content/File */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-4 sm:p-6">
                            <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                <FaFileAlt className="h-5 w-5 mr-2 text-indigo-500 dark:text-indigo-400" />
                                Essay Submission
                            </h2>

                            {essay.file_info && essay.file_info.exists ? (
                                <div className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                                    <div className="flex flex-col sm:flex-row items-start sm:items-center justify-between mb-4">
                                        <div className="flex items-center mb-2 sm:mb-0">
                                            {getFileIcon()}
                                            <span className="ml-2 text-gray-900 dark:text-gray-100 font-medium">
                                                {essay.file_info.name ||
                                                    "Essay Document"}
                                            </span>
                                            <span className="ml-2 text-xs text-gray-500 dark:text-gray-400">
                                                (
                                                {formatFileSize(
                                                    essay.file_info.size
                                                )}
                                                )
                                            </span>
                                        </div>

                                        {essay.file_info.download_url && (
                                            <a
                                                href={
                                                    essay.file_info.download_url
                                                }
                                                className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-800 transition-colors"
                                                download
                                            >
                                                <FaDownload className="h-4 w-4 mr-1" />
                                                Download File
                                            </a>
                                        )}
                                    </div>

                                    <div className="prose dark:prose-invert max-w-none">
                                        {essay.file_info.is_pdf ||
                                        essay.file_info.is_docx ? (
                                            <div className="p-4 bg-gray-100 dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-700">
                                                <div className="flex items-center text-gray-700 dark:text-gray-300">
                                                    <FiAlertCircle className="h-5 w-5 mr-2 text-amber-500" />
                                                    <p>
                                                        This file type cannot be
                                                        displayed directly.
                                                        Please download the file
                                                        to view its contents.
                                                    </p>
                                                </div>
                                            </div>
                                        ) : (
                                            <div className="whitespace-pre-line text-gray-700 dark:text-gray-300">
                                                {essay.content ||
                                                    "No preview available. Please download the file to view its contents."}
                                            </div>
                                        )}
                                    </div>
                                </div>
                            ) : (
                                <div className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                                    <div className="flex items-center justify-center p-6 text-gray-500 dark:text-gray-400">
                                        <FiAlertCircle className="h-6 w-6 mr-2" />
                                        <p>
                                            No file was found for this
                                            submission.
                                        </p>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Previous Feedback (if any) */}
                    {essay.feedback && (
                        <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                            <div className="p-4 sm:p-6">
                                <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                    <FiMessageSquare className="h-5 w-5 mr-2 text-indigo-500 dark:text-indigo-400" />
                                    Previous Feedback
                                </h2>
                                <div className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                                    <p className="text-gray-700 dark:text-gray-300 whitespace-pre-line">
                                        {essay.feedback}
                                    </p>
                                    <p className="text-sm text-gray-500 dark:text-gray-400 mt-2">
                                        Provided on:{" "}
                                        {essay.reviewed_at
                                            ? formatDate(essay.reviewed_at)
                                            : "N/A"}
                                    </p>
                                </div>
                            </div>
                        </div>
                    )}
                </div>
            </div>

            {/* Reject Modal */}
            <Modal
                show={showRejectModal}
                onClose={() => setShowRejectModal(false)}
            >
                <div className="p-6">
                    <div className="flex items-center mb-4">
                        <FaTimesCircle className="h-6 w-6 text-red-500 mr-2" />
                        <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100">
                            Reject Essay Submission
                        </h2>
                    </div>

                    <p className="mb-4 text-gray-600 dark:text-gray-400">
                        Please provide detailed feedback to help the writer
                        understand why their essay was rejected and how they can
                        improve.
                    </p>

                    <div className="mb-4">
                        <InputLabel htmlFor="feedback" value="Feedback" />
                        <Textarea
                            id="feedback"
                            className="mt-1 block w-full"
                            value={data.feedback}
                            onChange={(e) =>
                                setData("feedback", e.target.value)
                            }
                            required
                            rows={6}
                        />
                        {errors.feedback && (
                            <p className="text-sm text-red-600 mt-1">
                                {errors.feedback}
                            </p>
                        )}
                    </div>

                    <div className="mt-6 flex flex-col sm:flex-row justify-end space-y-2 sm:space-y-0 sm:space-x-2">
                        <SecondaryButton
                            onClick={() => setShowRejectModal(false)}
                            className="w-full sm:w-auto"
                        >
                            Cancel
                        </SecondaryButton>
                        <PrimaryButton
                            onClick={submitReview}
                            disabled={processing || !data.feedback}
                            className="bg-red-600 hover:bg-red-700 focus:bg-red-700 active:bg-red-900 focus:ring-red-500 w-full sm:w-auto"
                        >
                            Confirm Rejection
                        </PrimaryButton>
                    </div>
                </div>
            </Modal>

            {/* Approve Modal */}
            <Modal
                show={showApproveModal}
                onClose={() => setShowApproveModal(false)}
            >
                <div className="p-6">
                    <div className="flex items-center mb-4">
                        <FaCheckCircle className="h-6 w-6 text-green-500 mr-2" />
                        <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100">
                            Approve Writer
                        </h2>
                    </div>

                    <p className="mb-4 text-gray-600 dark:text-gray-400">
                        You are about to approve this writer's essay submission.
                        This will complete their qualification process and allow
                        them to bid on orders.
                    </p>

                    <div className="mb-4">
                        <InputLabel
                            htmlFor="feedback"
                            value="Feedback (Optional)"
                        />
                        <Textarea
                            id="feedback"
                            className="mt-1 block w-full"
                            value={data.feedback}
                            onChange={(e) =>
                                setData("feedback", e.target.value)
                            }
                            rows={4}
                            placeholder="Provide any feedback or comments for the writer (optional)"
                        />
                    </div>

                    <div className="mt-6 flex flex-col sm:flex-row justify-end space-y-2 sm:space-y-0 sm:space-x-2">
                        <SecondaryButton
                            onClick={() => setShowApproveModal(false)}
                            className="w-full sm:w-auto"
                        >
                            Cancel
                        </SecondaryButton>
                        <PrimaryButton
                            onClick={submitReview}
                            disabled={processing}
                            className="bg-green-600 hover:bg-green-700 focus:bg-green-700 active:bg-green-900 focus:ring-green-500 w-full sm:w-auto"
                        >
                            Confirm Approval
                        </PrimaryButton>
                    </div>
                </div>
            </Modal>
        </AdminLayout>
    );
}
