import React, { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import { Award, ArrowLeft, Save, X, HelpCircle, Info } from "lucide-react";
import { Button } from "@/Components/ui/Button";
import { FormInput } from "@/Components/ui/FormInput";
import { Textarea } from "@/Components/ui/Textarea";
import Alert from "@/Components/ui/Alert";
import { Label } from "@/Components/ui/Label";
import { Switch } from "@/Components/ui/Switch";
import {
    Tooltip,
    TooltipProvider,
    TooltipTrigger,
    TooltipContent,
} from "@/Components/ui/Tooltip";
import { Card, CardContent, CardHeader, CardTitle } from "@/Components/ui/Card";

const Create = ({ maxOrder = 0 }) => {
    const { data, setData, post, processing, errors } = useForm({
        name: "",
        min_orders: maxOrder + 1,
        discount_percentage: 5,
        description: "",
        is_active: true,
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.loyalty-tiers.store"));
    };

    return (
        <AdminLayout>
            <Head title="Create Loyalty Tier" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Award className="h-6 w-6 mr-2 text-blue-500" />
                                Create Loyalty Tier
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Add a new loyalty tier to reward returning
                                customers
                            </p>
                        </div>
                        <Button
                            variant="outline"
                            className="inline-flex items-center"
                            asChild
                        >
                            <Link
                                href={route("admin.loyalty-tiers.index")}
                                className="flex items-center"
                            >
                                <ArrowLeft className="w-4 h-4 mr-2" />
                                Back to Tiers
                            </Link>
                        </Button>
                    </div>

                    {errors.general && (
                        <Alert
                            className="mt-6"
                            variant="destructive"
                            message={errors.general}
                        />
                    )}

                    <Card className="mt-6">
                        <CardHeader>
                            <CardTitle>Tier Details</CardTitle>
                        </CardHeader>
                        <CardContent>
                            <form onSubmit={handleSubmit} className="space-y-6">
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div className="space-y-4">
                                        <FormInput
                                            id="name"
                                            name="name"
                                            label="Tier Name"
                                            value={data.name}
                                            onChange={(e) =>
                                                setData("name", e.target.value)
                                            }
                                            error={errors.name}
                                            placeholder="e.g., Silver, Gold, Platinum"
                                            required
                                        />

                                        <div>
                                            <div className="flex items-center gap-2">
                                                <Label
                                                    htmlFor="min_orders"
                                                    className="text-sm font-medium"
                                                >
                                                    Minimum Orders
                                                </Label>
                                                <TooltipProvider>
                                                    <Tooltip>
                                                        <TooltipTrigger asChild>
                                                            <span>
                                                                <HelpCircle className="h-4 w-4 text-gray-400" />
                                                            </span>
                                                        </TooltipTrigger>
                                                        <TooltipContent>
                                                            <p className="w-[200px]">
                                                                The minimum
                                                                number of orders
                                                                a customer must
                                                                place to qualify
                                                                for this loyalty
                                                                tier.
                                                            </p>
                                                        </TooltipContent>
                                                    </Tooltip>
                                                </TooltipProvider>
                                            </div>
                                            <div className="mt-1">
                                                <input
                                                    type="number"
                                                    id="min_orders"
                                                    name="min_orders"
                                                    min="1"
                                                    step="1"
                                                    value={data.min_orders}
                                                    onChange={(e) =>
                                                        setData(
                                                            "min_orders",
                                                            parseInt(
                                                                e.target.value
                                                            ) || 1
                                                        )
                                                    }
                                                    className={`w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 ${
                                                        errors.min_orders
                                                            ? "border-red-500"
                                                            : ""
                                                    }`}
                                                    required
                                                />
                                            </div>
                                            {errors.min_orders && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.min_orders}
                                                </p>
                                            )}
                                        </div>

                                        <div>
                                            <div className="flex items-center gap-2">
                                                <Label
                                                    htmlFor="discount_percentage"
                                                    className="text-sm font-medium"
                                                >
                                                    Discount Percentage
                                                </Label>
                                                <TooltipProvider>
                                                    <Tooltip>
                                                        <TooltipTrigger asChild>
                                                            <span>
                                                                <HelpCircle className="h-4 w-4 text-gray-400" />
                                                            </span>
                                                        </TooltipTrigger>
                                                        <TooltipContent>
                                                            <p className="w-[200px]">
                                                                The percentage
                                                                discount users
                                                                in this tier
                                                                will receive.
                                                            </p>
                                                        </TooltipContent>
                                                    </Tooltip>
                                                </TooltipProvider>
                                            </div>
                                            <div className="mt-1 relative">
                                                <input
                                                    type="number"
                                                    id="discount_percentage"
                                                    name="discount_percentage"
                                                    min="0"
                                                    max="100"
                                                    step="0.1"
                                                    value={
                                                        data.discount_percentage
                                                    }
                                                    onChange={(e) =>
                                                        setData(
                                                            "discount_percentage",
                                                            parseFloat(
                                                                e.target.value
                                                            ) || 0
                                                        )
                                                    }
                                                    className={`w-full pr-8 rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 ${
                                                        errors.discount_percentage
                                                            ? "border-red-500"
                                                            : ""
                                                    }`}
                                                    required
                                                />
                                                <div className="absolute inset-y-0 right-0 flex items-center px-3 pointer-events-none">
                                                    <span className="text-gray-500 dark:text-gray-400">
                                                        %
                                                    </span>
                                                </div>
                                            </div>
                                            {errors.discount_percentage && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.discount_percentage}
                                                </p>
                                            )}
                                        </div>
                                    </div>

                                    <div className="space-y-4">
                                        <div>
                                            <Label
                                                htmlFor="description"
                                                className="text-sm font-medium"
                                            >
                                                Description
                                            </Label>
                                            <div className="mt-1">
                                                <Textarea
                                                    id="description"
                                                    name="description"
                                                    rows={5}
                                                    value={
                                                        data.description || ""
                                                    }
                                                    onChange={(e) =>
                                                        setData(
                                                            "description",
                                                            e.target.value
                                                        )
                                                    }
                                                    className={`w-full rounded-md border-gray-300 dark:border-gray-600 dark:bg-gray-800 dark:text-white shadow-sm focus:border-blue-500 focus:ring-blue-500 ${
                                                        errors.description
                                                            ? "border-red-500"
                                                            : ""
                                                    }`}
                                                    placeholder="Explain the benefits of this tier for customers..."
                                                />
                                            </div>
                                            {errors.description && (
                                                <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                                    {errors.description}
                                                </p>
                                            )}
                                        </div>

                                        <div className="flex items-center justify-between">
                                            <div className="flex items-center gap-2">
                                                <Label
                                                    htmlFor="is_active"
                                                    className="text-sm font-medium"
                                                >
                                                    Active Status
                                                </Label>
                                                <TooltipProvider>
                                                    <Tooltip>
                                                        <TooltipTrigger asChild>
                                                            <span>
                                                                <HelpCircle className="h-4 w-4 text-gray-400" />
                                                            </span>
                                                        </TooltipTrigger>
                                                        <TooltipContent>
                                                            <p className="w-[200px]">
                                                                When active,
                                                                this tier will
                                                                be used for
                                                                determining
                                                                customer loyalty
                                                                discounts.
                                                            </p>
                                                        </TooltipContent>
                                                    </Tooltip>
                                                </TooltipProvider>
                                            </div>
                                            <Switch
                                                id="is_active"
                                                checked={data.is_active}
                                                onCheckedChange={(value) =>
                                                    setData("is_active", value)
                                                }
                                            />
                                        </div>

                                        <div className="bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4 mt-4">
                                            <div className="flex">
                                                <Info className="h-5 w-5 text-blue-600 dark:text-blue-400 mr-2 flex-shrink-0" />
                                                <div className="text-sm text-blue-700 dark:text-blue-300">
                                                    <p className="font-medium">
                                                        Loyalty Tier Information
                                                    </p>
                                                    <p className="mt-1">
                                                        Higher tiers typically
                                                        offer better discounts.
                                                        Customers will
                                                        automatically qualify
                                                        for the highest tier
                                                        they're eligible for
                                                        based on their order
                                                        count.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <div className="flex justify-end space-x-3">
                                    <Button
                                        type="button"
                                        variant="outline"
                                        onClick={() => window.history.back()}
                                        disabled={processing}
                                        className="inline-flex items-center"
                                    >
                                        <X className="w-4 h-4 mr-2" />
                                        Cancel
                                    </Button>
                                    <Button
                                        type="submit"
                                        disabled={processing}
                                        className="inline-flex items-center"
                                    >
                                        <Save className="w-4 h-4 mr-2" />
                                        {processing
                                            ? "Creating..."
                                            : "Create Tier"}
                                    </Button>
                                </div>
                            </form>
                        </CardContent>
                    </Card>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Create;
