import React, { useState, useEffect } from "react";
import { Head, Link, router, usePage } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import { Tooltip } from "@/Components/ui/Tooltip";
import {
    Award,
    Plus,
    Search,
    Filter,
    ArrowUp,
    ArrowDown,
    MoreHorizontal,
    BarChart,
    Edit,
    Trash2,
    Eye,
    CheckCircle,
    XCircle,
    BarChart2,
    ToggleLeft,
    ToggleRight,
    RefreshCw,
} from "lucide-react";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import { Input } from "@/Components/ui/Input";
import {
    DropdownMenu,
    DropdownMenuTrigger,
    DropdownMenuContent,
    DropdownMenuItem,
    DropdownMenuSeparator,
} from "@/Components/DropdownMenu";

import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/Components/ui/Select";
import Pagination from "@/Components/Pagination";
import LoadingIndicator from "@/Components/ui/LoadingIndicator";
import { confirmDialog } from "@/lib/confirmDialog";
import debounce from "lodash.debounce";
import { Checkbox } from "@/Components/ui/Checkbox";
import EmptyState from "@/Components/ui/EmptyState";
import Alert from "@/Components/ui/Alert";

const Index = ({ tiers, counts, filters, auth, flash }) => {
    const [selectedTiers, setSelectedTiers] = useState([]);
    const [isLoading, setIsLoading] = useState(false);
    const [localFilters, setLocalFilters] = useState(filters);
    const { errors } = usePage().props;

    // Update search params debounced
    const debouncedSearchChange = debounce((value) => {
        setLocalFilters((prev) => ({ ...prev, search: value }));
        updateQueryParams({ ...localFilters, search: value });
    }, 300);

    // Handle search input change
    const handleSearchChange = (e) => {
        const value = e.target.value;
        debouncedSearchChange(value);
    };

    // Update filter status
    const handleStatusChange = (value) => {
        const newFilters = { ...localFilters, status: value };
        setLocalFilters(newFilters);
        updateQueryParams(newFilters);
    };

    // Update sorting
    const handleSortChange = (field) => {
        const direction =
            localFilters.sort_field === field &&
            localFilters.sort_direction === "asc"
                ? "desc"
                : "asc";
        const newFilters = {
            ...localFilters,
            sort_field: field,
            sort_direction: direction,
        };
        setLocalFilters(newFilters);
        updateQueryParams(newFilters);
    };

    // Update URL query params and reload data
    const updateQueryParams = (params) => {
        setIsLoading(true);
        router.get(route("admin.loyalty-tiers.index"), params, {
            preserveState: true,
            preserveScroll: true,
            only: ["tiers", "counts", "filters"],
            onFinish: () => setIsLoading(false),
        });
    };

    // Select all tiers
    const selectAllTiers = (checked) => {
        if (checked) {
            setSelectedTiers(tiers.data.map((tier) => tier.id));
        } else {
            setSelectedTiers([]);
        }
    };

    // Toggle tier selection
    const toggleTierSelection = (tierId) => {
        setSelectedTiers((prev) =>
            prev.includes(tierId)
                ? prev.filter((id) => id !== tierId)
                : [...prev, tierId]
        );
    };

    // Handle bulk activation/deactivation
    const handleBulkToggleStatus = (status) => {
        if (selectedTiers.length === 0) return;

        const statusText = status ? "activate" : "deactivate";
        const confirmText = `Are you sure you want to ${statusText} ${selectedTiers.length} loyalty tier(s)?`;

        confirmDialog({
            title: `${status ? "Activate" : "Deactivate"} Loyalty Tiers`,
            message: confirmText,
            onConfirm: () => {
                router.put(
                    route("admin.loyalty-tiers.bulk-toggle-status"),
                    {
                        ids: selectedTiers,
                        status: status,
                    },
                    {
                        preserveScroll: true,
                        onSuccess: () => {
                            setSelectedTiers([]);
                        },
                    }
                );
            },
        });
    };

    // Handle bulk deletion
    const handleBulkDelete = () => {
        if (selectedTiers.length === 0) return;

        confirmDialog({
            title: "Delete Loyalty Tiers",
            message: `Are you sure you want to delete ${selectedTiers.length} loyalty tier(s)? This action cannot be undone.`,
            onConfirm: () => {
                router.delete(route("admin.loyalty-tiers.bulk-delete"), {
                    data: {
                        ids: selectedTiers,
                    },
                    preserveScroll: true,
                    onSuccess: () => {
                        setSelectedTiers([]);
                    },
                });
            },
        });
    };

    // Toggle individual tier status
    const handleToggleStatus = (tier) => {
        router.put(
            route("admin.loyalty-tiers.toggle-status", tier.id),
            {},
            { preserveScroll: true }
        );
    };

    // Handle delete for a single tier
    const handleDelete = (tier) => {
        confirmDialog({
            title: "Delete Loyalty Tier",
            message: `Are you sure you want to delete '${tier.name}'? This action cannot be undone.`,
            onConfirm: () => {
                router.delete(route("admin.loyalty-tiers.destroy", tier.id), {
                    preserveScroll: true,
                });
            },
        });
    };

    // Format discount for display
    const formatDiscount = (percentage) => {
        if (percentage === null || percentage === undefined) return "0.0%";
        return `${parseFloat(percentage).toFixed(1)}%`;
    };

    // Reset all filters
    const resetFilters = () => {
        const defaultFilters = {
            search: "",
            status: "all",
            sort_field: "min_orders",
            sort_direction: "asc",
        };
        setLocalFilters(defaultFilters);
        updateQueryParams(defaultFilters);
    };

    // Get status badge for tier
    const getStatusBadge = (isActive) => {
        return isActive ? (
            <Badge
                variant="outline"
                className="bg-green-100 text-green-800 border-green-200 dark:bg-green-900/20 dark:text-green-400 dark:border-green-800"
            >
                <CheckCircle className="w-3.5 h-3.5 mr-1" />
                Active
            </Badge>
        ) : (
            <Badge
                variant="outline"
                className="bg-gray-100 text-gray-800 border-gray-200 dark:bg-gray-700/50 dark:text-gray-400 dark:border-gray-700"
            >
                <XCircle className="w-3.5 h-3.5 mr-1" />
                Inactive
            </Badge>
        );
    };

    return (
        <AdminLayout>
            <Head title="Loyalty Tiers" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Award className="h-6 w-6 mr-2 text-blue-500" />
                                Loyalty Tiers
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Manage loyalty tiers and their associated
                                discount rates
                            </p>
                        </div>
                        <div className="flex flex-wrap items-center gap-3">
                            <Button
                                className="inline-flex items-center"
                                asChild
                            >
                                <Link
                                    href={route(
                                        "admin.loyalty-tiers.statistics"
                                    )}
                                    className="flex items-center"
                                >
                                    <BarChart2 className="w-4 h-4 mr-2" />
                                    Statistics
                                </Link>
                            </Button>
                            <Button
                                className="inline-flex items-center"
                                asChild
                            >
                                <Link
                                    href={route("admin.loyalty-tiers.create")}
                                    className="flex items-center"
                                >
                                    <Plus className="w-4 h-4 mr-2" />
                                    Add Tier
                                </Link>
                            </Button>
                        </div>
                    </div>

                    {flash?.success && (
                        <Alert
                            className="mt-6"
                            variant="success"
                            message={flash.success}
                        />
                    )}

                    {errors?.general && (
                        <Alert
                            className="mt-6"
                            variant="destructive"
                            message={errors.general}
                        />
                    )}

                    {/* Filters Section */}
                    <div className="mt-6 bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600 shadow rounded-lg p-4 sm:p-6">
                        <div className="flex flex-col sm:flex-row justify-between gap-4">
                            <div className="relative w-full sm:max-w-xs">
                                <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <Search className="h-5 w-5 text-gray-400" />
                                </div>
                                <Input
                                    type="text"
                                    placeholder="Search tiers..."
                                    defaultValue={filters.search || ""}
                                    onChange={handleSearchChange}
                                    className="pl-10"
                                />
                            </div>

                            <div className="flex flex-col sm:flex-row gap-4">
                                <div className="flex items-center gap-2">
                                    <Filter className="h-5 w-5 text-gray-400" />
                                    <Select
                                        value={localFilters.status || "all"}
                                        onValueChange={handleStatusChange}
                                    >
                                        <SelectTrigger className="w-[150px]">
                                            <SelectValue placeholder="Status" />
                                        </SelectTrigger>
                                        <SelectContent>
                                            <SelectItem value="all">
                                                All ({counts?.all || 0})
                                            </SelectItem>
                                            <SelectItem value="active">
                                                Active ({counts?.active || 0})
                                            </SelectItem>
                                            <SelectItem value="inactive">
                                                Inactive (
                                                {counts?.inactive || 0})
                                            </SelectItem>
                                        </SelectContent>
                                    </Select>
                                </div>

                                <Button
                                    variant="outline"
                                    size="sm"
                                    onClick={resetFilters}
                                    className="min-w-[90px]"
                                >
                                    <RefreshCw className="h-4 w-4 mr-2" />
                                    Reset
                                </Button>
                            </div>
                        </div>

                        {/* Status filters badges */}
                        <div className="flex flex-wrap gap-2 mt-3">
                            <Badge
                                variant={
                                    localFilters.status === "all"
                                        ? "default"
                                        : "outline"
                                }
                                className="cursor-pointer"
                                onClick={() => handleStatusChange("all")}
                            >
                                All ({counts?.all || 0})
                            </Badge>
                            <Badge
                                variant={
                                    localFilters.status === "active"
                                        ? "default"
                                        : "outline"
                                }
                                className="cursor-pointer"
                                onClick={() => handleStatusChange("active")}
                            >
                                <CheckCircle className="w-3.5 h-3.5 mr-1" />
                                Active ({counts?.active || 0})
                            </Badge>
                            <Badge
                                variant={
                                    localFilters.status === "inactive"
                                        ? "default"
                                        : "outline"
                                }
                                className="cursor-pointer"
                                onClick={() => handleStatusChange("inactive")}
                            >
                                <XCircle className="w-3.5 h-3.5 mr-1" />
                                Inactive ({counts?.inactive || 0})
                            </Badge>
                        </div>
                    </div>

                    {/* Bulk Actions */}
                    {selectedTiers.length > 0 && (
                        <div className="mt-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg p-4 flex flex-wrap items-center justify-between gap-4">
                            <div className="text-sm text-blue-700 dark:text-blue-300 font-medium">
                                {selectedTiers.length} tier
                                {selectedTiers.length !== 1 && "s"} selected
                            </div>
                            <div className="flex flex-wrap gap-3">
                                <Button
                                    size="sm"
                                    variant="outline"
                                    className="border-blue-600 text-blue-600 hover:bg-blue-50 dark:border-blue-500 dark:text-blue-400 dark:hover:bg-blue-900/20"
                                    onClick={() => handleBulkToggleStatus(true)}
                                >
                                    <CheckCircle className="w-4 h-4 mr-2" />
                                    Activate
                                </Button>
                                <Button
                                    size="sm"
                                    variant="outline"
                                    className="border-gray-600 text-gray-600 hover:bg-gray-50 dark:border-gray-500 dark:text-gray-400 dark:hover:bg-gray-800/50"
                                    onClick={() =>
                                        handleBulkToggleStatus(false)
                                    }
                                >
                                    <XCircle className="w-4 h-4 mr-2" />
                                    Deactivate
                                </Button>
                                <Button
                                    size="sm"
                                    variant="outline"
                                    className="border-red-600 text-red-600 hover:bg-red-50 dark:border-red-500 dark:text-red-400 dark:hover:bg-red-900/20"
                                    onClick={handleBulkDelete}
                                >
                                    <Trash2 className="w-4 h-4 mr-2" />
                                    Delete
                                </Button>
                            </div>
                        </div>
                    )}

                    {/* Tiers Grid/List */}
                    <div className="mt-4 bg-white dark:bg-gray-800 shadow overflow-hidden rounded-lg relative">
                        {isLoading && (
                            <div className="absolute inset-0 bg-gray-900/10 dark:bg-gray-900/50 z-10 flex items-center justify-center">
                                <LoadingIndicator size="lg" />
                            </div>
                        )}

                        {tiers?.data?.length === 0 ? (
                            <EmptyState
                                icon={<Award className="h-12 w-12" />}
                                title="No loyalty tiers found"
                                description="Create your first loyalty tier to reward returning customers with discounts."
                                actionText="Add Tier"
                                actionUrl={route("admin.loyalty-tiers.create")}
                            />
                        ) : (
                            <>
                                {/* Desktop Table Headers */}
                                <div className="hidden md:grid grid-cols-12 gap-4 px-6 py-3 border-b border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-800/50 text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                    <div className="col-span-1 flex items-center">
                                        <Checkbox
                                            checked={
                                                selectedTiers.length ===
                                                    tiers?.data?.length &&
                                                tiers?.data?.length > 0
                                            }
                                            onChange={(e) =>
                                                selectAllTiers(e.target.checked)
                                            }
                                            aria-label="Select all tiers"
                                        />
                                    </div>
                                    <div
                                        className="col-span-2 flex items-center cursor-pointer"
                                        onClick={() => handleSortChange("name")}
                                    >
                                        <span>Name</span>
                                        {localFilters.sort_field === "name" &&
                                            (localFilters.sort_direction ===
                                            "asc" ? (
                                                <ArrowUp className="ml-1 h-4 w-4" />
                                            ) : (
                                                <ArrowDown className="ml-1 h-4 w-4" />
                                            ))}
                                    </div>
                                    <div
                                        className="col-span-2 flex items-center cursor-pointer"
                                        onClick={() =>
                                            handleSortChange("min_orders")
                                        }
                                    >
                                        <span>Min. Orders</span>
                                        {localFilters.sort_field ===
                                            "min_orders" &&
                                            (localFilters.sort_direction ===
                                            "asc" ? (
                                                <ArrowUp className="ml-1 h-4 w-4" />
                                            ) : (
                                                <ArrowDown className="ml-1 h-4 w-4" />
                                            ))}
                                    </div>
                                    <div
                                        className="col-span-2 flex items-center cursor-pointer"
                                        onClick={() =>
                                            handleSortChange(
                                                "discount_percentage"
                                            )
                                        }
                                    >
                                        <span>Discount</span>
                                        {localFilters.sort_field ===
                                            "discount_percentage" &&
                                            (localFilters.sort_direction ===
                                            "asc" ? (
                                                <ArrowUp className="ml-1 h-4 w-4" />
                                            ) : (
                                                <ArrowDown className="ml-1 h-4 w-4" />
                                            ))}
                                    </div>
                                    <div className="col-span-2">Status</div>
                                    <div className="col-span-2">Coupons</div>
                                    <div className="col-span-1 text-right">
                                        Actions
                                    </div>
                                </div>

                                {/* Desktop Table Body */}
                                <div className="hidden md:block divide-y divide-gray-200 dark:divide-gray-700">
                                    {tiers?.data?.map((tier) => (
                                        <div
                                            key={tier.id}
                                            className="grid grid-cols-12 gap-4 px-6 py-4 text-sm bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-750"
                                        >
                                            <div className="col-span-1 flex items-center">
                                                <Checkbox
                                                    checked={selectedTiers.includes(
                                                        tier.id
                                                    )}
                                                    onChange={() =>
                                                        toggleTierSelection(
                                                            tier.id
                                                        )
                                                    }
                                                    aria-label={`Select ${tier.name}`}
                                                />
                                            </div>
                                            <div className="col-span-2 font-medium text-gray-900 dark:text-white">
                                                <Link
                                                    href={route(
                                                        "admin.loyalty-tiers.show",
                                                        tier.id
                                                    )}
                                                    className="hover:text-blue-600 dark:hover:text-blue-400"
                                                >
                                                    {tier.name ||
                                                        "Unnamed Tier"}
                                                </Link>
                                            </div>
                                            <div className="col-span-2 text-gray-600 dark:text-gray-300">
                                                {tier.min_orders || 0}+ orders
                                            </div>
                                            <div className="col-span-2 text-green-600 dark:text-green-400 font-medium">
                                                {formatDiscount(
                                                    tier.discount_percentage
                                                )}
                                            </div>
                                            <div className="col-span-2">
                                                {getStatusBadge(tier.is_active)}
                                            </div>
                                            <div className="col-span-2 flex items-center">
                                                <Badge
                                                    variant="outline"
                                                    className="bg-blue-50 text-blue-700 border-blue-100 dark:bg-blue-900/20 dark:text-blue-400 dark:border-blue-800"
                                                >
                                                    {tier.coupons_count || 0}{" "}
                                                    coupons
                                                </Badge>
                                            </div>
                                            <div className="col-span-1 flex justify-end">
                                                <div className="flex space-x-2">
                                                    <Tooltip content="View Details">
                                                        <Link
                                                            href={route(
                                                                "admin.loyalty-tiers.show",
                                                                tier.id
                                                            )}
                                                            className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300"
                                                        >
                                                            <Eye className="h-5 w-5" />
                                                        </Link>
                                                    </Tooltip>

                                                    <Tooltip content="Edit Tier">
                                                        <Link
                                                            href={route(
                                                                "admin.loyalty-tiers.edit",
                                                                tier.id
                                                            )}
                                                            className="text-amber-600 hover:text-amber-800 dark:text-amber-400 dark:hover:text-amber-300"
                                                        >
                                                            <Edit className="h-5 w-5" />
                                                        </Link>
                                                    </Tooltip>

                                                    <Tooltip
                                                        content={
                                                            tier.is_active
                                                                ? "Deactivate Tier"
                                                                : "Activate Tier"
                                                        }
                                                    >
                                                        <button
                                                            onClick={() =>
                                                                handleToggleStatus(
                                                                    tier
                                                                )
                                                            }
                                                            className={
                                                                tier.is_active
                                                                    ? "text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-300"
                                                                    : "text-green-600 hover:text-green-800 dark:text-green-400 dark:hover:text-green-300"
                                                            }
                                                        >
                                                            {tier.is_active ? (
                                                                <ToggleLeft className="h-5 w-5" />
                                                            ) : (
                                                                <ToggleRight className="h-5 w-5" />
                                                            )}
                                                        </button>
                                                    </Tooltip>

                                                    <Tooltip content="Delete Tier">
                                                        <button
                                                            onClick={() =>
                                                                handleDelete(
                                                                    tier
                                                                )
                                                            }
                                                            className="text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300"
                                                        >
                                                            <Trash2 className="h-5 w-5" />
                                                        </button>
                                                    </Tooltip>
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>

                                {/* Mobile Card View */}
                                <div className="md:hidden divide-y divide-gray-200 dark:divide-gray-700">
                                    {tiers?.data?.map((tier) => (
                                        <div
                                            key={tier.id}
                                            className="p-4 bg-white dark:bg-gray-800"
                                        >
                                            <div className="flex justify-between items-start mb-2">
                                                <div className="flex items-center gap-2">
                                                    <Checkbox
                                                        checked={selectedTiers.includes(
                                                            tier.id
                                                        )}
                                                        onChange={() =>
                                                            toggleTierSelection(
                                                                tier.id
                                                            )
                                                        }
                                                        aria-label={`Select ${tier.name}`}
                                                    />
                                                    <div>
                                                        <Link
                                                            href={route(
                                                                "admin.loyalty-tiers.show",
                                                                tier.id
                                                            )}
                                                            className="text-lg font-medium text-gray-900 dark:text-white hover:text-blue-600 dark:hover:text-blue-400"
                                                        >
                                                            {tier.name ||
                                                                "Unnamed Tier"}
                                                        </Link>
                                                        <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                                            {tier.min_orders ||
                                                                0}
                                                            + orders ·{" "}
                                                            {formatDiscount(
                                                                tier.discount_percentage
                                                            )}
                                                        </p>
                                                    </div>
                                                </div>
                                                {getStatusBadge(tier.is_active)}
                                            </div>

                                            <div className="mt-2">
                                                <Badge
                                                    variant="outline"
                                                    className="bg-blue-50 text-blue-700 border-blue-100 dark:bg-blue-900/20 dark:text-blue-400 dark:border-blue-800"
                                                >
                                                    {tier.coupons_count || 0}{" "}
                                                    coupons
                                                </Badge>
                                            </div>

                                            <div className="flex justify-end items-center mt-3 space-x-3">
                                                <Tooltip content="View Details">
                                                    <Link
                                                        href={route(
                                                            "admin.loyalty-tiers.show",
                                                            tier.id
                                                        )}
                                                        className="text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300"
                                                    >
                                                        <Eye className="h-5 w-5" />
                                                    </Link>
                                                </Tooltip>

                                                <Tooltip content="Edit Tier">
                                                    <Link
                                                        href={route(
                                                            "admin.loyalty-tiers.edit",
                                                            tier.id
                                                        )}
                                                        className="text-amber-600 hover:text-amber-800 dark:text-amber-400 dark:hover:text-amber-300"
                                                    >
                                                        <Edit className="h-5 w-5" />
                                                    </Link>
                                                </Tooltip>

                                                <Tooltip
                                                    content={
                                                        tier.is_active
                                                            ? "Deactivate Tier"
                                                            : "Activate Tier"
                                                    }
                                                >
                                                    <button
                                                        onClick={() =>
                                                            handleToggleStatus(
                                                                tier
                                                            )
                                                        }
                                                        className={
                                                            tier.is_active
                                                                ? "text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-300"
                                                                : "text-green-600 hover:text-green-800 dark:text-green-400 dark:hover:text-green-300"
                                                        }
                                                    >
                                                        {tier.is_active ? (
                                                            <ToggleLeft className="h-5 w-5" />
                                                        ) : (
                                                            <ToggleRight className="h-5 w-5" />
                                                        )}
                                                    </button>
                                                </Tooltip>

                                                <Tooltip content="Delete Tier">
                                                    <button
                                                        onClick={() =>
                                                            handleDelete(tier)
                                                        }
                                                        className="text-red-600 hover:text-red-800 dark:text-red-400 dark:hover:text-red-300"
                                                    >
                                                        <Trash2 className="h-5 w-5" />
                                                    </button>
                                                </Tooltip>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            </>
                        )}

                        {/* Pagination */}
                        {tiers?.data?.length > 0 && tiers?.links && (
                            <div className="px-6 py-4 border-t border-gray-200 dark:border-gray-700">
                                <Pagination links={tiers.links} />
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Index;
