import React, { useState } from "react";
import { Head, Link, router } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Award,
    ArrowLeft,
    Edit,
    Trash2,
    Users,
    Tag,
    BarChart2,
    Clock,
    Calendar,
    Percent,
    ChevronRight,
    CheckCircle,
    XCircle,
    ToggleLeft,
    ToggleRight,
    Eye,
    ExternalLink,
    AlertTriangle,
    Copy,
} from "lucide-react";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import Alert from "@/Components/ui/Alert";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import {
    Tooltip,
    TooltipContent,
    TooltipProvider,
    TooltipTrigger,
} from "@/Components/ui/Tooltip";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/Components/ui/Tabs";
import Pagination from "@/Components/Pagination";
import { confirmDialog } from "@/lib/confirmDialog";
import { formatDate } from "@/lib/formatters";
import EmptyState from "@/Components/ui/EmptyState";

const Show = ({
    loyaltyTier,
    statistics,
    coupons,
    activeCoupons,
    flash,
    errors,
}) => {
    const [activeTab, setActiveTab] = useState("overview");

    // Handle tier deletion
    const handleDelete = () => {
        confirmDialog({
            title: "Delete Loyalty Tier",
            message: `Are you sure you want to delete the '${loyaltyTier.name}' tier? This action cannot be undone.`,
            onConfirm: () => {
                router.delete(
                    route("admin.loyalty-tiers.destroy", loyaltyTier.id)
                );
            },
        });
    };

    // Handle tier status toggle
    const handleToggleStatus = () => {
        router.put(
            route("admin.loyalty-tiers.toggle-status", loyaltyTier.id),
            {},
            {
                preserveScroll: true,
            }
        );
    };

    // Format a number with commas
    const formatNumber = (num) => {
        return new Intl.NumberFormat().format(num);
    };

    // Copy coupon code to clipboard
    const copyToClipboard = (text) => {
        navigator.clipboard.writeText(text);
        // You could add a toast notification here
    };

    // Get status badge
    const getStatusBadge = (isActive) => {
        return isActive ? (
            <Badge
                variant="outline"
                className="bg-green-100 text-green-800 border-green-200"
            >
                <CheckCircle className="w-3.5 h-3.5 mr-1" />
                Active
            </Badge>
        ) : (
            <Badge
                variant="outline"
                className="bg-gray-100 text-gray-800 border-gray-200"
            >
                <XCircle className="w-3.5 h-3.5 mr-1" />
                Inactive
            </Badge>
        );
    };

    return (
        <AdminLayout>
            <Head title={`${loyaltyTier.name} Tier Details`} />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <Award className="h-6 w-6 mr-2 text-blue-500" />
                                {loyaltyTier.name} Tier
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Loyalty tier for customers with{" "}
                                {loyaltyTier.min_orders}+ orders
                            </p>
                        </div>
                        <div className="flex flex-wrap gap-2">
                            <Button
                                variant="outline"
                                size="sm"
                                asChild
                                className="h-9"
                            >
                                <Link href={route("admin.loyalty-tiers.index")}>
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Tiers
                                </Link>
                            </Button>

                            <Button
                                variant="outline"
                                size="sm"
                                className="h-9"
                                onClick={handleToggleStatus}
                            >
                                {loyaltyTier.is_active ? (
                                    <>
                                        <ToggleLeft className="w-4 h-4 mr-2" />
                                        Deactivate
                                    </>
                                ) : (
                                    <>
                                        <ToggleRight className="w-4 h-4 mr-2" />
                                        Activate
                                    </>
                                )}
                            </Button>

                            <Button
                                variant="outline"
                                size="sm"
                                asChild
                                className="h-9"
                            >
                                <Link
                                    href={route(
                                        "admin.loyalty-tiers.edit",
                                        loyaltyTier.id
                                    )}
                                >
                                    <Edit className="w-4 h-4 mr-2" />
                                    Edit
                                </Link>
                            </Button>

                            {statistics.total_coupons === 0 && (
                                <Button
                                    variant="destructive"
                                    size="sm"
                                    className="h-9"
                                    onClick={handleDelete}
                                >
                                    <Trash2 className="w-4 h-4 mr-2" />
                                    Delete
                                </Button>
                            )}
                        </div>
                    </div>

                    {flash.success && (
                        <Alert
                            className="mt-6"
                            variant="success"
                            message={flash.success}
                        />
                    )}

                    {errors.general && (
                        <Alert
                            className="mt-6"
                            variant="destructive"
                            message={errors.general}
                        />
                    )}

                    {/* Status Badge */}
                    <div className="mt-4">
                        {getStatusBadge(loyaltyTier.is_active)}

                        {statistics.total_coupons > 0 && (
                            <Badge
                                variant="outline"
                                className="ml-2 bg-blue-50 text-blue-700 border-blue-100"
                            >
                                <Tag className="w-3.5 h-3.5 mr-1" />
                                {statistics.total_coupons} coupon
                                {statistics.total_coupons !== 1 && "s"}
                            </Badge>
                        )}

                        <Badge
                            variant="outline"
                            className="ml-2 bg-purple-50 text-purple-700 border-purple-100"
                        >
                            <Users className="w-3.5 h-3.5 mr-1" />
                            {formatNumber(statistics.eligible_users)} eligible
                            user{statistics.eligible_users !== 1 && "s"}
                        </Badge>
                    </div>

                    {/* Main Content Tabs */}
                    <div className="mt-6">
                        <Tabs
                            defaultValue="overview"
                            onValueChange={setActiveTab}
                            className="w-full"
                        >
                            <TabsList className="grid w-full grid-cols-2 lg:w-auto lg:inline-flex">
                                <TabsTrigger value="overview">
                                    Overview
                                </TabsTrigger>
                                <TabsTrigger value="coupons">
                                    Coupons ({statistics.total_coupons})
                                </TabsTrigger>
                            </TabsList>

                            {/* Overview Tab */}
                            <TabsContent
                                value="overview"
                                className="mt-6 space-y-6"
                            >
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                                    {/* Tier Details Card */}
                                    <Card className="col-span-1 md:col-span-2">
                                        <CardHeader>
                                            <CardTitle>Tier Details</CardTitle>
                                        </CardHeader>
                                        <CardContent>
                                            <dl className="grid grid-cols-1 sm:grid-cols-2 gap-x-4 gap-y-6">
                                                <div>
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Name
                                                    </dt>
                                                    <dd className="mt-1 text-lg font-semibold text-gray-900 dark:text-white">
                                                        {loyaltyTier.name}
                                                    </dd>
                                                </div>

                                                <div>
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Minimum Orders
                                                    </dt>
                                                    <dd className="mt-1 text-lg font-semibold text-gray-900 dark:text-white flex items-center">
                                                        {loyaltyTier.min_orders}
                                                        <span className="ml-1 text-sm font-normal text-gray-500 dark:text-gray-400">
                                                            orders
                                                        </span>
                                                    </dd>
                                                </div>

                                                <div>
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Discount
                                                    </dt>
                                                    <dd className="mt-1 text-lg font-semibold text-green-600 dark:text-green-400 flex items-center">
                                                        {parseFloat(
                                                            loyaltyTier.discount_percentage
                                                        ).toFixed(1)}
                                                        %
                                                        <Percent className="h-4 w-4 ml-1" />
                                                    </dd>
                                                </div>

                                                <div>
                                                    <dt className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Status
                                                    </dt>
                                                    <dd className="mt-1">
                                                        {getStatusBadge(
                                                            loyaltyTier.is_active
                                                        )}
                                                    </dd>
                                                </div>
                                            </dl>
                                        </CardContent>
                                    </Card>

                                    {/* Statistics Cards */}
                                    <Card>
                                        <CardHeader className="pb-2">
                                            <CardTitle className="text-base">
                                                Active Coupons
                                            </CardTitle>
                                        </CardHeader>
                                        <CardContent>
                                            <div className="text-3xl font-bold text-blue-600 dark:text-blue-400">
                                                {statistics.active_coupons}
                                            </div>
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                                {statistics.active_coupons === 0
                                                    ? "No active coupons"
                                                    : statistics.active_coupons ===
                                                      1
                                                    ? "1 active coupon"
                                                    : `${statistics.active_coupons} active coupons`}
                                            </p>
                                        </CardContent>
                                    </Card>

                                    <Card>
                                        <CardHeader className="pb-2">
                                            <CardTitle className="text-base">
                                                Coupon Uses
                                            </CardTitle>
                                        </CardHeader>
                                        <CardContent>
                                            <div className="text-3xl font-bold text-purple-600 dark:text-purple-400">
                                                {formatNumber(
                                                    statistics.total_uses
                                                )}
                                            </div>
                                            <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                                Times coupons were used
                                            </p>
                                        </CardContent>
                                    </Card>

                                    {/* Description Card */}
                                    <Card className="col-span-1 md:col-span-2">
                                        <CardHeader>
                                            <CardTitle>Description</CardTitle>
                                        </CardHeader>
                                        <CardContent>
                                            {loyaltyTier.description ? (
                                                <div className="prose dark:prose-invert max-w-none">
                                                    <p>
                                                        {
                                                            loyaltyTier.description
                                                        }
                                                    </p>
                                                </div>
                                            ) : (
                                                <p className="text-gray-500 dark:text-gray-400 italic">
                                                    No description provided.
                                                </p>
                                            )}
                                        </CardContent>
                                    </Card>

                                    {/* Active Coupons Card */}
                                    <Card className="col-span-1 md:col-span-2 lg:col-span-4">
                                        <CardHeader className="flex flex-row items-center justify-between">
                                            <div>
                                                <CardTitle>
                                                    Active Coupons
                                                </CardTitle>
                                                <CardDescription>
                                                    Currently active coupons for
                                                    this loyalty tier
                                                </CardDescription>
                                            </div>
                                            {activeCoupons.length > 0 && (
                                                <Button
                                                    variant="outline"
                                                    size="sm"
                                                    asChild
                                                    className="ml-2"
                                                >
                                                    <Link
                                                        href="#"
                                                        onClick={() =>
                                                            setActiveTab(
                                                                "coupons"
                                                            )
                                                        }
                                                    >
                                                        View All
                                                        <ChevronRight className="ml-1 h-4 w-4" />
                                                    </Link>
                                                </Button>
                                            )}
                                        </CardHeader>
                                        <CardContent>
                                            {activeCoupons.length === 0 ? (
                                                <div className="text-center py-6">
                                                    <div className="mx-auto h-12 w-12 text-gray-400 dark:text-gray-600">
                                                        <Tag className="h-12 w-12" />
                                                    </div>
                                                    <h3 className="mt-2 text-sm font-semibold text-gray-900 dark:text-white">
                                                        No active coupons
                                                    </h3>
                                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                                        There are no active
                                                        coupons for this loyalty
                                                        tier.
                                                    </p>
                                                    <div className="mt-4">
                                                        <Button asChild>
                                                            <Link
                                                                href={route(
                                                                    "admin.coupons.create"
                                                                )}
                                                            >
                                                                Create Coupon
                                                            </Link>
                                                        </Button>
                                                    </div>
                                                </div>
                                            ) : (
                                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                                                    {activeCoupons
                                                        .slice(0, 6)
                                                        .map((coupon) => (
                                                            <div
                                                                key={coupon.id}
                                                                className="bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-4 hover:shadow-md transition-shadow"
                                                            >
                                                                <div className="flex justify-between">
                                                                    <div className="font-medium text-gray-900 dark:text-white flex items-center gap-2">
                                                                        <Tag className="h-4 w-4 text-blue-500" />
                                                                        <span className="font-mono">
                                                                            {
                                                                                coupon.coupon_code
                                                                            }
                                                                        </span>
                                                                        <TooltipProvider>
                                                                            <Tooltip>
                                                                                <TooltipTrigger
                                                                                    asChild
                                                                                >
                                                                                    <Button
                                                                                        variant="ghost"
                                                                                        size="xs"
                                                                                        className="h-6 w-6 p-0"
                                                                                        onClick={() =>
                                                                                            copyToClipboard(
                                                                                                coupon.coupon_code
                                                                                            )
                                                                                        }
                                                                                    >
                                                                                        <Copy className="h-3.5 w-3.5 text-gray-500" />
                                                                                        <span className="sr-only">
                                                                                            Copy
                                                                                            code
                                                                                        </span>
                                                                                    </Button>
                                                                                </TooltipTrigger>
                                                                                <TooltipContent>
                                                                                    <p>
                                                                                        Copy
                                                                                        to
                                                                                        clipboard
                                                                                    </p>
                                                                                </TooltipContent>
                                                                            </Tooltip>
                                                                        </TooltipProvider>
                                                                    </div>
                                                                    <Badge
                                                                        variant={
                                                                            coupon.is_active
                                                                                ? "success"
                                                                                : "secondary"
                                                                        }
                                                                    >
                                                                        {coupon.is_active
                                                                            ? "Active"
                                                                            : "Inactive"}
                                                                    </Badge>
                                                                </div>
                                                                <div className="mt-2 text-sm text-gray-600 dark:text-gray-300">
                                                                    {coupon.discount_type ===
                                                                    "percentage" ? (
                                                                        <span className="text-green-600 dark:text-green-400 font-medium">
                                                                            {parseFloat(
                                                                                coupon.discount_amount
                                                                            ).toFixed(
                                                                                1
                                                                            )}
                                                                            %
                                                                            off
                                                                        </span>
                                                                    ) : (
                                                                        <span className="text-green-600 dark:text-green-400 font-medium">
                                                                            $
                                                                            {parseFloat(
                                                                                coupon.discount_amount
                                                                            ).toFixed(
                                                                                2
                                                                            )}{" "}
                                                                            off
                                                                        </span>
                                                                    )}
                                                                </div>
                                                                <div className="mt-1 flex justify-between">
                                                                    <span className="text-xs text-gray-500 dark:text-gray-400 flex items-center">
                                                                        <Clock className="h-3 w-3 mr-1" />
                                                                        {
                                                                            coupon.uses_count
                                                                        }{" "}
                                                                        /{" "}
                                                                        {coupon.max_uses ||
                                                                            "∞"}{" "}
                                                                        uses
                                                                    </span>
                                                                    <Button
                                                                        variant="ghost"
                                                                        size="xs"
                                                                        asChild
                                                                        className="text-xs text-blue-600 dark:text-blue-400 h-5 px-1"
                                                                    >
                                                                        <Link
                                                                            href={route(
                                                                                "admin.coupons.show",
                                                                                coupon.id
                                                                            )}
                                                                        >
                                                                            <Eye className="h-3 w-3 mr-1" />
                                                                            Details
                                                                        </Link>
                                                                    </Button>
                                                                </div>
                                                            </div>
                                                        ))}
                                                </div>
                                            )}
                                        </CardContent>
                                        {activeCoupons.length > 6 && (
                                            <CardFooter className="flex justify-center">
                                                <Button
                                                    variant="outline"
                                                    onClick={() =>
                                                        setActiveTab("coupons")
                                                    }
                                                >
                                                    View All{" "}
                                                    {activeCoupons.length}{" "}
                                                    Coupons
                                                </Button>
                                            </CardFooter>
                                        )}
                                    </Card>

                                    {/* Additional Metadata Card */}
                                    <Card className="col-span-1 md:col-span-2 lg:col-span-4">
                                        <CardHeader>
                                            <CardTitle>
                                                Tier Information
                                            </CardTitle>
                                        </CardHeader>
                                        <CardContent>
                                            <dl className="grid grid-cols-1 sm:grid-cols-2 md:grid-cols-4 gap-4 text-sm">
                                                <div className="bg-gray-50 dark:bg-gray-800/50 p-3 rounded-lg">
                                                    <dt className="text-gray-500 dark:text-gray-400">
                                                        Created
                                                    </dt>
                                                    <dd className="mt-1 font-medium text-gray-900 dark:text-white flex items-center">
                                                        <Calendar className="h-4 w-4 mr-1 text-blue-500" />
                                                        {formatDate(
                                                            loyaltyTier.created_at
                                                        )}
                                                    </dd>
                                                </div>

                                                <div className="bg-gray-50 dark:bg-gray-800/50 p-3 rounded-lg">
                                                    <dt className="text-gray-500 dark:text-gray-400">
                                                        Last Updated
                                                    </dt>
                                                    <dd className="mt-1 font-medium text-gray-900 dark:text-white flex items-center">
                                                        <Calendar className="h-4 w-4 mr-1 text-blue-500" />
                                                        {formatDate(
                                                            loyaltyTier.updated_at
                                                        )}
                                                    </dd>
                                                </div>

                                                <div className="bg-gray-50 dark:bg-gray-800/50 p-3 rounded-lg">
                                                    <dt className="text-gray-500 dark:text-gray-400">
                                                        Eligible Users
                                                    </dt>
                                                    <dd className="mt-1 font-medium text-gray-900 dark:text-white flex items-center">
                                                        <Users className="h-4 w-4 mr-1 text-purple-500" />
                                                        {formatNumber(
                                                            statistics.eligible_users
                                                        )}{" "}
                                                        users
                                                    </dd>
                                                </div>

                                                <div className="bg-gray-50 dark:bg-gray-800/50 p-3 rounded-lg">
                                                    <dt className="text-gray-500 dark:text-gray-400">
                                                        Total Redemptions
                                                    </dt>
                                                    <dd className="mt-1 font-medium text-gray-900 dark:text-white flex items-center">
                                                        <Tag className="h-4 w-4 mr-1 text-green-500" />
                                                        {formatNumber(
                                                            statistics.total_uses
                                                        )}{" "}
                                                        uses
                                                    </dd>
                                                </div>
                                            </dl>
                                        </CardContent>
                                    </Card>
                                </div>
                            </TabsContent>

                            {/* Coupons Tab */}
                            <TabsContent value="coupons" className="mt-6">
                                <Card>
                                    <CardHeader>
                                        <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                                            <div>
                                                <CardTitle>
                                                    Associated Coupons
                                                </CardTitle>
                                                <CardDescription>
                                                    Coupons linked to the{" "}
                                                    {loyaltyTier.name} loyalty
                                                    tier
                                                </CardDescription>
                                            </div>
                                            <Button asChild>
                                                <Link
                                                    href={route(
                                                        "admin.coupons.create"
                                                    )}
                                                >
                                                    <Tag className="w-4 h-4 mr-2" />
                                                    Create Coupon
                                                </Link>
                                            </Button>
                                        </div>
                                    </CardHeader>
                                    <CardContent>
                                        {coupons.data.length === 0 ? (
                                            <EmptyState
                                                icon={
                                                    <Tag className="h-12 w-12" />
                                                }
                                                title="No coupons found"
                                                description="This loyalty tier doesn't have any coupons yet."
                                                actionText="Create Coupon"
                                                actionUrl={route(
                                                    "admin.coupons.create"
                                                )}
                                            />
                                        ) : (
                                            <div className="overflow-x-auto">
                                                <div className="inline-block min-w-full align-middle">
                                                    <div className="overflow-hidden ring-1 ring-black ring-opacity-5 sm:rounded-lg">
                                                        <table className="min-w-full divide-y divide-gray-300 dark:divide-gray-700">
                                                            <thead className="bg-gray-50 dark:bg-gray-800/50">
                                                                <tr>
                                                                    <th
                                                                        scope="col"
                                                                        className="py-3.5 pl-4 pr-3 text-left text-sm font-semibold text-gray-900 dark:text-white sm:pl-6"
                                                                    >
                                                                        Code
                                                                    </th>
                                                                    <th
                                                                        scope="col"
                                                                        className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                                    >
                                                                        Discount
                                                                    </th>
                                                                    <th
                                                                        scope="col"
                                                                        className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                                    >
                                                                        Usage
                                                                    </th>
                                                                    <th
                                                                        scope="col"
                                                                        className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                                    >
                                                                        Expiry
                                                                    </th>
                                                                    <th
                                                                        scope="col"
                                                                        className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                                    >
                                                                        Status
                                                                    </th>
                                                                    <th
                                                                        scope="col"
                                                                        className="relative py-3.5 pl-3 pr-4 sm:pr-6"
                                                                    >
                                                                        <span className="sr-only">
                                                                            Actions
                                                                        </span>
                                                                    </th>
                                                                </tr>
                                                            </thead>
                                                            <tbody className="divide-y divide-gray-200 dark:divide-gray-700 bg-white dark:bg-gray-800">
                                                                {coupons.data.map(
                                                                    (
                                                                        coupon
                                                                    ) => (
                                                                        <tr
                                                                            key={
                                                                                coupon.id
                                                                            }
                                                                            className="hover:bg-gray-50 dark:hover:bg-gray-750"
                                                                        >
                                                                            <td className="whitespace-nowrap py-4 pl-4 pr-3 text-sm font-medium text-gray-900 dark:text-white sm:pl-6">
                                                                                <div className="flex items-center gap-2">
                                                                                    <span className="font-mono">
                                                                                        {
                                                                                            coupon.coupon_code
                                                                                        }
                                                                                    </span>
                                                                                    <TooltipProvider>
                                                                                        <Tooltip>
                                                                                            <TooltipTrigger
                                                                                                asChild
                                                                                            >
                                                                                                <Button
                                                                                                    variant="ghost"
                                                                                                    size="xs"
                                                                                                    className="h-6 w-6 p-0"
                                                                                                    onClick={() =>
                                                                                                        copyToClipboard(
                                                                                                            coupon.coupon_code
                                                                                                        )
                                                                                                    }
                                                                                                >
                                                                                                    <Copy className="h-3.5 w-3.5 text-gray-500" />
                                                                                                    <span className="sr-only">
                                                                                                        Copy
                                                                                                        code
                                                                                                    </span>
                                                                                                </Button>
                                                                                            </TooltipTrigger>
                                                                                            <TooltipContent>
                                                                                                <p>
                                                                                                    Copy
                                                                                                    to
                                                                                                    clipboard
                                                                                                </p>
                                                                                            </TooltipContent>
                                                                                        </Tooltip>
                                                                                    </TooltipProvider>
                                                                                </div>
                                                                            </td>
                                                                            <td className="whitespace-nowrap px-3 py-4 text-sm">
                                                                                {coupon.discount_type ===
                                                                                "percentage" ? (
                                                                                    <span className="text-green-600 dark:text-green-400 font-medium">
                                                                                        {parseFloat(
                                                                                            coupon.discount_amount
                                                                                        ).toFixed(
                                                                                            1
                                                                                        )}

                                                                                        %
                                                                                    </span>
                                                                                ) : (
                                                                                    <span className="text-green-600 dark:text-green-400 font-medium">
                                                                                        $
                                                                                        {parseFloat(
                                                                                            coupon.discount_amount
                                                                                        ).toFixed(
                                                                                            2
                                                                                        )}
                                                                                    </span>
                                                                                )}
                                                                            </td>
                                                                            <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-600 dark:text-gray-300">
                                                                                {
                                                                                    coupon.uses_count
                                                                                }{" "}
                                                                                /{" "}
                                                                                {coupon.max_uses ||
                                                                                    "∞"}
                                                                            </td>
                                                                            <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-600 dark:text-gray-300">
                                                                                {coupon.expiry_date
                                                                                    ? formatDate(
                                                                                          coupon.expiry_date
                                                                                      )
                                                                                    : "Never"}
                                                                            </td>
                                                                            <td className="whitespace-nowrap px-3 py-4 text-sm">
                                                                                {getStatusBadge(
                                                                                    coupon.is_active
                                                                                )}
                                                                            </td>
                                                                            <td className="relative whitespace-nowrap py-4 pl-3 pr-4 text-right text-sm font-medium sm:pr-6">
                                                                                <Button
                                                                                    variant="ghost"
                                                                                    size="sm"
                                                                                    asChild
                                                                                    className="text-blue-600 dark:text-blue-400"
                                                                                >
                                                                                    <Link
                                                                                        href={route(
                                                                                            "admin.coupons.show",
                                                                                            coupon.id
                                                                                        )}
                                                                                    >
                                                                                        <Eye className="h-4 w-4 mr-1" />
                                                                                        View
                                                                                    </Link>
                                                                                </Button>
                                                                            </td>
                                                                        </tr>
                                                                    )
                                                                )}
                                                            </tbody>
                                                        </table>
                                                    </div>
                                                </div>
                                            </div>
                                        )}
                                    </CardContent>
                                    {coupons.links &&
                                        coupons.links.length > 3 && (
                                            <CardFooter className="border-t border-gray-200 dark:border-gray-700">
                                                <div className="w-full">
                                                    <Pagination
                                                        links={coupons.links}
                                                    />
                                                </div>
                                            </CardFooter>
                                        )}
                                </Card>
                            </TabsContent>
                        </Tabs>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Show;
