import React, { useState } from "react";
import { Head, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Award,
    ArrowLeft,
    Users,
    Tag,
    BarChart2,
    PieChart,
    TrendingUp,
    Calendar,
    Filter,
    Download,
    RefreshCw,
} from "lucide-react";
import { Button } from "@/Components/ui/Button";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/Components/ui/Tabs";
import {
    Select,
    SelectContent,
    SelectItem,
    SelectTrigger,
    SelectValue,
} from "@/Components/ui/Select";
import { Badge } from "@/Components/ui/Badge";
import { Progress } from "@/Components/ui/Progress";
import Alert from "@/Components/ui/Alert";

// Placeholder for chart components - you would import your preferred charting library
// For example: import { Bar, Pie, Line } from "react-chartjs-2";
// or import { BarChart, PieChart, LineChart } from "recharts";

const Statistics = ({
    stats = {},
    tiers = [],
    dateRanges = {},
    currentRange,
}) => {
    const [timeRange, setTimeRange] = useState(currentRange || "30days");
    const [isRefreshing, setIsRefreshing] = useState(false);

    // Handle refresh data
    const refreshData = () => {
        setIsRefreshing(true);
        // Simulate data refresh
        setTimeout(() => {
            setIsRefreshing(false);
        }, 1500);
    };

    // Format large numbers
    const formatNumber = (num) => {
        return new Intl.NumberFormat().format(num);
    };

    // Format percentage
    const formatPercent = (num) => {
        return `${parseFloat(num).toFixed(1)}%`;
    };

    // Handle export data
    const exportData = (format) => {
        // Implementation for exporting data
        console.log(`Exporting data in ${format} format`);
    };

    return (
        <AdminLayout>
            <Head title="Loyalty Program Statistics" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white flex items-center">
                                <BarChart2 className="h-6 w-6 mr-2 text-blue-500" />
                                Loyalty Program Statistics
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Performance metrics and analytics for your
                                loyalty tiers
                            </p>
                        </div>
                        <div className="flex flex-wrap gap-3">
                            <Button
                                variant="outline"
                                className="inline-flex items-center"
                                asChild
                            >
                                <Link href={route("admin.loyalty-tiers.index")}>
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to Tiers
                                </Link>
                            </Button>

                            <div className="flex items-center gap-2">
                                <Select
                                    value={timeRange}
                                    onValueChange={setTimeRange}
                                >
                                    <SelectTrigger className="w-[150px]">
                                        <Calendar className="w-4 h-4 mr-2" />
                                        <SelectValue placeholder="Select period" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        {Object.keys(dateRanges).map((key) => (
                                            <SelectItem key={key} value={key}>
                                                {dateRanges[key].label}
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>

                                <Button
                                    variant="ghost"
                                    size="icon"
                                    onClick={refreshData}
                                    disabled={isRefreshing}
                                >
                                    <RefreshCw
                                        className={`h-4 w-4 ${
                                            isRefreshing ? "animate-spin" : ""
                                        }`}
                                    />
                                </Button>
                            </div>
                        </div>
                    </div>

                    {/* Summary Cards */}
                    <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-4 gap-6 mt-6">
                        <Card>
                            <CardHeader className="pb-2">
                                <CardTitle className="text-base">
                                    Total Loyalty Users
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-3xl font-bold text-blue-600 dark:text-blue-400">
                                    {formatNumber(stats.total_loyalty_users)}
                                </div>
                                <div className="flex items-center mt-1">
                                    <span
                                        className={`text-xs ${
                                            stats.loyalty_users_change >= 0
                                                ? "text-green-500"
                                                : "text-red-500"
                                        } flex items-center`}
                                    >
                                        {stats.loyalty_users_change >= 0 ? (
                                            <TrendingUp className="h-3 w-3 mr-1" />
                                        ) : (
                                            <TrendingUp className="h-3 w-3 mr-1 transform rotate-180" />
                                        )}
                                        {formatPercent(
                                            Math.abs(stats.loyalty_users_change)
                                        )}
                                    </span>
                                    <span className="text-xs text-gray-500 dark:text-gray-400 ml-1">
                                        vs previous period
                                    </span>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardHeader className="pb-2">
                                <CardTitle className="text-base">
                                    Active Coupons
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-3xl font-bold text-purple-600 dark:text-purple-400">
                                    {formatNumber(stats.total_active_coupons)}
                                </div>
                                <div className="flex items-center mt-1">
                                    <span
                                        className={`text-xs ${
                                            stats.coupons_change >= 0
                                                ? "text-green-500"
                                                : "text-red-500"
                                        } flex items-center`}
                                    >
                                        {stats.coupons_change >= 0 ? (
                                            <TrendingUp className="h-3 w-3 mr-1" />
                                        ) : (
                                            <TrendingUp className="h-3 w-3 mr-1 transform rotate-180" />
                                        )}
                                        {formatPercent(
                                            Math.abs(stats.coupons_change)
                                        )}
                                    </span>
                                    <span className="text-xs text-gray-500 dark:text-gray-400 ml-1">
                                        vs previous period
                                    </span>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardHeader className="pb-2">
                                <CardTitle className="text-base">
                                    Coupon Redemptions
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-3xl font-bold text-green-600 dark:text-green-400">
                                    {formatNumber(stats.total_redemptions)}
                                </div>
                                <div className="flex items-center mt-1">
                                    <span
                                        className={`text-xs ${
                                            stats.redemptions_change >= 0
                                                ? "text-green-500"
                                                : "text-red-500"
                                        } flex items-center`}
                                    >
                                        {stats.redemptions_change >= 0 ? (
                                            <TrendingUp className="h-3 w-3 mr-1" />
                                        ) : (
                                            <TrendingUp className="h-3 w-3 mr-1 transform rotate-180" />
                                        )}
                                        {formatPercent(
                                            Math.abs(stats.redemptions_change)
                                        )}
                                    </span>
                                    <span className="text-xs text-gray-500 dark:text-gray-400 ml-1">
                                        vs previous period
                                    </span>
                                </div>
                            </CardContent>
                        </Card>

                        <Card>
                            <CardHeader className="pb-2">
                                <CardTitle className="text-base">
                                    Discount Value
                                </CardTitle>
                            </CardHeader>
                            <CardContent>
                                <div className="text-3xl font-bold text-amber-600 dark:text-amber-400">
                                    ${formatNumber(stats.total_discount_value)}
                                </div>
                                <div className="flex items-center mt-1">
                                    <span
                                        className={`text-xs ${
                                            stats.discount_value_change >= 0
                                                ? "text-green-500"
                                                : "text-red-500"
                                        } flex items-center`}
                                    >
                                        {stats.discount_value_change >= 0 ? (
                                            <TrendingUp className="h-3 w-3 mr-1" />
                                        ) : (
                                            <TrendingUp className="h-3 w-3 mr-1 transform rotate-180" />
                                        )}
                                        {formatPercent(
                                            Math.abs(
                                                stats.discount_value_change
                                            )
                                        )}
                                    </span>
                                    <span className="text-xs text-gray-500 dark:text-gray-400 ml-1">
                                        vs previous period
                                    </span>
                                </div>
                            </CardContent>
                        </Card>
                    </div>

                    {/* Tabs for different charts and analytics */}
                    <div className="mt-6">
                        <Tabs defaultValue="overview" className="w-full">
                            <TabsList className="grid w-full grid-cols-3 lg:w-auto lg:inline-flex">
                                <TabsTrigger value="overview">
                                    Overview
                                </TabsTrigger>
                                <TabsTrigger value="tiers">
                                    Tier Analysis
                                </TabsTrigger>
                                <TabsTrigger value="users">
                                    User Distribution
                                </TabsTrigger>
                            </TabsList>

                            {/* Overview Tab */}
                            <TabsContent
                                value="overview"
                                className="mt-6 space-y-6"
                            >
                                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                                    {/* Coupon Redemption Trend Chart */}
                                    <Card className="col-span-1 lg:col-span-2">
                                        <CardHeader>
                                            <div className="flex flex-wrap justify-between items-center">
                                                <div>
                                                    <CardTitle>
                                                        Redemption Trend
                                                    </CardTitle>
                                                    <CardDescription>
                                                        Coupon redemptions over
                                                        time
                                                    </CardDescription>
                                                </div>
                                                <Button
                                                    variant="outline"
                                                    size="sm"
                                                    onClick={() =>
                                                        exportData("csv")
                                                    }
                                                >
                                                    <Download className="w-4 h-4 mr-2" />
                                                    Export
                                                </Button>
                                            </div>
                                        </CardHeader>
                                        <CardContent className="h-80">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <BarChart2 className="h-8 w-8 mb-2" />
                                                    Line chart for redemption
                                                    trend would appear here
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>

                                    {/* Distribution by Tier */}
                                    <Card>
                                        <CardHeader>
                                            <CardTitle>Users by Tier</CardTitle>
                                            <CardDescription>
                                                Distribution of users across
                                                loyalty tiers
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent className="h-72">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <PieChart className="h-8 w-8 mb-2" />
                                                    Pie chart for user
                                                    distribution would appear
                                                    here
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>

                                    {/* Redemption by Tier */}
                                    <Card>
                                        <CardHeader>
                                            <CardTitle>
                                                Redemptions by Tier
                                            </CardTitle>
                                            <CardDescription>
                                                Coupon usage across different
                                                loyalty tiers
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent className="h-72">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <BarChart2 className="h-8 w-8 mb-2" />
                                                    Bar chart for redemptions by
                                                    tier would appear here
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>
                                </div>
                            </TabsContent>

                            {/* Tier Analysis Tab */}
                            <TabsContent value="tiers" className="mt-6">
                                <Card>
                                    <CardHeader>
                                        <CardTitle>
                                            Tier Performance Comparison
                                        </CardTitle>
                                        <CardDescription>
                                            Comparative analysis of all loyalty
                                            tiers
                                        </CardDescription>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="overflow-x-auto">
                                            <table className="min-w-full divide-y divide-gray-300 dark:divide-gray-700">
                                                <thead className="bg-gray-50 dark:bg-gray-800/50">
                                                    <tr>
                                                        <th
                                                            scope="col"
                                                            className="py-3.5 pl-4 pr-3 text-left text-sm font-semibold text-gray-900 dark:text-white sm:pl-6"
                                                        >
                                                            Tier
                                                        </th>
                                                        <th
                                                            scope="col"
                                                            className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                        >
                                                            Users
                                                        </th>
                                                        <th
                                                            scope="col"
                                                            className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                        >
                                                            Coupons
                                                        </th>
                                                        <th
                                                            scope="col"
                                                            className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                        >
                                                            Redemptions
                                                        </th>
                                                        <th
                                                            scope="col"
                                                            className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                        >
                                                            Value
                                                        </th>
                                                        <th
                                                            scope="col"
                                                            className="px-3 py-3.5 text-left text-sm font-semibold text-gray-900 dark:text-white"
                                                        >
                                                            Conversion
                                                        </th>
                                                        <th
                                                            scope="col"
                                                            className="relative py-3.5 pl-3 pr-4 sm:pr-6"
                                                        >
                                                            <span className="sr-only">
                                                                Actions
                                                            </span>
                                                        </th>
                                                    </tr>
                                                </thead>
                                                <tbody className="divide-y divide-gray-200 dark:divide-gray-700 bg-white dark:bg-gray-800">
                                                    {tiers.map((tier) => (
                                                        <tr
                                                            key={tier.id}
                                                            className="hover:bg-gray-50 dark:hover:bg-gray-750"
                                                        >
                                                            <td className="whitespace-nowrap py-4 pl-4 pr-3 text-sm font-medium text-gray-900 dark:text-white sm:pl-6">
                                                                <div className="flex items-center">
                                                                    <Award className="h-5 w-5 mr-2 text-blue-500" />
                                                                    {tier.name}
                                                                    <Badge
                                                                        variant={
                                                                            tier.is_active
                                                                                ? "outline"
                                                                                : "secondary"
                                                                        }
                                                                        className="ml-2"
                                                                    >
                                                                        {tier.is_active
                                                                            ? "Active"
                                                                            : "Inactive"}
                                                                    </Badge>
                                                                </div>
                                                                <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                                    {
                                                                        tier.min_orders
                                                                    }
                                                                    + orders ·{" "}
                                                                    {parseFloat(
                                                                        tier.discount_percentage
                                                                    ).toFixed(
                                                                        1
                                                                    )}
                                                                    % discount
                                                                </div>
                                                            </td>
                                                            <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-700 dark:text-gray-300">
                                                                {formatNumber(
                                                                    tier.user_count ||
                                                                        0
                                                                )}
                                                                <div className="w-24 mt-1">
                                                                    <Progress
                                                                        value={
                                                                            tier.user_percentage ||
                                                                            0
                                                                        }
                                                                        className="h-1"
                                                                    />
                                                                    <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                                        {formatPercent(
                                                                            tier.user_percentage ||
                                                                                0
                                                                        )}{" "}
                                                                        of total
                                                                    </div>
                                                                </div>
                                                            </td>
                                                            <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-700 dark:text-gray-300">
                                                                {formatNumber(
                                                                    tier.coupon_count ||
                                                                        0
                                                                )}
                                                            </td>
                                                            <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-700 dark:text-gray-300">
                                                                {formatNumber(
                                                                    tier.redemption_count ||
                                                                        0
                                                                )}
                                                                <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                                    {formatPercent(
                                                                        tier.redemption_percentage ||
                                                                            0
                                                                    )}{" "}
                                                                    of total
                                                                </div>
                                                            </td>
                                                            <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-700 dark:text-gray-300">
                                                                $
                                                                {formatNumber(
                                                                    tier.discount_value ||
                                                                        0
                                                                )}
                                                            </td>
                                                            <td className="whitespace-nowrap px-3 py-4 text-sm text-gray-700 dark:text-gray-300">
                                                                {formatPercent(
                                                                    tier.conversion_rate ||
                                                                        0
                                                                )}
                                                            </td>
                                                            <td className="relative whitespace-nowrap py-4 pl-3 pr-4 text-right text-sm font-medium sm:pr-6">
                                                                <Button
                                                                    variant="ghost"
                                                                    size="sm"
                                                                    asChild
                                                                    className="text-blue-600 dark:text-blue-400"
                                                                >
                                                                    <Link
                                                                        href={route(
                                                                            "admin.loyalty-tiers.show",
                                                                            tier.id
                                                                        )}
                                                                    >
                                                                        View
                                                                    </Link>
                                                                </Button>
                                                            </td>
                                                        </tr>
                                                    ))}
                                                </tbody>
                                            </table>
                                        </div>
                                    </CardContent>
                                </Card>

                                {/* Performance metrics cards */}
                                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-6 mt-6">
                                    <Card>
                                        <CardHeader>
                                            <CardTitle>
                                                Tier Effectiveness
                                            </CardTitle>
                                            <CardDescription>
                                                Comparing redemption rates
                                                across tiers
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent className="h-80">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <BarChart2 className="h-8 w-8 mb-2" />
                                                    Bar chart showing tier
                                                    effectiveness
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>

                                    <Card>
                                        <CardHeader>
                                            <CardTitle>
                                                Discount Impact
                                            </CardTitle>
                                            <CardDescription>
                                                Order value before and after
                                                discounts by tier
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent className="h-80">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <BarChart2 className="h-8 w-8 mb-2" />
                                                    Comparative chart showing
                                                    discount impact
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>

                                    <Card>
                                        <CardHeader>
                                            <CardTitle>
                                                Tier Progression
                                            </CardTitle>
                                            <CardDescription>
                                                User progression through loyalty
                                                tiers
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent className="h-80">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <BarChart2 className="h-8 w-8 mb-2" />
                                                    Funnel chart showing tier
                                                    progression
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>
                                </div>
                            </TabsContent>

                            {/* User Distribution Tab */}
                            <TabsContent value="users" className="mt-6">
                                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6">
                                    <Card className="col-span-1 lg:col-span-2">
                                        <CardHeader>
                                            <CardTitle>
                                                User Tier Distribution
                                            </CardTitle>
                                            <CardDescription>
                                                How users are distributed across
                                                different loyalty tiers
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent className="h-96">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <PieChart className="h-8 w-8 mb-2" />
                                                    Distribution chart would
                                                    appear here
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>

                                    {/* User Growth Card */}
                                    <Card>
                                        <CardHeader>
                                            <CardTitle>
                                                Tier Growth Trend
                                            </CardTitle>
                                            <CardDescription>
                                                User count growth by tier over
                                                time
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent className="h-80">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <BarChart2 className="h-8 w-8 mb-2" />
                                                    Line chart showing tier
                                                    growth trends
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>

                                    {/* Engagement Metrics */}
                                    <Card>
                                        <CardHeader>
                                            <CardTitle>
                                                User Engagement by Tier
                                            </CardTitle>
                                            <CardDescription>
                                                Comparing user engagement across
                                                tiers
                                            </CardDescription>
                                        </CardHeader>
                                        <CardContent className="h-80">
                                            {/* Replace with actual chart component */}
                                            <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                                <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                    <BarChart2 className="h-8 w-8 mb-2" />
                                                    Bar chart for user
                                                    engagement metrics
                                                </p>
                                            </div>
                                        </CardContent>
                                    </Card>
                                </div>

                                {/* Top Tier Users */}
                                <Card className="mt-6">
                                    <CardHeader>
                                        <CardTitle>
                                            Top Tier Distribution
                                        </CardTitle>
                                        <CardDescription>
                                            Geographic and demographic
                                            distribution of higher tier users
                                        </CardDescription>
                                    </CardHeader>
                                    <CardContent className="h-80">
                                        {/* Replace with actual chart/map component */}
                                        <div className="w-full h-full bg-gray-50 dark:bg-gray-800/50 rounded-md flex items-center justify-center">
                                            <p className="text-gray-500 dark:text-gray-400 flex flex-col items-center">
                                                <Users className="h-8 w-8 mb-2" />
                                                Geographic distribution map
                                                would appear here
                                            </p>
                                        </div>
                                    </CardContent>
                                </Card>
                            </TabsContent>
                        </Tabs>
                    </div>

                    {/* Data Quality Alert */}
                    <Alert
                        className="mt-6"
                        variant="info"
                        title="Data Information"
                        message={`Statistics shown are for the ${dateRanges[
                            timeRange
                        ]?.label.toLowerCase()} period. Data is refreshed every 24 hours.`}
                    />
                </div>
            </div>
        </AdminLayout>
    );
};

export default Statistics;
