import React, { useMemo, useState, useEffect } from "react";
import { Head } from "@inertiajs/react";
import { useForm } from "@inertiajs/react";
import { Button } from "@/Components/ui/Button";
import { OrderPreview } from "@/Components/OrderPreview";
import { DeadlineSelect } from "@/Components/ui/DeadlineSelect";
import { CustomSelect } from "@/Components/ui/CustomSelect";
import { FormInput } from "@/Components/ui/FormInput";
import { FormTextarea } from "@/Components/ui/FormTextarea";
import { NormalSelect } from "@/Components/ui/NormalSelect";
import { ClientSelect } from "@/Components/ui/ClientSelect";
import AuthenticatedLayout from "@/Layouts/AuthenticatedLayout";
import AdminLayout from "@/Layouts/AdminLayout";

export default function Create({
    academicLevels,
    deadlines,
    subjects,
    paperTypes,
    couponEnabled,
    couponCode,
    clients,
}) {
    const { data, setData, post, processing, errors, progress } = useForm({
        title: "",
        user_id: "",
        instructions: "",
        pages: 1,
        subject: "",
        aclevel: "",
        papertype: "",
        ppslides: 0,
        deadline: "",
        clientDeadline: "",
        clientTimezone: "",
        spacing: "",
        totalAmount: 0,
        styles: "",
        sources: 0,
        language: "",
        files: [],
        urgency: "",
        couponCode: "",
        netAmount: 0,
        discount: 0,
    });

    console.log(data);

    //handle form submit
    const handleSubmit = (e) => {
        e.preventDefault();

        const formData = new FormData();

        // Append all form fields
        Object.keys(data).forEach((key) => {
            if (key !== "files") {
                formData.append(key, data[key]);
            }
        });

        // Append multiple files
        data.files.forEach((file) => {
            formData.append("files[]", file);
        });

        post("/admin/orders", formData, {
            forceFormData: true,
            preserveScroll: true,
        });
    };

    const parseValue = (str) => {
        if (!str) return 0;
        return parseFloat(str.split("#")[0]) || 0;
    };

    const [discountInfo, setDiscountInfo] = useState({
        discountType: null,
        discountAmount: 0,
        error: null,
    });

    const validateCoupon = async () => {
        if (!data.couponCode) {
            setDiscountInfo({
                discountType: null,
                discountAmount: 0,
                error: null,
            });
            return;
        }

        try {
            const response = await axios.post("/coupon-validate", {
                code: data.couponCode,
            });
            if (response.data.valid) {
                setDiscountInfo({
                    discountType: response.data.type,
                    discountAmount: response.data.amount,
                    error: null,
                });
            } else {
                setDiscountInfo({
                    discountType: null,
                    discountAmount: 0,
                    error: "Invalid coupon code",
                });
            }
        } catch (error) {
            setDiscountInfo({
                discountType: null,
                discountAmount: 0,
                error: "Error validating coupon",
            });
        }
    };
    const {
        basePrice,
        totalAmount,
        costPerPage,
        costPerSlide,
        discount,
        netAmount,
    } = useMemo(() => {
        const pages = data.pages || 1;
        const slides = data.ppslides || 0;
        const pricePerSlide = 4;

        // Parse values from form data strings
        const urgency = parseValue(data.deadline);
        const aclevel = parseValue(data.aclevel);
        const spacing = parseValue(data.spacing);
        const subject = parseValue(data.subject);
        const paperType = parseValue(data.papertype);

        // Return 0 if any required value is missing
        if (!urgency || !aclevel || !spacing || !subject || !paperType)
            return {
                basePrice: 0,
                totalAmount: 0,
                costPerPage: 0,
                netAmount: 0,
                discount: 0,
            };

        // Calculate cost per page
        const costPerPage = aclevel * urgency * spacing * subject * paperType;

        // Calculate base price and slide price
        const basePrice = costPerPage * pages;
        const slidePrice = slides * pricePerSlide;

        const subtotal = basePrice + slidePrice;
        let discount = 0;

        if (discountInfo.discountType === "percentage") {
            discount = (subtotal * discountInfo.discountAmount) / 100;
        }

        return {
            basePrice: basePrice.toFixed(2),
            totalAmount: subtotal.toFixed(2),
            netAmount: (subtotal - discount).toFixed(2),
            costPerPage: costPerPage.toFixed(2),
            costPerSlide: slidePrice.toFixed(2),
            discount: discount.toFixed(2),
        };
    }, [
        data.pages,
        data.ppslides,
        data.deadline,
        data.aclevel,
        data.spacing,
        data.subject,
        data.papertype,

        discountInfo,
    ]);

    useEffect(() => {
        setData("totalAmount", parseFloat(totalAmount));
    }, [totalAmount]);

    useEffect(() => {
        setData("netAmount", parseFloat(netAmount));
    }, [netAmount]);

    useEffect(() => {
        if (discountInfo.discountType === "percentage") {
            const discountValue =
                (parseFloat(totalAmount) * discountInfo.discountAmount) / 100;
            setData("discount", discountValue);
        }
    }, [discountInfo, totalAmount]);

    useEffect(() => {
        if (data.deadline) {
            const parts = data.deadline.split("#");
            const urgencyValue = `${parts[1]} ${parts[2]}`;
            setData("urgency", urgencyValue);
        }
    }, [data.deadline]);

    return (
        <AdminLayout
            header={
                <h2 className="text-xl font-semibold leading-tight text-gray-800 dark:text-gray-200">
                    Create an Order
                </h2>
            }
        >
            <Head title="Create Order:Buy Essay Online" />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-1">
                    <div className="overflow-hidden shadow-sm sm:rounded-lg ">
                        <div className="p-6 ">
                            <div className="lg:pr-96">
                                <form
                                    onSubmit={handleSubmit}
                                    className="space-y-6 bg-gradient-to-r from-orange-200 to-cyan-200 dark:bg-none dark:bg-gray-800 p-6 rounded-lg"
                                >
                                    {/* Title - Full width */}
                                    <div>
                                        <ClientSelect
                                            id="user_id"
                                            name="user_id"
                                            label="Client"
                                            value={data.user_id || ""}
                                            setData={setData}
                                            options={clients}
                                            error={errors.user_id}
                                            placeholder="Select Client"
                                            className="dark:text-gray-200"
                                            required={true}
                                        />
                                    </div>
                                    {/* Title - Full width */}
                                    <div className="col-span-full">
                                        <FormInput
                                            id="title"
                                            name="title"
                                            label="Title"
                                            value={data.title}
                                            setData={setData}
                                            error={errors.title}
                                            placeholder="Enter title"
                                            className="dark:text-gray-200"
                                            required
                                        />
                                    </div>

                                    {/* Grid container for two-column layout */}
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <CustomSelect
                                            id="aclevel"
                                            name="aclevel"
                                            label="Academic Level"
                                            value={data.aclevel}
                                            setData={setData}
                                            options={academicLevels}
                                            error={errors.aclevel}
                                            className="dark:text-gray-200"
                                            required
                                        />

                                        <CustomSelect
                                            id="subject"
                                            name="subject"
                                            label="Subject"
                                            value={data.subject}
                                            setData={setData}
                                            options={subjects}
                                            error={errors.subject}
                                            className="dark:text-gray-200"
                                            required
                                        />

                                        <CustomSelect
                                            id="papertype"
                                            name="papertype"
                                            label="Type of Paper"
                                            value={data.papertype}
                                            setData={setData}
                                            options={paperTypes}
                                            error={errors.papertype}
                                            className="dark:text-gray-200"
                                            required
                                        />

                                        <CustomSelect
                                            id="spacing"
                                            name="spacing"
                                            label="Document Spacing"
                                            value={data.spacing}
                                            setData={setData}
                                            options={[
                                                {
                                                    id: 1,
                                                    value: "1",
                                                    name: "Double Spaced",
                                                },
                                                {
                                                    id: 2,
                                                    value: "2",
                                                    name: "Single Spaced",
                                                },
                                            ]}
                                            error={errors.spacing}
                                            placeholder="Select spacing"
                                            className="dark:text-gray-200"
                                            required={true}
                                        />
                                    </div>
                                    <div>
                                        <DeadlineSelect
                                            id="deadline"
                                            name="deadline"
                                            label="Deadline"
                                            value={data.deadline}
                                            setData={setData}
                                            options={deadlines}
                                            error={errors.deadline}
                                            className="dark:text-gray-200"
                                            required={true}
                                        />
                                    </div>

                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <FormInput
                                            id="pages"
                                            name="pages"
                                            label="Number of Pages"
                                            type="number"
                                            value={data.pages}
                                            setData={setData}
                                            error={errors.pages}
                                            placeholder="1"
                                            min="1"
                                            className="dark:text-gray-200"
                                            required={true}
                                        />
                                        {/* PowerPoint Slides */}
                                        <FormInput
                                            id="ppslides"
                                            name="ppslides"
                                            label="PowerPoint Slides"
                                            type="number"
                                            value={data.ppslides}
                                            setData={setData}
                                            error={errors.ppslides}
                                            placeholder="0"
                                            min="0"
                                            className="dark:text-gray-200"
                                            required={true}
                                        />
                                    </div>

                                    <div className="space-y-4">
                                        <label className="block text-sm font-medium text-gray-100">
                                            Upload Files
                                        </label>
                                        <input
                                            type="file"
                                            multiple
                                            className="w-full text-gray-100"
                                            onChange={(e) =>
                                                setData(
                                                    "files",
                                                    Array.from(e.target.files)
                                                )
                                            }
                                        />

                                        {/* Progress bar */}
                                        {progress && (
                                            <div className="w-full bg-gray-700 rounded-full h-2.5">
                                                <div
                                                    className="bg-blue-600 h-2.5 rounded-full transition-all duration-300"
                                                    style={{
                                                        width: `${progress.percentage}%`,
                                                    }}
                                                />
                                            </div>
                                        )}

                                        {/* Files list */}
                                        {data.files.length > 0 && (
                                            <div className="mt-4 space-y-2">
                                                <p className="text-sm text-gray-300">
                                                    Selected files:
                                                </p>
                                                <ul className="list-disc pl-5 space-y-1">
                                                    {Array.from(data.files).map(
                                                        (file, index) => (
                                                            <li
                                                                key={index}
                                                                className="text-sm text-gray-300"
                                                            >
                                                                {file.name} (
                                                                {(
                                                                    file.size /
                                                                    1024
                                                                ).toFixed(
                                                                    2
                                                                )}{" "}
                                                                KB)
                                                            </li>
                                                        )
                                                    )}
                                                </ul>
                                            </div>
                                        )}
                                    </div>

                                    {/* instructions - Full width */}
                                    <FormTextarea
                                        id="instructions"
                                        name="instructions"
                                        label="Instructions"
                                        value={data.instructions}
                                        setData={setData}
                                        error={errors.instructions}
                                        placeholder="Enter instructions"
                                        className="dark:text-gray-200"
                                        required={true}
                                    />
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <NormalSelect
                                            id="styles"
                                            name="styles"
                                            label="Writing Style"
                                            value={data.styles}
                                            setData={setData}
                                            options={[
                                                {
                                                    id: 1,
                                                    value: "APA",
                                                    name: "APA",
                                                },
                                                {
                                                    id: 2,
                                                    value: "MLA",
                                                    name: "MLA",
                                                },
                                                {
                                                    id: 3,
                                                    value: "Chicago",
                                                    name: "Chicago",
                                                },
                                                {
                                                    id: 4,
                                                    value: "Harvard",
                                                    name: "Harvard",
                                                },
                                                {
                                                    id: 5,
                                                    value: "Turabian",
                                                    name: "Turabian",
                                                },
                                                {
                                                    id: 6,
                                                    value: "Oxford",
                                                    name: "Oxford",
                                                },
                                                {
                                                    id: 7,
                                                    value: "Vancouver",
                                                    name: "Vancouver",
                                                },
                                            ]}
                                            error={errors.styles}
                                            placeholder="Select Style"
                                            className="dark:text-gray-200"
                                            required={true}
                                        />
                                        <FormInput
                                            id="sources"
                                            name="sources"
                                            label="Sources"
                                            type="number"
                                            value={data.sources}
                                            setData={setData}
                                            error={errors.sources}
                                            placeholder="0"
                                            min="0"
                                            className="dark:text-gray-200"
                                        />
                                    </div>
                                    <div>
                                        <NormalSelect
                                            id="language"
                                            name="language"
                                            label="Preferred Language"
                                            value={data.language}
                                            setData={setData}
                                            options={[
                                                {
                                                    id: 1,
                                                    value: "1",
                                                    name: "American English",
                                                },
                                                {
                                                    id: 2,
                                                    value: "2",
                                                    name: "British English",
                                                },
                                                {
                                                    id: 3,
                                                    value: "3",
                                                    name: "Australian English",
                                                },
                                                {
                                                    id: 4,
                                                    value: "4",
                                                    name: "Canadian English",
                                                },
                                            ]}
                                            error={errors.language}
                                            placeholder="Select Language"
                                            className="dark:text-gray-200"
                                            required={true}
                                        />
                                    </div>
                                    <div className="flex flex-col gap-2 w-full">
                                        {couponEnabled && (
                                            <div className="mb-4">
                                                <label className="block text-sm font-medium text-gray-100 mb-2">
                                                    Coupon Code:{" "}
                                                    <span className="text-xs text-green-400">
                                                        (Your Coupon Code:{" "}
                                                        {couponCode})
                                                    </span>
                                                </label>
                                                <div className="mt-1 flex w-full">
                                                    <FormInput
                                                        id="couponCode"
                                                        name="couponCode"
                                                        value={data.couponCode}
                                                        setData={setData}
                                                        placeholder="Enter coupon code"
                                                        className="dark:text-gray-200 flex-1"
                                                    />
                                                    <Button
                                                        type="button"
                                                        onClick={validateCoupon}
                                                        className="ml-2 px-4 bg-blue-700 hover:bg-blue-600 dark:bg-blue-500 dark:hover:bg-blue-400"
                                                    >
                                                        Apply
                                                    </Button>
                                                </div>
                                                {discountInfo.error && (
                                                    <div className="text-sm text-red-500 mt-1">
                                                        {discountInfo.error}
                                                    </div>
                                                )}
                                                {discountInfo.discountType && (
                                                    <div className="text-sm text-green-500 mt-1">
                                                        Discount applied:{" "}
                                                        {
                                                            discountInfo.discountAmount
                                                        }
                                                        %
                                                    </div>
                                                )}
                                            </div>
                                        )}
                                    </div>

                                    <div className="text-right mr-4 flex justify-between">
                                        <div className="mt-1 text-gray-100 text-lg font-semibold">
                                            <label className="block text-sm font-medium  text-green-400">
                                                Gross Amount
                                            </label>
                                            <div className="mt-1 text-gray-100 text-md font-semibold">
                                                ${totalAmount}
                                            </div>
                                        </div>

                                        <div className="mt-1 text-gray-100  font-semibold">
                                            <label className="block text-sm font-medium text-green-400">
                                                Discount Awarded
                                            </label>
                                            <div className="mt-1 text-gray-100 text-md font-semibold">
                                                ${discount}
                                            </div>
                                        </div>
                                        <div>
                                            <label className="block text-sm font-medium text-green-400">
                                                Net Amount
                                            </label>

                                            <div className="mt-1 text-gray-100 text-md font-semibold">
                                                ${netAmount}
                                            </div>
                                        </div>
                                    </div>

                                    <Button
                                        type="submit"
                                        className="w-full bg-blue-700 hover:bg-blue-600 dark:bg-blue-500 dark:hover:bg-blue-400"
                                        disabled={processing}
                                    >
                                        {processing
                                            ? "Creating..."
                                            : "Create Order"}
                                    </Button>
                                </form>
                            </div>
                        </div>
                        <OrderPreview
                            data={data}
                            totalAmount={totalAmount}
                            basePrice={basePrice}
                            costPerPage={costPerPage}
                            costPerSlide={costPerSlide}
                            netAmount={netAmount}
                            discount={discount}
                        />
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
