import { useEffect, useState, useMemo } from "react";
import AdminLayout from "@/Layouts/AdminLayout";
import CleanPagination from "@/Components/CleanPagination";
import { Link, router, usePage } from "@inertiajs/react";
import { 
    Search, 
    Edit, 
    View, 
    Trash2, 
    Plus, 
    ListOrdered, 
    Filter, 
    SortAsc, 
    SortDesc,
    Calendar,
    DollarSign,
    User,
    FileText,
    Clock,
    CheckCircle,
    XCircle,
    AlertCircle,
    TrendingUp,
    TrendingDown
} from "lucide-react";
import SectionContainer from "@/Components/SectionContainer";
import StatsCard from "@/Components/StatsCard";
import ExpiresIn from "@/Components/ExpiresIn";

export default function Index({
    orders,
    query,
    filters,
    totalOrders,
    assignedOrders,
    pendingOrders,
}) {
    const [search, setSearch] = useState(query || "");
    const [paymentStatus, setPaymentStatus] = useState(filters?.paymentStatus || "");
    const [orderStatus, setOrderStatus] = useState(filters?.orderStatus || "");
    const [sortField, setSortField] = useState("dateposted");
    const [sortDirection, setSortDirection] = useState("desc");
    const [showFilters, setShowFilters] = useState(false);
    const [perPage, setPerPage] = useState(10);
    
    const { flash } = usePage().props;
    const [flashMsg, setFlashMsg] = useState(flash.message);

    useEffect(() => {
        if (flashMsg) {
            const timer = setTimeout(() => {
                setFlashMsg("");
            }, 3000);
            return () => clearTimeout(timer);
        }
    }, [flashMsg]);

    useEffect(() => {
        setFlashMsg(flash.message);
    }, [flash.message]);

    // Calculate additional statistics
    const statistics = useMemo(() => {
        if (!orders?.data) return {};
        
        const ordersData = orders.data;
        const totalAmount = ordersData.reduce((sum, order) => sum + (parseFloat(order.net_amount) || 0), 0);
        const paidOrders = ordersData.filter(order => order.payment_status === 'paid').length;
        const urgentOrders = ordersData.filter(order => {
            if (!order.deadline) return false;
            // Exclude completed and approved orders
            if (['approved', 'cancelled'].includes(order.order_status)) return false;
            const deadline = new Date(order.deadline);
            const now = new Date();
            const hoursUntilDeadline = (deadline - now) / (1000 * 60 * 60);
            return hoursUntilDeadline <= 24 && hoursUntilDeadline > 0;
        }).length;
        const overdueOrders = ordersData.filter(order => {
            if (!order.deadline) return false;
            const deadline = new Date(order.deadline);
            const now = new Date();
            return deadline < now && order.order_status !== 'completed' && order.order_status !== 'cancelled';
        }).length;

        return {
            totalAmount: totalAmount.toFixed(2),
            paidOrders,
            urgentOrders,
            overdueOrders
        };
    }, [orders]);

    const handleFilters = (newFilters) => {
        const params = {
            query: search.trim() || undefined,
            paymentStatus: newFilters.paymentStatus !== undefined ? newFilters.paymentStatus : paymentStatus || undefined,
            orderStatus: newFilters.orderStatus !== undefined ? newFilters.orderStatus : orderStatus || undefined,
            sortField: sortField,
            sortDirection: sortDirection,
            perPage: perPage
        };

        const cleanParams = Object.fromEntries(
            Object.entries(params).filter(([_, value]) => value !== undefined && value !== "")
        );

        router.get(route("admin.orders.search"), cleanParams, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const handleSearch = (e) => {
        e.preventDefault();
        handleFilters({});
    };

    const handleSearchChange = (e) => {
        const value = e.target.value;
        setSearch(value);

        if (value.trim() === "") {
            const currentParams = { ...route().params };
            delete currentParams.query;
            router.get(route("admin.orders.search"), currentParams, {
                preserveState: true,
                preserveScroll: true,
            });
        }
    };

    const handleFilterChange = (filterType, value) => {
        if (filterType === "paymentStatus") {
            setPaymentStatus(value);
        } else if (filterType === "orderStatus") {
            setOrderStatus(value);
        }

        handleFilters({ [filterType]: value });
    };

    const handleSort = (field) => {
        const newDirection = field === sortField && sortDirection === "asc" ? "desc" : "asc";
        setSortField(field);
        setSortDirection(newDirection);
        handleFilters({});
    };

    const handlePerPageChange = (newPerPage) => {
        setPerPage(newPerPage);
        handleFilters({});
    };

    const handleDelete = (orderId) => {
        if (confirm("Are you sure you want to delete this order?")) {
            router.delete(route("admin.orders.destroy", orderId));
        }
    };

    const getStatusBadgeVariant = (status) => {
        const statusMap = {
            'bidding': 'bg-blue-100 text-blue-800 border-blue-200',
            'assigned': 'bg-purple-100 text-purple-800 border-purple-200',
            'in_progress': 'bg-yellow-100 text-yellow-800 border-yellow-200',
            'submitted': 'bg-indigo-100 text-indigo-800 border-indigo-200',
            'under_review': 'bg-orange-100 text-orange-800 border-orange-200',
            'approved': 'bg-green-100 text-green-800 border-green-200',
            'revision_requested': 'bg-red-100 text-red-800 border-red-200',
            'completed': 'bg-green-100 text-green-800 border-green-200',
            'cancelled': 'bg-gray-100 text-gray-800 border-gray-200'
        };
        return statusMap[status] || 'bg-gray-100 text-gray-800 border-gray-200';
    };

    const getPaymentStatusBadgeVariant = (status) => {
        const statusMap = {
            'paid': 'bg-green-100 text-green-800 border-green-200',
            'unpaid': 'bg-red-100 text-red-800 border-red-200',
            'partially_paid': 'bg-yellow-100 text-yellow-800 border-yellow-200',
            'refunded': 'bg-gray-100 text-gray-800 border-gray-200'
        };
        return statusMap[status] || 'bg-gray-100 text-gray-800 border-gray-200';
    };

    const getSortIcon = (field) => {
        if (field !== sortField) return <SortAsc className="w-4 h-4 text-gray-400" />;
        return sortDirection === "asc" ? 
            <SortAsc className="w-4 h-4 text-blue-600" /> : 
            <SortDesc className="w-4 h-4 text-blue-600" />;
    };

    return (
        <AdminLayout>
            {/* Enhanced Statistics */}
            <SectionContainer
                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:from-gray-800 dark:to-gray-900 border border-gray-300 dark:border-gray-600"
                title="Order Statistics"
                titleClassName="text-lg font-bold text-gray-900 dark:text-white bg-white bg-opacity-30 dark:bg-gray-900 dark:bg-opacity-40 border-b border-gray-300 dark:border-gray-700"
            >
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                    <StatsCard
                        title="Total Orders"
                        value={totalOrders}
                        icon={<ListOrdered className="w-6 h-6" />}
                        className="from-blue-500 to-cyan-600 dark:from-gray-800 dark:to-gray-900"
                    />
                    <StatsCard
                        title="Assigned Orders"
                        value={assignedOrders}
                        icon={<User className="w-6 h-6" />}
                        className="from-purple-500 to-pink-600 dark:from-gray-800 dark:to-gray-900"
                    />
                    <StatsCard
                        title="Pending Orders"
                        value={pendingOrders}
                        icon={<Clock className="w-6 h-6" />}
                        className="from-yellow-500 to-orange-600 dark:from-gray-800 dark:to-gray-900"
                    />
                    <StatsCard
                        title="Total Revenue"
                        value={`$${statistics.totalAmount || '0.00'}`}
                        icon={<DollarSign className="w-6 h-6" />}
                        className="from-green-500 to-emerald-600 dark:from-gray-800 dark:to-gray-900"
                    />
                </div>
                
                {/* Additional Quick Stats */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-4">
                    <div className="bg-white bg-opacity-50 dark:bg-gray-800 dark:bg-opacity-50 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center gap-3">
                            <div className="p-2 bg-green-100 dark:bg-green-900 rounded-lg">
                                <CheckCircle className="w-5 h-5 text-green-600 dark:text-green-400" />
                            </div>
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">Paid Orders</p>
                                <p className="text-xl font-bold text-gray-900 dark:text-white">{statistics.paidOrders}</p>
                            </div>
                        </div>
                    </div>
                    
                    <div className="bg-white bg-opacity-50 dark:bg-gray-800 dark:bg-opacity-50 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center gap-3">
                            <div className="p-2 bg-orange-100 dark:bg-orange-900 rounded-lg">
                                <AlertCircle className="w-5 h-5 text-orange-600 dark:text-orange-400" />
                            </div>
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">Urgent Orders</p>
                                <p className="text-xl font-bold text-gray-900 dark:text-white">{statistics.urgentOrders}</p>
                            </div>
                        </div>
                    </div>
                    
                    <div className="bg-white bg-opacity-50 dark:bg-gray-800 dark:bg-opacity-50 p-4 rounded-lg border border-gray-200 dark:border-gray-700">
                        <div className="flex items-center gap-3">
                            <div className="p-2 bg-red-100 dark:bg-red-900 rounded-lg">
                                <XCircle className="w-5 h-5 text-red-600 dark:text-red-400" />
                            </div>
                            <div>
                                <p className="text-sm text-gray-600 dark:text-gray-400">Overdue Orders</p>
                                <p className="text-xl font-bold text-gray-900 dark:text-white">{statistics.overdueOrders}</p>
                            </div>
                        </div>
                    </div>
                </div>
            </SectionContainer>

            {/* Flash Messages */}
            {flashMsg && (
                <div className="bg-green-400 border border-green-400 text-white px-4 py-3 rounded relative mb-4" role="alert">
                    <span className="block sm:inline">{flashMsg}</span>
                </div>
            )}

            {/* Enhanced Orders Table */}
            <SectionContainer
                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:from-gray-800 dark:to-gray-900 border border-gray-300 dark:border-gray-600"
                title="All Orders"
                titleClassName="text-gray-800 dark:text-gray-200 font-semibold bg-white bg-opacity-30 dark:bg-gray-900 dark:bg-opacity-40 border-b border-gray-300 dark:border-gray-700"
            >
                {/* Enhanced Controls */}
                <div className="flex flex-col lg:flex-row justify-between items-start lg:items-center gap-4 mb-6">
                    <div className="flex items-center gap-2">
                        <button
                            onClick={() => setShowFilters(!showFilters)}
                            className="flex items-center gap-2 px-4 py-2 bg-white bg-opacity-70 dark:bg-gray-800 dark:bg-opacity-70 text-gray-700 dark:text-gray-300 rounded-lg border border-gray-300 dark:border-gray-600 hover:bg-white dark:hover:bg-gray-700 transition-colors"
                        >
                            <Filter className="w-4 h-4" />
                            {showFilters ? 'Hide' : 'Show'} Filters
                        </button>
                        
                        <select
                            value={perPage}
                            onChange={(e) => handlePerPageChange(parseInt(e.target.value))}
                            className="px-3 py-2 bg-white bg-opacity-70 dark:bg-gray-800 dark:bg-opacity-70 text-gray-700 dark:text-gray-300 rounded-lg border border-gray-300 dark:border-gray-600"
                        >
                            <option value={5}>5 per page</option>
                            <option value={10}>10 per page</option>
                            <option value={25}>25 per page</option>
                            <option value={50}>50 per page</option>
                        </select>
                    </div>

                    <Link
                        href={route("admin.orders.create")}
                        className="flex items-center justify-center gap-2 px-6 py-3 bg-gradient-to-r from-blue-600 to-cyan-600 text-white rounded-lg hover:from-blue-700 hover:to-cyan-700 transition-all duration-200 shadow-lg hover:shadow-xl"
                    >
                        <Plus className="h-5 w-5" />
                        <span>Add New Order</span>
                    </Link>
                </div>

                {/* Enhanced Search and Filters */}
                <div className={`space-y-4 mb-6 transition-all duration-300 ${showFilters ? 'block' : 'hidden'}`}>
                    {/* Search Form */}
                    <form onSubmit={handleSearch} className="flex flex-col sm:flex-row items-stretch sm:items-center gap-3">
                        <div className="relative flex-1">
                            <input
                                type="search"
                                placeholder="Search by order number, title, subject, or client..."
                                value={search}
                                onChange={handleSearchChange}
                                className="w-full pl-10 pr-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:focus:ring-blue-400"
                            />
                            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 w-5 h-5 text-gray-500" />
                        </div>
                        <button
                            type="submit"
                            className="px-6 py-3 bg-gradient-to-r from-blue-600 to-cyan-600 text-white rounded-lg hover:from-blue-700 hover:to-cyan-700 transition-all duration-200 font-medium"
                        >
                            Search
                        </button>
                    </form>

                    {/* Filter Options */}
                    <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4">
                        <select
                            value={paymentStatus}
                            onChange={(e) => handleFilterChange("paymentStatus", e.target.value)}
                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        >
                            <option value="">All Payment Status</option>
                            <option value="paid">Paid</option>
                            <option value="unpaid">Unpaid</option>
                            <option value="partially_paid">Partially Paid</option>
                            <option value="refunded">Refunded</option>
                        </select>

                        <select
                            value={orderStatus}
                            onChange={(e) => handleFilterChange("orderStatus", e.target.value)}
                            className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                        >
                            <option value="">All Order Status</option>
                            <option value="bidding">Bidding</option>
                            <option value="assigned">Assigned</option>
                            <option value="in_progress">In Progress</option>
                            <option value="submitted">Submitted</option>
                            <option value="under_review">Under Review</option>
                            <option value="approved">Approved</option>
                            <option value="revision_requested">Revision Requested</option>
                            <option value="completed">Completed</option>
                            <option value="cancelled">Cancelled</option>
                        </select>
                    </div>
                </div>

                {/* Enhanced Table */}
                <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg overflow-hidden border border-gray-200 dark:border-gray-700">
                    <div className="overflow-x-auto">
                        <table className="w-full text-sm text-left">
                            <thead>
                                <tr className="bg-gradient-to-r from-blue-600 to-cyan-600 text-white">
                                    <th className="p-4 cursor-pointer hover:bg-blue-700 transition-colors" onClick={() => handleSort('order_number')}>
                                        <div className="flex items-center gap-2">
                                            Order Number
                                            {getSortIcon('order_number')}
                                        </div>
                                    </th>
                                    <th className="p-4 cursor-pointer hover:bg-blue-700 transition-colors" onClick={() => handleSort('title')}>
                                        <div className="flex items-center gap-2">
                                            Title
                                            {getSortIcon('title')}
                                        </div>
                                    </th>
                                    <th className="p-4 cursor-pointer hover:bg-blue-700 transition-colors" onClick={() => handleSort('pages')}>
                                        <div className="flex items-center gap-2">
                                            Pages
                                            {getSortIcon('pages')}
                                        </div>
                                    </th>
                                    <th className="p-4 cursor-pointer hover:bg-blue-700 transition-colors" onClick={() => handleSort('dateposted')}>
                                        <div className="flex items-center gap-2">
                                            Posted Date
                                            {getSortIcon('dateposted')}
                                        </div>
                                    </th>
                                    <th className="p-4 cursor-pointer hover:bg-blue-700 transition-colors" onClick={() => handleSort('deadline')}>
                                        <div className="flex items-center gap-2">
                                            Client Deadline
                                            {getSortIcon('deadline')}
                                        </div>
                                    </th>
                                    <th className="p-4 cursor-pointer hover:bg-blue-700 transition-colors" onClick={() => handleSort('writer_deadline')}>
                                        <div className="flex items-center gap-2">
                                            Writer Deadline
                                            {getSortIcon('writer_deadline')}
                                        </div>
                                    </th>
                                    <th className="p-4">Status</th>
                                    <th className="p-4 cursor-pointer hover:bg-blue-700 transition-colors" onClick={() => handleSort('net_amount')}>
                                        <div className="flex items-center gap-2">
                                            Amount
                                            {getSortIcon('net_amount')}
                                        </div>
                                    </th>
                                    <th className="p-4 text-center">Actions</th>
                                </tr>
                            </thead>
                            <tbody className="divide-y divide-gray-200 dark:divide-gray-700">
                                {orders?.data && orders.data.length > 0 ? (
                                    orders.data.map((order) => (
                                        <tr
                                            key={order.id}
                                            className="bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors duration-200"
                                        >
                                            <td className="p-4">
                                                <div className="font-mono font-semibold text-blue-600 dark:text-blue-400">
                                                    #{order.order_number}
                                                </div>
                                            </td>
                                            <td className="p-4">
                                                <div className="max-w-xs">
                                                    <div className="font-medium text-gray-900 dark:text-white truncate" title={order.title}>
                                                        {order.title || 'No Title'}
                                                    </div>
                                                    {order.subject && (
                                                        <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                                            {order.subject}
                                                        </div>
                                                    )}
                                                </div>
                                            </td>
                                            <td className="p-4">
                                                <div className="flex items-center gap-2">
                                                    <FileText className="w-4 h-4 text-gray-500" />
                                                    <span className="font-medium">{order.pages}</span>
                                                </div>
                                            </td>
                                            <td className="p-4">
                                                <div className="flex items-center gap-2">
                                                    <Calendar className="w-4 h-4 text-gray-500" />
                                                    <span className="text-sm">
                                                        {order.dateposted ? new Date(order.dateposted).toLocaleDateString() : 'N/A'}
                                                    </span>
                                                </div>
                                            </td>
                                            <td className="p-4">
                                                <ExpiresIn deadline={order.deadline} />
                                            </td>
                                            <td className="p-4">
                                                <ExpiresIn deadline={order.writer_deadline} />
                                            </td>
                                            <td className="p-4">
                                                <div className="space-y-2">
                                                    <div className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusBadgeVariant(order.order_status)}`}>
                                                        {order.order_status?.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase()) || 'Unknown'}
                                                    </div>
                                                    <div className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getPaymentStatusBadgeVariant(order.payment_status)}`}>
                                                        {order.payment_status?.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase()) || 'Unknown'}
                                                    </div>
                                                </div>
                                            </td>
                                            <td className="p-4">
                                                <div className="flex items-center gap-2">
                                                    <DollarSign className="w-4 h-4 text-green-600" />
                                                    <span className="font-semibold text-green-600 dark:text-green-400">
                                                        ${parseFloat(order.net_amount || 0).toFixed(2)}
                                                    </span>
                                                </div>
                                            </td>
                                            <td className="p-4">
                                                <div className="flex justify-center gap-2">
                                                    <Link
                                                        href={route("admin.orders.show", order.id)}
                                                        className="p-2 text-blue-600 hover:text-blue-800 hover:bg-blue-50 dark:hover:bg-blue-900/20 rounded-lg transition-colors"
                                                        title="View Order"
                                                    >
                                                        <View className="w-4 h-4" />
                                                    </Link>
                                                    <Link
                                                        href={route("admin.orders.edit", order.id)}
                                                        className="p-2 text-green-600 hover:text-green-800 hover:bg-green-50 dark:hover:bg-green-900/20 rounded-lg transition-colors"
                                                        title="Edit Order"
                                                    >
                                                        <Edit className="w-4 h-4" />
                                                    </Link>
                                                    <button
                                                        onClick={() => handleDelete(order.id)}
                                                        className="p-2 text-red-600 hover:text-red-800 hover:bg-red-50 dark:hover:bg-red-900/20 rounded-lg transition-colors"
                                                        title="Delete Order"
                                                    >
                                                        <Trash2 className="w-4 h-4" />
                                                    </button>
                                                </div>
                                            </td>
                                        </tr>
                                    ))
                                ) : (
                                    <tr>
                                        <td colSpan="9" className="text-center p-8">
                                            <div className="flex flex-col items-center gap-3">
                                                <ListOrdered className="w-12 h-12 text-gray-400" />
                                                <p className="text-gray-500 dark:text-gray-400 text-lg">No orders found</p>
                                                <p className="text-gray-400 dark:text-gray-500 text-sm">
                                                    Try adjusting your search criteria or filters
                                                </p>
                                            </div>
                                        </td>
                                    </tr>
                                )}
                            </tbody>
                        </table>
                    </div>
                </div>

                {/* Enhanced Pagination */}
                {orders?.data && orders.data.length > 0 && (
                    <div className="mt-6">
                        <CleanPagination
                            links={orders.links}
                            total={orders.total}
                            perPage={orders.per_page}
                            currentPage={orders.current_page}
                        />
                    </div>
                )}
            </SectionContainer>

            {/* Mobile/Tablet Card View */}
            <SectionContainer
                className="lg:hidden bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:from-gray-800 dark:to-gray-900 border border-gray-300 dark:border-gray-600"
                title="Mobile Orders View"
                titleClassName="text-gray-800 dark:text-gray-200 font-semibold bg-white bg-opacity-30 dark:bg-gray-900 dark:bg-opacity-40 border-b border-gray-300 dark:border-gray-700"
            >
                {/* Mobile Controls */}
                <div className="space-y-4 mb-6">
                    <Link
                        href={route("admin.orders.create")}
                        className="flex items-center justify-center w-full gap-2 px-4 py-3 bg-gradient-to-r from-blue-600 to-cyan-600 text-white rounded-lg hover:from-blue-700 hover:to-cyan-700 transition-all duration-200 shadow-lg"
                    >
                        <Plus className="h-5 w-5" />
                        <span>Add New Order</span>
                    </Link>

                    <button
                        onClick={() => setShowFilters(!showFilters)}
                        className="flex items-center justify-center w-full gap-2 px-4 py-3 bg-white bg-opacity-70 dark:bg-gray-800 dark:bg-opacity-70 text-gray-700 dark:text-gray-300 rounded-lg border border-gray-300 dark:border-gray-600 hover:bg-white dark:hover:bg-gray-700 transition-colors"
                    >
                        <Filter className="h-5 w-5" />
                        {showFilters ? 'Hide' : 'Show'} Filters
                    </button>

                    {showFilters && (
                        <div className="space-y-4 p-4 bg-white bg-opacity-50 dark:bg-gray-800 dark:bg-opacity-50 rounded-lg border border-gray-200 dark:border-gray-700">
                            <form onSubmit={handleSearch} className="space-y-3">
                                <div className="relative">
                                    <input
                                        type="search"
                                        placeholder="Search orders..."
                                        value={search}
                                        onChange={handleSearchChange}
                                        className="w-full pl-10 pr-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-white"
                                    />
                                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500" />
                                </div>
                                <button
                                    type="submit"
                                    className="w-full px-4 py-3 bg-gradient-to-r from-blue-600 to-cyan-600 text-white rounded-lg hover:from-blue-700 hover:to-cyan-700 transition-all duration-200"
                                >
                                    Search
                                </button>
                            </form>

                            <div className="grid grid-cols-1 gap-3">
                                <select
                                    value={paymentStatus}
                                    onChange={(e) => handleFilterChange("paymentStatus", e.target.value)}
                                    className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-white"
                                >
                                    <option value="">All Payment Status</option>
                                    <option value="paid">Paid</option>
                                    <option value="unpaid">Unpaid</option>
                                    <option value="partially_paid">Partially Paid</option>
                                    <option value="refunded">Refunded</option>
                                </select>

                                <select
                                    value={orderStatus}
                                    onChange={(e) => handleFilterChange("orderStatus", e.target.value)}
                                    className="w-full px-4 py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-white"
                                >
                                    <option value="">All Order Status</option>
                                    <option value="bidding">Bidding</option>
                                    <option value="assigned">Assigned</option>
                                    <option value="in_progress">In Progress</option>
                                    <option value="submitted">Submitted</option>
                                    <option value="under_review">Under Review</option>
                                    <option value="approved">Approved</option>
                                    <option value="revision_requested">Revision Requested</option>
                                    <option value="completed">Completed</option>
                                    <option value="cancelled">Cancelled</option>
                                </select>
                            </div>
                        </div>
                    )}
                </div>

                {/* Mobile Orders List */}
                {orders?.data && orders.data.length > 0 ? (
                    <div className="space-y-4">
                        {orders.data.map((order) => (
                            <div
                                key={order.id}
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-4 space-y-3 border border-gray-200 dark:border-gray-700"
                            >
                                {/* Order Header */}
                                <div className="flex justify-between items-start">
                                    <div>
                                        <h3 className="font-semibold text-sm dark:text-white">
                                            Order #{order.order_number}
                                        </h3>
                                        <p className="text-xs text-gray-500 dark:text-gray-400">
                                            {order.dateposted ? new Date(order.dateposted).toLocaleDateString() : 'N/A'}
                                        </p>
                                    </div>
                                    <div className="text-right">
                                        <div className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusBadgeVariant(order.order_status)}`}>
                                            {order.order_status?.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase()) || 'Unknown'}
                                        </div>
                                    </div>
                                </div>

                                {/* Order Details */}
                                <div className="space-y-2">
                                    {order.title && (
                                        <div className="text-sm text-gray-900 dark:text-white font-medium">
                                            {order.title}
                                        </div>
                                    )}
                                    {order.subject && (
                                        <div className="text-xs text-gray-500 dark:text-gray-400">
                                            Subject: {order.subject}
                                        </div>
                                    )}
                                    <div className="grid grid-cols-2 gap-2 text-xs text-gray-700 dark:text-gray-300">
                                        <div className="flex items-center gap-1">
                                            <FileText className="w-3 h-3" />
                                            <span className="font-medium">Pages:</span> {order.pages}
                                        </div>
                                        <div className="flex items-center gap-1">
                                            <DollarSign className="w-3 h-3" />
                                            <span className="font-medium">Amount:</span> ${parseFloat(order.net_amount || 0).toFixed(2)}
                                        </div>
                                    </div>
                                    <div className="text-xs text-gray-500 dark:text-gray-400">
                                        <div className="flex items-center gap-1 mb-1">
                                            <Clock className="w-3 h-3" />
                                            <span>Client Deadline:</span>
                                        </div>
                                        <ExpiresIn deadline={order.deadline} />
                                    </div>
                                </div>

                                {/* Actions */}
                                <div className="flex justify-between items-center border-t pt-3 dark:border-gray-700">
                                    <div className="flex gap-3">
                                        <Link
                                            href={route("admin.orders.show", order.id)}
                                            className="text-blue-500 hover:text-blue-700 flex items-center gap-1"
                                        >
                                            <View className="w-4 h-4" />
                                            <span className="text-xs">View</span>
                                        </Link>
                                        <Link
                                            href={route("admin.orders.edit", order.id)}
                                            className="text-green-500 hover:text-green-700 flex items-center gap-1"
                                        >
                                            <Edit className="w-4 h-4" />
                                            <span className="text-xs">Edit</span>
                                        </Link>
                                    </div>
                                    <button
                                        onClick={() => handleDelete(order.id)}
                                        className="text-red-500 hover:text-red-700 flex items-center gap-1"
                                    >
                                        <Trash2 className="w-4 h-4" />
                                        <span className="text-xs">Delete</span>
                                    </button>
                                </div>
                            </div>
                        ))}
                    </div>
                ) : (
                    <div className="text-center p-6 bg-white dark:bg-gray-800 rounded-lg shadow-md border border-gray-200 dark:border-gray-700">
                        <ListOrdered className="mx-auto mb-4 w-12 h-12 text-gray-400" />
                        <p className="text-gray-600 dark:text-gray-300">No orders found</p>
                    </div>
                )}

                {/* Mobile Pagination */}
                {orders?.data && orders.data.length > 0 && (
                    <div className="mt-6">
                        <CleanPagination
                            links={orders.links}
                            total={orders.total}
                            perPage={orders.per_page}
                            currentPage={orders.current_page}
                        />
                    </div>
                )}
            </SectionContainer>
        </AdminLayout>
    );
}
