import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import OrderTimeline from '@/Components/OrderTimeline';
import StatusManagement from '@/Components/StatusManagement';
import AdminFileUpload from '@/Components/AdminFileUpload';
import AdminFileManagement from '@/Components/AdminFileManagement';
import { 
    FileText, 
    User, 
    Calendar, 
    DollarSign, 
    Clock,
    Edit3,
    Save,
    X,
    AlertCircle,
    CheckCircle,
    Eye,
    Download,
    Upload,
    MoveLeft,
    Vote,
    Notebook,
    Timer
} from 'lucide-react';
import axios from 'axios';

const AdminOrderShow = ({ order, auth }) => {
    // Safety check - if order is null, show error message
    if (!order) {
        return (
            <AdminLayout>
                <div className="py-12">
                    <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                        <div className="bg-white overflow-hidden shadow-sm sm:rounded-lg">
                            <div className="p-6 text-gray-900">
                                <h2 className="text-lg font-semibold text-red-600 mb-4">Order Not Found</h2>
                                <p className="text-gray-600 mb-4">The requested order could not be found or may have been deleted.</p>
                                <Link 
                                    href="/admin/orders" 
                                    className="inline-flex items-center px-4 py-2 bg-gray-800 border border-transparent rounded-md font-semibold text-xs text-white uppercase tracking-widest hover:bg-gray-700"
                                >
                                    <MoveLeft className="w-4 h-4 mr-2" />
                                    Back to Orders
                                </Link>
                            </div>
                        </div>
                    </div>
                </div>
            </AdminLayout>
        );
    }



    const [activeTab, setActiveTab] = useState('overview');
    
    // Modal states
    const [isModalOpen, setIsModalOpen] = useState(false);
    const [isWriterDeadlineModalOpen, setIsWriterDeadlineModalOpen] = useState(false);
    const [isPaymentModalOpen, setIsPaymentModalOpen] = useState(false);
    const [isOrderStatusModalOpen, setIsOrderStatusModalOpen] = useState(false);
    
    // Form states
    const [newDeadline, setNewDeadline] = useState('');
    const [newWriterDeadline, setNewWriterDeadline] = useState('');
    const [paymentStatus, setPaymentStatus] = useState(order.payment_status);
    const [orderStatus, setOrderStatus] = useState(order.order_status);
    
    // Loading states
    const [isDeadlineLoading, setIsDeadlineLoading] = useState(false);
    const [isOrderStatusLoading, setIsOrderStatusLoading] = useState(false);
    
    // Error states
    const [errorMessage, setErrorMessage] = useState('');
    const [paymentUpdateError, setPaymentUpdateError] = useState('');
    const [orderStatusUpdateError, setOrderStatusUpdateError] = useState('');
    
    // Success states
    const [successMessage, setSuccessMessage] = useState('');
    
    // Helper function to convert backend datetime to datetime-local format
    const toDatetimeLocalFormat = (datetime) => {
        if (!datetime) return '';
        // Convert "YYYY-MM-DD HH:MM:SS" to "YYYY-MM-DDTHH:MM"
        return datetime.substring(0, 16).replace(' ', 'T');
    };

    // Inline edit states (for new timeline interface)
    const [isEditingDeadline, setIsEditingDeadline] = useState(false);
    const [isEditingWriterDeadline, setIsEditingWriterDeadline] = useState(false);
    const [isEditingPaymentStatus, setIsEditingPaymentStatus] = useState(false);
    const [editForm, setEditForm] = useState({
        deadline: toDatetimeLocalFormat(order.deadline),
        writer_deadline: toDatetimeLocalFormat(order.writer_deadline),
        payment_status: order.payment_status
    });

    const tabs = [
        { id: 'overview', name: 'Overview', icon: Eye },
        { id: 'timeline', name: 'Timeline', icon: Clock },
        { id: 'files', name: 'Files', icon: FileText }
    ];

    const formatDate = (dateString) => {
        if (!dateString) return null;
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const formatCurrency = (amount, currency = 'USD') => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: currency
        }).format(amount);
    };

    const getStatusColor = (status) => {
        const colors = {
            'pending': 'bg-gray-100 text-gray-800',
            'bidding': 'bg-blue-100 text-blue-800',
            'assigned': 'bg-purple-100 text-purple-800',
            'in_progress': 'bg-orange-100 text-orange-800',
            'submitted': 'bg-indigo-100 text-indigo-800',
            'under_review': 'bg-yellow-100 text-yellow-800',
            'approved': 'bg-green-100 text-green-800',
            'revision_requested': 'bg-orange-100 text-orange-800',
            'cancelled': 'bg-red-100 text-red-800'
        };
        return colors[status] || 'bg-gray-100 text-gray-800';
    };

    const getPaymentStatusColor = (status) => {
        const colors = {
            'paid': 'bg-green-100 text-green-800',
            'unpaid': 'bg-red-100 text-red-800',
            'refunded': 'bg-gray-100 text-gray-800',
            'partially_refunded': 'bg-yellow-100 text-yellow-800'
        };
        return colors[status] || 'bg-gray-100 text-gray-800';
    };

    const handleSaveDeadline = async () => {
        try {
            // Convert datetime-local format (YYYY-MM-DDTHH:MM) to backend format (YYYY-MM-DD HH:MM:SS)
            const formattedDeadline = editForm.deadline.replace('T', ' ') + ':00';
            
            if (new Date(formattedDeadline) <= new Date(order.dateposted)) {
                console.error('New deadline must be after the order posted date');
                alert('New deadline must be after the order posted date');
                return;
            }
            
            await axios.post(`/admin/orders/${order.id}/update-deadline`, {
                deadline: formattedDeadline,
            });
            
            setIsEditingDeadline(false);
            window.location.reload();
        } catch (error) {
            console.error('Failed to update deadline:', error);
            console.error('Error response:', error.response?.data);
            console.error('Validation errors:', error.response?.data?.errors);
            alert(error.response?.data?.message || 'Failed to update deadline');
        }
    };

    const handleSaveWriterDeadline = async () => {
        try {
            // Convert datetime-local format (YYYY-MM-DDTHH:MM) to backend format (YYYY-MM-DD HH:MM:SS)
            const formattedDeadline = editForm.writer_deadline.replace('T', ' ') + ':00';
            
            if (new Date(formattedDeadline) <= new Date(order.dateposted)) {
                console.error('New writer deadline must be after the order posted date');
                alert('New writer deadline must be after the order posted date');
                return;
            }
            
            await axios.post(`/admin/orders/${order.id}/update-writer-deadline`, {
                writer_deadline: formattedDeadline,
            });
            
            setIsEditingWriterDeadline(false);
            window.location.reload();
        } catch (error) {
            console.error('Failed to update writer deadline:', error);
            console.error('Error response:', error.response?.data);
            console.error('Validation errors:', error.response?.data?.errors);
            alert(error.response?.data?.message || 'Failed to update writer deadline');
        }
    };

    const handleSavePaymentStatus = async () => {
        try {
            axios.post(`/admin/orders/${order.id}/update-payment-status`, {
                payment_status: editForm.payment_status,
            })
            .then((response) => {
                // Update the order object with the new payment status
                order.payment_status = editForm.payment_status;
                setIsEditingPaymentStatus(false);
                // Refresh the page to show the updated status
                window.location.reload();
            })
            .catch((error) => {
                console.error('Failed to update payment status:', error);
            });
        } catch (error) {
            console.error('Failed to update payment status:', error);
        }
    };

    // Helper functions from old file
    const capitalize = (string) => {
        return string.charAt(0).toUpperCase() + string.slice(1);
    };

    const isFirefox = () => {
        return navigator.userAgent.toLowerCase().indexOf('firefox') > -1;
    };

    const formatDateForInput = (dateString) => {
        if (!dateString) return '';
        const dateParts = dateString.split(':');
        if (dateParts.length > 2) {
            dateString = dateParts.slice(0, 2).join(':');
        }
        return dateString.substring(0, 16).replace(' ', 'T');
    };

    // Modal functions
    const openModal = () => {
        setIsModalOpen(true);
        setNewDeadline(formatDateForInput(order.deadline));
        setErrorMessage('');
    };

    const closeModal = () => {
        setIsModalOpen(false);
    };

    const handleDeadlineChange = (e) => {
        setNewDeadline(e.target.value);
    };

    const updateDeadline = () => {
        const formattedDeadline = newDeadline.replace('T', ' ') + ':00';
        
        if (new Date(formattedDeadline) <= new Date(order.dateposted)) {
            setErrorMessage('New deadline must be after the order posted date');
            return;
        }
        
        setIsDeadlineLoading(true);
        
        axios.post(`/admin/orders/${order.id}/update-deadline`, {
            deadline: formattedDeadline,
        })
        .then((response) => {
            order.deadline = formattedDeadline;
            setIsDeadlineLoading(false);
            closeModal();
            window.location.reload();
        })
        .catch((error) => {
            setIsDeadlineLoading(false);
            setErrorMessage(error.response?.data?.message || 'Failed to update deadline');
        });
    };

    const openWriterModal = () => {
        setIsWriterDeadlineModalOpen(true);
        setNewWriterDeadline(formatDateForInput(order.writer_deadline));
        setErrorMessage('');
    };

    const closeWriterModal = () => {
        setIsWriterDeadlineModalOpen(false);
    };

    const handleWriterDeadlineChange = (e) => {
        setNewWriterDeadline(e.target.value);
    };

    const updateWriterDeadline = () => {
        const formattedDeadline = newWriterDeadline.replace('T', ' ') + ':00';
        
        if (new Date(formattedDeadline) <= new Date(order.dateposted)) {
            setErrorMessage('New writer deadline must be after the order posted date');
            return;
        }
        
        setIsDeadlineLoading(true);
        
        axios.post(`/admin/orders/${order.id}/update-writer-deadline`, {
            writer_deadline: formattedDeadline,
        })
        .then((response) => {
            order.writer_deadline = formattedDeadline;
            setIsDeadlineLoading(false);
            closeWriterModal();
            window.location.reload();
        })
        .catch((error) => {
            setIsDeadlineLoading(false);
            setErrorMessage(error.response?.data?.message || 'Failed to update writer deadline');
        });
    };

    const openPaymentModal = () => {
        setIsPaymentModalOpen(true);
        setPaymentStatus(order.payment_status);
        setPaymentUpdateError('');
    };

    const closePaymentModal = () => {
        setIsPaymentModalOpen(false);
    };

    const handlePaymentStatusChange = (e) => {
        setPaymentStatus(e.target.value);
    };

    const updatePaymentStatus = () => {
        axios
            .post(`/admin/orders/${order.id}/update-payment-status`, {
                payment_status: paymentStatus,
            })
            .then((response) => {
                // Update the order object with the new payment status
                order.payment_status = paymentStatus;
                closePaymentModal();
                // Refresh the page to show the updated status
                window.location.reload();
            })
            .catch((error) => {
                setPaymentUpdateError(
                    error.response?.data?.message ||
                        "Failed to update payment status"
                );
            });
    };

    const openOrderStatusModal = () => {
        setIsOrderStatusModalOpen(true);
        setOrderStatus(order.order_status);
        setOrderStatusUpdateError('');
    };

    const closeOrderStatusModal = () => {
        setIsOrderStatusModalOpen(false);
    };

    const handleOrderStatusChange = (e) => {
        setOrderStatus(e.target.value);
    };

    const updateOrderStatus = () => {
        setIsOrderStatusLoading(true); // Set loading state to true when starting the request

        axios
            .post(`/admin/orders/${order.id}/update-order-status`, {
                order_status: orderStatus,
            })
            .then((response) => {
                // Update the order object with the new order status
                order.order_status = orderStatus;
                setIsOrderStatusLoading(false); // Reset loading state
                closeOrderStatusModal();
                // Refresh the page to show the updated status
                window.location.reload();
            })
            .catch((error) => {
                setIsOrderStatusLoading(false); // Reset loading state even on error
                setOrderStatusUpdateError(
                    error.response?.data?.message ||
                        "Failed to update order status"
                );
            });
    };

    const getNextPossibleStatuses = () => {
        // This would be provided by the backend based on current status
        const statusTransitions = {
            'pending': ['bidding', 'cancelled'],
            'bidding': ['assigned', 'cancelled'],
            'assigned': ['in_progress', 'cancelled'],
            'in_progress': ['submitted', 'cancelled'],
            'submitted': ['under_review', 'revision_requested'],
            'under_review': ['approved', 'revision_requested'],
            'revision_requested': ['in_progress', 'cancelled'],
            'approved': [],
            'cancelled': []
        };
        return statusTransitions[order.order_status] || [];
    };

    return (
        <AdminLayout
            user={auth.user}
            header={<h2 className="font-semibold text-xl text-gray-800 dark:text-gray-200 leading-tight">Order Details</h2>}
        >
            <Head title={`Order #${order.order_number}`} />

            <div className="w-full mx-auto px-2 sm:px-2 lg:px-8 py-2 space-y-6 dark:bg-gray-900">
                {/* Back to orders section */}
                <div className="flex justify-end items-center mb-6">
                    <Link
                        href="/admin/orders"
                        className="flex items-center px-3 py-1.5 sm:px-4 sm:py-2 text-indigo-600 dark:text-gray-100 bg-indigo-50 dark:bg-indigo-900 hover:bg-indigo-100 dark:hover:bg-indigo-800 rounded-full shadow-sm border border-indigo-200 dark:border-indigo-700 transition-colors duration-200 text-sm"
                    >
                        <MoveLeft className="mr-1 h-3 w-3 sm:h-4 sm:w-4" />
                        <span className="font-medium">All Orders</span>
                    </Link>
                </div>

                {/* Success Message */}
                {successMessage && (
                    <div className="mb-4 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4">
                        <div className="flex items-center space-x-2 text-green-800 dark:text-green-200">
                            <CheckCircle className="w-5 h-5" />
                            <span className="font-medium">{successMessage}</span>
                        </div>
                    </div>
                )}

                {/* Header Section */}
                <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                    <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center space-y-4 sm:space-y-0">
                        <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-4">
                            <h1 className="text-lg sm:text-lg font-bold dark:text-white">
                                Order #{order.order_number}
                            </h1>
                            <div className="flex flex-wrap gap-2">
                                <span className={`px-3 py-1 rounded-full text-sm font-medium flex items-center ${
                                    order.payment_status === 'paid'
                                        ? 'bg-green-500 text-gray-50 dark:bg-green-800 dark:text-green-100'
                                        : 'bg-red-500 text-red-800 dark:bg-red-800 dark:text-red-100'
                                }`}>
                                    <Vote className="mr-1 h-4 w-4" />
                                    <span>{capitalize(order.payment_status)}</span>
                                </span>

                                <span className={`px-3 py-1 rounded-full text-sm font-medium flex items-center ${getStatusColor(order.order_status)}`}>
                                    <Clock className="w-4 h-4 mr-1" />
                                    <span>{capitalize(order.order_status)}</span>
                                </span>
                            </div>
                        </div>
                        <div className="flex flex-wrap gap-2 w-full sm:w-auto">
                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                Posted: {formatDate(order.dateposted)}
                            </div>
                        </div>
                    </div>
                </div>

                {/* Title Section */}
                <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 dark:bg-gray-800 rounded-lg shadow-sm p-4 sm:p-6">
                    <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center space-y-4 sm:space-y-0">
                        <div className="flex flex-col sm:flex-row sm:items-center gap-2 sm:gap-4">
                            <p className="text-xl sm:text-2xl font-bold dark:text-white">
                                Title: {order.title}
                            </p>
                        </div>
                    </div>
                </div>


             

                {/* Tabs */}
                <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                    <div className="border-b border-gray-200 dark:border-gray-700">
                            <nav className="-mb-px flex space-x-8 px-6">
                                {tabs.map((tab) => {
                                    const Icon = tab.icon;
                                    return (
                                        <button
                                            key={tab.id}
                                            onClick={() => setActiveTab(tab.id)}
                                            className={`py-4 px-1 border-b-2 font-medium text-sm flex items-center ${
                                                activeTab === tab.id
                                                    ? 'border-blue-500 text-blue-600 dark:text-blue-400'
                                                    : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 dark:text-gray-400 dark:hover:text-gray-300'
                                            }`}
                                        >
                                            <Icon className="w-4 h-4 mr-2" />
                                            {tab.name}
                                        </button>
                                    );
                                })}
                            </nav>
                        </div>

                        <div className="p-6">
                            {/* Overview Tab */}
                            {activeTab === 'overview' && (
                                <div className="grid grid-cols-1 lg:grid-cols-3 gap-6">
                                    {/* Main Content */}
                                    <div className="lg:col-span-2 space-y-6">
                                        {/* Order Details */}
                                        <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-lg p-6">
                                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                                Order Information
                                            </h3>
                                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Title:</span>
                                                    <p className="text-gray-900 dark:text-white font-medium">{order.title}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Type:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.type_of_paper}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Subject:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.subject}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Academic Level:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.academic_level}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Pages:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.pages}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Urgency:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.urgency}</p>
                                                </div>
                                            </div>
                                            
                                            <div className="mt-4">
                                                <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Instructions:</span>
                                                <p className="text-gray-900 dark:text-white mt-1 whitespace-pre-wrap">{order.instructions}</p>
                                            </div>
                                        </div>

                                        {/* Client Information */}
                                        <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-lg p-6">
                                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                                Client Information
                                            </h3>
                                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Name:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.user?.name}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Email:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.user?.email}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Phone:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.user?.phone || 'Not provided'}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Country:</span>
                                                    <p className="text-gray-900 dark:text-white">{order.user?.country || 'Not specified'}</p>
                                                </div>
                                            </div>
                                        </div>

                                        {/* Writer Information */}
                                        {order.writer && (
                                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-lg p-6">
                                                <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                                    Writer Information
                                                </h3>
                                                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Name:</span>
                                                        <p className="text-gray-900 dark:text-white">{order.writer?.name}</p>
                                                    </div>
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Email:</span>
                                                        <p className="text-gray-900 dark:text-white">{order.writer?.email}</p>
                                                    </div>
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Level:</span>
                                                        <p className="text-gray-900 dark:text-white">{order.writer?.writer_profile?.level?.name || 'Not specified'}</p>
                                                    </div>
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Experience:</span>
                                                        <p className="text-gray-900 dark:text-white">{order.writer?.writer_profile?.experience_years || 'Not specified'} years</p>
                                                    </div>
                                                </div>
                                            </div>
                                        )}

                                        {/* Payment Information */}
                                        <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-lg p-6">
                                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                                Payment Information
                                            </h3>
                                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Order Amount:</span>
                                                    <p className="text-gray-900 dark:text-white font-medium">{formatCurrency(order.order_amount, order.currency)}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Net Amount:</span>
                                                    <p className="text-gray-900 dark:text-white font-medium">{formatCurrency(order.net_amount, order.currency)}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Writer Amount:</span>
                                                    <p className="text-gray-900 dark:text-white">{formatCurrency(order.writer_amount, order.currency)}</p>
                                                </div>
                                                <div>
                                                    <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Discount:</span>
                                                    <p className="text-gray-900 dark:text-white">{formatCurrency(order.discount, order.currency)}</p>
                                                </div>
                                                {order.payment_method && (
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Payment Method:</span>
                                                        <p className="text-gray-900 dark:text-white">{order.payment_method}</p>
                                                    </div>
                                                )}
                                                {order.payment_date && (
                                                    <div>
                                                        <span className="text-sm font-medium text-gray-500 dark:text-gray-400">Payment Date:</span>
                                                        <p className="text-gray-900 dark:text-white">{formatDate(order.payment_date)}</p>
                                                    </div>
                                                )}
                                            </div>
                                        </div>

                                        {/* Important Dates */}
                                        <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 rounded-lg p-6">
                                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                                Important Dates
                                            </h3>
                                            <div className="space-y-4">
                                                {/* Client Deadline */}
                                                <div className="flex items-center justify-between">
                                                    <div className="flex items-center">
                                                        <Calendar className="w-4 h-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Client Deadline:</span>
                                                    </div>
                                                    <div className="flex items-center">
                                                        {isEditingDeadline ? (
                                                            <div className="flex items-center space-x-2">
                                                                <input
                                                                    type="datetime-local"
                                                                    value={editForm.deadline}
                                                                    onChange={(e) => setEditForm({...editForm, deadline: e.target.value})}
                                                                    className="px-2 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded dark:bg-gray-700 dark:text-white"
                                                                />
                                                                <button
                                                                    onClick={handleSaveDeadline}
                                                                    className="p-1 text-green-600 hover:text-green-700"
                                                                >
                                                                    <Save className="w-4 h-4" />
                                                                </button>
                                                                <button
                                                                    onClick={() => setIsEditingDeadline(false)}
                                                                    className="p-1 text-gray-600 hover:text-gray-700"
                                                                >
                                                                    <X className="w-4 h-4" />
                                                                </button>
                                                            </div>
                                                        ) : (
                                                            <div className="flex items-center">
                                                                <span className="text-gray-900 dark:text-white">{formatDate(order.deadline)}</span>
                                                                <button
                                                                    onClick={() => setIsEditingDeadline(true)}
                                                                    className="ml-2 p-1 text-gray-600 hover:text-gray-700"
                                                                >
                                                                    <Edit3 className="w-4 h-4" />
                                                                </button>
                                                            </div>
                                                        )}
                                                    </div>
                                                </div>

                                                {/* Writer Deadline */}
                                                <div className="flex items-center justify-between">
                                                    <div className="flex items-center">
                                                        <Clock className="w-4 h-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Writer Deadline:</span>
                                                    </div>
                                                    <div className="flex items-center">
                                                        {isEditingWriterDeadline ? (
                                                            <div className="flex items-center space-x-2">
                                                                <input
                                                                    type="datetime-local"
                                                                    value={editForm.writer_deadline}
                                                                    onChange={(e) => setEditForm({...editForm, writer_deadline: e.target.value})}
                                                                    className="px-2 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded dark:bg-gray-700 dark:text-white"
                                                                />
                                                                <button
                                                                    onClick={handleSaveWriterDeadline}
                                                                    className="p-1 text-green-600 hover:text-green-700"
                                                                >
                                                                    <Save className="w-4 h-4" />
                                                                </button>
                                                                <button
                                                                    onClick={() => setIsEditingWriterDeadline(false)}
                                                                    className="p-1 text-gray-600 hover:text-gray-700"
                                                                >
                                                                    <X className="w-4 h-4" />
                                                                </button>
                                                            </div>
                                                        ) : (
                                                            <div className="flex items-center">
                                                                <span className="text-gray-900 dark:text-white">
                                                                    {order.writer_deadline ? formatDate(order.writer_deadline) : 'Not set'}
                                                                </span>
                                                                <button
                                                                    onClick={() => setIsEditingWriterDeadline(true)}
                                                                    className="ml-2 p-1 text-gray-600 hover:text-gray-700"
                                                                >
                                                                    <Edit3 className="w-4 h-4" />
                                                                </button>
                                                            </div>
                                                        )}
                                                    </div>
                                                </div>

                                                {/* Payment Status */}
                                                <div className="flex items-center justify-between">
                                                    <div className="flex items-center">
                                                        <DollarSign className="w-4 h-4 text-gray-500 dark:text-gray-400 mr-2" />
                                                        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">Payment Status:</span>
                                                    </div>
                                                    <div className="flex items-center">
                                                        {isEditingPaymentStatus ? (
                                                            <div className="flex items-center space-x-2">
                                                                <select
                                                                    value={editForm.payment_status}
                                                                    onChange={(e) => setEditForm({...editForm, payment_status: e.target.value})}
                                                                    className="px-2 py-1 text-sm border border-gray-300 dark:border-gray-600 rounded dark:bg-gray-700 dark:text-white"
                                                                >
                                                                    <option value="paid">Paid</option>
                                                                    <option value="unpaid">Unpaid</option>
                                                                    <option value="refunded">Refunded</option>
                                                                    <option value="partially_refunded">Partially Refunded</option>
                                                                </select>
                                                                <button
                                                                    onClick={handleSavePaymentStatus}
                                                                    className="p-1 text-green-600 hover:text-green-700"
                                                                >
                                                                    <Save className="w-4 h-4" />
                                                                </button>
                                                                <button
                                                                    onClick={() => setIsEditingPaymentStatus(false)}
                                                                    className="p-1 text-gray-600 hover:text-gray-700"
                                                                >
                                                                    <X className="w-4 h-4" />
                                                                </button>
                                                            </div>
                                                        ) : (
                                                            <div className="flex items-center">
                                                                <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${getPaymentStatusColor(order.payment_status)}`}>
                                                                    {order.payment_status.toUpperCase()}
                                                                </span>
                                                                <button
                                                                    onClick={() => setIsEditingPaymentStatus(true)}
                                                                    className="ml-2 p-1 text-gray-600 hover:text-gray-700"
                                                                >
                                                                    <Edit3 className="w-4 h-4" />
                                                                </button>
                                                            </div>
                                                        )}
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Sidebar */}
                                    <div className="space-y-6">
                                        <StatusManagement 
                                            order={order} 
                                            nextPossibleStatuses={getNextPossibleStatuses()}
                                        />
                                    </div>
                                </div>
                            )}

                            {/* Timeline Tab */}
                            {activeTab === 'timeline' && (
                                <OrderTimeline order={order} />
                            )}

                            {/* Files Tab */}
                            {activeTab === 'files' && (
                                <div className="space-y-6">
                                    {/* Status Change Notice */}
                                    {order.order_status === 'submitted' && order.writer && (
                                        <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                                            <div className="flex items-center space-x-2 text-blue-800 dark:text-blue-200">
                                                <AlertCircle className="w-5 h-5" />
                                                <div>
                                                    <p className="font-medium">Order Status Change Notice</p>
                                                    <p className="text-sm">
                                                        When you access or download writer-submitted files, the order status will automatically change from "Submitted" to "Under Review".
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    )}
                                    
                                    <AdminFileUpload 
                                        orderId={order.id} 
                                        onFileUploaded={(files) => {
                                            // Refresh the page to show new files
                                            window.location.reload();
                                        }}
                                    />
                                    

                                    <AdminFileManagement 
                                        orderId={order.id}
                                        files={order.orderFiles || []}
                                        orderStatus={order.order_status}
                                        onFileDeleted={(file) => {
                                            // Remove file from local state
                                            const updatedFiles = order.orderFiles.filter(f => f.id !== file.id);
                                            order.orderFiles = updatedFiles;
                                        }}
                                        onStatusChange={(newStatus) => {
                                            // Update order status locally
                                            order.order_status = newStatus;
                                            // Show success message
                                            setSuccessMessage(`Order status successfully changed to "${newStatus}"`);
                                            // Clear success message after 5 seconds
                                            setTimeout(() => setSuccessMessage(''), 5000);
                                            // Force re-render
                                            setActiveTab(activeTab);
                                        }}
                                        isAdmin={true}
                                    />
                                </div>
                            )}
                        </div>
                    </div>
                </div>




                {/* Modals */}
                {/* Client Deadline Modal */}
                {isModalOpen && (
                    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
                        <div className="bg-white dark:bg-gray-800 rounded-lg p-6 w-full max-w-md">
                            <h3 className="text-lg font-semibold mb-4 dark:text-white">Adjust Deadline</h3>
                            {errorMessage && (
                                <div className="mb-4 p-3 bg-red-100 text-red-700 rounded-md">{errorMessage}</div>
                            )}
                            <div className="mb-4">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Current Deadline</label>
                                <p className="text-gray-600 dark:text-gray-400">{formatDate(order.deadline)}</p>
                            </div>
                            <div className="mb-4">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">New Deadline</label>
                                {isFirefox() ? (
                                    <div className="grid grid-cols-2 gap-2">
                                        <input
                                            type="date"
                                            value={newDeadline.split('T')[0]}
                                            onChange={(e) => setNewDeadline(`${e.target.value}T${newDeadline.split('T')[1] || '00:00'}`)}
                                            className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                        />
                                        <input
                                            type="time"
                                            value={newDeadline.split('T')[1] || ''}
                                            onChange={(e) => setNewDeadline(`${newDeadline.split('T')[0]}T${e.target.value}`)}
                                            className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                        />
                                    </div>
                                ) : (
                                    <input
                                        type="datetime-local"
                                        value={newDeadline}
                                        onChange={handleDeadlineChange}
                                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                    />
                                )}
                            </div>
                            <div className="flex justify-end space-x-3">
                                <button onClick={closeModal} className="px-4 py-2 bg-gray-200 text-gray-800 rounded-md hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600">
                                    Cancel
                                </button>
                                <button
                                    onClick={updateDeadline}
                                    disabled={isDeadlineLoading}
                                    className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-800 transition-colors flex items-center"
                                >
                                    {isDeadlineLoading ? (
                                        <>
                                            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                            Updating...
                                        </>
                                    ) : (
                                        'Update Deadline'
                                    )}
                                </button>
                            </div>
                        </div>
                    </div>
                )}

                {/* Writer Deadline Modal */}
                {isWriterDeadlineModalOpen && (
                    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
                        <div className="bg-white dark:bg-gray-800 rounded-lg p-6 w-full max-w-md">
                            <h3 className="text-lg font-semibold mb-4 dark:text-white">Adjust Writer Deadline</h3>
                            {errorMessage && (
                                <div className="mb-4 p-3 bg-red-100 text-red-700 rounded-md">{errorMessage}</div>
                            )}
                            <div className="mb-4">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Current Writer Deadline</label>
                                <p className="text-gray-600 dark:text-gray-400">{formatDate(order.writer_deadline)}</p>
                            </div>
                            <div className="mb-4">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">New Writer Deadline</label>
                                {isFirefox() ? (
                                    <div className="grid grid-cols-2 gap-2">
                                        <input
                                            type="date"
                                            value={newWriterDeadline.split('T')[0]}
                                            onChange={(e) => setNewWriterDeadline(`${e.target.value}T${newWriterDeadline.split('T')[1] || '00:00'}`)}
                                            className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                        />
                                        <input
                                            type="time"
                                            value={newWriterDeadline.split('T')[1] || ''}
                                            onChange={(e) => setNewWriterDeadline(`${newWriterDeadline.split('T')[0]}T${e.target.value}`)}
                                            className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                        />
                                    </div>
                                ) : (
                                    <input
                                        type="datetime-local"
                                        value={newWriterDeadline}
                                        onChange={handleWriterDeadlineChange}
                                        className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                    />
                                )}
                            </div>
                            <div className="flex justify-end space-x-3">
                                <button onClick={closeWriterModal} className="px-4 py-2 bg-gray-200 text-gray-800 rounded-md hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600">
                                    Cancel
                                </button>
                                <button
                                    onClick={updateWriterDeadline}
                                    disabled={isDeadlineLoading}
                                    className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-800 transition-colors flex items-center"
                                >
                                    {isDeadlineLoading ? (
                                        <>
                                            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                            Updating...
                                        </>
                                    ) : (
                                        'Update Writer Deadline'
                                    )}
                                </button>
                            </div>
                        </div>
                    </div>
                )}

                {/* Payment Status Modal */}
                {isPaymentModalOpen && (
                    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
                        <div className="bg-white dark:bg-gray-800 rounded-lg p-6 w-full max-w-md">
                            <h3 className="text-lg font-semibold mb-4 dark:text-white">Change Payment Status</h3>
                            {paymentUpdateError && (
                                <div className="mb-4 p-3 bg-red-100 text-red-700 rounded-md">{paymentUpdateError}</div>
                            )}
                            <div className="mb-4">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Current Status</label>
                                <p className="text-gray-600 dark:text-gray-400">{order.payment_status}</p>
                            </div>
                            <div className="mb-4">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">New Status</label>
                                <select
                                    value={paymentStatus}
                                    onChange={handlePaymentStatusChange}
                                    className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white"
                                >
                                    <option value="paid">Paid</option>
                                    <option value="unpaid">Unpaid</option>
                                    <option value="refunded">Refunded</option>
                                    <option value="partially_refunded">Partially Refunded</option>
                                </select>
                            </div>
                            <div className="flex justify-end space-x-3">
                                <button onClick={closePaymentModal} className="px-4 py-2 bg-gray-200 text-gray-800 rounded-md hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600">
                                    Cancel
                                </button>
                                <button onClick={updatePaymentStatus} className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-800">
                                    Update Status
                                </button>
                            </div>
                        </div>
                    </div>
                )}

                {/* Order Status Modal */}
                {isOrderStatusModalOpen && (
                    <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
                        <div className="bg-white dark:bg-gray-800 rounded-lg p-6 w-full max-w-md mx-4">
                            <h3 className="text-lg font-semibold mb-4 dark:text-white">Change Order Status</h3>
                            {orderStatusUpdateError && (
                                <div className="mb-4 p-3 bg-red-100 text-red-700 rounded-md">{orderStatusUpdateError}</div>
                            )}
                            <div className="mb-4">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">Current Status</label>
                                <p className="text-gray-600 dark:text-gray-400">{capitalize(order.order_status)}</p>
                            </div>
                            <div className="mb-4">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">New Status</label>
                                <select
                                    value={orderStatus}
                                    onChange={handleOrderStatusChange}
                                    className="w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md dark:bg-gray-700 dark:text-white focus:ring-blue-500 focus:border-blue-500"
                                >
                                    <option value="bidding">Bidding</option>
                                    <option value="assigned">Assigned</option>
                                    <option value="completed">Completed</option>
                                    <option value="revision">Revision</option>
                                    <option value="editing">Editing</option>
                                    <option value="cancelled">Cancelled</option>
                                    <option value="approved">Approved</option>
                                </select>
                            </div>
                            <div className="flex justify-end space-x-3">
                                <button onClick={closeOrderStatusModal} className="px-4 py-2 bg-gray-200 text-gray-800 rounded-md hover:bg-gray-300 dark:bg-gray-700 dark:text-gray-200 dark:hover:bg-gray-600 transition-colors">
                                    Cancel
                                </button>
                                <button
                                    onClick={updateOrderStatus}
                                    disabled={isOrderStatusLoading}
                                    className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-800 transition-colors flex items-center"
                                >
                                    {isOrderStatusLoading ? (
                                        <>
                                            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                            Updating...
                                        </>
                                    ) : (
                                        'Update Status'
                                    )}
                                </button>
                            </div>
                        </div>
                    </div>
                )}
            
        </AdminLayout>
    );
};

export default AdminOrderShow;
