import React, { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import InputError from "@/Components/InputError";
import InputLabel from "@/Components/InputLabel";
import TextInput from "@/Components/TextInput";
import SelectInput from "@/Components/SelectInput";
import Checkbox from "@/Components/Checkbox";
import { CreditCard, ArrowLeft, Save, AlertCircle, Info } from "lucide-react";

const Create = ({ availableServiceTypes }) => {
    const [selectedType, setSelectedType] = useState("");

    const { data, setData, post, processing, errors, reset } = useForm({
        name: "",
        display_name: "",
        is_active: false,
        config_data: {},
    });

    // Handle form submit
    const handleSubmit = (e) => {
        e.preventDefault();
        post(route("admin.payment-methods.store"), {
            onSuccess: () => reset(),
        });
    };

    // Handle payment method type selection
    const handleTypeChange = (e) => {
        const type = e.target.value;
        setSelectedType(type);
        setData("name", type);

        // Initialize config data based on the selected type
        if (type === "paypal") {
            setData("config_data", {
                mode: "sandbox",
                client_id: "",
                client_secret: "",
                live_client_id: "",
                live_client_secret: "",
            });
        } else if (type === "stripe") {
            setData("config_data", {
                publishable_key: "",
                secret_key: "",
                webhook_secret: "",
            });
        } else if (type === "paypal_direct") {
            setData("config_data", {
                mode: "sandbox",
                business_email: "",
                sandbox_business_email: "",
                form_action_sandbox:
                    "https://www.sandbox.paypal.com/cgi-bin/webscr",
                form_action_live: "https://www.paypal.com/cgi-bin/webscr",
                ipn_verify_sandbox:
                    "https://ipnpb.sandbox.paypal.com/cgi-bin/webscr",
                ipn_verify_live: "https://ipnpb.paypal.com/cgi-bin/webscr",
            });
        }
    };

    // Render config fields based on the selected payment method type
    const renderConfigFields = () => {
        if (!selectedType) return null;

        if (selectedType === "paypal") {
            return (
                <div className="space-y-4">
                    <InputLabel value="Mode" />
                    <SelectInput
                        value={data.config_data.mode || "sandbox"}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                mode: e.target.value,
                            })
                        }
                    >
                        <option value="sandbox">Sandbox</option>
                        <option value="live">Live</option>
                    </SelectInput>

                    <InputLabel value="Sandbox Client ID" />
                    <TextInput
                        value={data.config_data.client_id || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                client_id: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Sandbox Client Secret" />
                    <TextInput
                        value={data.config_data.client_secret || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                client_secret: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Live Client ID" />
                    <TextInput
                        value={data.config_data.live_client_id || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                live_client_id: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Live Client Secret" />
                    <TextInput
                        value={data.config_data.live_client_secret || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                live_client_secret: e.target.value,
                            })
                        }
                    />
                </div>
            );
        }

        if (selectedType === "stripe") {
            return (
                <div className="space-y-4">
                    <InputLabel value="Publishable Key" />
                    <TextInput
                        value={data.config_data.publishable_key || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                publishable_key: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Secret Key" />
                    <TextInput
                        value={data.config_data.secret_key || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                secret_key: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Webhook Secret" />
                    <TextInput
                        value={data.config_data.webhook_secret || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                webhook_secret: e.target.value,
                            })
                        }
                    />
                </div>
            );
        }

        if (selectedType === "paypal_direct") {
            return (
                <div className="space-y-4">
                    <InputLabel value="Mode" />
                    <SelectInput
                        value={data.config_data.mode || "sandbox"}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                mode: e.target.value,
                            })
                        }
                    >
                        <option value="sandbox">Sandbox</option>
                        <option value="live">Live</option>
                    </SelectInput>

                    <InputLabel value="Sandbox Business Email" />
                    <TextInput
                        value={data.config_data.sandbox_business_email || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                sandbox_business_email: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Live Business Email" />
                    <TextInput
                        value={data.config_data.business_email || ""}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                business_email: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Sandbox Form Action URL" />
                    <TextInput
                        value={data.config_data.form_action_sandbox || "https://www.sandbox.paypal.com/cgi-bin/webscr"}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                form_action_sandbox: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Live Form Action URL" />
                    <TextInput
                        value={data.config_data.form_action_live || "https://www.paypal.com/cgi-bin/webscr"}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                form_action_live: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Sandbox IPN Verify URL" />
                    <TextInput
                        value={data.config_data.ipn_verify_sandbox || "https://ipnpb.sandbox.paypal.com/cgi-bin/webscr"}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                ipn_verify_sandbox: e.target.value,
                            })
                        }
                    />

                    <InputLabel value="Live IPN Verify URL" />
                    <TextInput
                        value={data.config_data.ipn_verify_live || "https://ipnpb.paypal.com/cgi-bin/webscr"}
                        onChange={(e) =>
                            setData("config_data", {
                                ...data.config_data,
                                ipn_verify_live: e.target.value,
                            })
                        }
                    />
                </div>
            );
        }

        return null;
    };

    return (
        <AdminLayout>
            <Head title="Create Payment Method" />

            <div className="mb-4 flex items-center space-x-2">
                <Link
                    href={route("admin.payment-methods.index")}
                    className="inline-flex items-center px-3 py-1.5 text-sm text-blue-600 bg-blue-50 hover:bg-blue-100 rounded-full"
                >
                    <ArrowLeft className="w-4 h-4 mr-2" /> Back
                </Link>
            </div>

            <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                <form onSubmit={handleSubmit} className="p-6">
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <div className="space-y-6">
                            <div>
                                <InputLabel
                                    htmlFor="payment_type"
                                    value="Payment Method Type"
                                />
                                <SelectInput
                                    id="payment_type"
                                    name="payment_type"
                                    className="mt-1 block w-full"
                                    value={selectedType}
                                    onChange={handleTypeChange}
                                    required
                                >
                                    <option value="">Select a payment method...</option>
                                    {Object.entries(
                                        availableServiceTypes || {}
                                    ).map(([key, label]) => (
                                        <option key={key} value={key}>
                                            {label}
                                        </option>
                                    ))}
                                </SelectInput>
                                <InputError message={errors.name} className="mt-2" />
                            </div>

                            <div>
                                <InputLabel htmlFor="display_name" value="Display Name" />
                                <TextInput
                                    id="display_name"
                                    value={data.display_name}
                                    onChange={(e) => setData("display_name", e.target.value)}
                                    className="mt-1 block w-full"
                                    required
                                />
                                <InputError message={errors.display_name} className="mt-2" />
                            </div>

                            <div className="flex items-center space-x-2">
                                <Checkbox
                                    id="is_active"
                                    checked={data.is_active}
                                    onChange={(e) => setData("is_active", e.target.checked)}
                                />
                                <InputLabel htmlFor="is_active" value="Active" />
                            </div>
                        </div>

                        <div className="space-y-6">
                            <InputLabel value="Configuration" />
                            {renderConfigFields()}
                        </div>
                    </div>

                    <div className="mt-6 flex justify-end">
                        <button
                            type="submit"
                            disabled={processing}
                            className="inline-flex items-center px-4 py-2 bg-blue-600 hover:bg-blue-700 text-white rounded-md"
                        >
                            <Save className="w-4 h-4 mr-2" /> Save
                        </button>
                    </div>
                </form>
            </div>
        </AdminLayout>
    );
};

export default Create;
