import React, { useState, useEffect } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import InputError from "@/Components/InputError";
import InputLabel from "@/Components/InputLabel";
import TextInput from "@/Components/TextInput";
import SelectInput from "@/Components/SelectInput";
import Checkbox from "@/Components/Checkbox";
import {
    CreditCard,
    ArrowLeft,
    Save,
    AlertCircle,
    Info,
    RefreshCw,
    CheckCircle,
} from "lucide-react";

const Edit = ({ paymentMethod }) => {
    const [isTestingConnection, setIsTestingConnection] = useState(false);
    const [testResult, setTestResult] = useState(null);

    // Initialize form with existing payment method data
    const { data, setData, put, processing, errors } = useForm({
        display_name: paymentMethod.display_name || "",
        is_active: paymentMethod.is_active || false,
        config_data: paymentMethod.config_data || {},
    });

    // Handle form submit
    const handleSubmit = (e) => {
        e.preventDefault();
        put(route("admin.payment-methods.update", paymentMethod.id));
    };

    // Test payment gateway connection
    const testConnection = async () => {
        setIsTestingConnection(true);
        setTestResult(null);

        try {
            const response = await axios.post(
                route("admin.payment-methods.test-connection", paymentMethod.id)
            );

            setTestResult({
                success: response.data.success,
                message: response.data.message,
                details: response.data.details,
            });
        } catch (error) {
            setTestResult({
                success: false,
                message:
                    error.response?.data?.message || "Connection test failed",
                details: error.response?.data?.details,
            });
        } finally {
            setIsTestingConnection(false);
        }
    };

    // Render config fields based on the payment method type
    const renderConfigFields = () => {
        switch (paymentMethod.name) {
            case "paypal":
                return (
                    <div className="space-y-4">
                        <div>
                            <InputLabel htmlFor="mode" value="Environment" />
                            <SelectInput
                                id="mode"
                                name="mode"
                                className="mt-1 block w-full"
                                value={data.config_data.mode || "sandbox"}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        mode: e.target.value,
                                    })
                                }
                            >
                                <option value="sandbox">
                                    Sandbox (Testing)
                                </option>
                                <option value="live">Live (Production)</option>
                            </SelectInput>
                            <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                Select the PayPal environment you want to use
                            </p>
                        </div>

                        <div className="p-4 bg-yellow-50 dark:bg-yellow-900/30 rounded-md">
                            <h3 className="font-medium flex items-center mb-2 text-yellow-800 dark:text-yellow-400">
                                <Info size={18} className="mr-2" />
                                Sandbox Credentials (Testing)
                            </h3>

                            <div className="mt-3">
                                <InputLabel
                                    htmlFor="client_id"
                                    value="Sandbox Client ID"
                                />
                                <TextInput
                                    id="client_id"
                                    type="text"
                                    className="mt-1 block w-full"
                                    value={data.config_data.client_id || ""}
                                    onChange={(e) =>
                                        setData("config_data", {
                                            ...data.config_data,
                                            client_id: e.target.value,
                                        })
                                    }
                                />
                                <InputError
                                    message={errors["config_data.client_id"]}
                                    className="mt-2"
                                />
                            </div>

                            <div className="mt-3">
                                <InputLabel
                                    htmlFor="client_secret"
                                    value="Sandbox Client Secret"
                                />
                                <TextInput
                                    id="client_secret"
                                    type="password"
                                    className="mt-1 block w-full"
                                    placeholder="••••••••••••••••"
                                    value={data.config_data.client_secret || ""}
                                    onChange={(e) =>
                                        setData("config_data", {
                                            ...data.config_data,
                                            client_secret: e.target.value,
                                        })
                                    }
                                />
                                <InputError
                                    message={
                                        errors["config_data.client_secret"]
                                    }
                                    className="mt-2"
                                />
                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                    Leave blank to keep the current value
                                </p>
                            </div>
                        </div>

                        <div className="p-4 bg-green-50 dark:bg-green-900/30 rounded-md">
                            <h3 className="font-medium flex items-center mb-2 text-green-800 dark:text-green-400">
                                <Info size={18} className="mr-2" />
                                Live Credentials (Production)
                            </h3>

                            <div className="mt-3">
                                <InputLabel
                                    htmlFor="live_client_id"
                                    value="Live Client ID"
                                />
                                <TextInput
                                    id="live_client_id"
                                    type="text"
                                    className="mt-1 block w-full"
                                    value={
                                        data.config_data.live_client_id || ""
                                    }
                                    onChange={(e) =>
                                        setData("config_data", {
                                            ...data.config_data,
                                            live_client_id: e.target.value,
                                        })
                                    }
                                />
                                <InputError
                                    message={
                                        errors["config_data.live_client_id"]
                                    }
                                    className="mt-2"
                                />
                            </div>

                            <div className="mt-3">
                                <InputLabel
                                    htmlFor="live_client_secret"
                                    value="Live Client Secret"
                                />
                                <TextInput
                                    id="live_client_secret"
                                    type="password"
                                    className="mt-1 block w-full"
                                    placeholder="••••••••••••••••"
                                    value={
                                        data.config_data.live_client_secret ||
                                        ""
                                    }
                                    onChange={(e) =>
                                        setData("config_data", {
                                            ...data.config_data,
                                            live_client_secret: e.target.value,
                                        })
                                    }
                                />
                                <InputError
                                    message={
                                        errors["config_data.live_client_secret"]
                                    }
                                    className="mt-2"
                                />
                                <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                    Leave blank to keep the current value
                                </p>
                            </div>
                        </div>
                    </div>
                );

            case "stripe":
                return (
                    <div className="space-y-4">
                        <div>
                            <InputLabel
                                htmlFor="publishable_key"
                                value="Publishable Key"
                            />
                            <TextInput
                                id="publishable_key"
                                type="text"
                                className="mt-1 block w-full"
                                value={data.config_data.publishable_key || ""}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        publishable_key: e.target.value,
                                    })
                                }
                            />
                            <InputError
                                message={errors["config_data.publishable_key"]}
                                className="mt-2"
                            />
                            <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                The publishable key is used in your frontend
                                code
                            </p>
                        </div>

                        <div>
                            <InputLabel
                                htmlFor="secret_key"
                                value="Secret Key"
                            />
                            <TextInput
                                id="secret_key"
                                type="password"
                                className="mt-1 block w-full"
                                placeholder="••••••••••••••••"
                                value={data.config_data.secret_key || ""}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        secret_key: e.target.value,
                                    })
                                }
                            />
                            <InputError
                                message={errors["config_data.secret_key"]}
                                className="mt-2"
                            />
                            <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                Leave blank to keep the current value
                            </p>
                        </div>

                        <div>
                            <InputLabel
                                htmlFor="webhook_secret"
                                value="Webhook Secret"
                            />
                            <TextInput
                                id="webhook_secret"
                                type="password"
                                className="mt-1 block w-full"
                                placeholder="••••••••••••••••"
                                value={data.config_data.webhook_secret || ""}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        webhook_secret: e.target.value,
                                    })
                                }
                            />
                            <InputError
                                message={errors["config_data.webhook_secret"]}
                                className="mt-2"
                            />
                            <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                Leave blank to keep the current value
                            </p>
                        </div>
                    </div>
                );

            case "paypal_direct":
                return (
                    <div className="space-y-4">
                        <div>
                            <InputLabel htmlFor="mode" value="Environment" />
                            <SelectInput
                                id="mode"
                                name="mode"
                                className="mt-1 block w-full"
                                value={data.config_data.mode || "sandbox"}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        mode: e.target.value,
                                    })
                                }
                            >
                                <option value="sandbox">Sandbox (Testing)</option>
                                <option value="live">Live (Production)</option>
                            </SelectInput>
                        </div>

                        <div>
                            <InputLabel value="Sandbox Business Email" />
                            <TextInput
                                value={data.config_data.sandbox_business_email || ""}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        sandbox_business_email: e.target.value,
                                    })
                                }
                                className="mt-1 block w-full"
                            />
                        </div>

                        <div>
                            <InputLabel value="Live Business Email" />
                            <TextInput
                                value={data.config_data.business_email || ""}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        business_email: e.target.value,
                                    })
                                }
                                className="mt-1 block w-full"
                            />
                        </div>

                        <div>
                            <InputLabel value="Sandbox Form Action URL" />
                            <TextInput
                                value={data.config_data.form_action_sandbox || "https://www.sandbox.paypal.com/cgi-bin/webscr"}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        form_action_sandbox: e.target.value,
                                    })
                                }
                                className="mt-1 block w-full"
                            />
                        </div>

                        <div>
                            <InputLabel value="Live Form Action URL" />
                            <TextInput
                                value={data.config_data.form_action_live || "https://www.paypal.com/cgi-bin/webscr"}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        form_action_live: e.target.value,
                                    })
                                }
                                className="mt-1 block w-full"
                            />
                        </div>

                        <div>
                            <InputLabel value="Sandbox IPN Verify URL" />
                            <TextInput
                                value={data.config_data.ipn_verify_sandbox || "https://ipnpb.sandbox.paypal.com/cgi-bin/webscr"}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        ipn_verify_sandbox: e.target.value,
                                    })
                                }
                                className="mt-1 block w-full"
                            />
                        </div>

                        <div>
                            <InputLabel value="Live IPN Verify URL" />
                            <TextInput
                                value={data.config_data.ipn_verify_live || "https://ipnpb.paypal.com/cgi-bin/webscr"}
                                onChange={(e) =>
                                    setData("config_data", {
                                        ...data.config_data,
                                        ipn_verify_live: e.target.value,
                                    })
                                }
                                className="mt-1 block w-full"
                            />
                        </div>
                    </div>
                );

            default:
                return (
                    <div className="p-4 bg-yellow-100 dark:bg-yellow-900/30 rounded-md">
                        <p className="text-yellow-800 dark:text-yellow-400 flex items-center">
                            <AlertCircle size={18} className="mr-2" />
                            Unknown payment method type: {paymentMethod.name}
                        </p>
                    </div>
                );
        }
    };

    return (
        <AdminLayout>
            <Head title={`Edit ${paymentMethod.display_name}`} />

            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex justify-between items-center mb-6">
                        <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                            Edit Payment Method: {paymentMethod.display_name}
                        </h1>
                        <Link
                            href={route("admin.payment-methods.index")}
                            className="px-4 py-2 bg-gray-200 dark:bg-gray-700 hover:bg-gray-300 dark:hover:bg-gray-600 rounded-md text-gray-800 dark:text-gray-300 inline-flex items-center transition duration-150 ease-in-out"
                        >
                            <ArrowLeft size={16} className="mr-2" />
                            Back to List
                        </Link>
                    </div>

                    {errors.message && (
                        <div className="bg-red-100 dark:bg-red-900 border border-red-400 dark:border-red-700 text-red-700 dark:text-red-300 px-4 py-3 rounded relative mb-4 flex items-start">
                            <AlertCircle
                                size={20}
                                className="mr-2 flex-shrink-0 mt-0.5"
                            />
                            <span>{errors.message}</span>
                        </div>
                    )}

                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <form onSubmit={handleSubmit} className="p-6">
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div className="space-y-6">
                                    <div>
                                        <InputLabel
                                            htmlFor="name"
                                            value="Payment Method Type"
                                        />
                                        <div className="mt-1 flex items-center">
                                            <div className="flex-shrink-0 h-10 w-10 flex items-center justify-center bg-blue-100 dark:bg-blue-900 rounded-full">
                                                <CreditCard className="h-5 w-5 text-blue-600 dark:text-blue-400" />
                                            </div>
                                            <div className="ml-3">
                                                <p className="text-md font-medium text-gray-800 dark:text-gray-200 capitalize">
                                                    {paymentMethod.name}
                                                </p>
                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                    Internal identifier (cannot
                                                    be changed)
                                                </p>
                                            </div>
                                        </div>
                                    </div>

                                    <div>
                                        <InputLabel
                                            htmlFor="display_name"
                                            value="Display Name"
                                        />
                                        <TextInput
                                            id="display_name"
                                            type="text"
                                            className="mt-1 block w-full"
                                            value={data.display_name}
                                            onChange={(e) =>
                                                setData(
                                                    "display_name",
                                                    e.target.value
                                                )
                                            }
                                            required
                                        />
                                        <InputError
                                            message={errors.display_name}
                                            className="mt-2"
                                        />
                                        <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                            This is the name shown to customers
                                            during checkout
                                        </p>
                                    </div>

                                    <div className="flex items-center">
                                        <Checkbox
                                            id="is_active"
                                            name="is_active"
                                            checked={data.is_active}
                                            onChange={(e) =>
                                                setData(
                                                    "is_active",
                                                    e.target.checked
                                                )
                                            }
                                        />
                                        <InputLabel
                                            htmlFor="is_active"
                                            value="Active"
                                            className="ml-2"
                                        />
                                    </div>

                                    {data.is_active && (
                                        <div className="mt-4">
                                            <button
                                                type="button"
                                                className="px-4 py-2 bg-purple-600 hover:bg-purple-700 rounded-md text-white inline-flex items-center"
                                                onClick={testConnection}
                                                disabled={isTestingConnection}
                                            >
                                                <RefreshCw
                                                    size={16}
                                                    className={`mr-2 ${
                                                        isTestingConnection
                                                            ? "animate-spin"
                                                            : ""
                                                    }`}
                                                />
                                                {isTestingConnection
                                                    ? "Testing..."
                                                    : "Test Connection"}
                                            </button>

                                            {testResult && (
                                                <div
                                                    className={`mt-3 p-3 rounded-md ${
                                                        testResult.success
                                                            ? "bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-400"
                                                            : "bg-red-100 dark:bg-red-900/30 text-red-800 dark:text-red-400"
                                                    }`}
                                                >
                                                    <div className="flex items-start">
                                                        {testResult.success ? (
                                                            <CheckCircle
                                                                size={20}
                                                                className="mr-2 flex-shrink-0 mt-0.5"
                                                            />
                                                        ) : (
                                                            <AlertCircle
                                                                size={20}
                                                                className="mr-2 flex-shrink-0 mt-0.5"
                                                            />
                                                        )}
                                                        <div>
                                                            <p className="font-medium">
                                                                {
                                                                    testResult.message
                                                                }
                                                            </p>
                                                            {testResult.details && (
                                                                <ul className="mt-1 text-sm">
                                                                    {Object.entries(
                                                                        testResult.details
                                                                    ).map(
                                                                        ([
                                                                            key,
                                                                            value,
                                                                        ]) => (
                                                                            <li
                                                                                key={
                                                                                    key
                                                                                }
                                                                            >
                                                                                <strong>
                                                                                    {key.replace(
                                                                                        /_/g,
                                                                                        " "
                                                                                    )}

                                                                                    : {" "}
                                                                                </strong>
                                                                                {/* Format based on value type */}
                                                                                {typeof value ===
                                                                                    "object" &&
                                                                                value !==
                                                                                    null
                                                                                    ? Array.isArray(
                                                                                          value
                                                                                      )
                                                                                        ? value.join(
                                                                                              ", "
                                                                                          )
                                                                                        : JSON.stringify(
                                                                                              value
                                                                                          )
                                                                                    : String(
                                                                                          value
                                                                                      )}
                                                                            </li>
                                                                        )
                                                                    )}
                                                                </ul>
                                                            )}
                                                        </div>
                                                    </div>
                                                </div>
                                            )}
                                        </div>
                                    )}
                                </div>

                                <div className="space-y-6">
                                    <div className="p-4 bg-blue-50 dark:bg-blue-900/30 rounded-md mb-4">
                                        <h3 className="font-medium flex items-center text-blue-800 dark:text-blue-400">
                                            <CreditCard
                                                size={18}
                                                className="mr-2"
                                            />
                                            Configuration
                                        </h3>
                                        <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                            For security reasons, sensitive
                                            fields will not show the current
                                            value. Leave password fields blank
                                            to keep the current values.
                                        </p>
                                    </div>

                                    {renderConfigFields()}
                                </div>
                            </div>

                            <div className="flex items-center justify-end mt-8">
                                <button
                                    type="submit"
                                    disabled={processing}
                                    className="px-4 py-2 bg-blue-600 hover:bg-blue-700 disabled:bg-blue-400 rounded-md text-white inline-flex items-center transition duration-150 ease-in-out"
                                >
                                    <Save size={16} className="mr-2" />
                                    {processing
                                        ? "Saving..."
                                        : "Update Payment Method"}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Edit;
