import React, { useState } from "react";
import { Head, Link, router } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    PlusCircle,
    Edit,
    Trash2,
    AlertCircle,
    CheckCircle,
    XCircle,
    RefreshCw,
    CreditCard,
    DollarSign,
    BarChart,
} from "lucide-react";
import axios from "axios";
import { formatCurrency } from "@/utils/Utils";

const PaymentMethods = ({ paymentMethods, errors, flash, success }) => {
    const [loading, setLoading] = useState({});
    const [testResults, setTestResults] = useState({});

    // Handle toggling payment method active status
    const toggleActive = async (id) => {
        setLoading((prev) => ({ ...prev, toggle: id }));

        try {
            const response = await axios.post(
                route("admin.payment-methods.toggle-status", id)
            );

            if (response.data.success) {
                // Update UI without a full page reload
                const updatedMethods = paymentMethods.map((method) =>
                    method.id === id
                        ? { ...method, is_active: response.data.is_active }
                        : method
                );

                router.reload({
                    only: ["paymentMethods"],
                    data: { paymentMethods: updatedMethods },
                    preserveState: true,
                });
            }
        } catch (error) {
            console.error("Error toggling payment method status:", error);
            alert(
                "Failed to toggle payment method status: " +
                    (error.response?.data?.message || error.message)
            );
        } finally {
            setLoading((prev) => ({ ...prev, toggle: null }));
        }
    };

    // Test payment method connection
    const testConnection = async (id) => {
        setLoading((prev) => ({ ...prev, test: id }));
        setTestResults((prev) => ({ ...prev, [id]: null }));

        try {
            const response = await axios.post(
                route("admin.payment-methods.test-connection", id)
            );
            setTestResults((prev) => ({
                ...prev,
                [id]: {
                    success: response.data.success,
                    message: response.data.message,
                    details: response.data.details,
                },
            }));
        } catch (error) {
            console.error("Error testing connection:", error);
            setTestResults((prev) => ({
                ...prev,
                [id]: {
                    success: false,
                    message:
                        error.response?.data?.message ||
                        "Connection test failed",
                },
            }));
        } finally {
            setLoading((prev) => ({ ...prev, test: null }));
        }
    };

    // Handle payment method deletion
    const confirmDelete = (id, name) => {
        if (
            confirm(
                `Are you sure you want to delete the payment method "${name}"? This action cannot be undone.`
            )
        ) {
            router.delete(route("admin.payment-methods.destroy", id));
        }
    };

    // Clear test results for a payment method
    const clearTestResult = (id) => {
        setTestResults((prev) => {
            const newResults = { ...prev };
            delete newResults[id];
            return newResults;
        });
    };

    return (
        <AdminLayout>
            <Head title="Payment Methods" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <Link
                        href={route("admin.payment.analytics")}
                        className="px-4 py-2 bg-indigo-600 hover:bg-indigo-700 rounded-md text-white inline-flex items-center transition duration-150 ease-in-out ml-2"
                    >
                        <BarChart size={16} className="mr-2" />
                        Analytics Dashboard
                    </Link>
                    <div className="flex justify-between items-center mb-6">
                        <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                            Payment Methods
                        </h1>
                        <Link
                            href={route("admin.payment-methods.create")}
                            className="px-4 py-2 bg-blue-600 hover:bg-blue-700 rounded-md text-white inline-flex items-center transition duration-150 ease-in-out"
                        >
                            <PlusCircle size={16} className="mr-2" />
                            Add Method
                        </Link>
                    </div>
                    {/* Error message display */}
                    {flash && flash.error && (
                        <div className="bg-red-100 dark:bg-red-900 border border-red-400 dark:border-red-700 text-red-700 dark:text-red-300 px-4 py-3 rounded relative mb-4 flex items-start">
                            <AlertCircle
                                size={20}
                                className="mr-2 flex-shrink-0 mt-0.5"
                            />
                            <span>{flash.error}</span>
                        </div>
                    )}
                    {/* Success message display */}
                    {flash && flash.message && (
                        <div className="bg-green-100 dark:bg-green-900 border border-green-400 dark:border-green-700 text-green-700 dark:text-green-300 px-4 py-3 rounded relative mb-4 flex items-start">
                            <CheckCircle
                                size={20}
                                className="mr-2 flex-shrink-0 mt-0.5"
                            />
                            <span>{flash.message}</span>
                        </div>
                    )}
                    {/* Payment methods table */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        {paymentMethods && paymentMethods.length > 0 ? (
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
                                    <thead className="bg-gray-50 dark:bg-gray-700">
                                        <tr>
                                            <th
                                                scope="col"
                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                            >
                                                Name
                                            </th>
                                            <th
                                                scope="col"
                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                            >
                                                Status
                                            </th>
                                            <th
                                                scope="col"
                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                            >
                                                Transactions
                                            </th>
                                            <th
                                                scope="col"
                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                            >
                                                Volume
                                            </th>
                                            <th
                                                scope="col"
                                                className="px-6 py-3 text-left text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider"
                                            >
                                                Actions
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="bg-white dark:bg-gray-800 divide-y divide-gray-200 dark:divide-gray-700">
                                        {paymentMethods.map((method) => (
                                            <tr key={method.id}>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <div className="flex items-center">
                                                        <div className="flex-shrink-0 h-10 w-10 flex items-center justify-center bg-gray-100 dark:bg-gray-700 rounded-full">
                                                            <CreditCard
                                                                size={20}
                                                                className="text-gray-500 dark:text-gray-400"
                                                            />
                                                        </div>
                                                        <div className="ml-4">
                                                            <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                {
                                                                    method.display_name
                                                                }
                                                            </div>
                                                            <div className="text-sm text-gray-500 dark:text-gray-400">
                                                                {method.name}
                                                            </div>
                                                        </div>
                                                    </div>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <span
                                                        className={`px-2 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                                            method.is_active
                                                                ? "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200"
                                                                : "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200"
                                                        }`}
                                                    >
                                                        {method.is_active
                                                            ? "Active"
                                                            : "Inactive"}
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                    {method.stats
                                                        ?.transaction_count ||
                                                        0}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-500 dark:text-gray-400">
                                                    {formatCurrency(
                                                        method.stats
                                                            ?.transaction_volume ||
                                                            0
                                                    )}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm font-medium space-x-2">
                                                    <div className="flex flex-wrap gap-2">
                                                        {/* Edit button */}
                                                        <Link
                                                            href={route(
                                                                "admin.payment-methods.edit",
                                                                method.id
                                                            )}
                                                            className="text-blue-600 hover:text-blue-900 dark:text-blue-400 dark:hover:text-blue-300"
                                                            title="Edit"
                                                        >
                                                            <Edit size={18} />
                                                        </Link>

                                                        {/* Delete button */}
                                                        <button
                                                            onClick={() =>
                                                                confirmDelete(
                                                                    method.id,
                                                                    method.display_name
                                                                )
                                                            }
                                                            className="text-red-600 hover:text-red-900 dark:text-red-400 dark:hover:text-red-300"
                                                            title="Delete"
                                                        >
                                                            <Trash2 size={18} />
                                                        </button>

                                                        {/* Toggle active status */}
                                                        <button
                                                            onClick={() =>
                                                                toggleActive(
                                                                    method.id
                                                                )
                                                            }
                                                            disabled={
                                                                loading.toggle ===
                                                                method.id
                                                            }
                                                            className={`${
                                                                method.is_active
                                                                    ? "text-red-600 hover:text-red-900 dark:text-red-400 dark:hover:text-red-300"
                                                                    : "text-green-600 hover:text-green-900 dark:text-green-400 dark:hover:text-green-300"
                                                            } ${
                                                                loading.toggle ===
                                                                method.id
                                                                    ? "opacity-50 cursor-not-allowed"
                                                                    : ""
                                                            }`}
                                                            title={
                                                                method.is_active
                                                                    ? "Deactivate"
                                                                    : "Activate"
                                                            }
                                                        >
                                                            {method.is_active ? (
                                                                <XCircle
                                                                    size={18}
                                                                />
                                                            ) : (
                                                                <CheckCircle
                                                                    size={18}
                                                                />
                                                            )}
                                                        </button>

                                                        {/* Test connection button */}
                                                        <button
                                                            onClick={() =>
                                                                testConnection(
                                                                    method.id
                                                                )
                                                            }
                                                            disabled={
                                                                loading.test ===
                                                                    method.id ||
                                                                !method.is_active
                                                            }
                                                            className={`text-purple-600 hover:text-purple-900 dark:text-purple-400 dark:hover:text-purple-300 ${
                                                                loading.test ===
                                                                    method.id ||
                                                                !method.is_active
                                                                    ? "opacity-50 cursor-not-allowed"
                                                                    : ""
                                                            }`}
                                                            title={
                                                                method.is_active
                                                                    ? "Test Connection"
                                                                    : "Activate to test connection"
                                                            }
                                                        >
                                                            <RefreshCw
                                                                size={18}
                                                                className={
                                                                    loading.test ===
                                                                    method.id
                                                                        ? "animate-spin"
                                                                        : ""
                                                                }
                                                            />
                                                        </button>
                                                    </div>

                                                    {/* Test connection results */}
                                                    {/* Test connection results */}
                                                    {testResults[method.id] && (
                                                        <div
                                                            className={`mt-2 p-2 text-xs rounded ${
                                                                testResults[
                                                                    method.id
                                                                ].success
                                                                    ? "bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200"
                                                                    : "bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200"
                                                            }`}
                                                        >
                                                            <div className="flex justify-between items-start">
                                                                <div className="flex items-start">
                                                                    {testResults[
                                                                        method
                                                                            .id
                                                                    ]
                                                                        .success ? (
                                                                        <CheckCircle
                                                                            size={
                                                                                14
                                                                            }
                                                                            className="mr-1 mt-0.5"
                                                                        />
                                                                    ) : (
                                                                        <AlertCircle
                                                                            size={
                                                                                14
                                                                            }
                                                                            className="mr-1 mt-0.5"
                                                                        />
                                                                    )}
                                                                    <span>
                                                                        {
                                                                            testResults[
                                                                                method
                                                                                    .id
                                                                            ]
                                                                                .message
                                                                        }
                                                                    </span>
                                                                </div>
                                                                <button
                                                                    onClick={() =>
                                                                        clearTestResult(
                                                                            method.id
                                                                        )
                                                                    }
                                                                    className="ml-2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-300"
                                                                >
                                                                    <XCircle
                                                                        size={
                                                                            14
                                                                        }
                                                                    />
                                                                </button>
                                                            </div>
                                                            {testResults[
                                                                method.id
                                                            ].details && (
                                                                <div className="mt-1 pl-5">
                                                                    {Object.entries(
                                                                        testResults[
                                                                            method
                                                                                .id
                                                                        ]
                                                                            .details
                                                                    ).map(
                                                                        ([
                                                                            key,
                                                                            value,
                                                                        ]) => (
                                                                            <div
                                                                                key={
                                                                                    key
                                                                                }
                                                                                className="text-xs"
                                                                            >
                                                                                <span className="font-medium">
                                                                                    {key.replace(
                                                                                        /_/g,
                                                                                        " "
                                                                                    )}

                                                                                    :
                                                                                </span>{" "}
                                                                                {/* Fix here: properly render objects or arrays */}
                                                                                {typeof value ===
                                                                                    "object" &&
                                                                                value !==
                                                                                    null
                                                                                    ? Array.isArray(
                                                                                          value
                                                                                      )
                                                                                        ? value.join(
                                                                                              ", "
                                                                                          )
                                                                                        : JSON.stringify(
                                                                                              value
                                                                                          )
                                                                                    : value.toString()}
                                                                            </div>
                                                                        )
                                                                    )}
                                                                </div>
                                                            )}
                                                        </div>
                                                    )}
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        ) : (
                            <div className="px-6 py-4 text-center text-gray-500 dark:text-gray-400">
                                <div className="flex flex-col items-center py-8">
                                    <DollarSign
                                        size={48}
                                        className="mb-4 text-gray-400 dark:text-gray-600"
                                    />
                                    <h3 className="mb-1 text-lg font-medium text-gray-900 dark:text-gray-100">
                                        No payment methods found
                                    </h3>
                                    <p className="mb-4">
                                        You haven't added any payment methods
                                        yet.
                                    </p>
                                    <Link
                                        href={route(
                                            "admin.payment-methods.create"
                                        )}
                                        className="px-4 py-2 bg-blue-600 hover:bg-blue-700 rounded-md text-white inline-flex items-center transition duration-150 ease-in-out"
                                    >
                                        <PlusCircle
                                            size={16}
                                            className="mr-2"
                                        />
                                        Add Your First Payment Method
                                    </Link>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default PaymentMethods;
