import React, { useState, useEffect } from "react";
import { Head, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import axios from "axios";
import {
    ArrowLeft,
    CreditCard,
    FileText,
    User,
    DollarSign,
    Calendar,
    CheckCircle,
    AlertCircle,
    Clock,
    XCircle,
    RefreshCw,
    DownloadCloud,
    Tag,
    Code,
    File,
    Trash2,
    Repeat,
    Database,
    X,
    Info,
    Loader,
    ShoppingCartIcon,
} from "lucide-react";
import { format } from "date-fns";

// Refund Modal Component
const RefundModal = ({ payment, isOpen, onClose, onSuccess }) => {
    const [amount, setAmount] = useState("");
    const [reason, setReason] = useState("");
    const [notes, setNotes] = useState("");
    const [errors, setErrors] = useState({});
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [confirmation, setConfirmation] = useState(false);
    const [refundableAmount, setRefundableAmount] = useState(0);

    // Fetch refundable amount when modal opens
    useEffect(() => {
        if (isOpen) {
            fetchRefundableAmount();
            setAmount("");
            setReason("");
            setNotes("");
            setErrors({});
            setConfirmation(false);
        }
    }, [isOpen, payment?.id]);

    const fetchRefundableAmount = async () => {
        try {
            const response = await axios.get(
                route("admin.payments.refund-history", payment.id)
            );
            if (response.data.success) {
                setRefundableAmount(response.data.payment.refundable_amount);

                // Pre-fill with maximum refundable amount
                setAmount(response.data.payment.refundable_amount.toString());
            }
        } catch (error) {
            console.error("Error fetching refundable amount:", error);
        }
    };

    const validateForm = () => {
        const newErrors = {};

        if (!amount || isNaN(parseFloat(amount)) || parseFloat(amount) <= 0) {
            newErrors.amount = "Please enter a valid amount greater than 0";
        } else if (parseFloat(amount) > refundableAmount) {
            newErrors.amount = `Amount cannot exceed the refundable amount (${formatCurrency(
                refundableAmount
            )})`;
        }

        if (!reason?.trim()) {
            newErrors.reason = "Please provide a reason for the refund";
        }

        setErrors(newErrors);
        return Object.keys(newErrors).length === 0;
    };

    const handleSubmit = async (e) => {
        e.preventDefault();

        if (!validateForm()) {
            return;
        }

        if (!confirmation) {
            setConfirmation(true);
            return;
        }

        setIsSubmitting(true);

        try {
            const response = await axios.post(
                route("admin.payments.refund", payment.id),
                {
                    amount: parseFloat(amount),
                    reason,
                    notes,
                }
            );

            if (response.data.success) {
                onSuccess(response.data);
                onClose();
            } else {
                setErrors({
                    api: response.data.message || "Failed to process refund",
                });
                setConfirmation(false);
            }
        } catch (error) {
            console.error("Refund error:", error);
            setErrors({
                api:
                    error.response?.data?.message ||
                    "An error occurred while processing the refund",
            });
            setConfirmation(false);
        } finally {
            setIsSubmitting(false);
        }
    };

    // Format currency
    const formatCurrency = (amount, currency = "USD") => {
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: currency,
        }).format(amount);
    };

    if (!isOpen) return null;

    return (
        <div className="fixed inset-0 overflow-y-auto z-50">
            <div className="flex items-center justify-center min-h-screen px-4 pt-4 pb-20 text-center sm:block sm:p-0">
                {/* Background overlay */}
                <div
                    className="fixed inset-0 transition-opacity bg-gray-500 bg-opacity-75 dark:bg-gray-900 dark:bg-opacity-75"
                    onClick={!isSubmitting ? onClose : undefined}
                    aria-hidden="true"
                ></div>

                {/* Modal panel */}
                <div className="inline-block overflow-hidden text-left align-bottom transition-all transform bg-white dark:bg-gray-800 rounded-lg shadow-xl sm:my-8 sm:align-middle sm:max-w-lg sm:w-full">
                    <div className="px-4 pt-5 pb-4 bg-white dark:bg-gray-800 sm:p-6 sm:pb-4">
                        <div className="sm:flex sm:items-start">
                            <div className="flex items-center justify-center flex-shrink-0 w-12 h-12 mx-auto bg-orange-100 dark:bg-orange-900 rounded-full sm:mx-0 sm:h-10 sm:w-10">
                                <Repeat className="w-6 h-6 text-orange-600 dark:text-orange-400" />
                            </div>
                            <div className="mt-3 text-center sm:mt-0 sm:ml-4 sm:text-left">
                                <h3 className="text-lg font-medium leading-6 text-gray-900 dark:text-gray-100">
                                    {confirmation
                                        ? "Confirm Refund"
                                        : "Process Refund"}
                                </h3>
                                <div className="mt-2">
                                    {confirmation ? (
                                        <p className="text-sm text-gray-500 dark:text-gray-400">
                                            You are about to refund{" "}
                                            <span className="font-bold text-gray-700 dark:text-gray-300">
                                                {formatCurrency(
                                                    parseFloat(amount)
                                                )}
                                            </span>{" "}
                                            to the customer. This action cannot
                                            be undone.
                                        </p>
                                    ) : (
                                        <p className="text-sm text-gray-500 dark:text-gray-400">
                                            Enter the amount you want to refund
                                            and provide a reason. Maximum
                                            refundable amount:{" "}
                                            {formatCurrency(refundableAmount)}
                                        </p>
                                    )}
                                </div>
                            </div>
                        </div>

                        {errors.api && (
                            <div className="mt-4 p-3 bg-red-50 dark:bg-red-900/30 border border-red-200 dark:border-red-800 rounded-md">
                                <div className="flex items-start">
                                    <AlertCircle className="w-5 h-5 text-red-600 dark:text-red-400 mr-2 flex-shrink-0" />
                                    <p className="text-sm text-red-600 dark:text-red-400">
                                        {errors.api}
                                    </p>
                                </div>
                            </div>
                        )}

                        {!confirmation && (
                            <form onSubmit={handleSubmit} className="mt-6">
                                <div className="mb-4">
                                    <label
                                        htmlFor="amount"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                    >
                                        Refund Amount
                                    </label>
                                    <div className="mt-1 relative rounded-md shadow-sm">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <span className="text-gray-500 dark:text-gray-400 sm:text-sm">
                                                $
                                            </span>
                                        </div>
                                        <input
                                            type="number"
                                            name="amount"
                                            id="amount"
                                            step="0.01"
                                            min="0.01"
                                            max={refundableAmount}
                                            className="pl-7 block w-full border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white sm:text-sm"
                                            placeholder="0.00"
                                            value={amount}
                                            onChange={(e) =>
                                                setAmount(e.target.value)
                                            }
                                            disabled={isSubmitting}
                                            required
                                        />
                                    </div>
                                    {errors.amount && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                            {errors.amount}
                                        </p>
                                    )}
                                </div>

                                <div className="mb-4">
                                    <label
                                        htmlFor="reason"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                    >
                                        Reason for Refund
                                    </label>
                                    <select
                                        id="reason"
                                        name="reason"
                                        className="mt-1 block w-full border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white sm:text-sm"
                                        value={reason}
                                        onChange={(e) =>
                                            setReason(e.target.value)
                                        }
                                        disabled={isSubmitting}
                                        required
                                    >
                                        <option value="">
                                            Select a reason
                                        </option>
                                        <option value="Customer requested">
                                            Customer requested
                                        </option>
                                        <option value="Order cancelled">
                                            Order cancelled
                                        </option>
                                        <option value="Service issue">
                                            Service issue
                                        </option>
                                        <option value="Duplicate payment">
                                            Duplicate payment
                                        </option>
                                        <option value="Other">Other</option>
                                    </select>
                                    {errors.reason && (
                                        <p className="mt-1 text-sm text-red-600 dark:text-red-400">
                                            {errors.reason}
                                        </p>
                                    )}
                                </div>

                                <div className="mb-4">
                                    <label
                                        htmlFor="notes"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                    >
                                        Additional Notes (Optional)
                                    </label>
                                    <textarea
                                        id="notes"
                                        name="notes"
                                        rows={3}
                                        className="mt-1 block w-full border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white sm:text-sm"
                                        placeholder="Enter any additional details about this refund..."
                                        value={notes}
                                        onChange={(e) =>
                                            setNotes(e.target.value)
                                        }
                                        disabled={isSubmitting}
                                    />
                                </div>
                            </form>
                        )}
                    </div>

                    <div className="px-4 py-3 bg-gray-50 dark:bg-gray-700 sm:px-6 sm:flex sm:flex-row-reverse">
                        <button
                            type="button"
                            className={`w-full inline-flex justify-center rounded-md border border-transparent shadow-sm px-4 py-2 text-base font-medium text-white focus:outline-none focus:ring-2 focus:ring-offset-2 sm:ml-3 sm:w-auto sm:text-sm ${
                                isSubmitting
                                    ? "bg-orange-300 dark:bg-orange-700 cursor-not-allowed"
                                    : "bg-orange-600 hover:bg-orange-700 dark:bg-orange-600 dark:hover:bg-orange-700 focus:ring-orange-500"
                            }`}
                            onClick={handleSubmit}
                            disabled={isSubmitting}
                        >
                            {isSubmitting ? (
                                <>
                                    <Loader className="w-4 h-4 mr-2 animate-spin" />
                                    Processing...
                                </>
                            ) : confirmation ? (
                                "Confirm Refund"
                            ) : (
                                "Process Refund"
                            )}
                        </button>
                        <button
                            type="button"
                            className="mt-3 w-full inline-flex justify-center rounded-md border border-gray-300 dark:border-gray-600 shadow-sm px-4 py-2 bg-white dark:bg-gray-800 text-base font-medium text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-500 sm:mt-0 sm:ml-3 sm:w-auto sm:text-sm"
                            onClick={onClose}
                            disabled={isSubmitting}
                        >
                            Cancel
                        </button>
                    </div>
                </div>
            </div>
        </div>
    );
};

// Refund History Component

const RefundHistory = ({ refunds, currency = "USD" }) => {
    // Format currency
    const formatCurrency = (amount, currency = "USD") => {
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: currency,
        }).format(amount);
    };

    // Get status badge class
    const getRefundStatusBadgeClass = (status) => {
        switch (status) {
            case "completed":
                return "bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200";
            case "pending":
                return "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-200";
            case "failed":
                return "bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-200";
            default:
                return "bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300";
        }
    };

    if (!refunds || refunds.length === 0) {
        return (
            <div className="text-center py-4 text-gray-500 dark:text-gray-400">
                No refunds have been processed for this payment.
            </div>
        );
    }

    return (
        <div className="mt-4">
            {/* Desktop view - CSS Grid layout */}
            <div className="hidden md:block">
                {/* Header */}
                <div className="grid grid-cols-12 gap-2 bg-gray-50 dark:bg-gray-700 rounded-t-lg">
                    <div className="col-span-2 px-6 py-3">
                        <span className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                            Date
                        </span>
                    </div>
                    <div className="col-span-2 px-6 py-3">
                        <span className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                            Amount
                        </span>
                    </div>
                    <div className="col-span-2 px-6 py-3">
                        <span className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                            Status
                        </span>
                    </div>
                    <div className="col-span-3 px-6 py-3">
                        <span className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                            Reason
                        </span>
                    </div>
                    <div className="col-span-3 px-6 py-3">
                        <span className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                            Processed By
                        </span>
                    </div>
                </div>

                {/* Body */}
                <div className="bg-white dark:bg-gray-800 rounded-b-lg divide-y divide-gray-200 dark:divide-gray-700">
                    {refunds.map((refund) => (
                        <div
                            key={refund.id}
                            className="grid grid-cols-12 gap-2 hover:bg-gray-50 dark:hover:bg-gray-700/60"
                        >
                            <div className="col-span-2 px-6 py-4 flex items-center">
                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                    {refund.processed_at
                                        ? format(
                                              new Date(refund.processed_at),
                                              "MMM dd, yyyy HH:mm"
                                          )
                                        : format(
                                              new Date(refund.created_at),
                                              "MMM dd, yyyy HH:mm"
                                          )}
                                </span>
                            </div>
                            <div className="col-span-2 px-6 py-4 flex items-center">
                                <span className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                    {formatCurrency(
                                        refund.amount,
                                        refund.currency || currency
                                    )}
                                </span>
                            </div>
                            <div className="col-span-2 px-6 py-4 flex items-center">
                                <span
                                    className={`px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full ${getRefundStatusBadgeClass(
                                        refund.status
                                    )}`}
                                >
                                    {refund.status}
                                </span>
                            </div>
                            <div className="col-span-3 px-6 py-4 flex items-center">
                                <span className="text-sm text-gray-500 dark:text-gray-400 line-clamp-2">
                                    {refund.reason || "N/A"}
                                </span>
                            </div>
                            <div className="col-span-3 px-6 py-4 flex items-center">
                                <span className="text-sm text-gray-500 dark:text-gray-400">
                                    {refund.processor?.name || "System"}
                                </span>
                            </div>
                        </div>
                    ))}
                </div>
            </div>

            {/* Mobile view - Card layout */}
            <div className="md:hidden space-y-4">
                {refunds.map((refund) => (
                    <div
                        key={refund.id}
                        className="bg-white dark:bg-gray-800 shadow rounded-lg p-4 border border-gray-200 dark:border-gray-700"
                    >
                        {/* Top section with amount and status */}
                        <div className="flex justify-between items-center mb-3">
                            <div className="font-medium text-gray-900 dark:text-gray-100">
                                {formatCurrency(
                                    refund.amount,
                                    refund.currency || currency
                                )}
                            </div>
                            <span
                                className={`px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full ${getRefundStatusBadgeClass(
                                    refund.status
                                )}`}
                            >
                                {refund.status}
                            </span>
                        </div>

                        {/* Details section */}
                        <div className="space-y-2 text-sm">
                            <div className="grid grid-cols-3 gap-1">
                                <span className="text-gray-500 dark:text-gray-400 font-medium">
                                    Date:
                                </span>
                                <span className="col-span-2 text-gray-700 dark:text-gray-300">
                                    {refund.processed_at
                                        ? format(
                                              new Date(refund.processed_at),
                                              "MMM dd, yyyy HH:mm"
                                          )
                                        : format(
                                              new Date(refund.created_at),
                                              "MMM dd, yyyy HH:mm"
                                          )}
                                </span>
                            </div>

                            <div className="grid grid-cols-3 gap-1">
                                <span className="text-gray-500 dark:text-gray-400 font-medium">
                                    Reason:
                                </span>
                                <span className="col-span-2 text-gray-700 dark:text-gray-300">
                                    {refund.reason || "N/A"}
                                </span>
                            </div>

                            <div className="grid grid-cols-3 gap-1">
                                <span className="text-gray-500 dark:text-gray-400 font-medium">
                                    Processed By:
                                </span>
                                <span className="col-span-2 text-gray-700 dark:text-gray-300">
                                    {refund.processor?.name || "System"}
                                </span>
                            </div>
                        </div>
                    </div>
                ))}
            </div>
        </div>
    );
};

const PaymentDetail = ({ payment }) => {
    const [showRawData, setShowRawData] = useState(false);
    const [showRefundModal, setShowRefundModal] = useState(false);
    const [refunds, setRefunds] = useState([]);
    const [refundDetails, setRefundDetails] = useState(null);
    const [loadingRefunds, setLoadingRefunds] = useState(false);

    // Format currency
    const formatCurrency = (amount, currency = "USD") => {
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: currency,
        }).format(amount);
    };

    // Get status badge class
    const getStatusBadgeClass = (status) => {
        switch (status) {
            case "completed":
                return "bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200";
            case "pending":
                return "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-200";
            case "failed":
                return "bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-200";
            case "refunded":
                return "bg-purple-100 text-purple-800 dark:bg-purple-800 dark:text-purple-200";
            default:
                return "bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300";
        }
    };

    // Get status icon
    const getStatusIcon = (status) => {
        switch (status) {
            case "completed":
                return <CheckCircle size={20} className="mr-2" />;
            case "pending":
                return <Clock size={20} className="mr-2" />;
            case "failed":
                return <AlertCircle size={20} className="mr-2" />;
            case "refunded":
                return <XCircle size={20} className="mr-2" />;
            default:
                return null;
        }
    };

    // Fetch refund history
    const fetchRefundHistory = async () => {
        setLoadingRefunds(true);
        try {
            const response = await axios.get(
                route("admin.payments.refund-history", payment.id)
            );
            if (response.data.success) {
                setRefunds(response.data.refunds || []);
                setRefundDetails(response.data.payment || null);
            }
        } catch (error) {
            console.error("Error fetching refund history:", error);
        } finally {
            setLoadingRefunds(false);
        }
    };

    // Load refund history when component mounts
    useEffect(() => {
        if (payment && payment.id) {
            fetchRefundHistory();
        }
    }, [payment?.id]);

    // Handle successful refund
    const handleRefundSuccess = (data) => {
        // Refresh refund history
        fetchRefundHistory();
    };

    return (
        <AdminLayout>
            <Head title={`Payment #${payment.id}`} />

            <div className="py-6 overflow-hidden">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8 overflow-hidden">
                    <div className="flex justify-between items-center mb-6">
                        <div className="flex items-center">
                            <Link
                                href={route("admin.payments.history")}
                                className="mr-4 text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200"
                            >
                                <ArrowLeft size={20} />
                            </Link>
                            <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                                Payment Details
                            </h1>
                        </div>
                        <div className="flex space-x-2">
                            <button
                                onClick={() => setShowRawData(!showRawData)}
                                className="px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md text-gray-700 dark:text-gray-200 text-sm flex items-center hover:bg-gray-50 dark:hover:bg-gray-700 transition duration-150 ease-in-out"
                            >
                                <Database size={16} className="mr-2" />
                                {showRawData
                                    ? "Hide Raw Data"
                                    : "View Raw Data"}
                            </button>

                            {payment.status === "completed" &&
                                refundDetails?.refundable_amount > 0 && (
                                    <button
                                        onClick={() => setShowRefundModal(true)}
                                        className="px-3 py-2 border border-orange-300 bg-orange-50 dark:bg-orange-900/20 dark:border-orange-700 rounded-md text-orange-700 dark:text-orange-300 text-sm flex items-center hover:bg-orange-100 dark:hover:bg-orange-900/40 transition duration-150 ease-in-out"
                                    >
                                        <Repeat size={16} className="mr-2" />
                                        Process Refund
                                    </button>
                                )}
                        </div>
                    </div>

                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                        {/* Payment Status Header */}
                        <div
                            className={`px-6 py-4 border-b border-gray-200 dark:border-gray-700 ${
                                payment.status === "completed"
                                    ? "bg-green-50 dark:bg-green-900/20"
                                    : payment.status === "pending"
                                    ? "bg-yellow-50 dark:bg-yellow-900/20"
                                    : payment.status === "failed"
                                    ? "bg-red-50 dark:bg-red-900/20"
                                    : "bg-purple-50 dark:bg-purple-900/20"
                            }`}
                        >
                            <div className="flex flex-col sm:flex-row sm:justify-between sm:items-center">
                                <div className="flex items-center mb-2 sm:mb-0">
                                    {getStatusIcon(payment.status)}
                                    <span className="text-lg font-medium capitalize">
                                        {payment.status} Payment
                                    </span>
                                </div>
                                <div className="flex items-center">
                                    {refundDetails &&
                                        refundDetails.refunded_amount > 0 && (
                                            <div className="mr-4 text-sm">
                                                <span className="text-gray-600 dark:text-gray-400">
                                                    Refunded:{" "}
                                                </span>
                                                <span className="text-purple-600 dark:text-purple-400 font-medium">
                                                    {formatCurrency(
                                                        refundDetails.refunded_amount,
                                                        payment.currency
                                                    )}
                                                </span>
                                            </div>
                                        )}
                                    <ShoppingCartIcon className="h-5 w-5 text-gray-500 dark:text-gray-400 mr-1" />
                                    <span className="text-xl font-bold text-gray-900 dark:text-gray-100">
                                        {formatCurrency(
                                            payment.amount,
                                            payment.currency
                                        )}
                                    </span>
                                </div>
                            </div>
                        </div>

                        {/* Main Content */}
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6 p-6">
                            {/* Transaction Details */}
                            <div className="col-span-2">
                                <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                    <CreditCard className="h-5 w-5 mr-2 text-blue-500" />
                                    Transaction Details
                                </h2>
                                <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4 space-y-4">
                                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                        <div>
                                            <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                Transaction ID
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-900 dark:text-gray-100 font-mono break-all">
                                                {payment.transaction_id ||
                                                    "N/A"}
                                            </p>
                                        </div>
                                        <div>
                                            <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                Payment Method
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-900 dark:text-gray-100 capitalize">
                                                {payment.payment_method}
                                            </p>
                                        </div>
                                        <div>
                                            <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                Date Created
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-900 dark:text-gray-100">
                                                {payment.created_at
                                                    ? format(
                                                          new Date(
                                                              payment.created_at
                                                          ),
                                                          "PPP p"
                                                      )
                                                    : "N/A"}
                                            </p>
                                        </div>
                                        <div>
                                            <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                Payment Date
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-900 dark:text-gray-100">
                                                {payment.payment_date
                                                    ? format(
                                                          new Date(
                                                              payment.payment_date
                                                          ),
                                                          "PPP p"
                                                      )
                                                    : "N/A"}
                                            </p>
                                        </div>
                                        <div>
                                            <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                Amount
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-900 dark:text-gray-100 font-medium">
                                                {formatCurrency(
                                                    payment.amount,
                                                    payment.currency
                                                )}
                                            </p>
                                        </div>
                                        <div>
                                            <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                Status
                                            </h3>
                                            <p className="mt-1">
                                                <span
                                                    className={`px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full ${getStatusBadgeClass(
                                                        payment.status
                                                    )}`}
                                                >
                                                    {payment.status}
                                                </span>
                                            </p>
                                        </div>
                                    </div>
                                </div>

                                {/* Refund History Section */}
                                <div className="mt-6">
                                    <div className="flex justify-between items-center mb-4">
                                        <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 flex items-center">
                                            <Repeat className="h-5 w-5 mr-2 text-purple-500" />
                                            Refund History
                                        </h2>
                                        {loadingRefunds && (
                                            <div className="flex items-center text-sm text-gray-500 dark:text-gray-400">
                                                <Loader className="h-4 w-4 mr-2 animate-spin" />
                                                Loading refunds...
                                            </div>
                                        )}
                                    </div>

                                    {refundDetails && (
                                        <div className="bg-gray-50 dark:bg-gray-700 rounded-lg overflow-x-auto">
                                            <RefundHistory
                                                refunds={refunds}
                                                currency={payment.currency}
                                            />
                                        </div>
                                    )}
                                </div>

                                {/* Order Information */}
                                {payment.order && (
                                    <div className="mt-6">
                                        <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                            <FileText className="h-5 w-5 mr-2 text-indigo-500" />
                                            Order Information
                                        </h2>
                                        <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                                            <div className="flex justify-between items-center mb-4">
                                                <Link
                                                    href={route(
                                                        "admin.orders.show",
                                                        payment.order.id
                                                    )}
                                                    className="text-blue-600 dark:text-blue-400 hover:underline flex items-center"
                                                >
                                                    <span className="text-lg font-medium">
                                                        Order #
                                                        {
                                                            payment.order
                                                                .order_number
                                                        }
                                                    </span>
                                                    <ArrowLeft className="h-4 w-4 ml-1 transform rotate-180" />
                                                </Link>
                                                <span
                                                    className={`px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                                        payment.order
                                                            .order_status ===
                                                        "bidding"
                                                            ? "bg-blue-100 text-blue-800 dark:bg-blue-800 dark:text-blue-200"
                                                            : payment.order
                                                                  .order_status ===
                                                              "writing"
                                                            ? "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-200"
                                                            : payment.order
                                                                  .order_status ===
                                                              "completed"
                                                            ? "bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200"
                                                            : "bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300"
                                                    }`}
                                                >
                                                    {payment.order.order_status}
                                                </span>
                                            </div>
                                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                                <div>
                                                    <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Order Title
                                                    </h3>
                                                    <p className="mt-1 text-sm text-gray-900 dark:text-gray-100">
                                                        {payment.order.title}
                                                    </p>
                                                </div>
                                                <div>
                                                    <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Payment Status
                                                    </h3>
                                                    <p className="mt-1 text-sm text-gray-900 dark:text-gray-100">
                                                        <span
                                                            className={`px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full ${
                                                                payment.order
                                                                    .payment_status ===
                                                                "paid"
                                                                    ? "bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200"
                                                                    : "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-200"
                                                            }`}
                                                        >
                                                            {
                                                                payment.order
                                                                    .payment_status
                                                            }
                                                        </span>
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                )}

                                {/* Client Information */}
                                {payment.client && (
                                    <div className="mt-6">
                                        <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                            <User className="h-5 w-5 mr-2 text-green-500" />
                                            Client Information
                                        </h2>
                                        <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                                            <div className="grid grid-cols-1 sm:grid-cols-2 gap-4">
                                                <div>
                                                    <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Name
                                                    </h3>
                                                    <p className="mt-1 text-sm text-gray-900 dark:text-gray-100">
                                                        <Link
                                                            href={route(
                                                                "admin.users.show",
                                                                payment.client
                                                                    .id
                                                            )}
                                                            className="text-blue-600 dark:text-blue-400 hover:underline"
                                                        >
                                                            {
                                                                payment.client
                                                                    .name
                                                            }
                                                        </Link>
                                                    </p>
                                                </div>
                                                <div>
                                                    <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Email
                                                    </h3>
                                                    <p className="mt-1 text-sm text-gray-900 dark:text-gray-100">
                                                        <a
                                                            href={`mailto:${payment.client.email}`}
                                                            className="text-blue-600 dark:text-blue-400 hover:underline"
                                                        >
                                                            {
                                                                payment.client
                                                                    .email
                                                            }
                                                        </a>
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                )}
                            </div>

                            {/* Payment Timeline/Status */}
                            <div className="col-span-1">
                                <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                    <Calendar className="h-5 w-5 mr-2 text-purple-500" />
                                    Payment Timeline
                                </h2>
                                <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-4">
                                    <ol className="relative border-l border-gray-200 dark:border-gray-600 ml-3">
                                        <li className="mb-6 ml-6">
                                            <span className="absolute flex items-center justify-center w-6 h-6 bg-blue-100 rounded-full -left-3 ring-8 ring-gray-50 dark:ring-gray-700 dark:bg-blue-900">
                                                <Clock className="w-3 h-3 text-blue-800 dark:text-blue-300" />
                                            </span>
                                            <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-gray-100">
                                                Payment Created
                                            </h3>
                                            <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                {payment.created_at
                                                    ? format(
                                                          new Date(
                                                              payment.created_at
                                                          ),
                                                          "PPp"
                                                      )
                                                    : "N/A"}
                                            </time>
                                            <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                Payment was initiated
                                            </p>
                                        </li>
                                        {payment.status === "completed" &&
                                            payment.payment_date && (
                                                <li className="mb-6 ml-6">
                                                    <span className="absolute flex items-center justify-center w-6 h-6 bg-green-100 rounded-full -left-3 ring-8 ring-gray-50 dark:ring-gray-700 dark:bg-green-900">
                                                        <CheckCircle className="w-3 h-3 text-green-800 dark:text-green-300" />
                                                    </span>
                                                    <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-gray-100">
                                                        Payment Completed
                                                    </h3>
                                                    <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                        {format(
                                                            new Date(
                                                                payment.payment_date
                                                            ),
                                                            "PPp"
                                                        )}
                                                    </time>
                                                    <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                        Payment was successfully
                                                        processed
                                                    </p>
                                                </li>
                                            )}

                                        {/* Refund Events in Timeline */}
                                        {refunds &&
                                            refunds.length > 0 &&
                                            refunds.map((refund) => (
                                                <li
                                                    key={refund.id}
                                                    className="mb-6 ml-6"
                                                >
                                                    <span className="absolute flex items-center justify-center w-6 h-6 bg-purple-100 rounded-full -left-3 ring-8 ring-gray-50 dark:ring-gray-700 dark:bg-purple-900">
                                                        <Repeat className="w-3 h-3 text-purple-800 dark:text-purple-300" />
                                                    </span>
                                                    <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-gray-100">
                                                        Refund{" "}
                                                        {refund.status ===
                                                        "completed"
                                                            ? "Processed"
                                                            : refund.status ===
                                                              "pending"
                                                            ? "Initiated"
                                                            : "Failed"}
                                                    </h3>
                                                    <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                        {refund.processed_at
                                                            ? format(
                                                                  new Date(
                                                                      refund.processed_at
                                                                  ),
                                                                  "PPp"
                                                              )
                                                            : format(
                                                                  new Date(
                                                                      refund.created_at
                                                                  ),
                                                                  "PPp"
                                                              )}
                                                    </time>
                                                    <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                        {formatCurrency(
                                                            refund.amount,
                                                            refund.currency ||
                                                                payment.currency
                                                        )}{" "}
                                                        refund{" "}
                                                        {refund.status ===
                                                        "completed"
                                                            ? "was issued"
                                                            : refund.status ===
                                                              "pending"
                                                            ? "is pending"
                                                            : "failed"}
                                                    </p>
                                                    {refund.reason && (
                                                        <p className="mt-1 text-xs text-gray-500 dark:text-gray-400">
                                                            Reason:{" "}
                                                            {refund.reason}
                                                        </p>
                                                    )}
                                                </li>
                                            ))}

                                        {payment.status === "failed" && (
                                            <li className="mb-6 ml-6">
                                                <span className="absolute flex items-center justify-center w-6 h-6 bg-red-100 rounded-full -left-3 ring-8 ring-gray-50 dark:ring-gray-700 dark:bg-red-900">
                                                    <AlertCircle className="w-3 h-3 text-red-800 dark:text-red-300" />
                                                </span>
                                                <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-gray-100">
                                                    Payment Failed
                                                </h3>
                                                <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                    {payment.updated_at
                                                        ? format(
                                                              new Date(
                                                                  payment.updated_at
                                                              ),
                                                              "PPp"
                                                          )
                                                        : "N/A"}
                                                </time>
                                                <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                    Payment processing failed
                                                </p>
                                                {payment.metadata?.error && (
                                                    <p className="mt-1 text-xs text-red-600 dark:text-red-400">
                                                        {payment.metadata.error}
                                                    </p>
                                                )}
                                            </li>
                                        )}

                                        {payment.status === "refunded" && (
                                            <li className="mb-6 ml-6">
                                                <span className="absolute flex items-center justify-center w-6 h-6 bg-purple-100 rounded-full -left-3 ring-8 ring-gray-50 dark:ring-gray-700 dark:bg-purple-900">
                                                    <Repeat className="w-3 h-3 text-purple-800 dark:text-purple-300" />
                                                </span>
                                                <h3 className="mb-1 text-sm font-semibold text-gray-900 dark:text-gray-100">
                                                    Payment Fully Refunded
                                                </h3>
                                                <time className="block text-xs font-normal leading-none text-gray-500 dark:text-gray-400">
                                                    {payment.updated_at
                                                        ? format(
                                                              new Date(
                                                                  payment.updated_at
                                                              ),
                                                              "PPp"
                                                          )
                                                        : "N/A"}
                                                </time>
                                                <p className="text-sm font-normal text-gray-500 dark:text-gray-400">
                                                    Payment was refunded to
                                                    customer
                                                </p>
                                            </li>
                                        )}
                                    </ol>
                                </div>
                            </div>
                        </div>

                        {/* Raw Data/Metadata */}
                        {showRawData && payment.metadata && (
                            <div className="px-6 py-4 border-t border-gray-200 dark:border-gray-700">
                                <h2 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                    <Code className="h-5 w-5 mr-2 text-gray-500" />
                                    Raw Payment Data
                                </h2>
                                <div className="bg-gray-800 rounded-lg p-4 overflow-x-auto max-w-full">
                                    <pre className="text-xs text-gray-300 font-mono whitespace-pre-wrap">
                                        {JSON.stringify(
                                            payment.metadata,
                                            null,
                                            2
                                        )}
                                    </pre>
                                </div>
                            </div>
                        )}
                    </div>
                </div>
            </div>

            {/* Refund Modal */}
            <RefundModal
                payment={payment}
                isOpen={showRefundModal}
                onClose={() => setShowRefundModal(false)}
                onSuccess={handleRefundSuccess}
            />
        </AdminLayout>
    );
};

export default PaymentDetail;
