import React, { useState } from "react";
import { Head, Link, router } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import Pagination from "@/Components/Pagination";
import {
    Search,
    Filter,
    RefreshCw,
    Download,
    FileText,
    CreditCard,
    Calendar,
    User,
    ChevronDown,
    Eye,
    X,
    AlertCircle,
    CheckCircle,
    DollarSign,
    Clock,
    XCircle,
    BarChart2,
    ShoppingBasket,
    ShoppingCartIcon,
} from "lucide-react";
import { format } from "date-fns";

const PaymentHistory = ({ payments, filters, paymentMethods, statuses }) => {
    const [showFilters, setShowFilters] = useState(false);
    const [formFilters, setFormFilters] = useState({
        search: filters.search || "",
        status: filters.status || "",
        method: filters.method || "",
        start_date: filters.start_date || "",
        end_date: filters.end_date || "",
        per_page: filters.per_page || 15,
    });

    // Apply filters
    const applyFilters = () => {
        router.get(route("admin.payments.history"), formFilters, {
            preserveState: true,
            preserveScroll: true,
            only: ["payments"],
        });
    };

    // Reset filters
    const resetFilters = () => {
        setFormFilters({
            search: "",
            status: "",
            method: "",
            start_date: "",
            end_date: "",
            per_page: 15,
        });

        router.get(
            route("admin.payments.history"),
            {
                per_page: 15,
            },
            {
                preserveState: true,
                preserveScroll: true,
                only: ["payments"],
            }
        );
    };

    // Export filtered data
    const exportData = () => {
        const queryString = new URLSearchParams(formFilters).toString();
        window.open(
            `${route("admin.payments.export")}?${queryString}`,
            "_blank"
        );
    };

    // Format currency
    const formatCurrency = (amount, currency = "USD") => {
        return new Intl.NumberFormat("en-US", {
            style: "currency",
            currency: currency,
        }).format(amount);
    };

    // Get status badge class
    const getStatusBadgeClass = (status) => {
        switch (status) {
            case "completed":
                return "bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-200";
            case "pending":
                return "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-200";
            case "failed":
                return "bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-200";
            case "refunded":
                return "bg-purple-100 text-purple-800 dark:bg-purple-800 dark:text-purple-200";
            default:
                return "bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300";
        }
    };

    // Get status icon
    const getStatusIcon = (status) => {
        switch (status) {
            case "completed":
                return <CheckCircle size={14} className="mr-1" />;
            case "pending":
                return <Clock size={14} className="mr-1" />;
            case "failed":
                return <AlertCircle size={14} className="mr-1" />;
            case "refunded":
                return <XCircle size={14} className="mr-1" />;
            default:
                return null;
        }
    };

    return (
        <AdminLayout>
            <Head title="Payment History" />

            <div className="py-6 overflow-hidden">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8 overflow-hidden">
                    <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6 gap-3">
                        <div>
                            <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                                Payment History
                            </h1>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Manage and view all payment transactions
                            </p>
                        </div>
                        <div className="flex flex-wrap gap-2 sm:space-x-2">
                            <Link
                                href={route("admin.payment.analytics")}
                                className="px-4 py-2 bg-blue-600 hover:bg-blue-700 rounded-md text-white flex items-center transition duration-150 ease-in-out"
                            >
                                <BarChart2 size={16} className="mr-2" />
                                Analytics
                            </Link>
                            <button
                                onClick={() => setShowFilters(!showFilters)}
                                className="px-4 py-2 bg-indigo-600 hover:bg-indigo-700 rounded-md text-white flex items-center transition duration-150 ease-in-out"
                            >
                                <Filter size={16} className="mr-2" />
                                {showFilters ? "Hide Filters" : "Show Filters"}
                            </button>
                            <button
                                onClick={exportData}
                                className="px-4 py-2 bg-green-600 hover:bg-green-700 rounded-md text-white flex items-center transition duration-150 ease-in-out"
                            >
                                <Download size={16} className="mr-2" />
                                Export CSV
                            </button>
                        </div>
                    </div>

                    {/* Filters */}
                    {showFilters && (
                        <div className="bg-white dark:bg-gray-800 p-4 rounded-lg shadow mb-6 overflow-hidden">
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-4">
                                <div>
                                    <label
                                        htmlFor="search"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                    >
                                        Search
                                    </label>
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <Search className="h-4 w-4 text-gray-400" />
                                        </div>
                                        <input
                                            type="text"
                                            id="search"
                                            className="pl-10 block w-full border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white sm:text-sm"
                                            placeholder="Transaction ID, Order #, Client"
                                            value={formFilters.search}
                                            onChange={(e) =>
                                                setFormFilters({
                                                    ...formFilters,
                                                    search: e.target.value,
                                                })
                                            }
                                        />
                                    </div>
                                </div>
                                <div>
                                    <label
                                        htmlFor="status"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                    >
                                        Status
                                    </label>
                                    <select
                                        id="status"
                                        className="block w-full border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white sm:text-sm"
                                        value={formFilters.status}
                                        onChange={(e) =>
                                            setFormFilters({
                                                ...formFilters,
                                                status: e.target.value,
                                            })
                                        }
                                    >
                                        <option value="">All Statuses</option>
                                        {statuses.map((status) => (
                                            <option key={status} value={status}>
                                                {status
                                                    .charAt(0)
                                                    .toUpperCase() +
                                                    status.slice(1)}
                                            </option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label
                                        htmlFor="method"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                    >
                                        Payment Method
                                    </label>
                                    <select
                                        id="method"
                                        className="block w-full border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white sm:text-sm"
                                        value={formFilters.method}
                                        onChange={(e) =>
                                            setFormFilters({
                                                ...formFilters,
                                                method: e.target.value,
                                            })
                                        }
                                    >
                                        <option value="">All Methods</option>
                                        {paymentMethods.map((method) => (
                                            <option key={method} value={method}>
                                                {method
                                                    .charAt(0)
                                                    .toUpperCase() +
                                                    method.slice(1)}
                                            </option>
                                        ))}
                                    </select>
                                </div>
                                <div>
                                    <label
                                        htmlFor="start_date"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                    >
                                        From Date
                                    </label>
                                    <input
                                        type="date"
                                        id="start_date"
                                        className="block w-full border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white sm:text-sm"
                                        value={formFilters.start_date}
                                        onChange={(e) =>
                                            setFormFilters({
                                                ...formFilters,
                                                start_date: e.target.value,
                                            })
                                        }
                                    />
                                </div>
                                <div>
                                    <label
                                        htmlFor="end_date"
                                        className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                    >
                                        To Date
                                    </label>
                                    <input
                                        type="date"
                                        id="end_date"
                                        className="block w-full border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white sm:text-sm"
                                        value={formFilters.end_date}
                                        onChange={(e) =>
                                            setFormFilters({
                                                ...formFilters,
                                                end_date: e.target.value,
                                            })
                                        }
                                    />
                                </div>
                            </div>
                            <div className="mt-4 flex justify-end space-x-2">
                                <button
                                    onClick={resetFilters}
                                    type="button"
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 shadow-sm text-sm font-medium rounded-md text-gray-700 dark:text-gray-200 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                >
                                    <X className="h-4 w-4 mr-2" />
                                    Reset
                                </button>
                                <button
                                    onClick={applyFilters}
                                    type="button"
                                    className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                >
                                    <RefreshCw className="h-4 w-4 mr-2" />
                                    Apply Filters
                                </button>
                            </div>
                        </div>
                    )}

                    {/* Payment Records Table */}
                    <div className="bg-white dark:bg-gray-800 shadow-sm rounded-lg overflow-hidden">
                        {/* Replace the table with responsive divs */}

                        {/* Table headers - visible only on medium screens and up */}
                        <div className="hidden md:grid md:grid-cols-8 bg-gray-50 dark:bg-gray-700 p-4 gap-4 border-b border-gray-200 dark:border-gray-700">
                            <div className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Transaction
                            </div>
                            <div className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Date
                            </div>
                            <div className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Order
                            </div>
                            <div className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Client
                            </div>
                            <div className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Amount
                            </div>
                            <div className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Method
                            </div>
                            <div className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Status
                            </div>
                            <div className="text-xs font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
                                Actions
                            </div>
                        </div>

                        {/* Payment rows */}
                        {payments.data.length > 0 ? (
                            payments.data.map((payment) => (
                                <div
                                    key={payment.id}
                                    className="border-b border-gray-200 dark:border-gray-700 hover:bg-gray-50 dark:hover:bg-gray-700"
                                >
                                    {/* Desktop view - grid layout matching the headers above */}
                                    <div className="hidden md:grid md:grid-cols-8 p-4 gap-4 items-center">
                                        {/* Transaction ID */}
                                        <div className="flex items-center">
                                            <CreditCard className="flex-shrink-0 h-5 w-5 text-gray-400 mr-2" />
                                            <div className="truncate">
                                                <span className="font-mono text-sm text-gray-900 dark:text-gray-100">
                                                    {payment.transaction_id ||
                                                        "—"}
                                                </span>
                                            </div>
                                        </div>

                                        {/* Date */}
                                        <div className="flex items-center text-sm text-gray-500 dark:text-gray-400">
                                            <Calendar className="flex-shrink-0 h-5 w-5 text-gray-400 mr-2" />
                                            {payment.created_at
                                                ? format(
                                                      new Date(
                                                          payment.created_at
                                                      ),
                                                      "MMM dd, yyyy"
                                                  )
                                                : "—"}
                                        </div>

                                        {/* Order */}
                                        <div className="flex items-center text-sm text-gray-500 dark:text-gray-400">
                                            <FileText className="flex-shrink-0 h-5 w-5 text-gray-400 mr-2" />
                                            {payment.order_number ? (
                                                <Link
                                                    href={route(
                                                        "admin.orders.show",
                                                        payment.order_id
                                                    )}
                                                    className="text-blue-600 dark:text-blue-400 hover:underline"
                                                >
                                                    #{payment.order_number}
                                                </Link>
                                            ) : (
                                                "—"
                                            )}
                                        </div>

                                        {/* Client */}
                                        <div className="flex items-center text-sm text-gray-500 dark:text-gray-400">
                                            <User className="flex-shrink-0 h-5 w-5 text-gray-400 mr-2" />
                                            {payment.client ? (
                                                <div className="truncate">
                                                    <Link
                                                        href={route(
                                                            "admin.users.show",
                                                            payment.client.id
                                                        )}
                                                        className="text-blue-600 dark:text-blue-400 hover:underline"
                                                        title={
                                                            payment.client.email
                                                        }
                                                    >
                                                        {payment.client.name}
                                                    </Link>
                                                </div>
                                            ) : (
                                                "—"
                                            )}
                                        </div>

                                        {/* Amount */}
                                        <div className="flex items-center text-sm font-medium text-gray-900 dark:text-gray-100">
                                            <ShoppingCartIcon className="flex-shrink-0 h-5 w-5 text-gray-400 mr-2" />
                                            <span>
                                                {formatCurrency(
                                                    payment.amount,
                                                    payment.currency
                                                )}
                                            </span>
                                        </div>

                                        {/* Method */}
                                        <div className="text-sm text-gray-500 dark:text-gray-400 capitalize">
                                            {payment.payment_method}
                                        </div>

                                        {/* Status */}
                                        <div>
                                            <span
                                                className={`px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full ${getStatusBadgeClass(
                                                    payment.status
                                                )}`}
                                            >
                                                {getStatusIcon(payment.status)}
                                                {payment.status}
                                            </span>
                                        </div>

                                        {/* Actions */}
                                        <div className="text-sm font-medium">
                                            <Link
                                                href={route(
                                                    "admin.payments.show",
                                                    payment.id
                                                )}
                                                className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300 inline-flex items-center"
                                            >
                                                <Eye className="h-4 w-4 mr-1" />
                                                View
                                            </Link>
                                        </div>
                                    </div>

                                    {/* Mobile view - card layout with stacked information */}
                                    <div className="block md:hidden p-4 mt-2 rounded-lg shadow-md">
                                        <div className="flex justify-between items-start mb-3">
                                            <span
                                                className={`px-2 py-1 inline-flex text-xs leading-5 font-semibold rounded-full ${getStatusBadgeClass(
                                                    payment.status
                                                )}`}
                                            >
                                                {getStatusIcon(payment.status)}
                                                {payment.status}
                                            </span>
                                            <span className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                                {formatCurrency(
                                                    payment.amount,
                                                    payment.currency
                                                )}
                                            </span>
                                        </div>

                                        <div className="grid grid-cols-2 gap-2 mb-3">
                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                Transaction ID:
                                            </div>
                                            <div className="text-xs font-mono text-gray-900 dark:text-gray-100 truncate">
                                                {payment.transaction_id || "—"}
                                            </div>

                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                Date:
                                            </div>
                                            <div className="text-xs text-gray-900 dark:text-gray-100">
                                                {payment.created_at
                                                    ? format(
                                                          new Date(
                                                              payment.created_at
                                                          ),
                                                          "MMM dd, yyyy"
                                                      )
                                                    : "—"}
                                            </div>

                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                Order:
                                            </div>
                                            <div className="text-xs text-gray-900 dark:text-gray-100">
                                                {payment.order_number ? (
                                                    <Link
                                                        href={route(
                                                            "admin.orders.show",
                                                            payment.order_id
                                                        )}
                                                        className="text-blue-600 dark:text-blue-400 hover:underline"
                                                    >
                                                        #{payment.order_number}
                                                    </Link>
                                                ) : (
                                                    "—"
                                                )}
                                            </div>

                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                Client:
                                            </div>
                                            <div className="text-xs text-gray-900 dark:text-gray-100 truncate">
                                                {payment.client ? (
                                                    <Link
                                                        href={route(
                                                            "admin.users.show",
                                                            payment.client.id
                                                        )}
                                                        className="text-blue-600 dark:text-blue-400 hover:underline"
                                                        title={
                                                            payment.client.email
                                                        }
                                                    >
                                                        {payment.client.name}
                                                    </Link>
                                                ) : (
                                                    "—"
                                                )}
                                            </div>

                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                Method:
                                            </div>
                                            <div className="text-xs text-gray-900 dark:text-gray-100 capitalize">
                                                {payment.payment_method}
                                            </div>
                                        </div>

                                        <div className="mt-3 flex justify-end">
                                            <Link
                                                href={route(
                                                    "admin.payments.show",
                                                    payment.id
                                                )}
                                                className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300 inline-flex items-center text-sm"
                                            >
                                                <Eye className="h-4 w-4 mr-1" />
                                                View Details
                                            </Link>
                                        </div>
                                    </div>
                                </div>
                            ))
                        ) : (
                            <div className="px-6 py-10 text-center text-sm text-gray-500 dark:text-gray-400">
                                <div className="flex flex-col items-center">
                                    <AlertCircle className="h-10 w-10 text-gray-400 mb-2" />
                                    <p className="mb-1 font-medium">
                                        No payment records found
                                    </p>
                                    <p className="text-xs">
                                        Try adjusting your search or filter
                                        criteria
                                    </p>
                                </div>
                            </div>
                        )}

                        {/* Pagination remains the same */}
                        {/* Pagination */}
                        <div className="bg-white dark:bg-gray-800 px-4 py-3 border-t border-gray-200 dark:border-gray-700 sm:px-6">
                            <div className="flex items-center justify-between">
                                <div className="flex-1 flex justify-between sm:hidden">
                                    {payments.prev_page_url && (
                                        <Link
                                            href={payments.prev_page_url}
                                            className="relative inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 text-sm font-medium rounded-md text-gray-700 dark:text-gray-200 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600"
                                        >
                                            Previous
                                        </Link>
                                    )}
                                    {payments.next_page_url && (
                                        <Link
                                            href={payments.next_page_url}
                                            className="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 dark:border-gray-600 text-sm font-medium rounded-md text-gray-700 dark:text-gray-200 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600"
                                        >
                                            Next
                                        </Link>
                                    )}
                                </div>
                                <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                                    <div>
                                        <p className="text-sm text-gray-700 dark:text-gray-400">
                                            Showing{" "}
                                            <span className="font-medium">
                                                {payments.from || 0}
                                            </span>{" "}
                                            to{" "}
                                            <span className="font-medium">
                                                {payments.to || 0}
                                            </span>{" "}
                                            of{" "}
                                            <span className="font-medium">
                                                {payments.total}
                                            </span>{" "}
                                            results
                                        </p>
                                    </div>
                                    <div>
                                        <Pagination links={payments.links} />
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default PaymentHistory;
