import React, { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Search,
    Filter,
    Plus,
    ChevronDown,
    ChevronUp,
    Edit,
    Trash2,
    Eye,
    ToggleLeft,
    ToggleRight,
    Download,
    RefreshCw,
    BarChart,
    CalendarRange,
} from "lucide-react";
import { Transition } from "@headlessui/react";
import Pagination from "@/Components/Pagination";
import Tooltip from "@/Components/ui/Tooltip";

import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import EmptyState from "@/Components/ui/EmptyState";
import Alert from "@/Components/ui/Alert";
import { toast } from "react-hot-toast";

const Index = ({ seasonalTypes, counts, filters }) => {
    const [showFilters, setShowFilters] = useState(false);
    const [selectedTypes, setSelectedTypes] = useState([]);
    const [sortConfig, setSortConfig] = useState({
        key: filters.sort_field || "name",
        direction: filters.sort_direction || "asc",
    });
    const [flashMessage, setFlashMessage] = useState(null);

    // Check for flash messages from the session
    React.useEffect(() => {
        const message = window.flash?.message;
        if (message) {
            setFlashMessage(message);
            toast.success(message);
        }

        const error = window.flash?.error;
        if (error) {
            toast.error(error);
        }
    }, []);

    // Form for filters
    const { data, setData, get, processing, errors } = useForm({
        search: filters.search || "",
        status: filters.status || "",
        date_range: filters.date_range || "",
        period: filters.period || "",
    });

    // Handle filter submission
    const handleFilterSubmit = (e) => {
        e.preventDefault();
        get(route("admin.seasonal-types.index"), {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Reset filters
    const resetFilters = () => {
        setData({
            search: "",
            status: "",
            date_range: "",
            period: "",
        });
        get(route("admin.seasonal-types.index"), {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Handle sorting
    const requestSort = (key) => {
        const direction =
            sortConfig.key === key && sortConfig.direction === "asc"
                ? "desc"
                : "asc";
        setSortConfig({ key, direction });

        get(
            route("admin.seasonal-types.index", {
                ...data,
                sort_field: key,
                sort_direction: direction,
            }),
            {
                preserveState: true,
                preserveScroll: true,
            }
        );
    };

    // Format date for display
    const formatDate = (dateString) => {
        if (!dateString) return "N/A";
        return new Date(dateString).toLocaleDateString(undefined, {
            month: "short",
            day: "numeric",
        });
    };

    // Toggle seasonal type selection
    const toggleTypeSelection = (typeId) => {
        setSelectedTypes((prev) =>
            prev.includes(typeId)
                ? prev.filter((id) => id !== typeId)
                : [...prev, typeId]
        );
    };

    // Toggle all seasonal types
    const toggleAllTypes = () => {
        if (selectedTypes.length === seasonalTypes.data.length) {
            setSelectedTypes([]);
        } else {
            setSelectedTypes(seasonalTypes.data.map((type) => type.id));
        }
    };

    // Bulk actions
    const handleBulkAction = (action) => {
        if (selectedTypes.length === 0) {
            toast.error("Please select at least one seasonal type");
            return;
        }

        if (action === "activate") {
            if (confirm("Activate all selected seasonal types?")) {
                // Submit the bulk action form
                const form = document.getElementById("bulk-toggle-form");
                document.getElementById("bulk-status").value = "1";
                form.submit();
            }
        } else if (action === "deactivate") {
            if (confirm("Deactivate all selected seasonal types?")) {
                // Submit the bulk action form
                const form = document.getElementById("bulk-toggle-form");
                document.getElementById("bulk-status").value = "0";
                form.submit();
            }
        } else if (action === "delete") {
            if (
                confirm(
                    "Delete all selected seasonal types? This action cannot be undone."
                )
            ) {
                // Submit the bulk delete form
                document.getElementById("bulk-delete-form").submit();
            }
        }
    };

    // Get status badge
    const getStatusBadge = (type) => {
        if (!type.is_active) {
            return (
                <Badge
                    variant="secondary"
                    className="bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200"
                >
                    Inactive
                </Badge>
            );
        }

        // Check if it's currently active season
        const now = new Date();
        const startMonth = new Date(type.default_start_date).getMonth();
        const startDay = new Date(type.default_start_date).getDate();
        const endMonth = new Date(type.default_end_date).getMonth();
        const endDay = new Date(type.default_end_date).getDate();

        // Handle cross-year seasons (like winter: Dec-Jan)
        let isCurrentSeason = false;
        const currentMonth = now.getMonth();
        const currentDay = now.getDate();

        if (startMonth < endMonth) {
            // Regular season within same year
            isCurrentSeason =
                (currentMonth > startMonth ||
                    (currentMonth === startMonth && currentDay >= startDay)) &&
                (currentMonth < endMonth ||
                    (currentMonth === endMonth && currentDay <= endDay));
        } else {
            // Cross-year season (e.g. winter)
            isCurrentSeason =
                currentMonth > startMonth ||
                (currentMonth === startMonth && currentDay >= startDay) ||
                currentMonth < endMonth ||
                (currentMonth === endMonth && currentDay <= endDay);
        }

        return isCurrentSeason ? (
            <Badge
                variant="success"
                className="bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-300"
            >
                Active Now
            </Badge>
        ) : (
            <Badge
                variant="info"
                className="bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300"
            >
                Active
            </Badge>
        );
    };

    // Get date range display
    const getDateRangeDisplay = (type) => {
        if (!type.default_start_date || !type.default_end_date) {
            return "Not set";
        }

        const startDate = new Date(type.default_start_date);
        const endDate = new Date(type.default_end_date);

        const formatMonthDay = (date) => {
            return date.toLocaleDateString(undefined, {
                month: "short",
                day: "numeric",
            });
        };

        return `${formatMonthDay(startDate)} - ${formatMonthDay(endDate)}`;
    };

    return (
        <AdminLayout>
            <Head title="Seasonal Type Management" />

            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between space-y-3 md:space-y-0 mb-6">
                        <div>
                            <h2 className="text-2xl font-semibold text-gray-900 dark:text-white">
                                Seasonal Type Management
                            </h2>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                Create and manage seasonal periods for
                                promotional campaigns
                            </p>
                        </div>
                        <div className="flex flex-col sm:flex-row space-y-2 sm:space-y-0 sm:space-x-2">
                            <Link
                                href={route("admin.seasonal-types.statistics")}
                            >
                                <Button
                                    type="button"
                                    variant="outline"
                                    className="w-full sm:w-auto"
                                >
                                    <BarChart className="w-4 h-4 mr-2" />
                                    Statistics
                                </Button>
                            </Link>
                            <Link href={route("admin.seasonal-types.create")}>
                                <Button className="w-full sm:w-auto">
                                    <Plus className="w-4 h-4 mr-2" />
                                    New Seasonal Type
                                </Button>
                            </Link>
                        </div>
                    </div>

                    {/* Flash Message */}
                    {flashMessage && (
                        <Alert
                            variant="success"
                            message={flashMessage}
                            className="mb-6"
                            dismissible
                            onDismiss={() => setFlashMessage(null)}
                        />
                    )}

                    {/* Search and Filters */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg shadow mb-6">
                        <div className="p-4 sm:p-6 border-b border-gray-200 dark:border-gray-700">
                            <div className="flex flex-col sm:flex-row sm:items-center space-y-3 sm:space-y-0 justify-between">
                                <div className="w-full sm:max-w-xs flex-grow sm:flex-grow-0">
                                    <label htmlFor="search" className="sr-only">
                                        Search seasonal types
                                    </label>
                                    <div className="relative">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <Search
                                                className="h-5 w-5 text-gray-400"
                                                aria-hidden="true"
                                            />
                                        </div>
                                        <input
                                            type="text"
                                            name="search"
                                            id="search"
                                            className="block w-full pl-10 pr-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md leading-5 bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm transition duration-150 ease-in-out"
                                            placeholder="Search seasonal types"
                                            value={data.search}
                                            onChange={(e) =>
                                                setData(
                                                    "search",
                                                    e.target.value
                                                )
                                            }
                                            onKeyUp={(e) => {
                                                if (e.key === "Enter") {
                                                    handleFilterSubmit(e);
                                                }
                                            }}
                                        />
                                    </div>
                                </div>
                                <div className="flex items-center space-x-2">
                                    <Button
                                        type="button"
                                        variant="outline"
                                        onClick={() =>
                                            setShowFilters(!showFilters)
                                        }
                                        className="inline-flex items-center"
                                    >
                                        <Filter className="h-4 w-4 mr-2" />
                                        {showFilters
                                            ? "Hide Filters"
                                            : "Show Filters"}
                                    </Button>
                                </div>
                            </div>

                            {/* Expandable Filters */}
                            <Transition
                                show={showFilters}
                                enter="transition ease-out duration-200"
                                enterFrom="transform opacity-0 scale-95"
                                enterTo="transform opacity-100 scale-100"
                                leave="transition ease-in duration-150"
                                leaveFrom="transform opacity-100 scale-100"
                                leaveTo="transform opacity-0 scale-95"
                            >
                                <form
                                    onSubmit={handleFilterSubmit}
                                    className="mt-4"
                                >
                                    <div className="grid grid-cols-1 sm:grid-cols-3 gap-4">
                                        <div>
                                            <label
                                                htmlFor="status"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Status
                                            </label>
                                            <select
                                                id="status"
                                                name="status"
                                                className="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                                value={data.status}
                                                onChange={(e) =>
                                                    setData(
                                                        "status",
                                                        e.target.value
                                                    )
                                                }
                                            >
                                                <option value="">
                                                    All Statuses
                                                </option>
                                                <option value="active">
                                                    Active
                                                </option>
                                                <option value="inactive">
                                                    Inactive
                                                </option>
                                            </select>
                                        </div>

                                        <div>
                                            <label
                                                htmlFor="period"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300"
                                            >
                                                Current Period
                                            </label>
                                            <select
                                                id="period"
                                                name="period"
                                                className="mt-1 block w-full pl-3 pr-10 py-2 text-base border-gray-300 dark:border-gray-600 dark:bg-gray-700 dark:text-white focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm rounded-md"
                                                value={data.period}
                                                onChange={(e) =>
                                                    setData(
                                                        "period",
                                                        e.target.value
                                                    )
                                                }
                                            >
                                                <option value="">
                                                    All Periods
                                                </option>
                                                <option value="current">
                                                    Currently Active
                                                </option>
                                            </select>
                                        </div>

                                        <div className="flex items-end space-x-2">
                                            <Button
                                                type="submit"
                                                disabled={processing}
                                                className="px-4 py-2"
                                            >
                                                {processing ? (
                                                    <RefreshCw className="w-4 h-4 mr-2 animate-spin" />
                                                ) : (
                                                    <Search className="w-4 h-4 mr-2" />
                                                )}
                                                Filter
                                            </Button>
                                            <Button
                                                type="button"
                                                variant="outline"
                                                onClick={resetFilters}
                                                className="px-4 py-2"
                                            >
                                                Reset
                                            </Button>
                                        </div>
                                    </div>
                                </form>
                            </Transition>
                        </div>

                        {/* Bulk Actions */}
                        {selectedTypes.length > 0 && (
                            <div className="bg-blue-50 dark:bg-blue-900/20 border-t border-b border-blue-100 dark:border-blue-800 p-4">
                                <div className="flex items-center justify-between">
                                    <span className="text-sm font-medium text-blue-800 dark:text-blue-200">
                                        {selectedTypes.length} seasonal types
                                        selected
                                    </span>
                                    <div className="flex space-x-2">
                                        <Button
                                            type="button"
                                            variant="outline"
                                            onClick={() =>
                                                handleBulkAction("activate")
                                            }
                                            className="text-xs px-2 py-1"
                                        >
                                            <ToggleRight className="w-4 h-4 mr-1" />
                                            Activate
                                        </Button>
                                        <Button
                                            type="button"
                                            variant="outline"
                                            onClick={() =>
                                                handleBulkAction("deactivate")
                                            }
                                            className="text-xs px-2 py-1"
                                        >
                                            <ToggleLeft className="w-4 h-4 mr-1" />
                                            Deactivate
                                        </Button>
                                        <Button
                                            type="button"
                                            variant="destructive"
                                            onClick={() =>
                                                handleBulkAction("delete")
                                            }
                                            className="text-xs px-2 py-1"
                                        >
                                            <Trash2 className="w-4 h-4 mr-1" />
                                            Delete
                                        </Button>
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Hidden Bulk Action Forms */}
                        <form
                            id="bulk-toggle-form"
                            method="POST"
                            action={route(
                                "admin.seasonal-types.bulk-toggle-status"
                            )}
                        >
                            <input type="hidden" name="_method" value="PUT" />
                            <input
                                type="hidden"
                                name="_token"
                                value={window.csrfToken}
                            />
                            <input
                                type="hidden"
                                name="ids"
                                value={JSON.stringify(selectedTypes)}
                            />
                            <input
                                type="hidden"
                                id="bulk-status"
                                name="status"
                                value="1"
                            />
                        </form>

                        <form
                            id="bulk-delete-form"
                            method="POST"
                            action={route("admin.seasonal-types.bulk-delete")}
                        >
                            <input
                                type="hidden"
                                name="_method"
                                value="DELETE"
                            />
                            <input
                                type="hidden"
                                name="_token"
                                value={window.csrfToken}
                            />
                            <input
                                type="hidden"
                                name="ids"
                                value={JSON.stringify(selectedTypes)}
                            />
                        </form>

                        {/* Seasonal Types Table (CSS Grid) */}
                        <div className="overflow-x-auto">
                            {seasonalTypes.data.length > 0 ? (
                                <>
                                    {/* Table Header - Desktop */}
                                    <div className="hidden md:grid md:grid-cols-6 gap-4 px-6 py-3 border-b border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-800/80">
                                        <div className="flex items-center">
                                            <input
                                                type="checkbox"
                                                className="h-4 w-4 text-blue-600 rounded border-gray-300 dark:border-gray-600 dark:bg-gray-700 focus:ring-blue-500"
                                                checked={
                                                    selectedTypes.length ===
                                                        seasonalTypes.data
                                                            .length &&
                                                    seasonalTypes.data.length >
                                                        0
                                                }
                                                onChange={toggleAllTypes}
                                            />
                                            <button
                                                className="ml-3 text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider flex items-center"
                                                onClick={() =>
                                                    requestSort("name")
                                                }
                                            >
                                                Name
                                                {sortConfig.key === "name" && (
                                                    <>
                                                        {sortConfig.direction ===
                                                        "asc" ? (
                                                            <ChevronUp className="h-4 w-4 ml-1" />
                                                        ) : (
                                                            <ChevronDown className="h-4 w-4 ml-1" />
                                                        )}
                                                    </>
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider flex items-center">
                                            <button
                                                className="flex items-center"
                                                onClick={() =>
                                                    requestSort(
                                                        "default_start_date"
                                                    )
                                                }
                                            >
                                                Date Range
                                                {sortConfig.key ===
                                                    "default_start_date" && (
                                                    <>
                                                        {sortConfig.direction ===
                                                        "asc" ? (
                                                            <ChevronUp className="h-4 w-4 ml-1" />
                                                        ) : (
                                                            <ChevronDown className="h-4 w-4 ml-1" />
                                                        )}
                                                    </>
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Description
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider flex items-center">
                                            <button
                                                className="flex items-center"
                                                onClick={() =>
                                                    requestSort("coupons_count")
                                                }
                                            >
                                                Coupons
                                                {sortConfig.key ===
                                                    "coupons_count" && (
                                                    <>
                                                        {sortConfig.direction ===
                                                        "asc" ? (
                                                            <ChevronUp className="h-4 w-4 ml-1" />
                                                        ) : (
                                                            <ChevronDown className="h-4 w-4 ml-1" />
                                                        )}
                                                    </>
                                                )}
                                            </button>
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                            Status
                                        </div>
                                        <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider text-right">
                                            Actions
                                        </div>
                                    </div>

                                    {/* Table Rows */}
                                    <div className="divide-y divide-gray-200 dark:divide-gray-700">
                                        {seasonalTypes.data.map((type) => (
                                            <React.Fragment key={type.id}>
                                                {/* Desktop Row */}
                                                <div className="hidden md:grid md:grid-cols-6 gap-4 px-6 py-4 hover:bg-gray-50 dark:hover:bg-gray-800/60">
                                                    <div className="flex items-center">
                                                        <input
                                                            type="checkbox"
                                                            className="h-4 w-4 text-blue-600 rounded border-gray-300 dark:border-gray-600 dark:bg-gray-700 focus:ring-blue-500"
                                                            checked={selectedTypes.includes(
                                                                type.id
                                                            )}
                                                            onChange={() =>
                                                                toggleTypeSelection(
                                                                    type.id
                                                                )
                                                            }
                                                        />
                                                        <span className="ml-3 text-sm font-medium text-gray-900 dark:text-gray-100">
                                                            {type.name}
                                                        </span>
                                                    </div>
                                                    <div className="text-sm text-gray-700 dark:text-gray-300 flex items-center">
                                                        <CalendarRange className="w-4 h-4 mr-1 text-blue-600 dark:text-blue-400" />
                                                        {getDateRangeDisplay(
                                                            type
                                                        )}
                                                    </div>
                                                    <div className="text-sm text-gray-700 dark:text-gray-300 truncate">
                                                        {type.description ||
                                                            "No description"}
                                                    </div>
                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                        {type.coupons_count ||
                                                            0}
                                                    </div>
                                                    <div>
                                                        {getStatusBadge(type)}
                                                    </div>
                                                    <div className="text-right">
                                                        <div className="flex items-center justify-end space-x-2">
                                                            <Tooltip content="View Details">
                                                                <Link
                                                                    href={route(
                                                                        "admin.seasonal-types.show",
                                                                        type.id
                                                                    )}
                                                                    className="text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                                                >
                                                                    <Eye className="h-5 w-5 text-blue-600" />
                                                                </Link>
                                                            </Tooltip>

                                                            <Tooltip content="Edit">
                                                                <Link
                                                                    href={route(
                                                                        "admin.seasonal-types.edit",
                                                                        type.id
                                                                    )}
                                                                    className="text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 transition-colors"
                                                                >
                                                                    <Edit className="h-5 w-5 text-blue-600" />
                                                                </Link>
                                                            </Tooltip>

                                                            <Tooltip
                                                                content={
                                                                    type.is_active
                                                                        ? "Deactivate"
                                                                        : "Activate"
                                                                }
                                                            >
                                                                <Link
                                                                    href={route(
                                                                        "admin.seasonal-types.toggle-status",
                                                                        type.id
                                                                    )}
                                                                    method="post"
                                                                    as="button"
                                                                    className="text-gray-600 dark:text-gray-400 hover:text-amber-600 dark:hover:text-amber-400 transition-colors"
                                                                >
                                                                    {type.is_active ? (
                                                                        <ToggleLeft className="h-5 w-5 text-green-600" />
                                                                    ) : (
                                                                        <ToggleRight className="h-5 w-5 text-red-600" />
                                                                    )}
                                                                </Link>
                                                            </Tooltip>

                                                            <Tooltip content="Delete">
                                                                <Link
                                                                    href={route(
                                                                        "admin.seasonal-types.destroy",
                                                                        type.id
                                                                    )}
                                                                    method="delete"
                                                                    as="button"
                                                                    className="text-gray-600 dark:text-gray-400 hover:text-red-600 dark:hover:text-red-400 transition-colors"
                                                                    onClick={(
                                                                        e
                                                                    ) => {
                                                                        if (
                                                                            !confirm(
                                                                                "Are you sure you want to delete this seasonal type?"
                                                                            )
                                                                        ) {
                                                                            e.preventDefault();
                                                                        }
                                                                    }}
                                                                >
                                                                    <Trash2 className="h-5 w-5 text-red-600" />
                                                                </Link>
                                                            </Tooltip>
                                                        </div>
                                                    </div>
                                                </div>

                                                {/* Mobile Card View */}
                                                <div className="block md:hidden p-4 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                                                    <div className="flex items-center justify-between mb-2">
                                                        <div className="flex items-center">
                                                            <input
                                                                type="checkbox"
                                                                className="h-4 w-4 text-blue-600 rounded border-gray-300 dark:border-gray-600 dark:bg-gray-700 focus:ring-blue-500"
                                                                checked={selectedTypes.includes(
                                                                    type.id
                                                                )}
                                                                onChange={() =>
                                                                    toggleTypeSelection(
                                                                        type.id
                                                                    )
                                                                }
                                                            />
                                                            <span className="ml-3 text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                {type.name}
                                                            </span>
                                                        </div>
                                                        {getStatusBadge(type)}
                                                    </div>

                                                    <div className="grid grid-cols-2 gap-2 mb-3">
                                                        <div>
                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                Date Range
                                                            </p>
                                                            <p className="text-sm text-gray-900 dark:text-gray-100 flex items-center">
                                                                <CalendarRange className="w-4 h-4 mr-1 text-blue-600 dark:text-blue-400" />
                                                                {getDateRangeDisplay(
                                                                    type
                                                                )}
                                                            </p>
                                                        </div>
                                                        <div>
                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                Coupons
                                                            </p>
                                                            <p className="text-sm text-gray-900 dark:text-gray-100">
                                                                {type.coupons_count ||
                                                                    0}
                                                            </p>
                                                        </div>
                                                    </div>

                                                    {type.description && (
                                                        <div className="mb-3">
                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                Description
                                                            </p>
                                                            <p className="text-sm text-gray-900 dark:text-gray-100">
                                                                {
                                                                    type.description
                                                                }
                                                            </p>
                                                        </div>
                                                    )}

                                                    <div className="flex justify-end space-x-2 border-t border-gray-100 dark:border-gray-700 pt-2">
                                                        <Tooltip content="View Details">
                                                            <Link
                                                                href={route(
                                                                    "admin.seasonal-types.show",
                                                                    type.id
                                                                )}
                                                                className="text-xs text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                                            >
                                                                <Eye className="w-4 h-4" />
                                                            </Link>
                                                        </Tooltip>
                                                        <Tooltip content="Edit">
                                                            <Link
                                                                href={route(
                                                                    "admin.seasonal-types.edit",
                                                                    type.id
                                                                )}
                                                                className="text-xs text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200"
                                                            >
                                                                <Edit className="w-4 h-4" />
                                                            </Link>
                                                        </Tooltip>

                                                        <Tooltip content="Toggle Status">
                                                            <Link
                                                                href={route(
                                                                    "admin.seasonal-types.toggle-status",
                                                                    type.id
                                                                )}
                                                                method="post"
                                                                as="button"
                                                                className="text-xs text-amber-600 dark:text-amber-400 hover:text-amber-800 dark:hover:text-amber-300"
                                                            >
                                                                {type.is_active ? (
                                                                    <ToggleLeft className="w-4 h-4" />
                                                                ) : (
                                                                    <ToggleRight className="w-4 h-4" />
                                                                )}
                                                            </Link>
                                                        </Tooltip>

                                                        <Tooltip content="Delete">
                                                            <Link
                                                                href={route(
                                                                    "admin.seasonal-types.destroy",
                                                                    type.id
                                                                )}
                                                                method="delete"
                                                                as="button"
                                                                className="text-xs text-red-600 dark:text-red-400 hover:text-red-800 dark:hover:text-red-300"
                                                                onClick={(
                                                                    e
                                                                ) => {
                                                                    if (
                                                                        !confirm(
                                                                            "Are you sure you want to delete this seasonal type?"
                                                                        )
                                                                    ) {
                                                                        e.preventDefault();
                                                                    }
                                                                }}
                                                            >
                                                                <Trash2 className="w-4 h-4 text-red-600" />
                                                            </Link>
                                                        </Tooltip>
                                                    </div>
                                                </div>
                                            </React.Fragment>
                                        ))}
                                    </div>
                                </>
                            ) : (
                                // Empty State
                                <EmptyState
                                    icon={CalendarRange}
                                    title="No seasonal types found"
                                    description={
                                        data.search ||
                                        data.status ||
                                        data.period
                                            ? "Try adjusting your search or filter criteria"
                                            : "Get started by creating a new seasonal type"
                                    }
                                    actionText="Create Seasonal Type"
                                    actionUrl={route(
                                        "admin.seasonal-types.create"
                                    )}
                                    actionIcon={Plus}
                                    className="py-12"
                                />
                            )}
                        </div>
                    </div>

                    {/* Pagination */}
                    {seasonalTypes.data.length > 0 && (
                        <div className="mt-4 flex justify-between items-center">
                            <div className="text-sm text-gray-700 dark:text-gray-300">
                                Showing{" "}
                                <span className="font-medium">
                                    {seasonalTypes.from}
                                </span>{" "}
                                to{" "}
                                <span className="font-medium">
                                    {seasonalTypes.to}
                                </span>{" "}
                                of{" "}
                                <span className="font-medium">
                                    {seasonalTypes.total}
                                </span>{" "}
                                seasonal types
                            </div>
                            <Pagination links={seasonalTypes.links} />
                        </div>
                    )}

                    {/* Export Options */}
                    <div className="mt-8 border-t border-gray-200 dark:border-gray-700 pt-4">
                        <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                            Export Options
                        </h3>
                        <div className="flex space-x-2">
                            <Button variant="outline" size="sm">
                                <Download className="w-4 h-4 mr-2" />
                                Export CSV
                            </Button>
                            <Button variant="outline" size="sm">
                                <Download className="w-4 h-4 mr-2" />
                                Export PDF
                            </Button>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Index;
