import React, { useState } from "react";
import { Head, Link } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Card,
    CardContent,
    CardDescription,
    CardFooter,
    CardHeader,
    CardTitle,
} from "@/Components/ui/Card";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import Pagination from "@/Components/Pagination";
import {
    ArrowLeft,
    CalendarRange,
    Calendar,
    Edit,
    Tag,
    BarChart,
    Percent,
    DollarSign,
    Eye,
    Clock,
    Users,
    CheckCircle,
    AlertTriangle,
    ToggleLeft,
    ToggleRight,
} from "lucide-react";
import { Tabs, TabsList, TabsTrigger, TabsContent } from "@/Components/ui/Tabs";
import EmptyState from "@/Components/ui/EmptyState";

const Show = ({
    seasonalType,
    statistics,
    coupons,
    activeCoupons,
    isCurrentlyActive,
    nextOccurrence,
}) => {
    const [activeTab, setActiveTab] = useState("overview");

    const formatDate = (dateString) => {
        if (!dateString) return "N/A";
        return new Date(dateString).toLocaleDateString(undefined, {
            year: "numeric",
            month: "short",
            day: "numeric",
        });
    };

    const formatMonthDay = (dateString) => {
        if (!dateString) return "N/A";
        return new Date(dateString).toLocaleDateString(undefined, {
            month: "short",
            day: "numeric",
        });
    };

    const getStatusBadge = () => {
        if (!seasonalType.is_active) {
            return (
                <Badge
                    variant="secondary"
                    className="bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200"
                >
                    Inactive
                </Badge>
            );
        }

        return isCurrentlyActive ? (
            <Badge
                variant="success"
                className="bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-300"
            >
                Active Now
            </Badge>
        ) : (
            <Badge
                variant="info"
                className="bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-300"
            >
                Active
            </Badge>
        );
    };

    // Get discount display
    const getDiscountDisplay = (coupon) => {
        if (coupon.discount_type === "percentage") {
            return (
                <div className="flex items-center">
                    <Percent className="w-4 h-4 mr-1 text-green-600 dark:text-green-400" />
                    <span>{coupon.discount_amount}%</span>
                </div>
            );
        }
        return (
            <div className="flex items-center">
                <DollarSign className="w-4 h-4 mr-1 text-green-600 dark:text-green-400" />
                <span>${parseFloat(coupon.discount_amount).toFixed(2)}</span>
            </div>
        );
    };

    return (
        <AdminLayout>
            <Head title={seasonalType.name} />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6">
                        <div>
                            <div className="flex items-center">
                                <CalendarRange className="w-6 h-6 mr-2 text-blue-500" />
                                <h2 className="text-2xl font-semibold text-gray-900 dark:text-white">
                                    {seasonalType.name}
                                </h2>
                                <div className="ml-3">{getStatusBadge()}</div>
                            </div>
                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">
                                {formatMonthDay(
                                    seasonalType.default_start_date
                                )}{" "}
                                -{" "}
                                {formatMonthDay(seasonalType.default_end_date)}
                            </p>
                        </div>
                        <div className="mt-4 md:mt-0 flex flex-wrap gap-2">
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route("admin.seasonal-types.index")}
                                    className="flex items-center"
                                >
                                    <ArrowLeft className="w-4 h-4 mr-2" />
                                    Back to List
                                </Link>
                            </Button>
                            <Button variant="outline" size="sm" asChild>
                                <Link
                                    href={route(
                                        "admin.seasonal-types.toggle-status",
                                        seasonalType.id
                                    )}
                                    className="flex items-center"
                                    method="post"
                                    as="button"
                                >
                                    {seasonalType.is_active ? (
                                        <>
                                            <ToggleLeft className="w-4 h-4 mr-2" />
                                            Deactivate
                                        </>
                                    ) : (
                                        <>
                                            <ToggleRight className="w-4 h-4 mr-2" />
                                            Activate
                                        </>
                                    )}
                                </Link>
                            </Button>
                            <Button size="sm" asChild>
                                <Link
                                    href={route(
                                        "admin.seasonal-types.edit",
                                        seasonalType.id
                                    )}
                                    className="flex items-center"
                                >
                                    <Edit className="w-4 h-4 mr-2" />
                                    Edit
                                </Link>
                            </Button>
                        </div>
                    </div>

                    <Tabs
                        value={activeTab}
                        onValueChange={setActiveTab}
                        className="w-full"
                    >
                        <TabsList className="mb-6">
                            <TabsTrigger value="overview">Overview</TabsTrigger>
                            <TabsTrigger value="coupons">
                                Coupons ({statistics.total_coupons})
                            </TabsTrigger>
                            <TabsTrigger value="active-coupons">
                                Active Coupons ({activeCoupons.length})
                            </TabsTrigger>
                        </TabsList>

                        <TabsContent value="overview" className="space-y-6">
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                                {/* Overview Card */}
                                <Card className="md:col-span-2">
                                    <CardHeader>
                                        <CardTitle>
                                            Season Information
                                        </CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="space-y-4">
                                            {seasonalType.description && (
                                                <div>
                                                    <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        Description
                                                    </h3>
                                                    <p className="mt-1 text-gray-900 dark:text-gray-100">
                                                        {
                                                            seasonalType.description
                                                        }
                                                    </p>
                                                </div>
                                            )}

                                            <div>
                                                <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Annual Date Range
                                                </h3>
                                                <p className="mt-1 text-gray-900 dark:text-gray-100 flex items-center">
                                                    <Calendar className="w-4 h-4 mr-2 text-blue-600 dark:text-blue-400" />
                                                    {formatMonthDay(
                                                        seasonalType.default_start_date
                                                    )}{" "}
                                                    -{" "}
                                                    {formatMonthDay(
                                                        seasonalType.default_end_date
                                                    )}
                                                </p>
                                            </div>

                                            {nextOccurrence && (
                                                <div>
                                                    <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                        {nextOccurrence.status ===
                                                        "current"
                                                            ? "Current Period"
                                                            : "Next Occurrence"}
                                                    </h3>
                                                    <p className="mt-1 text-gray-900 dark:text-gray-100 flex items-center">
                                                        <Clock className="w-4 h-4 mr-2 text-green-600 dark:text-green-400" />
                                                        {formatDate(
                                                            nextOccurrence.start
                                                        )}{" "}
                                                        -{" "}
                                                        {formatDate(
                                                            nextOccurrence.end
                                                        )}
                                                        {nextOccurrence.status ===
                                                            "current" && (
                                                            <Badge
                                                                variant="success"
                                                                className="ml-2"
                                                            >
                                                                Active Now
                                                            </Badge>
                                                        )}
                                                    </p>
                                                </div>
                                            )}

                                            <div>
                                                <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Created At
                                                </h3>
                                                <p className="mt-1 text-gray-900 dark:text-gray-100">
                                                    {formatDate(
                                                        seasonalType.created_at
                                                    )}
                                                </p>
                                            </div>

                                            <div>
                                                <h3 className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Last Updated
                                                </h3>
                                                <p className="mt-1 text-gray-900 dark:text-gray-100">
                                                    {formatDate(
                                                        seasonalType.updated_at
                                                    )}
                                                </p>
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>

                                {/* Stats Card */}
                                <Card>
                                    <CardHeader>
                                        <CardTitle>Statistics</CardTitle>
                                    </CardHeader>
                                    <CardContent>
                                        <div className="space-y-4">
                                            <div className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Total Coupons
                                                </div>
                                                <div className="mt-1 flex items-center">
                                                    <Tag className="w-5 h-5 mr-2 text-blue-600 dark:text-blue-400" />
                                                    <span className="text-2xl font-semibold text-gray-900 dark:text-white">
                                                        {
                                                            statistics.total_coupons
                                                        }
                                                    </span>
                                                </div>
                                            </div>

                                            <div className="p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Active Coupons
                                                </div>
                                                <div className="mt-1 flex items-center">
                                                    <CheckCircle className="w-5 h-5 mr-2 text-green-600 dark:text-green-400" />
                                                    <span className="text-2xl font-semibold text-gray-900 dark:text-white">
                                                        {
                                                            statistics.active_coupons
                                                        }
                                                    </span>
                                                </div>
                                            </div>

                                            <div className="p-4 bg-purple-50 dark:bg-purple-900/20 rounded-lg">
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Total Redemptions
                                                </div>
                                                <div className="mt-1 flex items-center">
                                                    <Users className="w-5 h-5 mr-2 text-purple-600 dark:text-purple-400" />
                                                    <span className="text-2xl font-semibold text-gray-900 dark:text-white">
                                                        {statistics.total_uses ||
                                                            0}
                                                    </span>
                                                </div>
                                            </div>

                                            <div className="p-4 bg-amber-50 dark:bg-amber-900/20 rounded-lg">
                                                <div className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                                    Total Discount Amount
                                                </div>
                                                <div className="mt-1 flex items-center">
                                                    <DollarSign className="w-5 h-5 mr-2 text-amber-600 dark:text-amber-400" />
                                                    <span className="text-2xl font-semibold text-gray-900 dark:text-white">
                                                        $
                                                        {parseFloat(
                                                            statistics.total_discount ||
                                                                0
                                                        ).toFixed(2)}
                                                    </span>
                                                </div>
                                            </div>
                                        </div>
                                    </CardContent>
                                </Card>
                            </div>
                        </TabsContent>

                        <TabsContent value="coupons">
                            <Card>
                                <CardHeader>
                                    <CardTitle>All Coupons</CardTitle>
                                    <CardDescription>
                                        All coupons associated with this
                                        seasonal type
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    {coupons.data.length > 0 ? (
                                        <>
                                            {/* Coupons Table (CSS Grid) */}
                                            <div className="overflow-x-auto">
                                                {/* Table Header */}
                                                <div className="hidden md:grid md:grid-cols-5 gap-4 px-6 py-3 bg-gray-50 dark:bg-gray-800/50 rounded-t-lg">
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                        Code
                                                    </div>
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                        Discount
                                                    </div>
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                        Expiry
                                                    </div>
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                        Usage
                                                    </div>
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider text-right">
                                                        Status
                                                    </div>
                                                </div>

                                                {/* Table Rows */}
                                                <div className="divide-y divide-gray-200 dark:divide-gray-700">
                                                    {coupons.data.map(
                                                        (coupon) => (
                                                            <React.Fragment
                                                                key={coupon.id}
                                                            >
                                                                {/* Desktop Row */}
                                                                <div className="hidden md:grid md:grid-cols-5 gap-4 px-6 py-4 hover:bg-gray-50 dark:hover:bg-gray-800/60">
                                                                    <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                        {
                                                                            coupon.coupon_code
                                                                        }
                                                                    </div>
                                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                                        {getDiscountDisplay(
                                                                            coupon
                                                                        )}
                                                                    </div>
                                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                                        {coupon.expiry_date
                                                                            ? formatDate(
                                                                                  coupon.expiry_date
                                                                              )
                                                                            : "No Expiry"}
                                                                    </div>
                                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                                        {coupon.uses_count ||
                                                                            0}
                                                                        {coupon.max_uses
                                                                            ? ` / ${coupon.max_uses}`
                                                                            : ""}
                                                                    </div>
                                                                    <div className="text-right">
                                                                        {coupon.is_active ? (
                                                                            <Badge variant="success">
                                                                                Active
                                                                            </Badge>
                                                                        ) : (
                                                                            <Badge variant="secondary">
                                                                                Inactive
                                                                            </Badge>
                                                                        )}
                                                                    </div>
                                                                </div>

                                                                {/* Mobile Card View */}
                                                                <div className="block md:hidden p-4 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-800/60">
                                                                    <div className="flex justify-between items-center mb-2">
                                                                        <div className="font-medium text-gray-900 dark:text-gray-100">
                                                                            {
                                                                                coupon.coupon_code
                                                                            }
                                                                        </div>
                                                                        {coupon.is_active ? (
                                                                            <Badge variant="success">
                                                                                Active
                                                                            </Badge>
                                                                        ) : (
                                                                            <Badge variant="secondary">
                                                                                Inactive
                                                                            </Badge>
                                                                        )}
                                                                    </div>
                                                                    <div className="grid grid-cols-2 gap-2 mt-2">
                                                                        <div>
                                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                                Discount
                                                                            </div>
                                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                                {getDiscountDisplay(
                                                                                    coupon
                                                                                )}
                                                                            </div>
                                                                        </div>
                                                                        <div>
                                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                                Expiry
                                                                            </div>
                                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                                {coupon.expiry_date
                                                                                    ? formatDate(
                                                                                          coupon.expiry_date
                                                                                      )
                                                                                    : "No Expiry"}
                                                                            </div>
                                                                        </div>
                                                                        <div>
                                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                                Usage
                                                                            </div>
                                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                                {coupon.uses_count ||
                                                                                    0}
                                                                                {coupon.max_uses
                                                                                    ? ` / ${coupon.max_uses}`
                                                                                    : ""}
                                                                            </div>
                                                                        </div>
                                                                        <div className="flex justify-end items-end">
                                                                            <Link
                                                                                href={route(
                                                                                    "admin.coupons.show",
                                                                                    coupon.id
                                                                                )}
                                                                                className="inline-flex items-center px-2 py-1 text-xs font-medium text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                                                            >
                                                                                <Eye className="w-3 h-3 mr-1" />
                                                                                View
                                                                            </Link>
                                                                        </div>
                                                                    </div>
                                                                </div>
                                                            </React.Fragment>
                                                        )
                                                    )}
                                                </div>
                                            </div>

                                            {/* Pagination */}
                                            <div className="mt-4">
                                                <Pagination
                                                    links={coupons.links}
                                                />
                                            </div>
                                        </>
                                    ) : (
                                        <EmptyState
                                            icon={Tag}
                                            title="No coupons found"
                                            description="There are no coupons associated with this seasonal type yet."
                                            actionText="Create Coupon"
                                            actionUrl={route(
                                                "admin.coupons.create"
                                            )}
                                        />
                                    )}
                                </CardContent>
                            </Card>
                        </TabsContent>

                        <TabsContent value="active-coupons">
                            <Card>
                                <CardHeader>
                                    <CardTitle>Active Coupons</CardTitle>
                                    <CardDescription>
                                        Currently active coupons for this
                                        seasonal type
                                    </CardDescription>
                                </CardHeader>
                                <CardContent>
                                    {activeCoupons.length > 0 ? (
                                        <>
                                            {/* Active Coupons Table (CSS Grid) */}
                                            <div className="overflow-x-auto">
                                                {/* Table Header */}
                                                <div className="hidden md:grid md:grid-cols-5 gap-4 px-6 py-3 bg-gray-50 dark:bg-gray-800/50 rounded-t-lg">
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                        Code
                                                    </div>
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                        Discount
                                                    </div>
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                        Expiry
                                                    </div>
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                        Usage
                                                    </div>
                                                    <div className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider text-right">
                                                        Actions
                                                    </div>
                                                </div>

                                                {/* Table Rows */}
                                                <div className="divide-y divide-gray-200 dark:divide-gray-700">
                                                    {activeCoupons.map(
                                                        (coupon) => (
                                                            <React.Fragment
                                                                key={coupon.id}
                                                            >
                                                                {/* Desktop Row */}
                                                                <div className="hidden md:grid md:grid-cols-5 gap-4 px-6 py-4 hover:bg-gray-50 dark:hover:bg-gray-800/60">
                                                                    <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
                                                                        {
                                                                            coupon.coupon_code
                                                                        }
                                                                    </div>
                                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                                        {getDiscountDisplay(
                                                                            coupon
                                                                        )}
                                                                    </div>
                                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                                        {coupon.expiry_date
                                                                            ? formatDate(
                                                                                  coupon.expiry_date
                                                                              )
                                                                            : "No Expiry"}
                                                                    </div>
                                                                    <div className="text-sm text-gray-700 dark:text-gray-300">
                                                                        {coupon.uses_count ||
                                                                            0}
                                                                        {coupon.max_uses
                                                                            ? ` / ${coupon.max_uses}`
                                                                            : ""}
                                                                    </div>
                                                                    <div className="text-right">
                                                                        <Button
                                                                            variant="outline"
                                                                            size="sm"
                                                                            asChild
                                                                        >
                                                                            <Link
                                                                                href={route(
                                                                                    "admin.coupons.show",
                                                                                    coupon.id
                                                                                )}
                                                                            >
                                                                                <Eye className="w-4 h-4 mr-1" />
                                                                                View
                                                                            </Link>
                                                                        </Button>
                                                                    </div>
                                                                </div>

                                                                {/* Mobile Card View */}
                                                                <div className="block md:hidden p-4 bg-white dark:bg-gray-800 hover:bg-gray-50 dark:hover:bg-gray-800/60">
                                                                    <div className="font-medium text-gray-900 dark:text-gray-100 mb-2">
                                                                        {
                                                                            coupon.coupon_code
                                                                        }
                                                                    </div>
                                                                    <div className="grid grid-cols-2 gap-2 mt-2">
                                                                        <div>
                                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                                Discount
                                                                            </div>
                                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                                {getDiscountDisplay(
                                                                                    coupon
                                                                                )}
                                                                            </div>
                                                                        </div>
                                                                        <div>
                                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                                Expiry
                                                                            </div>
                                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                                {coupon.expiry_date
                                                                                    ? formatDate(
                                                                                          coupon.expiry_date
                                                                                      )
                                                                                    : "No Expiry"}
                                                                            </div>
                                                                        </div>
                                                                        <div>
                                                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                                                Usage
                                                                            </div>
                                                                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                                                                {coupon.uses_count ||
                                                                                    0}
                                                                                {coupon.max_uses
                                                                                    ? ` / ${coupon.max_uses}`
                                                                                    : ""}
                                                                            </div>
                                                                        </div>
                                                                        <div className="flex justify-end items-end">
                                                                            <Link
                                                                                href={route(
                                                                                    "admin.coupons.show",
                                                                                    coupon.id
                                                                                )}
                                                                                className="inline-flex items-center px-2 py-1 text-xs font-medium text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                                                            >
                                                                                <Eye className="w-3 h-3 mr-1" />
                                                                                View
                                                                            </Link>
                                                                        </div>
                                                                    </div>
                                                                </div>
                                                            </React.Fragment>
                                                        )
                                                    )}
                                                </div>
                                            </div>
                                        </>
                                    ) : (
                                        <EmptyState
                                            icon={AlertTriangle}
                                            title="No active coupons"
                                            description="There are no active coupons for this seasonal type currently."
                                            actionText="Create Coupon"
                                            actionUrl={route(
                                                "admin.coupons.create"
                                            )}
                                        />
                                    )}
                                </CardContent>
                            </Card>
                        </TabsContent>
                    </Tabs>
                </div>
            </div>
        </AdminLayout>
    );
};

export default Show;
