import { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import InputLabel from "@/Components/InputLabel";
import TextInput from "@/Components/TextInput";
import InputError from "@/Components/InputError";
import { Textarea } from "@/Components/ui/Textarea";
import PrimaryButton from "@/Components/PrimaryButton";
import { ArrowLeft, Save, Plus, X } from "lucide-react";

export default function Edit({ question, categories }) {
    const [newCategory, setNewCategory] = useState("");
    const [showNewCategoryInput, setShowNewCategoryInput] = useState(false);

    const { data, setData, put, processing, errors } = useForm({
        question: question.question || "",
        category: question.category || "",
        option_a: question.option_a || "",
        option_b: question.option_b || "",
        option_c: question.option_c || "",
        option_d: question.option_d || "",
        correct_answer: question.correct_answer || "",
        explanation: question.explanation || "",
        is_active: question.is_active,
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        put(route("admin.test-questions.update", question.id));
    };

    const addNewCategory = () => {
        if (newCategory.trim()) {
            setData("category", newCategory.trim());
            setNewCategory("");
            setShowNewCategoryInput(false);
        }
    };

    return (
        <AdminLayout>
            <Head title="Edit Test Question" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex items-center justify-between mb-6">
                        <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                            Edit Test Question
                        </h1>
                        <Link
                            href={route("admin.test-questions.index")}
                            className="px-4 py-2 text-sm text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-800 transition-colors flex items-center"
                        >
                            <ArrowLeft className="h-4 w-4 mr-1" />
                            Back to Questions
                        </Link>
                    </div>

                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <form onSubmit={handleSubmit} className="p-6 space-y-6">
                            <div>
                                <InputLabel
                                    htmlFor="question"
                                    value="Question"
                                    className="text-gray-900 dark:text-gray-100"
                                />
                                <Textarea
                                    id="question"
                                    className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                    value={data.question}
                                    onChange={(e) =>
                                        setData("question", e.target.value)
                                    }
                                    required
                                />
                                <InputError
                                    message={errors.question}
                                    className="mt-2"
                                />
                            </div>

                            <div>
                                <InputLabel
                                    htmlFor="category"
                                    value="Category"
                                    className="text-gray-900 dark:text-gray-100"
                                />
                                {showNewCategoryInput ? (
                                    <div className="flex mt-1">
                                        <TextInput
                                            id="new-category"
                                            type="text"
                                            className="block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                            value={newCategory}
                                            onChange={(e) =>
                                                setNewCategory(e.target.value)
                                            }
                                            placeholder="Enter new category"
                                            autoFocus
                                        />
                                        <button
                                            type="button"
                                            onClick={addNewCategory}
                                            className="ml-2 px-3 py-2 bg-green-600 dark:bg-green-500 text-white rounded-md hover:bg-green-700 dark:hover:bg-green-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            <Save className="h-5 w-5" />
                                        </button>
                                        <button
                                            type="button"
                                            onClick={() =>
                                                setShowNewCategoryInput(false)
                                            }
                                            className="ml-2 px-3 py-2 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 rounded-md hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        >
                                            <X className="h-5 w-5" />
                                        </button>
                                    </div>
                                ) : (
                                    <div className="flex mt-1">
                                        <select
                                            id="category"
                                            className="block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                            value={data.category}
                                            onChange={(e) =>
                                                setData(
                                                    "category",
                                                    e.target.value
                                                )
                                            }
                                            required
                                        >
                                            <option value="">
                                                Select a category
                                            </option>
                                            {categories.map((category) => (
                                                <option
                                                    key={category}
                                                    value={category}
                                                >
                                                    {category}
                                                </option>
                                            ))}
                                        </select>
                                        <button
                                            type="button"
                                            onClick={() =>
                                                setShowNewCategoryInput(true)
                                            }
                                            className="ml-2 px-3 py-2 bg-blue-600 dark:bg-blue-500 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            title="Add new category"
                                        >
                                            <Plus className="h-5 w-5" />
                                        </button>
                                    </div>
                                )}
                                <InputError
                                    message={errors.category}
                                    className="mt-2"
                                />
                            </div>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div>
                                    <InputLabel
                                        htmlFor="option_a"
                                        value="Option A"
                                        className="text-gray-900 dark:text-gray-100"
                                    />
                                    <Textarea
                                        id="option_a"
                                        className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                        value={data.option_a}
                                        onChange={(e) =>
                                            setData("option_a", e.target.value)
                                        }
                                        required
                                    />
                                    <InputError
                                        message={errors.option_a}
                                        className="mt-2"
                                    />
                                </div>

                                <div>
                                    <InputLabel
                                        htmlFor="option_b"
                                        value="Option B"
                                        className="text-gray-900 dark:text-gray-100"
                                    />
                                    <Textarea
                                        id="option_b"
                                        className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                        value={data.option_b}
                                        onChange={(e) =>
                                            setData("option_b", e.target.value)
                                        }
                                        required
                                    />
                                    <InputError
                                        message={errors.option_b}
                                        className="mt-2"
                                    />
                                </div>

                                <div>
                                    <InputLabel
                                        htmlFor="option_c"
                                        value="Option C"
                                        className="text-gray-900 dark:text-gray-100"
                                    />
                                    <Textarea
                                        id="option_c"
                                        className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                        value={data.option_c}
                                        onChange={(e) =>
                                            setData("option_c", e.target.value)
                                        }
                                        required
                                    />
                                    <InputError
                                        message={errors.option_c}
                                        className="mt-2"
                                    />
                                </div>

                                <div>
                                    <InputLabel
                                        htmlFor="option_d"
                                        value="Option D"
                                        className="text-gray-900 dark:text-gray-100"
                                    />
                                    <Textarea
                                        id="option_d"
                                        className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                        value={data.option_d}
                                        onChange={(e) =>
                                            setData("option_d", e.target.value)
                                        }
                                        required
                                    />
                                    <InputError
                                        message={errors.option_d}
                                        className="mt-2"
                                    />
                                </div>
                            </div>

                            <div>
                                <InputLabel
                                    htmlFor="correct_answer"
                                    value="Correct Answer"
                                    className="text-gray-900 dark:text-gray-100"
                                />
                                <div className="mt-2 flex space-x-4">
                                    {["a", "b", "c", "d"].map((option) => (
                                        <label
                                            key={option}
                                            className="inline-flex items-center"
                                        >
                                            <input
                                                type="radio"
                                                name="correct_answer"
                                                value={option}
                                                checked={
                                                    data.correct_answer ===
                                                    option
                                                }
                                                onChange={() =>
                                                    setData(
                                                        "correct_answer",
                                                        option
                                                    )
                                                }
                                                className="border-gray-300 dark:border-gray-700 text-indigo-600 shadow-sm focus:ring-indigo-500 dark:focus:ring-indigo-600"
                                            />
                                            <span className="ml-2 text-gray-700 dark:text-gray-300 uppercase">
                                                {option}
                                            </span>
                                        </label>
                                    ))}
                                </div>
                                <InputError
                                    message={errors.correct_answer}
                                    className="mt-2"
                                />
                            </div>

                            <div>
                                <InputLabel
                                    htmlFor="explanation"
                                    value="Explanation (Optional)"
                                    className="text-gray-900 dark:text-gray-100"
                                />
                                <Textarea
                                    id="explanation"
                                    className="mt-1 block w-full border-gray-300 dark:border-gray-700 dark:bg-gray-900 dark:text-gray-300 focus:border-indigo-500 dark:focus:border-indigo-600 focus:ring-indigo-500 dark:focus:ring-indigo-600 rounded-md shadow-sm"
                                    value={data.explanation}
                                    onChange={(e) =>
                                        setData("explanation", e.target.value)
                                    }
                                />
                                <InputError
                                    message={errors.explanation}
                                    className="mt-2"
                                />
                            </div>

                            <div className="flex items-center">
                                <input
                                    id="is_active"
                                    type="checkbox"
                                    className="rounded border-gray-300 dark:border-gray-700 text-indigo-600 shadow-sm focus:ring-indigo-500 dark:focus:ring-indigo-600"
                                    checked={data.is_active}
                                    onChange={(e) =>
                                        setData("is_active", e.target.checked)
                                    }
                                />
                                <label
                                    htmlFor="is_active"
                                    className="ml-2 text-sm text-gray-700 dark:text-gray-300"
                                >
                                    Active (available for tests)
                                </label>
                            </div>

                            <div className="flex items-center justify-end mt-6">
                                <Link
                                    href={route("admin.test-questions.index")}
                                    className="px-4 py-2 text-sm text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-800 transition-colors"
                                >
                                    Cancel
                                </Link>
                                <PrimaryButton
                                    className="ml-4"
                                    disabled={processing}
                                >
                                    <Save className="h-4 w-4 mr-2" />
                                    Update Question
                                </PrimaryButton>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
// This code is a React component for editing test questions in an admin panel. It uses the Inertia.js library for form handling and navigation. The component includes input fields for the question, category, options, correct answer, and explanation. It also allows adding a new category if needed. The form submission updates the question in the database.
