import { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import AdminLayout from "@/Layouts/AdminLayout";
import Pagination from "@/Components/Pagination";
import {
    Search,
    Plus,
    Edit,
    Trash2,
    CheckCircle,
    XCircle,
    Filter,
    RefreshCw,
} from "lucide-react";
import { Transition } from "@headlessui/react";

export default function Index({ questions, categories, filters, flash }) {
    const [showFilters, setShowFilters] = useState(false);
    const { data, setData, get, processing } = useForm({
        search: filters.search || "",
        category: filters.category || "all",
        status: filters.status || "all",
    });

    const handleSearch = (e) => {
        e.preventDefault();
        get(route("admin.test-questions.index"), {
            preserveState: true,
        });
    };

    const resetFilters = () => {
        setData({
            search: "",
            category: "all",
            status: "all",
        });
        get(route("admin.test-questions.index"), {
            preserveState: true,
        });
    };

    return (
        <AdminLayout>
            <Head title="Test Questions Management" />

            <div className="py-12 bg-gray-50 ">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8 ">
                    <div className="flex justify-between items-center mb-6">
                        <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                            Test Questions
                        </h1>
                        <Link
                            href={route("admin.test-questions.create")}
                            className="px-4 py-2 bg-blue-600 dark:bg-blue-500 text-white rounded-md flex items-center hover:bg-blue-700 dark:hover:bg-blue-600 transition-colors"
                        >
                            <Plus className="h-4 w-4 mr-2" />
                            Add Question
                        </Link>
                    </div>

                    {flash.success && (
                        <div className="mb-6 bg-green-100 dark:bg-green-900/30 border-l-4 border-green-500 text-green-700 dark:text-green-300 p-4 rounded">
                            <p>{flash.success}</p>
                        </div>
                    )}

                    {flash.error && (
                        <div className="mb-6 bg-red-100 dark:bg-red-900/30 border-l-4 border-red-500 text-red-700 dark:text-red-300 p-4 rounded">
                            <p>{flash.error}</p>
                        </div>
                    )}

                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <div className="flex flex-col md:flex-row md:items-center md:justify-between mb-6 gap-4">
                                <form
                                    onSubmit={handleSearch}
                                    className="flex items-center flex-1"
                                >
                                    <div className="relative flex-1">
                                        <div className="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <Search className="h-5 w-5 text-gray-400" />
                                        </div>
                                        <input
                                            type="text"
                                            className="block w-full pl-10 pr-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md leading-5 bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-100 placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm transition-colors"
                                            placeholder="Search questions..."
                                            value={data.search}
                                            onChange={(e) =>
                                                setData(
                                                    "search",
                                                    e.target.value
                                                )
                                            }
                                        />
                                    </div>
                                    <button
                                        type="submit"
                                        className="ml-3 px-3 py-2 bg-blue-600 dark:bg-blue-500 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                        disabled={processing}
                                    >
                                        Search
                                    </button>
                                    <button
                                        type="button"
                                        onClick={() =>
                                            setShowFilters(!showFilters)
                                        }
                                        className="ml-2 px-3 py-2 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 rounded-md hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-500 dark:focus:ring-offset-gray-800 transition-colors"
                                    >
                                        <Filter className="h-5 w-5" />
                                    </button>
                                    {(data.category !== "all" ||
                                        data.status !== "all" ||
                                        data.search) && (
                                        <button
                                            type="button"
                                            onClick={resetFilters}
                                            className="ml-2 px-3 py-2 bg-gray-200 dark:bg-gray-700 text-gray-700 dark:text-gray-200 rounded-md hover:bg-gray-300 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-gray-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            title="Reset filters"
                                        >
                                            <RefreshCw className="h-5 w-5" />
                                        </button>
                                    )}
                                </form>
                            </div>

                            <Transition
                                show={showFilters}
                                enter="transition ease-out duration-100"
                                enterFrom="transform opacity-0 scale-95"
                                enterTo="transform opacity-100 scale-100"
                                leave="transition ease-in duration-75"
                                leaveFrom="transform opacity-100 scale-100"
                                leaveTo="transform opacity-0 scale-95"
                            >
                                <div className="mb-6 p-4 bg-gray-50 dark:bg-gray-700/50 rounded-md">
                                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label
                                                htmlFor="category"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                            >
                                                Category
                                            </label>
                                            <select
                                                id="category"
                                                className="block w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm transition-colors"
                                                value={data.category}
                                                onChange={(e) => {
                                                    setData(
                                                        "category",
                                                        e.target.value
                                                    );
                                                }}
                                            >
                                                <option value="all">
                                                    All Categories
                                                </option>
                                                {categories.map((category) => (
                                                    <option
                                                        key={category}
                                                        value={category}
                                                    >
                                                        {category}
                                                    </option>
                                                ))}
                                            </select>
                                        </div>
                                        <div>
                                            <label
                                                htmlFor="status"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1"
                                            >
                                                Status
                                            </label>
                                            <select
                                                id="status"
                                                className="block w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm bg-white dark:bg-gray-900 text-gray-900 dark:text-gray-100 focus:outline-none focus:ring-blue-500 focus:border-blue-500 sm:text-sm transition-colors"
                                                value={data.status}
                                                onChange={(e) => {
                                                    setData(
                                                        "status",
                                                        e.target.value
                                                    );
                                                }}
                                            >
                                                <option value="all">
                                                    All Statuses
                                                </option>
                                                <option value="active">
                                                    Active
                                                </option>
                                                <option value="inactive">
                                                    Inactive
                                                </option>
                                            </select>
                                        </div>
                                    </div>
                                    <div className="mt-4 flex justify-end">
                                        <button
                                            type="button"
                                            onClick={handleSearch}
                                            className="px-4 py-2 bg-blue-600 dark:bg-blue-500 text-white rounded-md hover:bg-blue-700 dark:hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800 transition-colors"
                                            disabled={processing}
                                        >
                                            Apply Filters
                                        </button>
                                    </div>
                                </div>
                            </Transition>

                            {questions.data.length > 0 ? (
                                <>
                                    {/* Desktop view - Grid layout (table alternative) */}
                                    <div className="hidden md:block">
                                        {/* Header */}
                                        <div className="grid grid-cols-12 gap-2 bg-gray-50 dark:bg-gray-800 rounded-t-lg">
                                            <div className="col-span-4 px-4 py-3">
                                                <span className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                    Question
                                                </span>
                                            </div>
                                            <div className="col-span-2 px-4 py-3">
                                                <span className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                    Category
                                                </span>
                                            </div>
                                            <div className="col-span-2 px-4 py-3">
                                                <span className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                    Answer
                                                </span>
                                            </div>
                                            <div className="col-span-2 px-4 py-3">
                                                <span className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                    Status
                                                </span>
                                            </div>
                                            <div className="col-span-2 px-4 py-3 text-right">
                                                <span className="text-xs font-medium text-gray-500 dark:text-gray-400 uppercase tracking-wider">
                                                    Actions
                                                </span>
                                            </div>
                                        </div>

                                        {/* Body */}
                                        <div className="bg-white dark:bg-gray-900 rounded-b-lg divide-y divide-gray-200 dark:divide-gray-800">
                                            {questions.data.map((question) => (
                                                <div
                                                    key={question.id}
                                                    className="grid grid-cols-12 gap-2 hover:bg-gray-50 dark:hover:bg-gray-800/60"
                                                >
                                                    <div className="col-span-4 px-4 py-3">
                                                        <div className="text-sm text-gray-900 dark:text-gray-100 line-clamp-2">
                                                            {question.question}
                                                        </div>
                                                    </div>
                                                    <div className="col-span-2 px-4 py-3 flex items-center">
                                                        <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200">
                                                            {question.category}
                                                        </span>
                                                    </div>
                                                    <div className="col-span-2 px-4 py-3 flex items-center text-sm text-gray-500 dark:text-gray-400 uppercase">
                                                        {
                                                            question.correct_answer
                                                        }
                                                    </div>
                                                    <div className="col-span-2 px-4 py-3 flex items-center">
                                                        {question.is_active ? (
                                                            <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200">
                                                                Active
                                                            </span>
                                                        ) : (
                                                            <span className="px-2 inline-flex text-xs leading-5 font-semibold rounded-full bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-200">
                                                                Inactive
                                                            </span>
                                                        )}
                                                    </div>
                                                    <div className="col-span-2 px-4 py-3 flex justify-end items-center space-x-2">
                                                        <Link
                                                            href={route(
                                                                "admin.test-questions.edit",
                                                                question.id
                                                            )}
                                                            className="text-indigo-600 dark:text-indigo-400 hover:text-indigo-900 dark:hover:text-indigo-300"
                                                            title="Edit"
                                                        >
                                                            <Edit className="h-5 w-5" />
                                                        </Link>
                                                        <Link
                                                            href={route(
                                                                "admin.test-questions.toggle-status",
                                                                question.id
                                                            )}
                                                            method="post"
                                                            as="button"
                                                            className={
                                                                question.is_active
                                                                    ? "text-red-600 dark:text-red-400 hover:text-red-900 dark:hover:text-red-300"
                                                                    : "text-green-600 dark:text-green-400 hover:text-green-900 dark:hover:text-green-300"
                                                            }
                                                            title={
                                                                question.is_active
                                                                    ? "Deactivate"
                                                                    : "Activate"
                                                            }
                                                        >
                                                            {question.is_active ? (
                                                                <XCircle className="h-5 w-5" />
                                                            ) : (
                                                                <CheckCircle className="h-5 w-5" />
                                                            )}
                                                        </Link>
                                                        <Link
                                                            href={route(
                                                                "admin.test-questions.destroy",
                                                                question.id
                                                            )}
                                                            method="delete"
                                                            as="button"
                                                            className="text-red-600 dark:text-red-400 hover:text-red-900 dark:hover:text-red-300"
                                                            title="Delete"
                                                            onClick={(e) => {
                                                                if (
                                                                    !confirm(
                                                                        "Are you sure you want to delete this question?"
                                                                    )
                                                                ) {
                                                                    e.preventDefault();
                                                                }
                                                            }}
                                                        >
                                                            <Trash2 className="h-5 w-5" />
                                                        </Link>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </div>

                                    {/* Mobile view - Card layout */}
                                    <div className="md:hidden space-y-4">
                                        {questions.data.map((question) => (
                                            <div
                                                key={question.id}
                                                className="bg-white dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-lg shadow p-4 relative"
                                            >
                                                {/* Status badge positioned at top right */}
                                                <div className="absolute top-4 right-4">
                                                    {question.is_active ? (
                                                        <span className="px-2 py-1 inline-flex text-xs leading-4 font-semibold rounded-full bg-green-100 dark:bg-green-900 text-green-800 dark:text-green-200">
                                                            Active
                                                        </span>
                                                    ) : (
                                                        <span className="px-2 py-1 inline-flex text-xs leading-4 font-semibold rounded-full bg-red-100 dark:bg-red-900 text-red-800 dark:text-red-200">
                                                            Inactive
                                                        </span>
                                                    )}
                                                </div>

                                                {/* Question content */}
                                                <div className="mb-4 pr-16">
                                                    {" "}
                                                    {/* Right padding to avoid overlap with status badge */}
                                                    <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                        Question:
                                                    </h3>
                                                    <p className="text-gray-900 dark:text-gray-100 text-sm line-clamp-3">
                                                        {question.question}
                                                    </p>
                                                </div>

                                                {/* Question details */}
                                                <div className="space-y-2 text-sm mb-4">
                                                    <div className="flex justify-between">
                                                        <span className="text-gray-700 dark:text-gray-300 font-medium">
                                                            Category:
                                                        </span>
                                                        <span className="px-2 py-0.5 inline-flex text-xs leading-5 font-semibold rounded-full bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200">
                                                            {question.category}
                                                        </span>
                                                    </div>
                                                    <div className="flex justify-between">
                                                        <span className="text-gray-700 dark:text-gray-300 font-medium">
                                                            Answer:
                                                        </span>
                                                        <span className="text-gray-900 dark:text-gray-100 uppercase">
                                                            {
                                                                question.correct_answer
                                                            }
                                                        </span>
                                                    </div>
                                                </div>

                                                {/* Actions */}
                                                <div className="flex justify-end space-x-4 pt-3 border-t border-gray-200 dark:border-gray-700">
                                                    <Link
                                                        href={route(
                                                            "admin.test-questions.edit",
                                                            question.id
                                                        )}
                                                        className="text-indigo-600 dark:text-indigo-400 hover:text-indigo-900 dark:hover:text-indigo-300 flex items-center"
                                                    >
                                                        <Edit className="h-4 w-4 mr-1" />
                                                        <span>Edit</span>
                                                    </Link>

                                                    <Link
                                                        href={route(
                                                            "admin.test-questions.toggle-status",
                                                            question.id
                                                        )}
                                                        method="post"
                                                        as="button"
                                                        className={
                                                            question.is_active
                                                                ? "text-red-600 dark:text-red-400 hover:text-red-900 dark:hover:text-red-300 flex items-center"
                                                                : "text-green-600 dark:text-green-400 hover:text-green-900 dark:hover:text-green-300 flex items-center"
                                                        }
                                                    >
                                                        {question.is_active ? (
                                                            <>
                                                                <XCircle className="h-4 w-4 mr-1" />
                                                                <span>
                                                                    Deactivate
                                                                </span>
                                                            </>
                                                        ) : (
                                                            <>
                                                                <CheckCircle className="h-4 w-4 mr-1" />
                                                                <span>
                                                                    Activate
                                                                </span>
                                                            </>
                                                        )}
                                                    </Link>

                                                    <Link
                                                        href={route(
                                                            "admin.test-questions.destroy",
                                                            question.id
                                                        )}
                                                        method="delete"
                                                        as="button"
                                                        className="text-red-600 dark:text-red-400 hover:text-red-900 dark:hover:text-red-300 flex items-center"
                                                        onClick={(e) => {
                                                            if (
                                                                !confirm(
                                                                    "Are you sure you want to delete this question?"
                                                                )
                                                            ) {
                                                                e.preventDefault();
                                                            }
                                                        }}
                                                    >
                                                        <Trash2 className="h-4 w-4 mr-1" />
                                                        <span>Delete</span>
                                                    </Link>
                                                </div>
                                            </div>
                                        ))}
                                    </div>

                                    <div className="mt-6">
                                        <Pagination links={questions.links} />
                                    </div>
                                </>
                            ) : (
                                <div className="text-center py-12">
                                    <svg
                                        className="mx-auto h-12 w-12 text-gray-400 dark:text-gray-600"
                                        fill="none"
                                        viewBox="0 0 24 24"
                                        stroke="currentColor"
                                    >
                                        <path
                                            strokeLinecap="round"
                                            strokeLinejoin="round"
                                            strokeWidth={2}
                                            d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z"
                                        />
                                    </svg>
                                    <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-gray-100">
                                        No questions found
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        Get started by creating a new test
                                        question.
                                    </p>
                                    <div className="mt-6">
                                        <Link
                                            href={route(
                                                "admin.test-questions.create"
                                            )}
                                            className="inline-flex items-center px-4 py-2 border border-transparent shadow-sm text-sm font-medium rounded-md text-white bg-blue-600 dark:bg-blue-500 hover:bg-blue-700 dark:hover:bg-blue-600 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500 dark:focus:ring-offset-gray-800"
                                        >
                                            <Plus className="h-5 w-5 mr-2" />
                                            Add Question
                                        </Link>
                                    </div>
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
