import { useEffect, useState } from "react";
import AdminLayout from "@/Layouts/AdminLayout";
import CleanPagination from "@/Components/CleanPagination";
import { Link, router, usePage } from "@inertiajs/react";
import {
    Search,
    Edit,
    View,
    Trash2,
    UserX,
    Plus,
    Mail,
    UserCircle,
    CheckCheck,
    X,
    ShoppingBag,
} from "lucide-react";
import SectionContainer from "@/Components/SectionContainer";
import StatsCard from "@/Components/StatsCard";

export default function UsersIndex({
    users,
    query,
    filters,
    userCount,
    writerCount,
    clientCount,
    adminCount,
}) {
    const [search, setSearch] = useState(query || "");
    const [userType, setUserType] = useState(filters?.userType || "");
    const [status, setStatus] = useState(filters?.status || "");
    const { flash } = usePage().props;
    const [flashMsg, setFlashMsg] = useState(flash.message);

    useEffect(() => {
        if (flashMsg) {
            const timer = setTimeout(() => {
                setFlashMsg("");
            }, 3000);
            return () => clearTimeout(timer);
        }
    }, [flashMsg]);

    useEffect(() => {
        setFlashMsg(flash.message);
    }, [flash.message]);

    const handleFilters = (newFilters) => {
        const params = {
            query: search.trim() || undefined,
            userType:
                (newFilters.userType === ""
                    ? undefined
                    : newFilters.userType) ||
                (userType === "" ? undefined : userType),
            status:
                (newFilters.status === "" ? undefined : newFilters.status) ||
                (status === "" ? undefined : status),
        };

        // Remove undefined values
        const cleanParams = Object.fromEntries(
            Object.entries(params).filter(([_, value]) => value !== undefined)
        );

        router.get(route("admin.users.search"), cleanParams, {
            preserveState: true,
        });
    };

    const handleSearch = (e) => {
        e.preventDefault();
        router.get(
            route("admin.users.search"),
            {
                ...route().params,
                query: search.trim() || undefined,
            },
            { preserveState: true }
        );
    };

    const handleSearchChange = (e) => {
        const value = e.target.value;
        setSearch(value);

        if (value.trim() === "") {
            router.get(route("admin.users.search"), {
                ...route().params,
                query: undefined,
            });
        }
    };

    const handleDelete = (userId) => {
        if (confirm("Are you sure you want to delete this user?")) {
            router.delete(route("admin.users.destroy", userId));
        }
    };

    return (
        <AdminLayout>
            <div className="dark:bg-gray-900 min-h-screen">
                {/*  Statistics */}
                <SectionContainer
                    className="bg-gradient-to-r from-orange-200 to-cyan-200 dark:bg-none dark:from-gray-800 dark:to-gray-900"
                    title="Statistics"
                    titleClassName="text-lg font-bold text-gray-900 dark:text-white bg-white bg-opacity-30 dark:bg-gray-900 dark:bg-opacity-40 border-b border-gray-300 dark:border-gray-700"
                >
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                        <StatsCard
                            title="All Users"
                            value={userCount}
                            className="from-blue-500 to-pink-600 dark:from-gray-800 dark:to-gray-900"
                        />

                        {/* Total Orders */}
                        <StatsCard
                            title="Admins"
                            value={adminCount}
                            className="from-green-500 to-blue-600 dark:from-gray-800 dark:to-gray-900"
                        />

                        {/* Total Writers */}
                        <StatsCard
                            title="Total Writers"
                            value={writerCount}
                            className="from-cyan-500 to-red-600 dark:from-gray-800 dark:to-gray-900"
                        />

                        {/* Total Revenue */}
                        <StatsCard
                            title="Total Clients"
                            value={clientCount}
                            className="from-yellow-500 to-green-600 dark:from-gray-800 dark:to-gray-900"
                        />
                    </div>
                </SectionContainer>
                {/* Header Section */}
                <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center gap-4 mb-6">
                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                        Manage Users
                    </h1>

                    <Link
                        href={route("admin.users.create")}
                        className="flex items-center justify-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors w-full sm:w-auto"
                    >
                        <Plus className="h-4 w-4" />
                        <span>Add New</span>
                    </Link>
                </div>

                {/* Search Form */}
                <form
                    onSubmit={handleSearch}
                    className="mb-6 flex flex-col sm:flex-row items-stretch sm:items-center gap-2"
                >
                    <div className="relative flex-1">
                        <input
                            type="search"
                            placeholder="Search users..."
                            value={search}
                            onChange={handleSearchChange}
                            className="border p-2 pl-10 rounded w-full dark:bg-gray-800 dark:text-white dark:border-gray-600"
                        />
                        <Search className="absolute left-3 top-3 w-5 h-5 text-gray-500" />
                    </div>
                    <button
                        type="submit"
                        className="bg-blue-500 text-white px-4 py-2 rounded hover:bg-blue-600"
                    >
                        Search
                    </button>
                </form>

                <div className="flex flex-col sm:flex-row gap-4 mb-6">
                    <select
                        value={userType}
                        onChange={(e) => {
                            const value = e.target.value;
                            setUserType(value);
                            router.get(
                                route("admin.users.search"),
                                {
                                    ...route().params,
                                    userType: value || undefined,
                                },
                                {
                                    preserveState: true,
                                }
                            );
                        }}
                        className="border py-2 rounded dark:bg-gray-800 dark:text-white dark:border-gray-600"
                    >
                        <option value="">All User Types</option>
                        <option value="admin">Admin</option>
                        <option value="writer">Writer</option>
                        <option value="client">Client</option>
                    </select>

                    <select
                        value={status}
                        onChange={(e) => {
                            const value = e.target.value;
                            setStatus(value);
                            router.get(
                                route("admin.users.search"),
                                {
                                    ...route().params,
                                    status: value || undefined,
                                },
                                {
                                    preserveState: true,
                                }
                            );
                        }}
                        className="border py-2 rounded dark:bg-gray-800 dark:text-white dark:border-gray-600"
                    >
                        <option value="">All Statuses</option>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                {flashMsg && (
                    <div
                        className="bg-green-400 border border-green-400 text-white px-4 py-3 rounded relative mb-2"
                        role="alert"
                    >
                        <span className="block sm:inline">{flashMsg}</span>
                    </div>
                )}
                {/* Desktop Table View */}
                <div className="hidden md:block overflow-x-auto rounded-lg shadow-lg bg-gradient-to-r from-orange-200 to-cyan-200 dark:bg-none dark:bg-gray-800">
                    <table className="w-full text-sm text-left border-collapse">
                        <thead>
                            <tr className="bg-gradient-to-r from-blue-900 to-cyan-500 dark:bg-gray-700 text-gray-100 dark:text-white">
                                <th className="p-3">ID</th>
                                <th className="p-3">Name</th>
                                <th className="p-3">Email</th>
                                <th className="p-3">User Type</th>
                                <th className="p-3">Status</th>
                                <th className="p-3">Verified</th>
                                <th className="p-3">Orders</th>
                                <th className="p-3 text-center">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            {users.data.length > 0 ? (
                                users.data.map((user) => (
                                    <tr
                                        key={user.id}
                                        className="border-b dark:border-gray-700 hover:bg-gray-100 dark:hover:bg-gray-700"
                                    >
                                        <td className="p-3">{user.id}</td>
                                        <td className="p-3">{user.name}</td>
                                        <td className="p-3">{user.email}</td>
                                        <td className="p-3 capitalize">
                                            {user.user_type}
                                        </td>
                                        <td className="p-3 capitalize flex items-center gap-2">
                                            <span
                                                className={`w-2 h-2 rounded-full ${
                                                    user.status === "active"
                                                        ? "bg-green-500"
                                                        : user.status ===
                                                          "inactive"
                                                        ? "bg-red-500"
                                                        : user.status ===
                                                          "banned"
                                                        ? "bg-gray-500"
                                                        : ""
                                                }`}
                                            ></span>
                                            {user.status}
                                        </td>
                                        <td className="p-2 text-center">
                                            {user.email_verified_at ? (
                                                <CheckCheck className="w-5 h-5 text-green-500" />
                                            ) : (
                                                <span className="text-red-500">
                                                    <X className="w-5 h-5 text-red-500" />
                                                </span>
                                            )}
                                        </td>
                                        <td className="p-3 text-center">
                                            <div className="flex items-center justify-center gap-1">
                                                <ShoppingBag className="w-4 h-4 text-blue-500" />
                                                <span>
                                                    {user.order_count || 0}
                                                </span>
                                            </div>
                                        </td>
                                        <td className="p-3 flex justify-center gap-3">
                                            <Link
                                                href={route(
                                                    "admin.users.show",
                                                    user.id
                                                )}
                                                className="text-blue-500 hover:text-blue-700"
                                            >
                                                <View className="w-5 h-5" />
                                            </Link>
                                            <Link
                                                href={route(
                                                    "admin.users.edit",
                                                    user.id
                                                )}
                                                className="text-blue-500 hover:text-blue-700"
                                            >
                                                <Edit className="w-5 h-5" />
                                            </Link>
                                            <button
                                                onClick={() =>
                                                    handleDelete(user.id)
                                                }
                                                className="text-red-500 hover:text-red-700"
                                            >
                                                <Trash2 className="w-5 h-5" />
                                            </button>
                                        </td>
                                    </tr>
                                ))
                            ) : (
                                <tr>
                                    <td colSpan="8" className="text-center p-4">
                                        No users found
                                    </td>
                                </tr>
                            )}
                        </tbody>
                    </table>
                </div>

                {/* Mobile/Tablet Card View */}
                <div className="md:hidden space-y-4">
                    {users.data.length > 0 ? (
                        users.data.map((user) => (
                            <div
                                key={user.id}
                                className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-4"
                            >
                                <div className="flex items-center justify-between mb-4">
                                    <div className="flex items-center gap-3">
                                        <UserCircle className="w-10 h-10 text-gray-500 dark:text-gray-400" />
                                        <div>
                                            <h3 className="font-medium text-gray-900 dark:text-white">
                                                {user.name}
                                            </h3>
                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                ID: {user.id}
                                            </p>
                                        </div>
                                    </div>
                                    <span className="px-3 py-1 text-xs font-medium capitalize bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 rounded-full">
                                        {user.user_type}
                                    </span>
                                </div>

                                <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400 mb-2">
                                    <Mail className="w-4 h-4" />
                                    <span className="text-sm">
                                        {user.email}
                                    </span>
                                </div>

                                <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400 mb-2">
                                    <span
                                        className={`w-2 h-2 rounded-full ${
                                            user.status === "active"
                                                ? "bg-green-500"
                                                : user.status === "inactive"
                                                ? "bg-red-500"
                                                : user.status === "banned"
                                                ? "bg-gray-500"
                                                : ""
                                        }`}
                                    ></span>
                                    <span className="text-sm capitalize">
                                        {user.status}
                                    </span>
                                </div>

                                <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400 mb-2">
                                    {user.email_verified_at ? (
                                        <CheckCheck className="w-4 h-4 text-green-500" />
                                    ) : (
                                        <X className="w-4 h-4 text-red-500" />
                                    )}
                                    <span className="text-sm">
                                        {user.email_verified_at
                                            ? "Verified"
                                            : "Not Verified"}
                                    </span>
                                </div>

                                <div className="flex items-center gap-2 text-gray-500 dark:text-gray-400 mb-4">
                                    <ShoppingBag className="w-4 h-4 text-blue-500" />
                                    <span className="text-sm">
                                        Orders: {user.order_count || 0}
                                    </span>
                                </div>

                                <div className="flex justify-end gap-3 border-t dark:border-gray-700 pt-4">
                                    <Link
                                        href={route(
                                            "admin.users.show",
                                            user.id
                                        )}
                                        className="text-blue-500 hover:text-blue-700"
                                    >
                                        <View className="w-5 h-5" />
                                    </Link>
                                    <Link
                                        href={route(
                                            "admin.users.edit",
                                            user.id
                                        )}
                                        className="text-blue-500 hover:text-blue-700"
                                    >
                                        <Edit className="w-5 h-5" />
                                    </Link>
                                    <button
                                        onClick={() => handleDelete(user.id)}
                                        className="text-red-500 hover:text-red-700"
                                    >
                                        <Trash2 className="w-5 h-5" />
                                    </button>
                                </div>
                            </div>
                        ))
                    ) : (
                        <div className="text-center p-4 bg-white dark:bg-gray-800 rounded-lg">
                            No users found
                        </div>
                    )}
                </div>

                <div className="mt-6">
                    <CleanPagination
                        links={users.links}
                        total={users.total}
                        perPage={users.per_page}
                        currentPage={users.current_page}
                    />
                </div>
            </div>
        </AdminLayout>
    );
}
