import { Head, Link, useForm } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { 
    ArrowLeft,
    Save,
    User,
    FileText,
    Calendar,
    DollarSign,
    Star,
    MessageSquare
} from 'lucide-react';

export default function EditWriterInfraction({ 
    infraction,
    writers, 
    orders, 
    infractionTypes, 
    severityLevels 
}) {
    const { data, setData, put, processing, errors } = useForm({
        writer_id: infraction.writer_id || '',
        order_id: infraction.order_id || '',
        infraction_type: infraction.infraction_type || '',
        severity_level: infraction.severity_level || '',
        description: infraction.description || '',
        evidence: infraction.evidence || '',
        admin_notes: infraction.admin_notes || '',
        occurred_at: infraction.occurred_at ? infraction.occurred_at.slice(0, 16) : '',
        estimated_cost_impact: infraction.estimated_cost_impact || '',
        client_satisfaction_impact: infraction.client_satisfaction_impact || '',
        impact_assessment: infraction.impact_assessment || ''
    });

    const handleSubmit = (e) => {
        e.preventDefault();
        put(route('admin.writer-infractions.update', infraction.id));
    };

    const getInfractionTypeLabel = (type) => {
        const typeLabels = {
            'deadline_violation': 'Deadline Violation',
            'quality_issues': 'Quality Issues',
            'plagiarism': 'Plagiarism',
            'communication_failure': 'Communication Failure',
            'policy_violation': 'Policy Violation',
            'unprofessional_behavior': 'Unprofessional Behavior',
            'client_complaint': 'Client Complaint',
            'other': 'Other'
        };
        return typeLabels[type] || type;
    };

    return (
        <AdminLayout>
            <Head title={`Edit Infraction #${infraction.id}`} />
            
            <div className="py-6">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center gap-4 mb-4">
                            <Link
                                href={route('admin.writer-infractions.show', infraction.id)}
                                className="inline-flex items-center text-gray-600 hover:text-gray-900 dark:text-gray-400 dark:hover:text-white"
                            >
                                <ArrowLeft size={16} className="mr-2" />
                                Back to Infraction
                            </Link>
                        </div>
                        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                            Edit Infraction #{infraction.id}
                        </h1>
                        <p className="mt-2 text-gray-600 dark:text-gray-400">
                            Update infraction details and information
                        </p>
                    </div>

                    {/* Form */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-6">
                        <form onSubmit={handleSubmit} className="space-y-6">
                            {/* Basic Information */}
                            <div>
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                    Basic Information
                                </h3>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    {/* Writer Selection */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            <User size={16} className="inline mr-2" />
                                            Writer *
                                        </label>
                                        <select
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                            value={data.writer_id}
                                            onChange={(e) => setData('writer_id', e.target.value)}
                                        >
                                            <option value="">Select a writer</option>
                                            {writers.map(writer => (
                                                <option key={writer.id} value={writer.id}>
                                                    {writer.name} ({writer.email})
                                                </option>
                                            ))}
                                        </select>
                                        {errors.writer_id && (
                                            <p className="mt-1 text-sm text-red-600">{errors.writer_id}</p>
                                        )}
                                    </div>

                                    {/* Order Selection */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            <FileText size={16} className="inline mr-2" />
                                            Related Order (Optional)
                                        </label>
                                        <select
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                            value={data.order_id}
                                            onChange={(e) => setData('order_id', e.target.value)}
                                        >
                                            <option value="">Select an order (optional)</option>
                                            {orders.map(order => (
                                                <option key={order.id} value={order.id}>
                                                    #{order.id} - {order.title}
                                                </option>
                                            ))}
                                        </select>
                                        {errors.order_id && (
                                            <p className="mt-1 text-sm text-red-600">{errors.order_id}</p>
                                        )}
                                    </div>

                                    {/* Infraction Type */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Infraction Type *
                                        </label>
                                        <select
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                            value={data.infraction_type}
                                            onChange={(e) => setData('infraction_type', e.target.value)}
                                        >
                                            <option value="">Select infraction type</option>
                                            {Object.entries(infractionTypes).map(([key, label]) => (
                                                <option key={key} value={key}>
                                                    {label}
                                                </option>
                                            ))}
                                        </select>
                                        {errors.infraction_type && (
                                            <p className="mt-1 text-sm text-red-600">{errors.infraction_type}</p>
                                        )}
                                    </div>

                                    {/* Severity Level */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            Severity Level *
                                        </label>
                                        <select
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                            value={data.severity_level}
                                            onChange={(e) => setData('severity_level', e.target.value)}
                                        >
                                            <option value="">Select severity level</option>
                                            {Object.entries(severityLevels).map(([key, label]) => (
                                                <option key={key} value={key}>
                                                    {label}
                                                </option>
                                            ))}
                                        </select>
                                        {errors.severity_level && (
                                            <p className="mt-1 text-sm text-red-600">{errors.severity_level}</p>
                                        )}
                                    </div>

                                    {/* Occurred At */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            <Calendar size={16} className="inline mr-2" />
                                            When Did This Occur?
                                        </label>
                                        <input
                                            type="datetime-local"
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                            value={data.occurred_at}
                                            onChange={(e) => setData('occurred_at', e.target.value)}
                                        />
                                        {errors.occurred_at && (
                                            <p className="mt-1 text-sm text-red-600">{errors.occurred_at}</p>
                                        )}
                                    </div>
                                </div>
                            </div>

                            {/* Description */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    <MessageSquare size={16} className="inline mr-2" />
                                    Description of Infraction *
                                </label>
                                <textarea
                                    rows={4}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    placeholder="Provide a detailed description of what happened..."
                                    value={data.description}
                                    onChange={(e) => setData('description', e.target.value)}
                                />
                                {errors.description && (
                                    <p className="mt-1 text-sm text-red-600">{errors.description}</p>
                                )}
                            </div>

                            {/* Evidence */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    <FileText size={16} className="inline mr-2" />
                                    Evidence (Optional)
                                </label>
                                <textarea
                                    rows={3}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    placeholder="Document any evidence, screenshots, or supporting information..."
                                    value={data.evidence}
                                    onChange={(e) => setData('evidence', e.target.value)}
                                />
                                {errors.evidence && (
                                    <p className="mt-1 text-sm text-red-600">{errors.evidence}</p>
                                )}
                            </div>

                            {/* Impact Assessment */}
                            <div>
                                <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                    Impact Assessment
                                </h3>
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    {/* Cost Impact */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            <DollarSign size={16} className="inline mr-2" />
                                            Estimated Cost Impact
                                        </label>
                                        <input
                                            type="number"
                                            step="0.01"
                                            min="0"
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                            placeholder="0.00"
                                            value={data.estimated_cost_impact}
                                            onChange={(e) => setData('estimated_cost_impact', e.target.value)}
                                        />
                                        {errors.estimated_cost_impact && (
                                            <p className="mt-1 text-sm text-red-600">{errors.estimated_cost_impact}</p>
                                        )}
                                    </div>

                                    {/* Client Satisfaction Impact */}
                                    <div>
                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                            <Star size={16} className="inline mr-2" />
                                            Client Satisfaction Impact (1-5)
                                        </label>
                                        <select
                                            className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                            value={data.client_satisfaction_impact}
                                            onChange={(e) => setData('client_satisfaction_impact', e.target.value)}
                                        >
                                            <option value="">Select impact level</option>
                                            <option value="1">1 - Very Low Impact</option>
                                            <option value="2">2 - Low Impact</option>
                                            <option value="3">3 - Moderate Impact</option>
                                            <option value="4">4 - High Impact</option>
                                            <option value="5">5 - Very High Impact</option>
                                        </select>
                                        {errors.client_satisfaction_impact && (
                                            <p className="mt-1 text-sm text-red-600">{errors.client_satisfaction_impact}</p>
                                        )}
                                    </div>
                                </div>

                                {/* Impact Assessment Notes */}
                                <div className="mt-4">
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                        <MessageSquare size={16} className="inline mr-2" />
                                        Impact Assessment Notes
                                    </label>
                                    <textarea
                                        rows={3}
                                        className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                        placeholder="Describe the overall impact on business operations, client relationships, etc..."
                                        value={data.impact_assessment}
                                        onChange={(e) => setData('impact_assessment', e.target.value)}
                                    />
                                    {errors.impact_assessment && (
                                        <p className="mt-1 text-sm text-red-600">{errors.impact_assessment}</p>
                                    )}
                                </div>
                            </div>

                            {/* Admin Notes */}
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    <MessageSquare size={16} className="inline mr-2" />
                                    Admin Notes (Optional)
                                </label>
                                <textarea
                                    rows={3}
                                    className="w-full border border-gray-300 rounded-md px-3 py-2 focus:outline-none focus:ring-2 focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:border-gray-600 dark:text-white"
                                    placeholder="Internal notes, recommendations, or additional context..."
                                    value={data.admin_notes}
                                    onChange={(e) => setData('admin_notes', e.target.value)}
                                />
                                {errors.admin_notes && (
                                    <p className="mt-1 text-sm text-red-600">{errors.admin_notes}</p>
                                )}
                            </div>

                            {/* Form Actions */}
                            <div className="flex items-center justify-end gap-4 pt-6 border-t border-gray-200 dark:border-gray-700">
                                <Link
                                    href={route('admin.writer-infractions.show', infraction.id)}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:bg-gray-800 dark:text-white dark:border-gray-600 dark:hover:bg-gray-700"
                                >
                                    Cancel
                                </Link>
                                <button
                                    type="submit"
                                    disabled={processing}
                                    className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 disabled:opacity-50 disabled:cursor-not-allowed"
                                >
                                    <Save size={16} className="mr-2" />
                                    {processing ? 'Saving...' : 'Update Infraction'}
                                </button>
                            </div>
                        </form>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
} 