import { Head, Link } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { 
    ArrowLeft,
    Edit,
    Trash2,
    User,
    FileText,
    Calendar,
    AlertTriangle,
    CheckCircle,
    Clock,
    XCircle
} from 'lucide-react';

export default function ShowWriterInfraction({ infraction, relatedInfractions, fineStatistics }) {
    const getStatusColor = (status) => {
        switch (status) {
            case 'pending':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'under_review':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'resolved':
                return 'bg-green-100 text-green-800 border-green-200';
            case 'dismissed':
                return 'bg-gray-100 text-gray-800 border-gray-200';
            case 'escalated':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const getSeverityColor = (severity) => {
        switch (severity) {
            case 'warning':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'minor':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'major':
                return 'bg-orange-100 text-orange-800 border-orange-200';
            case 'critical':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const getInfractionTypeLabel = (type) => {
        const typeLabels = {
            'deadline_violation': 'Deadline Violation',
            'quality_issues': 'Quality Issues',
            'plagiarism': 'Plagiarism',
            'communication_failure': 'Communication Failure',
            'policy_violation': 'Policy Violation',
            'unprofessional_behavior': 'Unprofessional Behavior',
            'client_complaint': 'Client Complaint',
            'other': 'Other'
        };
        return typeLabels[type] || type;
    };

    return (
        <AdminLayout>
            <Head title={`Infraction #${infraction.id}`} />
            
            <div className="py-6">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-8">
                        <div className="flex items-center gap-4 mb-4">
                            <Link
                                href={route('admin.writer-infractions.index')}
                                className="inline-flex items-center text-gray-600 hover:text-gray-900 dark:text-gray-400 dark:hover:text-white"
                            >
                                <ArrowLeft size={16} className="mr-2" />
                                Back to Infractions
                            </Link>
                        </div>
                        <div className="flex items-center justify-between">
                            <div>
                                <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                                    Infraction #{infraction.id}
                                </h1>
                                <p className="mt-2 text-gray-600 dark:text-gray-400">
                                    {getInfractionTypeLabel(infraction.infraction_type)} - {infraction.severity_level} severity
                                </p>
                            </div>
                            <div className="flex items-center gap-3">
                                <Link
                                    href={route('admin.writer-infractions.edit', infraction.id)}
                                    className="inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:bg-gray-800 dark:text-white dark:border-gray-600 dark:hover:bg-gray-700"
                                >
                                    <Edit size={16} className="mr-2" />
                                    Edit
                                </Link>
                            </div>
                        </div>
                    </div>

                    {/* Infraction Details */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-6 mb-6">
                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                            Infraction Details
                        </h3>
                        
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Writer
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {infraction.writer?.name || 'Unknown'}
                                </p>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Type
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {getInfractionTypeLabel(infraction.infraction_type)}
                                </p>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Severity
                                </label>
                                <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getSeverityColor(infraction.severity_level)}`}>
                                    {infraction.severity_level.charAt(0).toUpperCase() + infraction.severity_level.slice(1)}
                                </span>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Status
                                </label>
                                <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(infraction.status)}`}>
                                    {infraction.status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                                </span>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Reported At
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {new Date(infraction.reported_at).toLocaleString()}
                                </p>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Occurred At
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {infraction.occurred_at ? new Date(infraction.occurred_at).toLocaleString() : 'N/A'}
                                </p>
                            </div>
                        </div>

                        <div className="mt-6">
                            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                Description
                            </label>
                            <p className="text-sm text-gray-900 dark:text-white bg-gray-50 dark:bg-gray-700 p-3 rounded-md">
                                {infraction.description}
                            </p>
                        </div>

                        {infraction.evidence && (
                            <div className="mt-6">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Evidence
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white bg-gray-50 dark:bg-gray-700 p-3 rounded-md">
                                    {infraction.evidence}
                                </p>
                            </div>
                        )}

                        {infraction.admin_notes && (
                            <div className="mt-6">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Admin Notes
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white bg-gray-50 dark:bg-gray-700 p-3 rounded-md">
                                    {infraction.admin_notes}
                                </p>
                            </div>
                        )}
                    </div>

                    {/* Impact Assessment */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-6 mb-6">
                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                            Impact Assessment
                        </h3>
                        
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-6">
                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Cost Impact
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    ${infraction.estimated_cost_impact || '0.00'}
                                </p>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Client Satisfaction Impact
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {infraction.client_satisfaction_impact ? `${infraction.client_satisfaction_impact}/5` : 'N/A'}
                                </p>
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Repeat Count
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {infraction.repeat_count || 1}
                                </p>
                            </div>
                        </div>

                        {infraction.impact_assessment && (
                            <div className="mt-6">
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                    Impact Assessment Notes
                                </label>
                                <p className="text-sm text-gray-900 dark:text-white bg-gray-50 dark:bg-gray-700 p-3 rounded-md">
                                    {infraction.impact_assessment}
                                </p>
                            </div>
                        )}
                    </div>

                    {/* Related Infractions */}
                    {relatedInfractions.length > 0 && (
                        <div className="bg-white dark:bg-gray-800 rounded-lg shadow-md p-6">
                            <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                                Related Infractions
                            </h3>
                            
                            <div className="space-y-3">
                                {relatedInfractions.map(related => (
                                    <div key={related.id} className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700 rounded-md">
                                        <div>
                                            <p className="text-sm font-medium text-gray-900 dark:text-white">
                                                #{related.id} - {getInfractionTypeLabel(related.infraction_type)}
                                            </p>
                                            <p className="text-xs text-gray-600 dark:text-gray-400">
                                                {new Date(related.reported_at).toLocaleDateString()}
                                            </p>
                                        </div>
                                        <Link
                                            href={route('admin.writer-infractions.show', related.id)}
                                            className="text-indigo-600 hover:text-indigo-900 dark:text-indigo-400 dark:hover:text-indigo-300 text-sm"
                                        >
                                            View
                                        </Link>
                                    </div>
                                ))}
                            </div>
                        </div>
                    )}
                </div>
            </div>
        </AdminLayout>
    );
} 