import { useState } from "react";
import { Head, Link } from "@inertiajs/react";
import { Textarea } from "@/Components/ui/Textarea";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    ArrowLeft,
    CheckCircle,
    XCircle,
    ChartBar,
    User,
    Calendar,
    Mail,
    Clock,
    ChevronDown,
    ChevronUp,
} from "lucide-react";

export default function Show({ writer, score, categorizedResults, responses }) {
    const [expandedQuestions, setExpandedQuestions] = useState({});

    // Toggle question expansion
    const toggleQuestion = (id) => {
        setExpandedQuestions({
            ...expandedQuestions,
            [id]: !expandedQuestions[id],
        });
    };

    // Calculate color based on category performance
    const getCategoryColor = (percentage) => {
        if (percentage >= 90) return "text-green-600 dark:text-green-400";
        if (percentage >= 70) return "text-yellow-600 dark:text-yellow-400";
        return "text-red-600 dark:text-red-400";
    };

    // Calculate width for progress bars
    const getProgressWidth = (percentage) => {
        return `${Math.max(percentage, 3)}%`; // Minimum 3% width for visibility
    };

    // Format date
    const formatDate = (dateString) => {
        if (!dateString) return "—";
        return new Date(dateString).toLocaleString();
    };

    // Get background color for answer options
    const getAnswerBgColor = (
        questionId,
        option,
        correctAnswer,
        selectedAnswer
    ) => {
        if (option === correctAnswer) {
            return "bg-green-100 dark:bg-green-900/30 border-green-300 dark:border-green-700";
        }
        if (option === selectedAnswer && option !== correctAnswer) {
            return "bg-red-100 dark:bg-red-900/30 border-red-300 dark:border-red-700";
        }
        return "bg-white dark:bg-gray-800 border-gray-300 dark:border-gray-700";
    };

    return (
        <AdminLayout>
            <Head title={`Test Results - ${writer.name}`} />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    <div className="flex items-center justify-between mb-6">
                        <h1 className="text-2xl font-semibold text-gray-900 dark:text-gray-100">
                            Writer Test Results
                        </h1>
                        <Link
                            href={route("admin.writers.tests")}
                            className="px-4 py-2 text-sm text-gray-700 dark:text-gray-300 bg-white dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-md hover:bg-gray-50 dark:hover:bg-gray-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500 dark:focus:ring-offset-gray-800 transition-colors flex items-center"
                        >
                            <ArrowLeft className="h-4 w-4 mr-1" />
                            Back to List
                        </Link>
                    </div>

                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-6">
                            <div className="flex flex-col md:flex-row md:justify-between md:items-center">
                                <div className="mb-4 md:mb-0">
                                    <h2 className="text-xl font-semibold text-gray-900 dark:text-gray-100 mb-2">
                                        Writer Information
                                    </h2>
                                    <div className="space-y-2">
                                        <div className="flex items-center text-gray-700 dark:text-gray-300">
                                            <User className="h-5 w-5 mr-2 text-gray-500 dark:text-gray-400" />
                                            <span>{writer.name}</span>
                                        </div>
                                        <div className="flex items-center text-gray-700 dark:text-gray-300">
                                            <Mail className="h-5 w-5 mr-2 text-gray-500 dark:text-gray-400" />
                                            <span>{writer.email}</span>
                                        </div>
                                        <div className="flex items-center text-gray-700 dark:text-gray-300">
                                            <Calendar className="h-5 w-5 mr-2 text-gray-500 dark:text-gray-400" />
                                            <span>
                                                Test completed:{" "}
                                                {formatDate(
                                                    writer.completed_at
                                                )}
                                            </span>
                                        </div>
                                        {writer.can_retry_after && (
                                            <div className="flex items-center text-gray-700 dark:text-gray-300">
                                                <Clock className="h-5 w-5 mr-2 text-gray-500 dark:text-gray-400" />
                                                <span>
                                                    Can retry after:{" "}
                                                    {formatDate(
                                                        writer.can_retry_after
                                                    )}
                                                </span>
                                            </div>
                                        )}
                                    </div>
                                </div>

                                <div className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg">
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-2">
                                        Test Score
                                    </h3>
                                    <div className="flex items-center">
                                        <div className="text-3xl font-bold mr-2 flex items-center">
                                            {score.passed ? (
                                                <CheckCircle className="h-8 w-8 text-green-500 dark:text-green-400 mr-2" />
                                            ) : (
                                                <XCircle className="h-8 w-8 text-red-500 dark:text-red-400 mr-2" />
                                            )}
                                            <span
                                                className={
                                                    score.passed
                                                        ? "text-green-600 dark:text-green-400"
                                                        : "text-red-600 dark:text-red-400"
                                                }
                                            >
                                                {Math.round(score.percentage)}%
                                            </span>
                                        </div>
                                        <div className="text-gray-600 dark:text-gray-400">
                                            ({score.correct} of {score.total}{" "}
                                            correct)
                                        </div>
                                    </div>
                                    <div className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                                        Passing score: 90%
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Category Performance */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-6">
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-gray-100 mb-4 flex items-center">
                                <ChartBar className="h-5 w-5 mr-2" />
                                Performance by Category
                            </h2>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                                {Object.entries(categorizedResults).map(
                                    ([category, result]) => (
                                        <div
                                            key={category}
                                            className="bg-gray-50 dark:bg-gray-700 p-4 rounded-lg"
                                        >
                                            <div className="flex justify-between items-center mb-2">
                                                <h4 className="font-medium text-gray-800 dark:text-gray-200">
                                                    {category}
                                                </h4>
                                                <span
                                                    className={`font-medium ${getCategoryColor(
                                                        result.percentage
                                                    )}`}
                                                >
                                                    {Math.round(
                                                        result.percentage
                                                    )}
                                                    %
                                                </span>
                                            </div>

                                            <div className="flex items-center text-sm text-gray-500 dark:text-gray-400 mb-2">
                                                <span>
                                                    {result.correct} of{" "}
                                                    {result.total} correct
                                                </span>
                                            </div>

                                            <div className="w-full bg-gray-200 dark:bg-gray-600 rounded-full h-2.5">
                                                <div
                                                    className={`h-2.5 rounded-full ${
                                                        result.percentage >= 90
                                                            ? "bg-green-500 dark:bg-green-400"
                                                            : result.percentage >=
                                                              70
                                                            ? "bg-yellow-500 dark:bg-yellow-400"
                                                            : "bg-red-500 dark:bg-red-400"
                                                    }`}
                                                    style={{
                                                        width: getProgressWidth(
                                                            result.percentage
                                                        ),
                                                    }}
                                                ></div>
                                            </div>
                                        </div>
                                    )
                                )}
                            </div>
                        </div>
                    </div>

                    {/* Detailed Responses */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            <h2 className="text-xl font-semibold text-gray-900 dark:text-gray-100 mb-4">
                                Detailed Question Responses
                            </h2>

                            <div className="space-y-4">
                                {responses.map((response) => (
                                    <div
                                        key={response.id}
                                        className="border border-gray-200 dark:border-gray-700 rounded-lg overflow-hidden"
                                    >
                                        <div
                                            className={`p-4 cursor-pointer ${
                                                response.is_correct
                                                    ? "bg-green-50 dark:bg-green-900/20"
                                                    : "bg-red-50 dark:bg-red-900/20"
                                            }`}
                                            onClick={() =>
                                                toggleQuestion(response.id)
                                            }
                                        >
                                            <div className="flex justify-between items-start">
                                                <div className="flex items-start">
                                                    {response.is_correct ? (
                                                        <CheckCircle className="h-5 w-5 text-green-500 dark:text-green-400 mt-0.5 mr-2 flex-shrink-0" />
                                                    ) : (
                                                        <XCircle className="h-5 w-5 text-red-500 dark:text-red-400 mt-0.5 mr-2 flex-shrink-0" />
                                                    )}
                                                    <div>
                                                        <div className="font-medium text-gray-900 dark:text-gray-100">
                                                            {response.question}
                                                        </div>
                                                        <div className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                                            Category:{" "}
                                                            {response.category}
                                                        </div>
                                                    </div>
                                                </div>
                                                <div className="flex items-center">
                                                    <span className="text-sm text-gray-500 dark:text-gray-400 mr-2">
                                                        {expandedQuestions[
                                                            response.id
                                                        ]
                                                            ? "Hide"
                                                            : "Show"}{" "}
                                                        details
                                                    </span>
                                                    {expandedQuestions[
                                                        response.id
                                                    ] ? (
                                                        <ChevronUp className="h-5 w-5 text-gray-500 dark:text-gray-400" />
                                                    ) : (
                                                        <ChevronDown className="h-5 w-5 text-gray-500 dark:text-gray-400" />
                                                    )}
                                                </div>
                                            </div>
                                        </div>

                                        {expandedQuestions[response.id] && (
                                            <div className="p-4 border-t border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
                                                <div className="space-y-3">
                                                    {Object.entries(
                                                        response.options
                                                    ).map(([key, value]) => (
                                                        <div
                                                            key={key}
                                                            className={`p-3 border rounded-lg ${getAnswerBgColor(
                                                                response.id,
                                                                key,
                                                                response.correct_answer,
                                                                response.selected_answer
                                                            )}`}
                                                        >
                                                            <div className="flex items-center">
                                                                <div
                                                                    className={`w-5 h-5 rounded-full border flex items-center justify-center mr-3 ${
                                                                        key ===
                                                                        response.selected_answer
                                                                            ? "border-blue-500 bg-blue-500"
                                                                            : "border-gray-300 dark:border-gray-600"
                                                                    }`}
                                                                >
                                                                    {key ===
                                                                        response.selected_answer && (
                                                                        <div className="h-2 w-2 rounded-full bg-white"></div>
                                                                    )}
                                                                </div>
                                                                <span className="text-gray-700 dark:text-gray-300">
                                                                    <span className="font-medium uppercase mr-2">
                                                                        {key}.
                                                                    </span>
                                                                    {value}
                                                                </span>
                                                                {key ===
                                                                    response.correct_answer && (
                                                                    <CheckCircle className="h-5 w-5 text-green-500 dark:text-green-400 ml-2" />
                                                                )}
                                                            </div>
                                                        </div>
                                                    ))}
                                                </div>

                                                {response.explanation && (
                                                    <div className="mt-4 p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                                                        <h4 className="font-medium text-gray-900 dark:text-gray-100 mb-1">
                                                            Explanation:
                                                        </h4>
                                                        <p className="text-gray-700 dark:text-gray-300">
                                                            {
                                                                response.explanation
                                                            }
                                                        </p>
                                                    </div>
                                                )}

                                                <div className="mt-4 text-sm">
                                                    <div className="flex items-center">
                                                        <span className="text-gray-600 dark:text-gray-400 mr-2">
                                                            {response.is_correct
                                                                ? "Correct"
                                                                : "Incorrect"}
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </AdminLayout>
    );
}
