import React, { useState, useEffect } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { 
    DollarSign, 
    Clock, 
    CheckCircle, 
    XCircle, 
    AlertCircle,
    Download,
    Filter,
    Search,
    Calendar,
    User
} from 'lucide-react';
import SectionContainer from '@/Components/SectionContainer';
import StatsCard from '@/Components/StatsCard';

export default function WriterWithdrawalsIndex({ withdrawals, stats, filters }) {
    const [selectedWithdrawals, setSelectedWithdrawals] = useState([]);
    const [showFilters, setShowFilters] = useState(false);
    const [localFilters, setLocalFilters] = useState({
        status: filters.status || 'all',
        date: filters.date || '',
        writer: filters.writer || ''
    });

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'approved':
                return <CheckCircle size={16} className="text-green-500" />;
            case 'paid':
                return <CheckCircle size={16} className="text-blue-500" />;
            case 'rejected':
                return <XCircle size={16} className="text-red-500" />;
            case 'processing':
                return <Clock size={16} className="text-yellow-500" />;
            default:
                return <AlertCircle size={16} className="text-gray-500" />;
        }
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'approved':
                return 'bg-green-100 text-green-800';
            case 'paid':
                return 'bg-blue-100 text-blue-800';
            case 'rejected':
                return 'bg-red-100 text-red-800';
            case 'processing':
                return 'bg-yellow-100 text-yellow-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const applyFilters = () => {
        router.get('/admin/writer-withdrawals', localFilters, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const clearFilters = () => {
        setLocalFilters({
            status: 'all',
            date: '',
            writer: ''
        });
        router.get('/admin/writer-withdrawals', {}, {
            preserveState: true,
            preserveScroll: true
        });
    };

    const handleBulkApprove = () => {
        if (selectedWithdrawals.length === 0) return;

        if (confirm(`Are you sure you want to approve ${selectedWithdrawals.length} withdrawal requests?`)) {
            router.post('/admin/writer-withdrawals/bulk-approve', {
                withdrawal_ids: selectedWithdrawals
            });
        }
    };

    const toggleWithdrawalSelection = (id) => {
        setSelectedWithdrawals(prev => 
            prev.includes(id) 
                ? prev.filter(w => w !== id)
                : [...prev, id]
        );
    };

    const selectAll = () => {
        const pendingIds = withdrawals.data
            .filter(w => w.status === 'requested')
            .map(w => w.id);
        setSelectedWithdrawals(pendingIds);
    };

    const deselectAll = () => {
        setSelectedWithdrawals([]);
    };

    return (
        <AdminLayout>
            <Head title="Writer Withdrawal Management" />
            
            <div className="p-6 max-w-7xl mx-auto">
                {/* Header */}
                <div className="mb-8">
                    <h1 className="text-3xl font-bold text-gray-900 mb-2">Writer Withdrawal Management</h1>
                    <p className="text-gray-600">
                        Manage writer withdrawal requests and process payments
                    </p>
                </div>

                {/* Statistics Cards */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                    <StatsCard
                        title="Total Requests"
                        value={stats.total}
                        className="from-blue-500 to-blue-600"
                    />
                    <StatsCard
                        title="Pending Approval"
                        value={stats.pending}
                        className="from-yellow-500 to-yellow-600"
                    />
                    <StatsCard
                        title="Approved"
                        value={stats.approved}
                        className="from-green-500 to-green-600"
                    />
                    <StatsCard
                        title="Total Paid"
                        value={formatCurrency(stats.total_amount_paid)}
                        className="from-purple-500 to-purple-600"
                    />
                </div>

                {/* Filters and Actions */}
                <SectionContainer
                    className="bg-white shadow-sm border border-gray-200"
                    title="Filters & Actions"
                    titleClassName="text-lg font-semibold text-gray-900"
                >
                    <div className="p-6">
                        {/* Filter Toggle */}
                        <div className="flex items-center justify-between mb-4">
                            <button
                                onClick={() => setShowFilters(!showFilters)}
                                className="flex items-center text-sm text-gray-600 hover:text-gray-900"
                            >
                                <Filter size={16} className="mr-2" />
                                {showFilters ? 'Hide Filters' : 'Show Filters'}
                            </button>
                            
                            <div className="flex items-center space-x-3">
                                <button
                                    onClick={clearFilters}
                                    className="text-sm text-gray-600 hover:text-gray-900"
                                >
                                    Clear Filters
                                </button>
                                <button
                                    onClick={applyFilters}
                                    className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700"
                                >
                                    Apply Filters
                                </button>
                            </div>
                        </div>

                        {/* Filter Form */}
                        {showFilters && (
                            <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-6">
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                        Status
                                    </label>
                                    <select
                                        value={localFilters.status}
                                        onChange={(e) => setLocalFilters(prev => ({ ...prev, status: e.target.value }))}
                                        className="w-full border border-gray-300 rounded-md px-3 py-2"
                                    >
                                        <option value="all">All Statuses</option>
                                        <option value="requested">Requested</option>
                                        <option value="approved">Approved</option>
                                        <option value="processing">Processing</option>
                                        <option value="paid">Paid</option>
                                        <option value="rejected">Rejected</option>
                                    </select>
                                </div>
                                
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                        Date
                                    </label>
                                    <input
                                        type="date"
                                        value={localFilters.date}
                                        onChange={(e) => setLocalFilters(prev => ({ ...prev, date: e.target.value }))}
                                        className="w-full border border-gray-300 rounded-md px-3 py-2"
                                    />
                                </div>
                                
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 mb-1">
                                        Writer
                                    </label>
                                    <input
                                        type="text"
                                        placeholder="Search by name or email"
                                        value={localFilters.writer}
                                        onChange={(e) => setLocalFilters(prev => ({ ...prev, writer: e.target.value }))}
                                        className="w-full border border-gray-300 rounded-md px-3 py-2"
                                    />
                                </div>
                            </div>
                        )}

                        {/* Bulk Actions */}
                        {selectedWithdrawals.length > 0 && (
                            <div className="bg-blue-50 border border-blue-200 rounded-md p-4 mb-4">
                                <div className="flex items-center justify-between">
                                    <div className="flex items-center">
                                        <span className="text-sm text-blue-800">
                                            {selectedWithdrawals.length} withdrawal(s) selected
                                        </span>
                                    </div>
                                    <div className="flex items-center space-x-3">
                                        <button
                                            onClick={selectAll}
                                            className="text-sm text-blue-600 hover:text-blue-800"
                                        >
                                            Select All Pending
                                        </button>
                                        <button
                                            onClick={deselectAll}
                                            className="text-sm text-blue-600 hover:text-blue-800"
                                        >
                                            Deselect All
                                        </button>
                                        <button
                                            onClick={handleBulkApprove}
                                            className="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700"
                                        >
                                            Approve Selected
                                        </button>
                                    </div>
                                </div>
                            </div>
                        )}

                        {/* Export */}
                        <div className="flex justify-end">
                            <Link
                                href="/admin/writer-withdrawals-export"
                                className="inline-flex items-center px-4 py-2 bg-gray-600 text-white rounded-md hover:bg-gray-700"
                            >
                                <Download size={16} className="mr-2" />
                                Export Data
                            </Link>
                        </div>
                    </div>
                </SectionContainer>

                {/* Withdrawals Table */}
                <SectionContainer
                    className="bg-white shadow-sm border border-gray-200"
                    title="Withdrawal Requests"
                    titleClassName="text-lg font-semibold text-gray-900"
                >
                    <div className="overflow-x-auto">
                        <table className="min-w-full divide-y divide-gray-200">
                            <thead className="bg-gray-50">
                                <tr>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        <input
                                            type="checkbox"
                                            onChange={(e) => e.target.checked ? selectAll() : deselectAll()}
                                            className="rounded border-gray-300"
                                        />
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Writer
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Amount
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Status
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Requested Date
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Payment Method
                                    </th>
                                    <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                        Actions
                                    </th>
                                </tr>
                            </thead>
                            <tbody className="bg-white divide-y divide-gray-200">
                                {withdrawals.data.map((withdrawal) => (
                                    <tr key={withdrawal.id} className="hover:bg-gray-50">
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <input
                                                type="checkbox"
                                                checked={selectedWithdrawals.includes(withdrawal.id)}
                                                onChange={() => toggleWithdrawalSelection(withdrawal.id)}
                                                className="rounded border-gray-300"
                                            />
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="flex items-center">
                                                <div className="flex-shrink-0 h-10 w-10">
                                                    <div className="h-10 w-10 rounded-full bg-gray-300 flex items-center justify-center">
                                                        <User size={20} className="text-gray-600" />
                                                    </div>
                                                </div>
                                                <div className="ml-4">
                                                    <div className="text-sm font-medium text-gray-900">
                                                        {withdrawal.writer?.name || 'Unknown'}
                                                    </div>
                                                    <div className="text-sm text-gray-500">
                                                        {withdrawal.writer?.email || 'N/A'}
                                                    </div>
                                                </div>
                                            </div>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="text-sm font-medium text-gray-900">
                                                {formatCurrency(withdrawal.requested_amount)}
                                            </div>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap">
                                            <div className="flex items-center">
                                                {getStatusIcon(withdrawal.status)}
                                                <span className={`ml-2 inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(withdrawal.status)}`}>
                                                    {withdrawal.status}
                                                </span>
                                            </div>
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                            {formatDate(withdrawal.withdrawal_date)}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                            {withdrawal.payment_method}
                                        </td>
                                        <td className="px-6 py-4 whitespace-nowrap text-sm font-medium">
                                            <Link
                                                href={`/admin/writer-withdrawals/${withdrawal.id}`}
                                                className="text-blue-600 hover:text-blue-900"
                                            >
                                                View Details
                                            </Link>
                                        </td>
                                    </tr>
                                ))}
                            </tbody>
                        </table>
                    </div>

                    {/* Pagination */}
                    {withdrawals.links && (
                        <div className="px-6 py-3 border-t border-gray-200">
                            <div className="flex items-center justify-between">
                                <div className="text-sm text-gray-700">
                                    Showing {withdrawals.from} to {withdrawals.to} of {withdrawals.total} results
                                </div>
                                <div className="flex space-x-2">
                                    {withdrawals.links.map((link, index) => (
                                        <button
                                            key={index}
                                            onClick={() => router.get(link.url)}
                                            disabled={!link.url}
                                            className={`px-3 py-1 text-sm rounded-md ${
                                                link.active
                                                    ? 'bg-blue-600 text-white'
                                                    : link.url
                                                    ? 'bg-white text-gray-700 border border-gray-300 hover:bg-gray-50'
                                                    : 'bg-gray-100 text-gray-400 cursor-not-allowed'
                                            }`}
                                            dangerouslySetInnerHTML={{ __html: link.label }}
                                        />
                                    ))}
                                </div>
                            </div>
                        </div>
                    )}
                </SectionContainer>
            </div>
        </AdminLayout>
    );
} 