import React, { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import AdminLayout from '@/Layouts/AdminLayout';
import { 
    ArrowLeft,
    CheckCircle, 
    XCircle, 
    Clock, 
    AlertCircle,
    DollarSign,
    User,
    Calendar,
    CreditCard,
    FileText,
    Eye
} from 'lucide-react';
import SectionContainer from '@/Components/SectionContainer';

export default function WriterWithdrawalShow({ withdrawal }) {
    const [showRejectModal, setShowRejectModal] = useState(false);
    const [showPaymentModal, setShowPaymentModal] = useState(false);
    const [rejectionReason, setRejectionReason] = useState('');
    const [transactionReference, setTransactionReference] = useState('');
    const [paymentNotes, setPaymentNotes] = useState('');
    const [isProcessing, setIsProcessing] = useState(false);

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'long',
            day: 'numeric',
            hour: '2-digit',
            minute: '2-digit'
        });
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'approved':
                return <CheckCircle size={20} className="text-green-500" />;
            case 'paid':
                return <CheckCircle size={20} className="text-blue-500" />;
            case 'rejected':
                return <XCircle size={20} className="text-red-500" />;
            case 'processing':
                return <Clock size={20} className="text-yellow-500" />;
            default:
                return <AlertCircle size={20} className="text-gray-500" />;
        }
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'approved':
                return 'bg-green-100 text-green-800';
            case 'paid':
                return 'bg-blue-100 text-blue-800';
            case 'rejected':
                return 'bg-red-100 text-red-800';
            case 'processing':
                return 'bg-yellow-100 text-yellow-800';
            default:
                return 'bg-gray-100 text-gray-800';
        }
    };

    const handleApprove = async () => {
        if (confirm('Are you sure you want to approve this withdrawal request?')) {
            setIsProcessing(true);
            try {
                await router.post(`/admin/writer-withdrawals/${withdrawal.id}/approve`);
                // Refresh the page to show updated status
                router.reload();
            } catch (error) {
                console.error('Failed to approve withdrawal:', error);
            } finally {
                setIsProcessing(false);
            }
        }
    };

    const handleReject = async () => {
        if (!rejectionReason.trim()) {
            alert('Please provide a rejection reason');
            return;
        }

        if (confirm('Are you sure you want to reject this withdrawal request?')) {
            setIsProcessing(true);
            try {
                await router.post(`/admin/writer-withdrawals/${withdrawal.id}/reject`, {
                    rejection_reason: rejectionReason
                });
                setShowRejectModal(false);
                setRejectionReason('');
                // Refresh the page to show updated status
                router.reload();
            } catch (error) {
                console.error('Failed to reject withdrawal:', error);
            } finally {
                setIsProcessing(false);
            }
        }
    };

    const handleProcess = async () => {
        if (confirm('Mark this withdrawal as processing?')) {
            setIsProcessing(true);
            try {
                await router.post(`/admin/writer-withdrawals/${withdrawal.id}/process`);
                // Refresh the page to show updated status
                router.reload();
            } catch (error) {
                console.error('Failed to mark as processing:', error);
            } finally {
                setIsProcessing(false);
            }
        }
    };

    const handleMarkAsPaid = async () => {
        if (!transactionReference.trim()) {
            alert('Please provide a transaction reference');
            return;
        }

        if (confirm('Mark this withdrawal as paid?')) {
            setIsProcessing(true);
            try {
                await router.post(`/admin/writer-withdrawals/${withdrawal.id}/mark-paid`, {
                    transaction_reference: transactionReference,
                    payment_notes: paymentNotes
                });
                setShowPaymentModal(false);
                setTransactionReference('');
                setPaymentNotes('');
                // Refresh the page to show updated status
                router.reload();
            } catch (error) {
                console.error('Failed to mark as paid:', error);
            } finally {
                setIsProcessing(false);
            }
        }
    };

    const canApprove = withdrawal.status === 'requested';
    const canReject = ['requested', 'approved'].includes(withdrawal.status);
    const canProcess = withdrawal.status === 'approved';
    const canMarkAsPaid = ['approved', 'processing'].includes(withdrawal.status);

    return (
        <AdminLayout>
            <Head title={`Withdrawal #${withdrawal.id} - ${withdrawal.writer?.name}`} />
            
            <div className="p-6 max-w-4xl mx-auto">
                {/* Header */}
                <div className="mb-8">
                    <div className="flex items-center justify-between">
                        <div>
                            <Link
                                href="/admin/writer-withdrawals"
                                className="inline-flex items-center text-sm text-gray-600 hover:text-gray-900 mb-4"
                            >
                                <ArrowLeft size={16} className="mr-2" />
                                Back to Withdrawals
                            </Link>
                            <h1 className="text-3xl font-bold text-gray-900 mb-2">
                                Withdrawal Request #{withdrawal.id}
                            </h1>
                            <p className="text-gray-600">
                                Manage withdrawal request from {withdrawal.writer?.name}
                            </p>
                        </div>
                        
                        <div className="flex items-center space-x-3">
                            {getStatusIcon(withdrawal.status)}
                            <span className={`inline-flex px-3 py-1 text-sm font-semibold rounded-full ${getStatusColor(withdrawal.status)}`}>
                                {withdrawal.status}
                            </span>
                        </div>
                    </div>
                </div>

                {/* Action Buttons */}
                <SectionContainer
                    className="bg-white shadow-sm border border-gray-200 mb-6"
                    title="Actions"
                    titleClassName="text-lg font-semibold text-gray-900"
                >
                    <div className="p-6">
                        <div className="flex flex-wrap gap-3">
                            {canApprove && (
                                <button
                                    onClick={handleApprove}
                                    disabled={isProcessing}
                                    className="px-4 py-2 bg-green-600 text-white rounded-md hover:bg-green-700 disabled:opacity-50"
                                >
                                    <CheckCircle size={16} className="inline mr-2" />
                                    Approve
                                </button>
                            )}
                            
                            {canReject && (
                                <button
                                    onClick={() => setShowRejectModal(true)}
                                    disabled={isProcessing}
                                    className="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 disabled:opacity-50"
                                >
                                    <XCircle size={16} className="inline mr-2" />
                                    Reject
                                </button>
                            )}
                            
                            {canProcess && (
                                <button
                                    onClick={handleProcess}
                                    disabled={isProcessing}
                                    className="px-4 py-2 bg-yellow-600 text-white rounded-md hover:bg-yellow-700 disabled:opacity-50"
                                >
                                    <Clock size={16} className="inline mr-2" />
                                    Mark as Processing
                                </button>
                            )}
                            
                            {canMarkAsPaid && (
                                <button
                                    onClick={() => setShowPaymentModal(true)}
                                    disabled={isProcessing}
                                    className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50"
                                >
                                    <DollarSign size={16} className="inline mr-2" />
                                    Mark as Paid
                                </button>
                            )}
                        </div>
                    </div>
                </SectionContainer>

                {/* Withdrawal Details */}
                <SectionContainer
                    className="bg-white shadow-sm border border-gray-200 mb-6"
                    title="Withdrawal Details"
                    titleClassName="text-lg font-semibold text-gray-900"
                >
                    <div className="p-6">
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Request Information</h3>
                                <dl className="space-y-3">
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Requested Amount</dt>
                                        <dd className="text-lg font-semibold text-gray-900">
                                            {formatCurrency(withdrawal.requested_amount)}
                                        </dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Currency</dt>
                                        <dd className="text-sm text-gray-900">{withdrawal.currency}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Withdrawal Date</dt>
                                        <dd className="text-sm text-gray-900">{formatDate(withdrawal.withdrawal_date)}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Payment Method</dt>
                                        <dd className="text-sm text-gray-900">{withdrawal.payment_method}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Requested At</dt>
                                        <dd className="text-sm text-gray-900">{formatDate(withdrawal.created_at)}</dd>
                                    </div>
                                </dl>
                            </div>
                            
                            <div>
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Writer Information</h3>
                                <dl className="space-y-3">
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Name</dt>
                                        <dd className="text-sm text-gray-900">{withdrawal.writer?.name || 'N/A'}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Email</dt>
                                        <dd className="text-sm text-gray-900">{withdrawal.writer?.email || 'N/A'}</dd>
                                    </div>
                                    <div>
                                        <dt className="text-sm font-medium text-gray-500">Writer ID</dt>
                                        <dd className="text-sm text-gray-900">{withdrawal.writer_id}</dd>
                                    </div>
                                </dl>
                            </div>
                        </div>

                        {/* Payment Details */}
                        {withdrawal.payment_details && (
                            <div className="mt-6 pt-6 border-t border-gray-200">
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Payment Details</h3>
                                <div className="bg-gray-50 rounded-md p-4">
                                    <pre className="text-sm text-gray-700 whitespace-pre-wrap">
                                        {JSON.stringify(withdrawal.payment_details, null, 2)}
                                    </pre>
                                </div>
                            </div>
                        )}

                        {/* Status Timeline */}
                        <div className="mt-6 pt-6 border-t border-gray-200">
                            <h3 className="text-lg font-medium text-gray-900 mb-4">Status Timeline</h3>
                            <div className="space-y-3">
                                <div className="flex items-center">
                                    <div className="w-3 h-3 bg-green-500 rounded-full mr-3"></div>
                                    <span className="text-sm text-gray-600">Requested on {formatDate(withdrawal.created_at)}</span>
                                </div>
                                
                                {withdrawal.approved_at && (
                                    <div className="flex items-center">
                                        <div className="w-3 h-3 bg-blue-500 rounded-full mr-3"></div>
                                        <span className="text-sm text-gray-600">
                                            Approved by {withdrawal.approver?.name || 'Admin'} on {formatDate(withdrawal.approved_at)}
                                        </span>
                                    </div>
                                )}
                                
                                {withdrawal.processed_at && (
                                    <div className="flex items-center">
                                        <div className="w-3 h-3 bg-yellow-500 rounded-full mr-3"></div>
                                        <span className="text-sm text-gray-600">Processing started on {formatDate(withdrawal.processed_at)}</span>
                                    </div>
                                )}
                                
                                {withdrawal.paid_at && (
                                    <div className="flex items-center">
                                        <div className="w-3 h-3 bg-green-500 rounded-full mr-3"></div>
                                        <span className="text-sm text-gray-600">Paid on {formatDate(withdrawal.paid_at)}</span>
                                    </div>
                                )}
                                
                                {withdrawal.rejection_reason && (
                                    <div className="flex items-center">
                                        <div className="w-3 h-3 bg-red-500 rounded-full mr-3"></div>
                                        <span className="text-sm text-gray-600">Rejected: {withdrawal.rejection_reason}</span>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </SectionContainer>

                {/* Payment Records */}
                {withdrawal.payment_records && withdrawal.payment_records.length > 0 && (
                    <SectionContainer
                        className="bg-white shadow-sm border border-gray-200 mb-6"
                        title="Payment Records"
                        titleClassName="text-lg font-semibold text-gray-900"
                    >
                        <div className="p-6">
                            <div className="overflow-x-auto">
                                <table className="min-w-full divide-y divide-gray-200">
                                    <thead className="bg-gray-50">
                                        <tr>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Order
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Amount
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Status
                                            </th>
                                            <th className="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                Available At
                                            </th>
                                        </tr>
                                    </thead>
                                    <tbody className="bg-white divide-y divide-gray-200">
                                        {withdrawal.payment_records.map((record) => (
                                            <tr key={record.id}>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                    <Link
                                                        href={`/admin/orders/${record.order?.id}`}
                                                        className="text-blue-600 hover:text-blue-900"
                                                    >
                                                        Order #{record.order?.order_number}
                                                    </Link>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                    {formatCurrency(record.payment_amount)}
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap">
                                                    <span className={`inline-flex px-2 py-1 text-xs font-semibold rounded-full ${getStatusColor(record.status)}`}>
                                                        {record.status}
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4 whitespace-nowrap text-sm text-gray-900">
                                                    {formatDate(record.available_at)}
                                                </td>
                                            </tr>
                                        ))}
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </SectionContainer>
                )}

                {/* Reject Modal */}
                {showRejectModal && (
                    <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
                        <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
                            <div className="mt-3">
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Reject Withdrawal</h3>
                                <div className="mb-4">
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Rejection Reason *
                                    </label>
                                    <textarea
                                        value={rejectionReason}
                                        onChange={(e) => setRejectionReason(e.target.value)}
                                        className="w-full border border-gray-300 rounded-md px-3 py-2"
                                        rows="3"
                                        placeholder="Provide a reason for rejection..."
                                    />
                                </div>
                                <div className="flex justify-end space-x-3">
                                    <button
                                        onClick={() => setShowRejectModal(false)}
                                        className="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={handleReject}
                                        disabled={isProcessing || !rejectionReason.trim()}
                                        className="px-4 py-2 bg-red-600 text-white rounded-md hover:bg-red-700 disabled:opacity-50"
                                    >
                                        Reject
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )}

                {/* Payment Modal */}
                {showPaymentModal && (
                    <div className="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full z-50">
                        <div className="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
                            <div className="mt-3">
                                <h3 className="text-lg font-medium text-gray-900 mb-4">Mark as Paid</h3>
                                <div className="mb-4">
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Transaction Reference *
                                    </label>
                                    <input
                                        type="text"
                                        value={transactionReference}
                                        onChange={(e) => setTransactionReference(e.target.value)}
                                        className="w-full border border-gray-300 rounded-md px-3 py-2"
                                        placeholder="Enter transaction reference..."
                                    />
                                </div>
                                <div className="mb-4">
                                    <label className="block text-sm font-medium text-gray-700 mb-2">
                                        Payment Notes
                                    </label>
                                    <textarea
                                        value={paymentNotes}
                                        onChange={(e) => setPaymentNotes(e.target.value)}
                                        className="w-full border border-gray-300 rounded-md px-3 py-2"
                                        rows="2"
                                        placeholder="Optional payment notes..."
                                    />
                                </div>
                                <div className="flex justify-end space-x-3">
                                    <button
                                        onClick={() => setShowPaymentModal(false)}
                                        className="px-4 py-2 bg-gray-300 text-gray-700 rounded-md hover:bg-gray-400"
                                    >
                                        Cancel
                                    </button>
                                    <button
                                        onClick={handleMarkAsPaid}
                                        disabled={isProcessing || !transactionReference.trim()}
                                        className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 disabled:opacity-50"
                                    >
                                        Mark as Paid
                                    </button>
                                </div>
                            </div>
                        </div>
                    </div>
                )}
            </div>
        </AdminLayout>
    );
} 