import { Head, Link, usePage } from "@inertiajs/react";
import { useState, useEffect, useMemo } from "react";
import { motion, AnimatePresence } from "framer-motion";
import {
    Calendar,
    Clock,
    ArrowRight,
    User,
    Tag,
    ChevronLeft,
    Facebook,
    Twitter,
    Linkedin,
    Link2,
    ChevronUp,
    Edit,
} from "lucide-react";
import MainLayout from "@/Layouts/MainLayout";
import OptimizedImage from "@/Components/OptimizedImage";
import PriceCalculatorSection from "@/Components/CMS/PriceCalculatorSection";

export default function BlogPost({
    post,
    relatedPosts = [],
    meta = {},
    openGraph = {},
    twitterCard = {},
    breadcrumb = [],
}) {
    const { auth } = usePage().props;
    const [readingProgress, setReadingProgress] = useState(0);
    const [activeSection, setActiveSection] = useState(null);
    const [showBackToTop, setShowBackToTop] = useState(false);
    const [calculatorOpen, setCalculatorOpen] = useState(false);

    // Parse content for shortcodes and convert to React components
    const parsedContent = useMemo(() => {
        if (!post?.content) return [];

        const content = post.content;
        const shortcodePattern = /\[price-calculator(?:\s+([^\]]+))?\]/g;
        const segments = [];
        let lastIndex = 0;
        let match;

        while ((match = shortcodePattern.exec(content)) !== null) {
            // Add HTML content before shortcode
            if (match.index > lastIndex) {
                segments.push({
                    type: 'html',
                    content: content.substring(lastIndex, match.index)
                });
            }

            // Parse shortcode attributes (if any)
            const attributes = {};
            if (match[1]) {
                const attrPattern = /(\w+)="([^"]*)"/g;
                let attrMatch;
                while ((attrMatch = attrPattern.exec(match[1])) !== null) {
                    attributes[attrMatch[1]] = attrMatch[2];
                }
            }

            // Add shortcode component
            segments.push({
                type: 'shortcode',
                shortcode: 'price-calculator',
                attributes
            });

            lastIndex = match.index + match[0].length;
        }

        // Add remaining HTML content
        if (lastIndex < content.length) {
            segments.push({
                type: 'html',
                content: content.substring(lastIndex)
            });
        }

        return segments;
    }, [post?.content]);

    // Render a content segment based on type
    const renderContentSegment = (segment, index) => {
        if (segment.type === 'html') {
            return (
                <div
                    key={`html-${index}`}
                    className="prose prose-lg max-w-none dark:prose-invert"
                    dangerouslySetInnerHTML={{ __html: segment.content }}
                />
            );
        }

        if (segment.type === 'shortcode' && segment.shortcode === 'price-calculator') {
            // Default calculator configuration for blog posts
            const calculatorContent = {
                title: segment.attributes.title || 'Calculate Your Essay Price',
                subtitle: segment.attributes.subtitle || 'Get an instant quote for your assignment. No commitment required.',
                buttonText: segment.attributes.buttonText || 'Get Your Price',
                buttonVariant: segment.attributes.variant || 'primary',
                backgroundColor: segment.attributes.background || 'gray',
                alignment: segment.attributes.align || 'center',
                showEyebrow: false,
                showDescription: false,
            };

            return (
                <div key={`shortcode-${index}`} className="-mx-6 md:-mx-8 my-8">
                    <PriceCalculatorSection content={calculatorContent} />
                </div>
            );
        }

        return null;
    };

    useEffect(() => {
        const handleScroll = () => {
            setShowBackToTop(window.pageYOffset > 400);
        };

        window.addEventListener("scroll", handleScroll);
        return () => window.removeEventListener("scroll", handleScroll);
    }, []);

    // Add this function after your other functions (around line 80)
    const scrollToTop = () => {
        window.scrollTo({
            top: 0,
            behavior: "smooth",
        });
    };

    // Safety check for post
    if (!post) {
        return (
            <MainLayout>
                <div className="min-h-screen bg-gray-50 dark:bg-gray-900 flex items-center justify-center">
                    <div className="text-center">
                        <h1 className="text-2xl font-bold text-gray-900 dark:text-white mb-4">
                            Post not found
                        </h1>
                        <Link
                            href="/blog"
                            className="text-blue-600 dark:text-blue-400 hover:underline"
                        >
                            Return to blog
                        </Link>
                    </div>
                </div>
            </MainLayout>
        );
    }

    // Track reading progress
    useEffect(() => {
        const handleScroll = () => {
            const article = document.querySelector("article");
            if (!article) return;

            const articleTop = article.offsetTop;
            const articleHeight = article.offsetHeight;
            const windowHeight = window.innerHeight;
            const scrollTop = window.pageYOffset;

            const progress = Math.min(
                Math.max(
                    (scrollTop - articleTop + windowHeight / 2) / articleHeight,
                    0
                ),
                1
            );

            setReadingProgress(progress * 100);
        };

        window.addEventListener("scroll", handleScroll);
        return () => window.removeEventListener("scroll", handleScroll);
    }, []);

    // Track active section for TOC
    useEffect(() => {
        const handleScroll = () => {
            if (
                !post?.table_of_contents ||
                !Array.isArray(post.table_of_contents)
            )
                return;

            const headings = post.table_of_contents
                .map((item) => document.getElementById(item.id))
                .filter(Boolean);

            const scrollPosition = window.scrollY + 150;

            let currentSection = null;
            for (const heading of headings) {
                if (heading.offsetTop <= scrollPosition) {
                    currentSection = heading.id;
                }
            }

            setActiveSection(currentSection);
        };

        window.addEventListener("scroll", handleScroll);
        return () => window.removeEventListener("scroll", handleScroll);
    }, [post?.table_of_contents]);

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: { staggerChildren: 0.1 },
        },
    };

    // Format date helper
    const formatDate = (dateString) => {
        if (!dateString) return "Unknown date";
        return new Date(dateString).toLocaleDateString("en-US", {
            year: "numeric",
            month: "long",
            day: "numeric",
        });
    };

    // Share functions
    const shareOnFacebook = () => {
        window.open(
            `https://www.facebook.com/sharer/sharer.php?u=${encodeURIComponent(
                window.location.href
            )}`,
            "_blank"
        );
    };

    const shareOnTwitter = () => {
        const text = `${post.title} - ${post.excerpt || ""}`;
        window.open(
            `https://twitter.com/intent/tweet?text=${encodeURIComponent(
                text
            )}&url=${encodeURIComponent(window.location.href)}`,
            "_blank"
        );
    };

    const shareOnLinkedIn = () => {
        window.open(
            `https://www.linkedin.com/sharing/share-offsite/?url=${encodeURIComponent(
                window.location.href
            )}`,
            "_blank"
        );
    };

    const copyLink = () => {
        navigator.clipboard.writeText(window.location.href);
        alert("Link copied to clipboard!");
    };

    // Helper to convert relative URLs to absolute URLs
    const getAbsoluteUrl = (url) => {
        if (!url) return null;
        if (url.startsWith('http://') || url.startsWith('https://')) return url;
        if (url.startsWith('/')) return `https://academicscribe.com${url}`;
        return `https://academicscribe.com/${url}`;
    };

    // Prepare canonical URL
    const canonicalUrl = meta.canonical || `/blog/${post.slug}`;
    const fullCanonicalUrl = canonicalUrl.startsWith("http")
        ? canonicalUrl
        : `${window.location.origin}${canonicalUrl}`;

    return (
        <MainLayout>
            <Head>
                {/* Basic SEO Meta Tags */}
                <title>{meta.title || post.title}</title>
                <meta
                    name="description"
                    content={meta.description || post.excerpt || ""}
                />
                <meta
                    name="keywords"
                    content={
                        post.meta_data?.keywords ||
                        "academic writing, blog post"
                    }
                />
                <meta
                    name="author"
                    content={post.author?.name || "Academic Scribe"}
                />
                <link rel="canonical" href={fullCanonicalUrl} />

                {/* Open Graph Meta Tags - From Database */}
                <meta property="og:title" content={openGraph.title || meta.title || post.title} />
                <meta
                    property="og:description"
                    content={openGraph.description || meta.description || post.excerpt || ""}
                />
                <meta property="og:type" content={openGraph.type || "article"} />
                <meta property="og:url" content={openGraph.url || fullCanonicalUrl} />
                <meta property="og:site_name" content={openGraph.site_name || "Academic Scribe"} />
                {openGraph.image && (
                    <meta property="og:image" content={getAbsoluteUrl(openGraph.image)} />
                )}
                {!openGraph.image && post.featured_image && (
                    <meta property="og:image" content={getAbsoluteUrl(post.featured_image)} />
                )}
                {!openGraph.image && !post.featured_image && (
                    <meta property="og:image" content="https://academicscribe.com/images/blog-default-og.jpg" />
                )}
                <meta
                    property="og:image:alt"
                    content={openGraph.image_alt || post.featured_image_alt || post.title}
                />
                {openGraph.locale && (
                    <meta property="og:locale" content={openGraph.locale} />
                )}
                <meta
                    property="article:published_time"
                    content={post.published_at}
                />
                <meta
                    property="article:author"
                    content={post.author?.name || "Academic Scribe"}
                />
                <meta
                    property="article:section"
                    content={post.category?.name || "Academic Writing"}
                />

                {/* Twitter Card Meta Tags - From Database */}
                <meta name="twitter:card" content={twitterCard.card || "summary_large_image"} />
                {twitterCard.site && (
                    <meta name="twitter:site" content={twitterCard.site} />
                )}
                {!twitterCard.site && (
                    <meta name="twitter:site" content="@AcademicScribe" />
                )}
                {twitterCard.creator && (
                    <meta name="twitter:creator" content={twitterCard.creator} />
                )}
                <meta name="twitter:title" content={twitterCard.title || openGraph.title || meta.title || post.title} />
                <meta
                    name="twitter:description"
                    content={twitterCard.description || openGraph.description || meta.description || post.excerpt || ""}
                />
                {twitterCard.image && (
                    <meta name="twitter:image" content={getAbsoluteUrl(twitterCard.image)} />
                )}
                {twitterCard.image_alt && (
                    <meta name="twitter:image:alt" content={twitterCard.image_alt} />
                )}
                <meta
                    name="twitter:image"
                    content={
                        getAbsoluteUrl(post.featured_image) ||
                        "https://academicscribe.com/images/blog-default-twitter.jpg"
                    }
                />

                {/* Structured Data - Prioritize database over hardcoded */}
                <script
                    type="application/ld+json"
                    dangerouslySetInnerHTML={{
                        __html: JSON.stringify(
                            post.structured_data || {
                                "@context": "https://schema.org",
                                "@type": "BlogPosting",
                                headline: post.title,
                                description: post.excerpt || meta.description || "",
                                image: getAbsoluteUrl(post.featured_image),
                                datePublished: post.published_at,
                                dateModified: post.updated_at,
                                author: {
                                    "@type": "Person",
                                    name: post.author?.name || "Academic Scribe",
                                },
                                publisher: {
                                    "@type": "Organization",
                                    name: "Academic Scribe",
                                    url: "https://academicscribe.com",
                                },
                                mainEntityOfPage: {
                                    "@type": "WebPage",
                                    "@id": fullCanonicalUrl,
                                },
                            }
                        ),
                    }}
                />
            </Head>

            {/* Reading Progress Bar */}
            <div className="fixed top-0 left-0 w-full h-1 bg-gray-200 dark:bg-gray-800 z-50">
                <div
                    className="h-full bg-blue-600 transition-all duration-300"
                    style={{ width: `${readingProgress}%` }}
                />
            </div>

            <div className="min-h-screen bg-gray-50 dark:bg-gray-900">
                {/* Breadcrumb */}
                {breadcrumb &&
                    Array.isArray(breadcrumb) &&
                    breadcrumb.length > 0 && (
                        <div className="bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                            <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-4">
                                <nav className="flex" aria-label="Breadcrumb">
                                    <ol className="flex items-center space-x-2">
                                        {breadcrumb.map((item, index) => (
                                            <li
                                                key={index}
                                                className="flex items-center"
                                            >
                                                {index > 0 && (
                                                    <ChevronLeft className="w-4 h-4 text-gray-400 mx-2 rotate-180" />
                                                )}
                                                <Link
                                                    href={item.url}
                                                    className={`text-sm ${
                                                        index ===
                                                        breadcrumb.length - 1
                                                            ? "text-gray-900 dark:text-white font-medium"
                                                            : "text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300"
                                                    }`}
                                                >
                                                    {item.name}
                                                </Link>
                                            </li>
                                        ))}
                                    </ol>
                                </nav>
                            </div>
                        </div>
                    )}

                {/* Main Content */}
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-8">
                    <div className="grid grid-cols-1 lg:grid-cols-12 gap-8">
                        {/* Article Content */}
                        <div className="lg:col-span-8">
                            <motion.article
                                initial="hidden"
                                animate="visible"
                                variants={staggerContainer}
                                className="bg-white dark:bg-gray-800 rounded-xl shadow-sm overflow-hidden"
                            >
                                {/* Featured Image */}
                                {post.featured_image && (
                                    <motion.div
                                        variants={fadeIn}
                                        className="relative"
                                    >
                                        <OptimizedImage
                                            src={post.featured_image}
                                            alt={
                                                post.featured_image_alt ||
                                                post.title
                                            }
                                            className="w-full h-64 md:h-80 object-cover"
                                            priority={true}
                                        />
                                        <div className="absolute inset-0 bg-gradient-to-t from-black/20 to-transparent" />
                                    </motion.div>
                                )}

                                {/* Article Header */}
                                <div className="p-6 md:p-8">
                                    <motion.div
                                        variants={fadeIn}
                                        className="mb-6"
                                    >
                                        <div className="flex items-center justify-between mb-4">
                                            {/* Category Badge */}
                                            {post.category && (
                                                <span className="inline-flex items-center px-3 py-1 rounded-full text-sm font-medium bg-blue-100 text-blue-800 dark:bg-blue-900/30 dark:text-blue-400">
                                                    {post.category.name}
                                                </span>
                                            )}

                                            {/* Edit Link for Admins */}
                                            {(auth?.user?.user_type === 'admin' || auth?.user?.user_type === 'super_admin') && (
                                                <Link
                                                    href={`/admin/blog/posts/${post.id}/edit`}
                                                    className="inline-flex items-center gap-2 px-4 py-2 bg-gray-100 hover:bg-gray-200 dark:bg-gray-700 dark:hover:bg-gray-600 text-gray-700 dark:text-gray-300 text-sm font-medium rounded-lg transition-colors"
                                                >
                                                    <Edit className="w-4 h-4" />
                                                    Edit Post
                                                </Link>
                                            )}
                                        </div>

                                        {/* Title */}
                                        <h1 className="text-3xl md:text-4xl font-bold text-gray-900 dark:text-white mb-4">
                                            {post.title}
                                        </h1>

                                        {/* Subtitle */}
                                        {post.subtitle && (
                                            <p className="text-xl text-gray-600 dark:text-gray-300 mb-6">
                                                {post.subtitle}
                                            </p>
                                        )}

                                        {/* Post Meta */}
                                        <div className="flex flex-wrap items-center gap-4 text-sm text-gray-500 dark:text-gray-400">
                                            <div className="flex items-center">
                                                <User className="w-4 h-4 mr-2" />
                                                <span className="font-medium">{post.author?.name || "Academic Scribe"}</span>
                                            </div>
                                            <div className="flex items-center">
                                                <Calendar className="w-4 h-4 mr-2" />
                                                <span>Published {formatDate(post.published_at)}</span>
                                            </div>
                                            {post.updated_at && post.updated_at !== post.published_at && (
                                                <div className="flex items-center text-gray-400 dark:text-gray-500">
                                                    <Clock className="w-4 h-4 mr-2" />
                                                    <span>Updated {formatDate(post.updated_at)}</span>
                                                </div>
                                            )}
                                            <div className="flex items-center">
                                                <Clock className="w-4 h-4 mr-2" />
                                                {post.read_time || 5} min read
                                            </div>
                                        </div>

                                        {/* Tags */}
                                        {post.tags &&
                                            Array.isArray(post.tags) &&
                                            post.tags.length > 0 && (
                                                <div className="flex flex-wrap gap-2 mt-4">
                                                    {post.tags.map((tag) => (
                                                        <span
                                                            key={tag.id}
                                                            className="inline-flex items-center px-2 py-1 rounded-md text-xs font-medium bg-gray-100 text-gray-700 dark:bg-gray-700 dark:text-gray-300"
                                                        >
                                                            <Tag className="w-3 h-3 mr-1" />
                                                            {tag.name}
                                                        </span>
                                                    ))}
                                                </div>
                                            )}
                                    </motion.div>

                                    {/* Social Share */}
                                    <motion.div
                                        variants={fadeIn}
                                        className="flex items-center gap-4 py-4 border-t border-b border-gray-200 dark:border-gray-700 mb-8"
                                    >
                                        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                            Share this post:
                                        </span>
                                        <div className="flex gap-2">
                                            <button
                                                onClick={shareOnFacebook}
                                                className="p-2 rounded-full bg-blue-600 text-white hover:bg-blue-700 transition-colors"
                                            >
                                                <Facebook className="w-4 h-4" />
                                            </button>
                                            <button
                                                onClick={shareOnTwitter}
                                                className="p-2 rounded-full bg-sky-500 text-white hover:bg-sky-600 transition-colors"
                                            >
                                                <Twitter className="w-4 h-4" />
                                            </button>
                                            <button
                                                onClick={shareOnLinkedIn}
                                                className="p-2 rounded-full bg-blue-700 text-white hover:bg-blue-800 transition-colors"
                                            >
                                                <Linkedin className="w-4 h-4" />
                                            </button>
                                            <button
                                                onClick={copyLink}
                                                className="p-2 rounded-full bg-gray-600 text-white hover:bg-gray-700 transition-colors"
                                            >
                                                <Link2 className="w-4 h-4" />
                                            </button>
                                        </div>
                                    </motion.div>

                                    {/* Article Content */}
                                    <motion.div variants={fadeIn}>
                                        {parsedContent.length > 0 ? (
                                            parsedContent.map((segment, index) => 
                                                renderContentSegment(segment, index)
                                            )
                                        ) : (
                                            <div
                                                className="prose prose-lg max-w-none dark:prose-invert"
                                                dangerouslySetInnerHTML={{
                                                    __html: post.content || "",
                                                }}
                                            />
                                        )}
                                    </motion.div>
                                </div>
                            </motion.article>
                        </div>

                        {/* Sidebar */}
                        <div className="lg:col-span-4">
                            <div className="sticky top-8 max-h-[calc(100vh-4rem)] overflow-y-auto space-y-6">
                                {/* Author Info */}
                                {post.author && (
                                    <motion.div
                                        initial="hidden"
                                        animate="visible"
                                        variants={fadeIn}
                                        className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6"
                                    >
                                        <div className="flex items-center mb-4">
                                            {post.author.avatar && (
                                                <img
                                                    src={post.author.avatar}
                                                    alt={post.author.name}
                                                    className="w-12 h-12 rounded-full mr-4"
                                                />
                                            )}
                                            <div>
                                                <h3 className="font-semibold text-gray-900 dark:text-white">
                                                    {post.author.name}
                                                </h3>
                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                    Author
                                                </p>
                                            </div>
                                        </div>
                                        {post.author.bio && (
                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                {post.author.bio}
                                            </p>
                                        )}
                                    </motion.div>
                                )}

                                {/* Table of Contents */}
                                {post.table_of_contents &&
                                    Array.isArray(post.table_of_contents) &&
                                    post.table_of_contents.length > 0 && (
                                        <motion.div
                                            initial="hidden"
                                            animate="visible"
                                            variants={fadeIn}
                                            className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6"
                                        >
                                            <h3 className="font-semibold text-gray-900 dark:text-white mb-4 border-b border-gray-200 dark:border-gray-700 pb-2">
                                                Table of Contents
                                            </h3>
                                            <ul className="space-y-2">
                                                {post.table_of_contents.map(
                                                    (item, index) => (
                                                        <li key={index}>
                                                            <a
                                                                href={`#${item.id}`}
                                                                className={`block text-sm transition-colors py-1 px-2 rounded-md ${
                                                                    activeSection ===
                                                                    item.id
                                                                        ? "text-blue-600 dark:text-blue-400 bg-blue-50 dark:bg-blue-900/30 font-medium"
                                                                        : "text-gray-600 dark:text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-gray-50 dark:hover:bg-gray-700"
                                                                }`}
                                                                style={{
                                                                    paddingLeft: `${
                                                                        (item.level -
                                                                            1) *
                                                                            16 +
                                                                        8
                                                                    }px`,
                                                                }}
                                                                onClick={(
                                                                    e
                                                                ) => {
                                                                    e.preventDefault();
                                                                    const element =
                                                                        document.getElementById(
                                                                            item.id
                                                                        );
                                                                    if (
                                                                        element
                                                                    ) {
                                                                        const offsetTop =
                                                                            element.offsetTop -
                                                                            100;
                                                                        window.scrollTo(
                                                                            {
                                                                                top: offsetTop,
                                                                                behavior:
                                                                                    "smooth",
                                                                            }
                                                                        );
                                                                    }
                                                                }}
                                                            >
                                                                {item.title}
                                                            </a>
                                                        </li>
                                                    )
                                                )}
                                            </ul>
                                        </motion.div>
                                    )}

                                {/* Related Posts */}
                                {relatedPosts &&
                                    Array.isArray(relatedPosts) &&
                                    relatedPosts.length > 0 && (
                                        <motion.div
                                            initial="hidden"
                                            animate="visible"
                                            variants={fadeIn}
                                            className="bg-white dark:bg-gray-800 rounded-xl shadow-sm p-6"
                                        >
                                            <h3 className="font-semibold text-gray-900 dark:text-white mb-4 border-b border-gray-200 dark:border-gray-700 pb-2">
                                                Related Posts
                                            </h3>
                                            <div className="space-y-4">
                                                {relatedPosts.map(
                                                    (relatedPost) => (
                                                        <article
                                                            key={relatedPost.id}
                                                            className="group"
                                                        >
                                                            <div className="flex gap-3 p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                                                                {relatedPost.featured_image && (
                                                                    <OptimizedImage
                                                                        src={
                                                                            relatedPost.featured_image
                                                                        }
                                                                        alt={
                                                                            relatedPost.title
                                                                        }
                                                                        width={64}
                                                                        height={64}
                                                                        className="w-16 h-16 object-cover rounded-lg flex-shrink-0"
                                                                    />
                                                                )}
                                                                <div className="flex-1">
                                                                    <Link
                                                                        href={`/blog/${relatedPost.slug}`}
                                                                    >
                                                                        <h4 className="font-medium text-gray-900 dark:text-white hover:text-blue-600 dark:hover:text-blue-400 transition-colors line-clamp-2">
                                                                            {
                                                                                relatedPost.title
                                                                            }
                                                                        </h4>
                                                                    </Link>
                                                                    <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                                                        {formatDate(
                                                                            relatedPost.published_at
                                                                        )}
                                                                    </p>
                                                                </div>
                                                            </div>
                                                        </article>
                                                    )
                                                )}
                                            </div>
                                        </motion.div>
                                    )}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            <AnimatePresence>
                {showBackToTop && (
                    <motion.div
                        initial={{ opacity: 0, scale: 0.8, y: 20 }}
                        animate={{ opacity: 1, scale: 1, y: 0 }}
                        exit={{ opacity: 0, scale: 0.8, y: 20 }}
                        transition={{ duration: 0.3 }}
                        className="fixed bottom-8 right-8 z-50"
                    >
                        <button
                            onClick={scrollToTop}
                            className="group flex items-center px-4 py-3 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-full shadow-lg hover:shadow-xl text-gray-700 dark:text-gray-300 hover:text-blue-600 dark:hover:text-blue-400 transition-all duration-300 hover:scale-105"
                            aria-label="Back to top"
                        >
                            <ChevronUp className="w-4 h-4 mr-2 group-hover:-translate-y-0.5 transition-transform duration-200" />
                            <span className="text-sm font-medium hidden sm:inline">
                                Back to Top
                            </span>
                        </button>
                    </motion.div>
                )}
            </AnimatePresence>
        </MainLayout>
    );
}
