import { useState, useEffect } from "react";
import { Head, Link, useForm, router } from "@inertiajs/react";
import {
    Search,
    X,
    Filter,
    ArrowUpDown,
    Calendar,
    CalendarDays,
    Check,
    ChevronDown,
    ChevronsUpDown,
    Gavel,
    ListOrdered,
    Clock,
    DollarSign,
} from "lucide-react";
import ClientLayout from "@/Layouts/ClientLayout";
import CleanPagination from "@/Components/CleanPagination";
import { formatDistance } from "date-fns";

export default function BidsIndex({ bids, filters, stats }) {
    // State for filter dropdowns
    const [statusDropdownOpen, setStatusDropdownOpen] = useState(false);
    const [sortDropdownOpen, setSortDropdownOpen] = useState(false);
    const [dateRangeOpen, setDateRangeOpen] = useState(false);

    // Filter form
    const { data, setData, get, processing } = useForm({
        status: filters.status || "all",
        search: filters.search || "",
        from_date: filters.from_date || "",
        to_date: filters.to_date || "",
        sort_by: filters.sort_by || "created_at",
        sort_direction: filters.sort_direction || "desc",
    });

    // Status options
    const statusOptions = [
        { value: "all", label: "All Bids" },
        { value: "pending", label: "Pending" },
        { value: "accepted", label: "Accepted" },
        { value: "rejected", label: "Rejected" },
    ];

    // Sort options
    const sortOptions = [
        { value: "created_at", label: "Newest First", direction: "desc" },
        { value: "created_at", label: "Oldest First", direction: "asc" },
        { value: "bid_amount", label: "Highest Amount", direction: "desc" },
        { value: "bid_amount", label: "Lowest Amount", direction: "asc" },
        {
            value: "completion_time",
            label: "Earliest Completion",
            direction: "asc",
        },
        {
            value: "completion_time",
            label: "Latest Completion",
            direction: "desc",
        },
    ];

    // Get current sort option label
    const getCurrentSortLabel = () => {
        const option = sortOptions.find(
            (opt) =>
                opt.value === data.sort_by &&
                opt.direction === data.sort_direction
        );
        return option ? option.label : "Sort by";
    };

    // Get current status label
    const getCurrentStatusLabel = () => {
        const option = statusOptions.find((opt) => opt.value === data.status);
        return option ? option.label : "All Bids";
    };

    // Handle search submit
    const handleSearch = (e) => {
        e.preventDefault();
        applyFilters();
    };

    // Apply all filters
    const applyFilters = () => {
        get(route("client.bids.index"), {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Clear all filters
    const clearFilters = () => {
        setData({
            status: "all",
            search: "",
            from_date: "",
            to_date: "",
            sort_by: "created_at",
            sort_direction: "desc",
        });
        get(route("client.bids.index"), {
            preserveState: true,
            preserveScroll: true,
        });
    };

    // Set status filter
    const setStatus = (status) => {
        console.log("Setting status to:", status);
        setData("status", status);
        setStatusDropdownOpen(false);
        applyFilters();
    };

    // Replace the existing handleStatusChange function with this corrected version
    const handleStatusChange = (newStatus) => {
        // Log the new status for debugging
        console.log("Setting status to:", newStatus);

        // Update form data
        setData("status", newStatus);

        // Close the dropdown
        setStatusDropdownOpen(false);

        // Make a direct visit to the route with the explicit status parameter
        // This ensures the parameter is passed correctly to the backend
        router.get(
            route("client.bids.index"),
            { status: newStatus },
            {
                preserveState: true,
                preserveScroll: true,
                only: ["bids", "stats"],
            }
        );
    };

    // Set sort option
    const setSort = (sortBy, sortDirection) => {
        setData({
            ...data,
            sort_by: sortBy,
            sort_direction: sortDirection,
        });
        setSortDropdownOpen(false);
        applyFilters();
    };

    // Handle date range change
    const handleDateRangeChange = () => {
        setDateRangeOpen(false);
        applyFilters();
    };

    // Format currency with the correct symbol
    const formatCurrency = (amount, currencyCode = "USD") => {
        // Get the appropriate currency symbol
        const currencySymbol = getCurrencySymbol(currencyCode);

        // Format the number with 2 decimal places
        const formattedAmount = parseFloat(amount).toFixed(2);

        // Return the formatted currency string
        return `${currencySymbol}${formattedAmount}`;
    };

    // Helper function to get currency symbol from currency code
    const getCurrencySymbol = (currencyCode) => {
        switch (currencyCode?.toUpperCase()) {
            case "EUR":
                return "€";
            case "GBP":
                return "£";
            case "CAD":
                return "C$";
            case "AUD":
                return "A$";
            default:
                return "$";
        }
    };

    // Function to get status badge class
    const getStatusBadgeClass = (status) => {
        switch (status) {
            case "accepted":
                return "bg-green-500 text-white";
            case "rejected":
                return "bg-red-500 text-white";
            case "pending":
                return "bg-yellow-500 text-yellow-100";
            default:
                return "bg-gray-500 text-white";
        }
    };

    // Format date for display
    const formatDate = (dateString) => {
        if (!dateString) return "N/A";

        const date = new Date(dateString);
        if (isNaN(date.getTime())) return "Invalid Date";

        return date.toLocaleDateString("en-US", {
            year: "numeric",
            month: "short",
            day: "numeric",
            hour: "2-digit",
            minute: "2-digit",
        });
    };

    // Calculate time remaining until completion
    const timeUntilCompletion = (completionTime) => {
        if (!completionTime) return "No deadline set";

        const now = new Date();
        const completion = new Date(completionTime);

        if (isNaN(completion.getTime())) return "Invalid deadline";

        if (now > completion) return "Deadline passed";

        return formatDistance(completion, now, { addSuffix: true });
    };

    // Determine if we have active filters
    const hasActiveFilters =
        data.status !== "all" ||
        data.search !== "" ||
        data.from_date !== "" ||
        data.to_date !== "" ||
        data.sort_by !== "created_at" ||
        data.sort_direction !== "desc";

    return (
        <ClientLayout
            header={
                <div className="text-xl font-semibold leading-tight text-gray-800 dark:text-gray-100">
                    My Bids
                </div>
            }
        >
            <Head title="My Bids" />

            <div className="py-12">
                <div className="mx-auto max-w-7xl sm:px-6 lg:px-8">
                    <div className="overflow-hidden bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none border border-gray-300 dark:border-gray-600 shadow-sm sm:rounded-lg dark:bg-gray-800">
                        {/* Statistics Section */}
                        <div className="grid grid-cols-1 md:grid-cols-4 gap-4 p-6 bg-gradient-to-r from-indigo-100 to-cyan-100 dark:bg-none border-b border-gray-200 dark:border-gray-700 dark:bg-gray-800">
                            <div className="p-4 rounded-lg bg-white dark:bg-gray-700 shadow-sm border border-gray-200 dark:border-gray-600">
                                <h3 className="text-sm font-medium text-gray-600 dark:text-gray-300">
                                    Total Bids
                                </h3>
                                <p className="text-2xl font-bold text-indigo-600 dark:text-indigo-400">
                                    {stats.total}
                                </p>
                            </div>
                            <div className="p-4 rounded-lg bg-white dark:bg-gray-700 shadow-sm border border-gray-200 dark:border-gray-600">
                                <h3 className="text-sm font-medium text-gray-600 dark:text-gray-300">
                                    Pending
                                </h3>
                                <p className="text-2xl font-bold text-yellow-500 dark:text-yellow-400">
                                    {stats.pending}
                                </p>
                            </div>
                            <div className="p-4 rounded-lg bg-white dark:bg-gray-700 shadow-sm border border-gray-200 dark:border-gray-600">
                                <h3 className="text-sm font-medium text-gray-600 dark:text-gray-300">
                                    Accepted
                                </h3>
                                <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                                    {stats.accepted}
                                </p>
                            </div>
                            <div className="p-4 rounded-lg bg-white dark:bg-gray-700 shadow-sm border border-gray-200 dark:border-gray-600">
                                <h3 className="text-sm font-medium text-gray-600 dark:text-gray-300">
                                    Average Bid
                                </h3>
                                <p className="text-2xl font-bold text-cyan-600 dark:text-cyan-400">
                                    {formatCurrency(stats.average_amount)}
                                </p>
                            </div>
                        </div>

                        <div className="bg-gray-100 p-2 mb-4 rounded">
                            <p>Current status filter: {data.status}</p>
                            <p>Bids returned: {bids.data.length}</p>
                            <p>
                                Statuses:{" "}
                                {JSON.stringify(
                                    bids.data.map((bid) => bid.status)
                                )}
                            </p>
                        </div>

                        <div className="p-6 text-gray-900 dark:text-gray-200">
                            {/* Filters Section */}
                            <div className="mb-6">
                                <div className="flex flex-col md:flex-row gap-4">
                                    <div className="flex-1">
                                        <form
                                            onSubmit={handleSearch}
                                            className="relative"
                                        >
                                            <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 h-5 w-5" />
                                            <input
                                                type="search"
                                                placeholder="Search order number, title, or writer..."
                                                className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-indigo-500 dark:focus:ring-indigo-400 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100 placeholder-gray-500 dark:placeholder-gray-400"
                                                value={data.search}
                                                onChange={(e) =>
                                                    setData(
                                                        "search",
                                                        e.target.value
                                                    )
                                                }
                                            />
                                        </form>
                                    </div>
                                    <div className="flex flex-col sm:flex-row gap-2">
                                        {/* Status Filter */}
                                        <div className="relative">
                                            <button
                                                onClick={() =>
                                                    setStatusDropdownOpen(
                                                        !statusDropdownOpen
                                                    )
                                                }
                                                className="flex items-center justify-between w-full px-4 py-2 text-sm font-medium text-gray-700 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none dark:text-gray-200"
                                            >
                                                <span>
                                                    {getCurrentStatusLabel()}
                                                </span>
                                                <ChevronDown className="w-4 h-4 ml-2" />
                                            </button>
                                            {statusDropdownOpen && (
                                                <div className="absolute right-0 z-10 mt-2 w-full bg-white dark:bg-gray-800 shadow-lg rounded-md border border-gray-200 dark:border-gray-700 py-1">
                                                    {statusOptions.map(
                                                        (option) => (
                                                            <button
                                                                key={
                                                                    option.value
                                                                }
                                                                onClick={() =>
                                                                    handleStatusChange(
                                                                        option.value
                                                                    )
                                                                }
                                                                className={`block px-4 py-2 text-sm w-full text-left ${
                                                                    data.status ===
                                                                    option.value
                                                                        ? "bg-indigo-50 dark:bg-indigo-900/30 text-indigo-600 dark:text-indigo-400"
                                                                        : "text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700"
                                                                }`}
                                                            >
                                                                {option.label}
                                                                {data.status ===
                                                                    option.value && (
                                                                    <Check className="inline-block w-4 h-4 ml-2" />
                                                                )}
                                                            </button>
                                                        )
                                                    )}
                                                </div>
                                            )}
                                        </div>

                                        {/* Sort Dropdown */}
                                        <div className="relative">
                                            <button
                                                onClick={() =>
                                                    setSortDropdownOpen(
                                                        !sortDropdownOpen
                                                    )
                                                }
                                                className="flex items-center justify-between w-full px-4 py-2 text-sm font-medium text-gray-700 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none dark:text-gray-200"
                                            >
                                                <span>
                                                    {getCurrentSortLabel()}
                                                </span>
                                                <ChevronsUpDown className="w-4 h-4 ml-2" />
                                            </button>
                                            {sortDropdownOpen && (
                                                <div className="absolute right-0 z-10 mt-2 w-56 bg-white dark:bg-gray-800 shadow-lg rounded-md border border-gray-200 dark:border-gray-700 py-1">
                                                    {sortOptions.map(
                                                        (option) => (
                                                            <button
                                                                key={`${option.value}-${option.direction}`}
                                                                onClick={() =>
                                                                    setSort(
                                                                        option.value,
                                                                        option.direction
                                                                    )
                                                                }
                                                                className={`block px-4 py-2 text-sm w-full text-left ${
                                                                    data.sort_by ===
                                                                        option.value &&
                                                                    data.sort_direction ===
                                                                        option.direction
                                                                        ? "bg-indigo-50 dark:bg-indigo-900/30 text-indigo-600 dark:text-indigo-400"
                                                                        : "text-gray-700 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-gray-700"
                                                                }`}
                                                            >
                                                                {option.label}
                                                                {data.sort_by ===
                                                                    option.value &&
                                                                    data.sort_direction ===
                                                                        option.direction && (
                                                                        <Check className="inline-block w-4 h-4 ml-2" />
                                                                    )}
                                                            </button>
                                                        )
                                                    )}
                                                </div>
                                            )}
                                        </div>

                                        {/* Date Range Filter */}
                                        <div className="relative">
                                            <button
                                                onClick={() =>
                                                    setDateRangeOpen(
                                                        !dateRangeOpen
                                                    )
                                                }
                                                className="flex items-center justify-between w-full px-4 py-2 text-sm font-medium text-gray-700 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none dark:text-gray-200"
                                            >
                                                <CalendarDays className="w-4 h-4 mr-2" />
                                                <span>
                                                    {data.from_date
                                                        ? "Date Range"
                                                        : "Date Filter"}
                                                </span>
                                                <ChevronDown className="w-4 h-4 ml-2" />
                                            </button>
                                            {dateRangeOpen && (
                                                <div className="absolute right-0 z-10 mt-2 w-64 bg-white dark:bg-gray-800 shadow-lg rounded-md border border-gray-200 dark:border-gray-700 p-4">
                                                    <div className="mb-4">
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                            From Date
                                                        </label>
                                                        <input
                                                            type="date"
                                                            value={
                                                                data.from_date
                                                            }
                                                            onChange={(e) =>
                                                                setData(
                                                                    "from_date",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white text-sm"
                                                        />
                                                    </div>
                                                    <div className="mb-4">
                                                        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                            To Date
                                                        </label>
                                                        <input
                                                            type="date"
                                                            value={data.to_date}
                                                            onChange={(e) =>
                                                                setData(
                                                                    "to_date",
                                                                    e.target
                                                                        .value
                                                                )
                                                            }
                                                            className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white text-sm"
                                                        />
                                                    </div>
                                                    <div className="flex justify-between">
                                                        <button
                                                            type="button"
                                                            onClick={() => {
                                                                setData(
                                                                    "from_date",
                                                                    ""
                                                                );
                                                                setData(
                                                                    "to_date",
                                                                    ""
                                                                );
                                                                setDateRangeOpen(
                                                                    false
                                                                );
                                                                applyFilters();
                                                            }}
                                                            className="inline-flex items-center px-3 py-2 border border-gray-300 dark:border-gray-600 shadow-sm text-sm leading-4 font-medium rounded-md text-gray-700 dark:text-gray-200 bg-white dark:bg-gray-700 hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none"
                                                        >
                                                            Clear
                                                        </button>
                                                        <button
                                                            type="button"
                                                            onClick={
                                                                handleDateRangeChange
                                                            }
                                                            className="inline-flex items-center px-3 py-2 border border-transparent shadow-sm text-sm leading-4 font-medium rounded-md text-white bg-indigo-600 hover:bg-indigo-700 focus:outline-none"
                                                        >
                                                            Apply
                                                        </button>
                                                    </div>
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                </div>

                                {/* Active filters */}
                                {hasActiveFilters && (
                                    <div className="flex flex-wrap items-center gap-2 mt-4">
                                        <span className="text-sm font-medium text-gray-500 dark:text-gray-400">
                                            Active filters:
                                        </span>

                                        {data.status !== "all" && (
                                            <div className="flex items-center gap-1 px-2 py-1 text-xs font-medium text-indigo-800 bg-indigo-100 dark:bg-indigo-900 dark:text-indigo-300 rounded-full">
                                                Status:{" "}
                                                {getCurrentStatusLabel()}
                                                <button
                                                    onClick={() => {
                                                        setData(
                                                            "status",
                                                            "all"
                                                        );
                                                        applyFilters();
                                                    }}
                                                    className="ml-1 text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-200"
                                                >
                                                    <X className="w-3 h-3" />
                                                </button>
                                            </div>
                                        )}

                                        {data.search && (
                                            <div className="flex items-center gap-1 px-2 py-1 text-xs font-medium text-indigo-800 bg-indigo-100 dark:bg-indigo-900 dark:text-indigo-300 rounded-full">
                                                Search: {data.search}
                                                <button
                                                    onClick={() => {
                                                        setData("search", "");
                                                        applyFilters();
                                                    }}
                                                    className="ml-1 text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-200"
                                                >
                                                    <X className="w-3 h-3" />
                                                </button>
                                            </div>
                                        )}

                                        {data.from_date && (
                                            <div className="flex items-center gap-1 px-2 py-1 text-xs font-medium text-indigo-800 bg-indigo-100 dark:bg-indigo-900 dark:text-indigo-300 rounded-full">
                                                From: {data.from_date}
                                                <button
                                                    onClick={() => {
                                                        setData(
                                                            "from_date",
                                                            ""
                                                        );
                                                        applyFilters();
                                                    }}
                                                    className="ml-1 text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-200"
                                                >
                                                    <X className="w-3 h-3" />
                                                </button>
                                            </div>
                                        )}

                                        {data.to_date && (
                                            <div className="flex items-center gap-1 px-2 py-1 text-xs font-medium text-indigo-800 bg-indigo-100 dark:bg-indigo-900 dark:text-indigo-300 rounded-full">
                                                To: {data.to_date}
                                                <button
                                                    onClick={() => {
                                                        setData("to_date", "");
                                                        applyFilters();
                                                    }}
                                                    className="ml-1 text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-200"
                                                >
                                                    <X className="w-3 h-3" />
                                                </button>
                                            </div>
                                        )}

                                        {(data.sort_by !== "created_at" ||
                                            data.sort_direction !== "desc") && (
                                            <div className="flex items-center gap-1 px-2 py-1 text-xs font-medium text-indigo-800 bg-indigo-100 dark:bg-indigo-900 dark:text-indigo-300 rounded-full">
                                                Sort: {getCurrentSortLabel()}
                                                <button
                                                    onClick={() => {
                                                        setData({
                                                            ...data,
                                                            sort_by:
                                                                "created_at",
                                                            sort_direction:
                                                                "desc",
                                                        });
                                                        applyFilters();
                                                    }}
                                                    className="ml-1 text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-200"
                                                >
                                                    <X className="w-3 h-3" />
                                                </button>
                                            </div>
                                        )}

                                        <button
                                            onClick={clearFilters}
                                            className="px-2 py-1 text-xs font-medium text-red-800 bg-red-100 dark:bg-red-900/30 dark:text-red-300 rounded-full hover:bg-red-200 dark:hover:bg-red-800/50"
                                        >
                                            Clear all filters
                                        </button>
                                    </div>
                                )}
                            </div>

                            {/* Bids Grid/Cards */}
                            {bids.data.length > 0 ? (
                                <>
                                    <div className="hidden lg:grid grid-cols-6 gap-4 mb-4 font-medium text-gray-700 dark:text-gray-300 bg-gray-50 dark:bg-gray-700/50 p-4 rounded-lg">
                                        <div>Order</div>
                                        <div>Writer</div>
                                        <div>Amount</div>
                                        <div>Completion Time</div>
                                        <div>Status</div>
                                        <div className="text-center">
                                            Actions
                                        </div>
                                    </div>

                                    <div className="space-y-4 lg:space-y-0 lg:grid lg:grid-cols-1 lg:gap-4">
                                        {bids.data.map((bid) => (
                                            <div
                                                key={bid.id}
                                                className="bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700 overflow-hidden"
                                            >
                                                {/* Desktop view */}
                                                <div className="hidden lg:grid grid-cols-6 items-center p-4">
                                                    <div className="font-medium">
                                                        <Link
                                                            href={`/orders/${bid.order.id}`}
                                                            className="text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-300"
                                                        >
                                                            #
                                                            {
                                                                bid.order
                                                                    .order_number
                                                            }
                                                        </Link>
                                                        <p className="text-xs text-gray-500 dark:text-gray-400 truncate max-w-[200px]">
                                                            {bid.order.title}
                                                        </p>
                                                    </div>

                                                    <div>
                                                        <p className="font-medium text-gray-900 dark:text-gray-100">
                                                            {
                                                                bid.writer_nickname
                                                            }
                                                        </p>
                                                        {bid.writer_experience && (
                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                {
                                                                    bid.writer_experience
                                                                }{" "}
                                                                yrs exp.
                                                            </p>
                                                        )}
                                                    </div>

                                                    <div className="font-medium text-green-600 dark:text-green-400">
                                                        {formatCurrency(
                                                            bid.bid_amount
                                                        )}
                                                    </div>

                                                    <div>
                                                        <p className="text-sm">
                                                            {formatDate(
                                                                bid.completion_time
                                                            )}
                                                        </p>
                                                        <p className="text-xs text-gray-500 dark:text-gray-400">
                                                            {timeUntilCompletion(
                                                                bid.completion_time
                                                            )}
                                                        </p>
                                                    </div>

                                                    <div>
                                                        <span
                                                            className={`px-3 py-1 rounded-full text-xs font-medium ${getStatusBadgeClass(
                                                                bid.status
                                                            )}`}
                                                        >
                                                            {bid.status
                                                                .charAt(0)
                                                                .toUpperCase() +
                                                                bid.status.slice(
                                                                    1
                                                                )}
                                                        </span>
                                                    </div>

                                                    <div className="flex justify-center space-x-2">
                                                        <Link
                                                            href={`/orders/${bid.order.id}`}
                                                            className="px-3 py-2 bg-indigo-600 hover:bg-indigo-700 text-white rounded-md text-sm font-medium transition-colors duration-200"
                                                        >
                                                            View Order
                                                        </Link>
                                                    </div>
                                                </div>

                                                {/* Mobile view */}
                                                <div className="lg:hidden p-4 space-y-4">
                                                    <div className="flex justify-between items-center">
                                                        <div>
                                                            <Link
                                                                href={`/orders/${bid.order.id}`}
                                                                className="text-indigo-600 dark:text-indigo-400 text-lg font-medium hover:text-indigo-800 dark:hover:text-indigo-300"
                                                            >
                                                                Order #
                                                                {
                                                                    bid.order
                                                                        .order_number
                                                                }
                                                            </Link>
                                                            <p className="text-sm text-gray-500 dark:text-gray-400 truncate max-w-[220px]">
                                                                {
                                                                    bid.order
                                                                        .title
                                                                }
                                                            </p>
                                                        </div>
                                                        <span
                                                            className={`px-3 py-1 rounded-full text-xs font-medium ${getStatusBadgeClass(
                                                                bid.status
                                                            )}`}
                                                        >
                                                            {bid.status
                                                                .charAt(0)
                                                                .toUpperCase() +
                                                                bid.status.slice(
                                                                    1
                                                                )}
                                                        </span>
                                                    </div>
                                                    <div className="grid grid-cols-2 gap-4 pt-2">
                                                        <div>
                                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                Writer
                                                            </p>
                                                            <p className="font-medium text-gray-900 dark:text-gray-100">
                                                                {
                                                                    bid.writer_nickname
                                                                }
                                                            </p>
                                                            {bid.writer_experience && (
                                                                <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                    {
                                                                        bid.writer_experience
                                                                    }{" "}
                                                                    yrs
                                                                    experience
                                                                </p>
                                                            )}
                                                        </div>

                                                        <div>
                                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                Bid Amount
                                                            </p>
                                                            <p className="font-medium text-green-600 dark:text-green-400">
                                                                {formatCurrency(
                                                                    bid.bid_amount
                                                                )}
                                                            </p>
                                                        </div>

                                                        <div>
                                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                Completion Time
                                                            </p>
                                                            <p className="font-medium text-gray-900 dark:text-gray-100">
                                                                {formatDate(
                                                                    bid.completion_time
                                                                )}
                                                            </p>
                                                            <p className="text-xs text-gray-500 dark:text-gray-400">
                                                                {timeUntilCompletion(
                                                                    bid.completion_time
                                                                )}
                                                            </p>
                                                        </div>

                                                        <div>
                                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                Date Submitted
                                                            </p>
                                                            <p className="font-medium text-gray-900 dark:text-gray-100">
                                                                {formatDate(
                                                                    bid.created_at
                                                                )}
                                                            </p>
                                                        </div>
                                                    </div>

                                                    {bid.comments && (
                                                        <div className="mt-2 pt-2 border-t border-gray-100 dark:border-gray-700">
                                                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                                                Comments
                                                            </p>
                                                            <p className="text-sm text-gray-700 dark:text-gray-300">
                                                                {bid.comments}
                                                            </p>
                                                        </div>
                                                    )}

                                                    <div className="mt-4 pt-4 border-t border-gray-100 dark:border-gray-700">
                                                        <Link
                                                            href={`/orders/${bid.order.id}`}
                                                            className="w-full inline-flex justify-center items-center px-4 py-2 bg-indigo-600 hover:bg-indigo-700 text-white rounded-md text-sm font-medium transition-colors duration-200"
                                                        >
                                                            View Order Details
                                                        </Link>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </>
                            ) : (
                                <div className="text-center py-12 bg-white dark:bg-gray-800 rounded-lg shadow-sm border border-gray-200 dark:border-gray-700">
                                    <div className="inline-flex items-center justify-center w-16 h-16 rounded-full bg-indigo-100 text-indigo-600 dark:bg-indigo-900/30 dark:text-indigo-400 mb-4">
                                        <Gavel className="w-8 h-8" />
                                    </div>
                                    <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-2">
                                        No bids found
                                    </h3>
                                    <p className="text-gray-500 dark:text-gray-400 mb-6 max-w-md mx-auto">
                                        {data.search ||
                                        data.status !== "all" ||
                                        data.from_date ||
                                        data.to_date
                                            ? "No bids matching your filters were found. Try adjusting your search criteria."
                                            : "There are currently no bids on your orders. When writers bid on your orders, they'll appear here."}
                                    </p>
                                    {(data.search ||
                                        data.status !== "all" ||
                                        data.from_date ||
                                        data.to_date) && (
                                        <button
                                            onClick={clearFilters}
                                            className="inline-flex items-center px-4 py-2 bg-indigo-600 hover:bg-indigo-700 border border-transparent rounded-md font-medium text-sm text-white"
                                        >
                                            <Filter className="w-4 h-4 mr-2" />
                                            Clear Filters
                                        </button>
                                    )}
                                </div>
                            )}

                            {/* Pagination */}
                            {bids.data.length > 0 && (
                                <div className="mt-6">
                                    <CleanPagination
                                        links={bids.links}
                                        total={bids.total}
                                        perPage={bids.per_page}
                                        currentPage={bids.current_page}
                                    />
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
}
