import { useState } from "react";
import { Head, Link, router } from "@inertiajs/react";
import { Search, FileText, Clock, User, Calendar, AlertTriangle, Eye, MessageSquare, Gavel } from "lucide-react";
import ClientLayout from "@/Layouts/ClientLayout";
import { Button } from "@/Components/ui/Button";
import { Badge } from "@/Components/ui/Badge";
import Pagination from "@/Components/Pagination";

export default function ActiveProjects({ auth, activeProjects, filters }) {
    const [searchQuery, setSearchQuery] = useState(filters.search || "");

    const handleSearch = (e) => {
        e.preventDefault();
        router.get(route('projects.active'), { search: searchQuery }, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const clearSearch = () => {
        setSearchQuery("");
        router.get(route('projects.active'), {}, {
            preserveState: true,
            preserveScroll: true,
        });
    };

    const getStatusBadge = (status) => {
        const statusConfig = {
            'pending': { variant: 'secondary', text: 'Pending' },
            'pending_payment': { variant: 'warning', text: 'Payment Pending' },
            'bidding': { variant: 'info', text: 'Bidding Open' },
            'assigned': { variant: 'default', text: 'Assigned' },
            'in_progress': { variant: 'default', text: 'In Progress' },
            'submitted': { variant: 'success', text: 'Submitted' },
            'under_review': { variant: 'warning', text: 'Under Review' },
            'revision_requested': { variant: 'destructive', text: 'Revision Needed' }
        };

        const config = statusConfig[status] || { variant: 'default', text: status };
        return <Badge variant={config.variant}>{config.text}</Badge>;
    };

    const formatDate = (dateString) => {
        if (!dateString) return 'N/A';
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    };

    const formatDeadline = (deadline) => {
        if (!deadline) return 'N/A';
        const deadlineDate = new Date(deadline);
        const now = new Date();
        const diffTime = deadlineDate - now;
        const diffDays = Math.ceil(diffTime / (1000 * 60 * 60 * 24));
        
        if (diffDays < 0) {
            return <span className="text-red-600 font-medium">Overdue ({Math.abs(diffDays)} days)</span>;
        } else if (diffDays === 0) {
            return <span className="text-orange-600 font-medium">Due Today</span>;
        } else if (diffDays <= 3) {
            return <span className="text-orange-600 font-medium">Due in {diffDays} days</span>;
        } else {
            return <span className="text-gray-600">{formatDate(deadline)}</span>;
        }
    };

    const getProgressPercentage = (status) => {
        const progressMap = {
            'pending': 10,
            'pending_payment': 20,
            'bidding': 30,
            'assigned': 40,
            'in_progress': 60,
            'submitted': 80,
            'under_review': 90,
            'revision_requested': 70
        };
        return progressMap[status] || 0;
    };

    return (
        <ClientLayout>
            <Head title="Active Projects" />

            <div className="py-12">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header Section */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-6">
                            <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                                <div>
                                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                                        Active Projects
                                    </h1>
                                    <p className="text-gray-600 dark:text-gray-400 mt-1">
                                        Monitor the progress of your ongoing academic work
                                    </p>
                                </div>
                                <div className="flex items-center gap-3">
                                    <Link href="/orders">
                                        <Button variant="outline">
                                            <FileText className="w-4 h-4 mr-2" />
                                            All Orders
                                        </Button>
                                    </Link>
                                    <Link href="/projects/completed">
                                        <Button variant="outline">
                                            <FileText className="w-4 h-4 mr-2" />
                                            Completed
                                        </Button>
                                    </Link>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Search and Filters */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg mb-6">
                        <div className="p-6">
                            <form onSubmit={handleSearch} className="flex flex-col sm:flex-row gap-4">
                                <div className="flex-1 relative">
                                    <Search className="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400 w-4 h-4" />
                                    <input
                                        type="text"
                                        placeholder="Search by order number, title, subject, or paper type..."
                                        className="w-full pl-10 pr-4 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white"
                                        value={searchQuery}
                                        onChange={(e) => setSearchQuery(e.target.value)}
                                    />
                                </div>
                                <div className="flex gap-2">
                                    <Button type="submit" variant="default">
                                        <Search className="w-4 h-4 mr-2" />
                                        Search
                                    </Button>
                                    {filters.search && (
                                        <Button type="button" variant="outline" onClick={clearSearch}>
                                            Clear
                                        </Button>
                                    )}
                                </div>
                            </form>
                        </div>
                    </div>

                    {/* Projects List */}
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6">
                            {activeProjects.data.length === 0 ? (
                                <div className="text-center py-12">
                                    <Clock className="mx-auto h-12 w-12 text-gray-400" />
                                    <h3 className="mt-2 text-sm font-medium text-gray-900 dark:text-white">
                                        No active projects found
                                    </h3>
                                    <p className="mt-1 text-sm text-gray-500 dark:text-gray-400">
                                        {filters.search 
                                            ? "Try adjusting your search terms or clear the search to see all projects."
                                            : "You don't have any active projects at the moment. Your ongoing work will appear here."
                                        }
                                    </p>
                                    {filters.search && (
                                        <div className="mt-6">
                                            <Button variant="outline" onClick={clearSearch}>
                                                Clear Search
                                            </Button>
                                        </div>
                                    )}
                                </div>
                            ) : (
                                <div className="space-y-6">
                                    {activeProjects.data.map((project) => (
                                        <div
                                            key={project.id}
                                            className="border border-gray-200 dark:border-gray-700 rounded-lg p-6 hover:shadow-md transition-shadow duration-200"
                                        >
                                            <div className="flex flex-col lg:flex-row lg:items-start lg:justify-between gap-4">
                                                <div className="flex-1">
                                                    <div className="flex items-start justify-between mb-4">
                                                        <div className="flex-1">
                                                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-2">
                                                                {project.title}
                                                            </h3>
                                                            <div className="flex flex-wrap items-center gap-2 text-sm text-gray-600 dark:text-gray-400 mb-3">
                                                                <span className="flex items-center gap-1">
                                                                    <FileText className="w-4 h-4" />
                                                                    {project.type_of_paper}
                                                                </span>
                                                                <span>•</span>
                                                                <span>{project.subject}</span>
                                                                <span>•</span>
                                                                <span>{project.academic_level}</span>
                                                                <span>•</span>
                                                                <span>{project.pages} pages</span>
                                                            </div>
                                                        </div>
                                                        <div className="flex items-center gap-2">
                                                            {getStatusBadge(project.order_status)}
                                                            {project.is_overdue && (
                                                                <Badge variant="destructive">
                                                                    <AlertTriangle className="w-3 h-3 mr-1" />
                                                                    Overdue
                                                                </Badge>
                                                            )}
                                                        </div>
                                                    </div>

                                                    {/* Progress Bar */}
                                                    <div className="mb-4">
                                                        <div className="flex items-center justify-between text-sm text-gray-600 dark:text-gray-400 mb-2">
                                                            <span>Progress</span>
                                                            <span>{getProgressPercentage(project.order_status)}%</span>
                                                        </div>
                                                        <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2">
                                                            <div 
                                                                className="bg-blue-600 h-2 rounded-full transition-all duration-300"
                                                                style={{ width: `${getProgressPercentage(project.order_status)}%` }}
                                                            ></div>
                                                        </div>
                                                    </div>

                                                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 text-sm">
                                                        <div className="flex items-center gap-2">
                                                            <Calendar className="w-4 h-4 text-gray-400" />
                                                            <span className="text-gray-600 dark:text-gray-400">Deadline:</span>
                                                            <span className="font-medium">
                                                                {formatDeadline(project.deadline)}
                                                            </span>
                                                        </div>
                                                        
                                                        <div className="flex items-center gap-2">
                                                            <User className="w-4 h-4 text-gray-400" />
                                                            <span className="text-gray-600 dark:text-gray-400">Writer:</span>
                                                            <span className="font-medium text-gray-900 dark:text-white">
                                                                {project.writer_name || 'Unassigned'}
                                                            </span>
                                                        </div>
                                                        
                                                        <div className="flex items-center gap-2">
                                                            <Gavel className="w-4 h-4 text-gray-400" />
                                                            <span className="text-gray-600 dark:text-gray-400">Bids:</span>
                                                            <span className="font-medium text-gray-900 dark:text-white">
                                                                {project.bids_count || 0}
                                                            </span>
                                                        </div>
                                                        
                                                        <div className="flex items-center gap-2">
                                                            <Clock className="w-4 h-4 text-gray-400" />
                                                            <span className="text-gray-600 dark:text-gray-400">Posted:</span>
                                                            <span className="font-medium text-gray-900 dark:text-white">
                                                                {formatDate(project.dateposted)}
                                                            </span>
                                                        </div>
                                                    </div>

                                                    {project.instructions && (
                                                        <div className="mt-4">
                                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                                <span className="font-medium">Instructions:</span> {project.instructions}
                                                            </p>
                                                        </div>
                                                    )}
                                                </div>

                                                <div className="flex flex-col sm:flex-row gap-2">
                                                    <Link href={`/orders/${project.id}`}>
                                                        <Button variant="outline" size="sm">
                                                            <Eye className="w-4 h-4 mr-2" />
                                                            View Details
                                                        </Button>
                                                    </Link>
                                                    
                                                    {project.order_status === 'bidding' && (
                                                        <Link href="/client/bids">
                                                            <Button variant="default" size="sm">
                                                                <Gavel className="w-4 h-4 mr-2" />
                                                                View Bids
                                                            </Button>
                                                        </Link>
                                                    )}
                                                    
                                                    {project.writer_id && (
                                                        <Button variant="outline" size="sm">
                                                            <MessageSquare className="w-4 h-4 mr-2" />
                                                            Message Writer
                                                        </Button>
                                                    )}
                                                </div>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            )}

                            {/* Pagination */}
                            {activeProjects.data.length > 0 && (
                                <div className="mt-6">
                                    <Pagination links={activeProjects.links} />
                                </div>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
} 