import { Head, Link, useForm } from "@inertiajs/react";
import React, { useState } from "react";
import { FileText, ArrowLeft, AlertCircle, CheckCircle } from "lucide-react";

import ClientLayout from "@/Layouts/ClientLayout";
import { Button } from "@/Components/ui/Button";
import { Card, CardContent, CardHeader, CardTitle } from "@/Components/ui/Card";
import { Input } from "@/Components/ui/Input";
import { Label } from "@/Components/ui/Label";
import { Textarea } from "@/Components/ui/Textarea";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/Components/ui/Select";
import Alert from "@/Components/ui/Alert";

export default function ClientRevisionCreate({ order, revisionTypes, existingRevisions, maxRevisions }) {
    const { data, setData, post, processing, errors, reset } = useForm({
        revision_type: '',
        revision_reason: '',
        specific_instructions: '',
    });

    const [showSuccess, setShowSuccess] = useState(false);

    const handleSubmit = (e) => {
        e.preventDefault();
        
        post(route('client.revisions.store', order.id), {
            onSuccess: () => {
                setShowSuccess(true);
                reset();
            },
        });
    };

    const getRevisionTypeLabel = (type) => {
        return revisionTypes[type] || type;
    };

    const remainingRevisions = maxRevisions - existingRevisions;

    if (showSuccess) {
        return (
            <ClientLayout>
                <Head title="Revision Requested" />
                <div className="max-w-2xl mx-auto py-12 px-4 sm:px-6 lg:px-8">
                    <Card>
                        <CardContent className="text-center py-12">
                            <CheckCircle className="h-16 w-16 text-green-500 mx-auto mb-4" />
                            <h2 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
                                Revision Requested Successfully!
                            </h2>
                            <p className="text-gray-600 dark:text-gray-400 mb-6">
                                Your revision request has been submitted and our team has been notified. 
                                We'll assign a writer to work on your revision as soon as possible.
                            </p>
                            <div className="flex space-x-3 justify-center">
                                <Link href={route('client.revisions.index', order.id)}>
                                    <Button variant="outline">
                                        <FileText size={16} className="mr-2" />
                                        View All Revisions
                                    </Button>
                                </Link>
                                <Link href={route('orders.show', order.id)}>
                                    <Button>
                                        Back to Order
                                    </Button>
                                </Link>
                            </div>
                        </CardContent>
                    </Card>
                </div>
            </ClientLayout>
        );
    }

    return (
        <ClientLayout>
            <Head title="Request Revision" />

            <div className="max-w-4xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
                {/* Header */}
                <div className="mb-8">
                    <div className="flex items-center space-x-4 mb-4">
                        <Link href={route('client.revisions.index', order.id)}>
                            <Button variant="ghost" size="sm">
                                <ArrowLeft size={16} className="mr-2" />
                                Back to Revisions
                            </Button>
                        </Link>
                    </div>
                    <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                        Request Revision
                    </h1>
                    <p className="mt-2 text-gray-600 dark:text-gray-400">
                        Order #{order.order_number} - {order.title || 'Untitled Order'}
                    </p>
                </div>

                {/* Revision Limit Warning */}
                {remainingRevisions <= 2 && (
                    <Alert className="mb-6" message={
                        <><strong>Attention:</strong> This is revision #{existingRevisions + 1} of {maxRevisions} allowed revisions. 
                        {remainingRevisions === 1 && ' This will be your final revision request.'}
                        {remainingRevisions === 0 && ' You have reached the maximum number of revisions for this order.'}</>
                    } />
                )}

                {/* Order Summary */}
                <Card className="mb-6">
                    <CardHeader>
                        <CardTitle>Order Summary</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                            <div>
                                <span className="text-sm font-medium text-gray-500">Status:</span>
                                <p className="text-sm text-gray-900 dark:text-white">{order.order_status.replace('_', ' ').toUpperCase()}</p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">Writer:</span>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {order.writer ? order.writer.name : 'Unassigned'}
                                </p>
                            </div>
                            <div>
                                <span className="text-sm font-medium text-gray-500">Revisions Used:</span>
                                <p className="text-sm text-gray-900 dark:text-white">
                                    {existingRevisions} of {maxRevisions}
                                </p>
                            </div>
                        </div>
                    </CardContent>
                </Card>

                {/* Revision Form */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center">
                            <FileText className="h-5 w-5 mr-2" />
                            Revision Request Form
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        <form onSubmit={handleSubmit} className="space-y-6">
                            {/* Revision Type */}
                            <div>
                                <Label htmlFor="revision_type">Revision Type *</Label>
                                <Select
                                    value={data.revision_type}
                                    onValueChange={(value) => setData('revision_type', value)}
                                >
                                    <SelectTrigger>
                                        <SelectValue placeholder="Select revision type" />
                                    </SelectTrigger>
                                    <SelectContent>
                                        {Object.entries(revisionTypes).map(([key, description]) => (
                                            <SelectItem key={key} value={key}>
                                                <div>
                                                    <div className="font-medium">{description}</div>
                                                </div>
                                            </SelectItem>
                                        ))}
                                    </SelectContent>
                                </Select>
                                {errors.revision_type && (
                                    <p className="text-sm text-red-600 mt-1">{errors.revision_type}</p>
                                )}
                            </div>

                            {/* Revision Reason */}
                            <div>
                                <Label htmlFor="revision_reason">
                                    Revision Reason * 
                                    <span className="text-sm text-gray-500 ml-2">
                                        (Max 1000 characters)
                                    </span>
                                </Label>
                                <Textarea
                                    id="revision_reason"
                                    value={data.revision_reason}
                                    onChange={(e) => setData('revision_reason', e.target.value)}
                                    placeholder="Please describe the issues you found and what needs to be revised..."
                                    rows={4}
                                    maxLength={1000}
                                    className="mt-1"
                                />
                                <div className="flex justify-between items-center mt-1">
                                    {errors.revision_reason && (
                                        <p className="text-sm text-red-600">{errors.revision_reason}</p>
                                    )}
                                    <span className="text-sm text-gray-500 ml-auto">
                                        {data.revision_reason.length}/1000
                                    </span>
                                </div>
                            </div>

                            {/* Specific Instructions */}
                            <div>
                                <Label htmlFor="specific_instructions">
                                    Specific Instructions
                                    <span className="text-sm text-gray-500 ml-2">
                                        (Optional, Max 2000 characters)
                                    </span>
                                </Label>
                                <Textarea
                                    id="specific_instructions"
                                    value={data.specific_instructions}
                                    onChange={(e) => setData('specific_instructions', e.target.value)}
                                    placeholder="Provide any specific instructions or examples for the writer..."
                                    rows={4}
                                    maxLength={2000}
                                    className="mt-1"
                                />
                                <div className="flex justify-between items-center mt-1">
                                    {errors.specific_instructions && (
                                        <p className="text-sm text-red-600">{errors.specific_instructions}</p>
                                    )}
                                    <span className="text-sm text-gray-500 ml-auto">
                                        {data.specific_instructions.length}/2000
                                    </span>
                                </div>
                            </div>

                            {/* Form Actions */}
                            <div className="flex justify-end space-x-3 pt-4 border-t border-gray-200 dark:border-gray-700">
                                <Link href={route('client.revisions.index', order.id)}>
                                    <Button type="button" variant="outline">
                                        Cancel
                                    </Button>
                                </Link>
                                <Button 
                                    type="submit" 
                                    disabled={processing || !data.revision_type || !data.revision_reason}
                                >
                                    {processing ? 'Submitting...' : 'Submit Revision Request'}
                                </Button>
                            </div>
                        </form>
                    </CardContent>
                </Card>

                {/* Help Information */}
                <Card className="mt-6">
                    <CardHeader>
                        <CardTitle>Tips for Effective Revision Requests</CardTitle>
                    </CardHeader>
                    <CardContent>
                        <div className="space-y-3 text-sm text-gray-600 dark:text-gray-400">
                            <div className="flex items-start space-x-2">
                                <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
                                <span>Be specific about what needs to be changed or improved</span>
                            </div>
                            <div className="flex items-start space-x-2">
                                <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
                                <span>Provide examples or references when possible</span>
                            </div>
                            <div className="flex items-start space-x-2">
                                <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
                                <span>Explain why the changes are important for your needs</span>
                            </div>
                            <div className="flex items-start space-x-2">
                                <CheckCircle className="h-4 w-4 text-green-500 mt-0.5 flex-shrink-0" />
                                <span>Use clear, constructive language to help the writer understand</span>
                            </div>
                        </div>
                    </CardContent>
                </Card>
            </div>
        </ClientLayout>
    );
} 