import { Head, Link } from "@inertiajs/react";
import React from "react";
import { FileText, Clock, CheckCircle, AlertCircle, ArrowRight } from "lucide-react";

import ClientLayout from "@/Layouts/ClientLayout";
import { Badge } from "@/Components/ui/Badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/Components/ui/Card";

export default function ClientRevisionDashboard({ stats, recentRevisions }) {
    const getStatusBadge = (status) => {
        const statusConfig = {
            requested: { color: "bg-blue-100 text-blue-800", icon: Clock, label: "Requested" },
            in_progress: { color: "bg-yellow-100 text-yellow-800", icon: Clock, label: "In Progress" },
            completed: { color: "bg-green-100 text-green-800", icon: CheckCircle, label: "Completed" },
            rejected: { color: "bg-red-100 text-red-800", icon: AlertCircle, label: "Rejected" },
            escalated: { color: "bg-orange-100 text-orange-800", icon: AlertCircle, label: "Escalated" },
        };

        const config = statusConfig[status] || statusConfig.requested;
        const Icon = config.icon;

        return (
            <Badge className={`${config.color} dark:${config.color.replace('100', '900')} dark:${config.color.replace('800', '200')}`}>
                <Icon size={12} className="mr-1" />
                {config.label}
            </Badge>
        );
    };

    const getRevisionTypeLabel = (type) => {
        const typeLabels = {
            quality: "Quality Issues",
            requirements: "Requirements Not Met",
            formatting: "Formatting Issues",
            plagiarism: "Plagiarism Concerns",
            deadline: "Deadline Issues",
            communication: "Communication Issues",
            other: "Other Issues",
        };

        return typeLabels[type] || type;
    };

    return (
        <ClientLayout>
            <Head title="Revision Dashboard" />

            <div className="max-w-7xl mx-auto py-6 px-4 sm:px-6 lg:px-8">
                <div className="mb-8">
                    <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                        Revision Dashboard
                    </h1>
                    <p className="mt-2 text-gray-600 dark:text-gray-400">
                        Track and manage your revision requests
                    </p>
                </div>

                {/* Statistics Cards */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-6 mb-8">
                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Total Revisions</CardTitle>
                            <FileText className="h-4 w-4 text-gray-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold">{stats.total_revisions}</div>
                            <p className="text-xs text-gray-600 dark:text-gray-400">
                                All time revisions
                            </p>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Pending Revisions</CardTitle>
                            <Clock className="h-4 w-4 text-yellow-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-yellow-600">{stats.pending_revisions}</div>
                            <p className="text-xs text-gray-600 dark:text-gray-400">
                                Awaiting completion
                            </p>
                        </CardContent>
                    </Card>

                    <Card>
                        <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
                            <CardTitle className="text-sm font-medium">Completed Revisions</CardTitle>
                            <CheckCircle className="h-4 w-4 text-green-500" />
                        </CardHeader>
                        <CardContent>
                            <div className="text-2xl font-bold text-green-600">{stats.completed_revisions}</div>
                            <p className="text-xs text-gray-600 dark:text-gray-400">
                                Successfully completed
                            </p>
                        </CardContent>
                    </Card>
                </div>

                {/* Recent Revisions */}
                <Card>
                    <CardHeader>
                        <CardTitle className="flex items-center">
                            <FileText className="h-5 w-5 mr-2" />
                            Recent Revisions
                        </CardTitle>
                    </CardHeader>
                    <CardContent>
                        {recentRevisions.length === 0 ? (
                            <div className="text-center py-8">
                                <FileText className="h-12 w-12 text-gray-400 mx-auto mb-4" />
                                <p className="text-gray-500 dark:text-gray-400">
                                    No revisions yet. Your revision history will appear here.
                                </p>
                            </div>
                        ) : (
                            <div className="space-y-4">
                                {recentRevisions.map((revision) => (
                                    <div
                                        key={revision.id}
                                        className="flex items-center justify-between p-4 border border-gray-200 dark:border-gray-700 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors"
                                    >
                                        <div className="flex-1">
                                            <div className="flex items-center space-x-3 mb-2">
                                                <h3 className="font-medium text-gray-900 dark:text-white">
                                                    Order #{revision.order.order_number}
                                                </h3>
                                                {getStatusBadge(revision.status)}
                                            </div>
                                            <div className="flex items-center space-x-4 text-sm text-gray-600 dark:text-gray-400">
                                                <span>
                                                    <strong>Type:</strong> {getRevisionTypeLabel(revision.revision_type)}
                                                </span>
                                                <span>
                                                    <strong>Revision #:</strong> {revision.revision_number}
                                                </span>
                                                <span>
                                                    <strong>Created:</strong> {new Date(revision.created_at).toLocaleDateString()}
                                                </span>
                                            </div>
                                            {revision.assigned_to && (
                                                <div className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                                                    <strong>Assigned to:</strong> {revision.assigned_to.name}
                                                </div>
                                            )}
                                        </div>
                                        <Link
                                            href={route('client.revisions.show', [revision.order_id, revision.id])}
                                            className="flex items-center text-blue-600 hover:text-blue-800 dark:text-blue-400 dark:hover:text-blue-300 transition-colors"
                                        >
                                            View Details
                                            <ArrowRight size={16} className="ml-1" />
                                        </Link>
                                    </div>
                                ))}
                            </div>
                        )}
                    </CardContent>
                </Card>
            </div>
        </ClientLayout>
    );
} 