import React, { useState } from "react";
import { Head, useForm } from "@inertiajs/react";
import MainLayout from "@/Layouts/MainLayout";
import { toast } from "react-hot-toast";

export default function Contact(props) {
    const { seoData, flash, contentBlocks, structuredData, isCMS } = props;
    
    const { data, setData, post, processing, errors, reset } = useForm({
        name: "",
        email: "",
        subject: "",
        message: "",
        phone: "",
    });

    const [isSubmitting, setIsSubmitting] = useState(false);

    // Extract content from CMS blocks (same pattern as Essays.jsx)
    const getContentBlock = (blockKey) => {
        if (!isCMS || !contentBlocks) return null;
        const block = contentBlocks.find((b) => b.block_key === blockKey);
        return block ? block.content : null;
    };

    // CMS data with fallbacks
    const hero = getContentBlock('contact_hero') || {
        title: 'Contact Us',
        description: 'Have a question or need assistance? We\'re here to help. Get in touch with our team and we\'ll respond as soon as possible.'
    };

    const contactInfo = getContentBlock('contact_information') || {
        title: 'Get in Touch',
        contact_methods: [
            { type: 'email', icon: 'mail', label: 'Email', value: 'support@academicscribe.com', color: 'blue' },
            { type: 'time', icon: 'clock', label: 'Response Time', value: 'Within 24 hours', color: 'green' },
            { type: 'support', icon: 'chat', label: 'Support Hours', value: '24/7 Customer Support', color: 'purple' }
        ]
    };

    const whyChoose = getContentBlock('contact_why_choose') || {
        title: 'Why Choose Academic Scribe?',
        features: [
            { icon: 'check-circle', title: 'Quality Assurance', description: 'Professional writers with proven expertise', color: 'blue' },
            { icon: 'clock', title: 'Timely Delivery', description: 'Always on time, every time', color: 'green' },
            { icon: 'chat', title: '24/7 Support', description: 'Round-the-clock customer assistance', color: 'purple' }
        ]
    };

    const contactFaqs = getContentBlock('contact_faqs') || {
        title: 'Frequently Asked Questions',
        faqs: [
            { question: 'How quickly will I receive a response?', answer: 'We typically respond to all inquiries within 24 hours during business days.' },
            { question: 'What information should I include?', answer: 'Please provide as much detail as possible about your inquiry to help us assist you better.' },
            { question: 'Can I track my inquiry?', answer: 'Yes, you\'ll receive a confirmation email with a reference number for tracking.' },
            { question: 'Is my information secure?', answer: 'Absolutely. We use industry-standard security measures to protect your data.' }
        ]
    };

    // Handle flash messages
    React.useEffect(() => {
        if (flash?.success) {
            toast.success(flash.success, { duration: 5000 });
        }
        if (flash?.error) {
            toast.error(flash.error, { duration: 4000 });
        }
    }, [flash]);

    const handleSubmit = async (e) => {
        e.preventDefault();
        setIsSubmitting(true);

        try {
            const response = await fetch(route("contact.store"), {
                method: "POST",
                headers: {
                    "Content-Type": "application/json",
                    "X-CSRF-TOKEN": document
                        .querySelector('meta[name="csrf-token"]')
                        .getAttribute("content"),
                    Accept: "application/json",
                },
                body: JSON.stringify(data),
            });

            const result = await response.json();

            if (response.ok && result.success) {
                toast.success(
                    result.message || "Thank you for contacting us! We will get back to you soon.",
                    { duration: 5000, icon: "✅" }
                );
                reset();
            } else {
                if (result.errors) {
                    Object.entries(result.errors).forEach(([field, messages]) => {
                        messages.forEach((message) => {
                            toast.error(message, { duration: 4000, icon: "❌" });
                        });
                    });
                } else {
                    toast.error(
                        result.message || "Something went wrong. Please try again.",
                        { duration: 4000, icon: "❌" }
                    );
                }
            }
        } catch (error) {
            console.error("Contact form error:", error);
            toast.error(
                "Network error. Please check your connection and try again.",
                { duration: 4000, icon: "🔌" }
            );
        } finally {
            setIsSubmitting(false);
        }
    };

    // Icon mapping for contact methods
    const getIconSVG = (iconType) => {
        const icons = {
            mail: (
                <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M3 8l7.89 4.26a2 2 0 002.22 0L21 8M5 19h14a2 2 0 002-2V7a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z"
                />
            ),
            clock: (
                <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z"
                />
            ),
            chat: (
                <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M17 8h2a2 2 0 012 2v6a2 2 0 01-2 2h-2v4l-4-4H9a1.994 1.994 0 01-1.414-.586m0 0L11 14h4a2 2 0 002-2V6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2v4l.586-.586z"
                />
            ),
            'check-circle': (
                <path
                    strokeLinecap="round"
                    strokeLinejoin="round"
                    strokeWidth={2}
                    d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"
                />
            )
        };
        return icons[iconType] || icons['check-circle'];
    };

    return (
        <MainLayout pageData={{ ...seoData, structuredData }}>

            <div className="min-h-screen bg-gray-100 dark:bg-gray-900 py-6 sm:py-8 md:py-12 lg:py-16">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Hero Section - CMS Driven */}
                    <div className="text-center mb-8 sm:mb-10 md:mb-12 lg:mb-16">
                        <h1 className="text-2xl sm:text-3xl md:text-4xl lg:text-5xl xl:text-6xl font-bold text-gray-900 dark:text-white mb-3 sm:mb-4 md:mb-6">
                            {hero.title}
                        </h1>
                        <div 
                            className="text-sm sm:text-base md:text-lg lg:text-xl text-gray-600 dark:text-gray-300 max-w-2xl md:max-w-3xl lg:max-w-4xl mx-auto px-2 sm:px-4 content-area"
                            dangerouslySetInnerHTML={{ __html: hero.description }}
                        />
                    </div>

                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 sm:gap-8 lg:gap-10">
                        {/* Contact Information - CMS Driven */}
                        <div className="lg:col-span-1 order-2 lg:order-1">
                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:from-gray-700 dark:to-gray-700 rounded-lg shadow-lg border border-gray-200 dark:border-gray-600 p-4 sm:p-6 lg:p-8 h-full">
                                <h2 className="text-lg sm:text-xl md:text-2xl font-semibold text-gray-900 dark:text-gray-100 mb-4 sm:mb-6">
                                    {contactInfo.title}
                                </h2>
                                <div className="space-y-4 sm:space-y-6">
                                    {contactInfo.contact_methods.map((method, index) => {
                                        const colorClasses = {
                                            blue: 'bg-blue-100 dark:bg-blue-500/20 text-blue-600 dark:text-blue-300',
                                            green: 'bg-green-100 dark:bg-green-500/20 text-green-600 dark:text-green-300',
                                            purple: 'bg-purple-100 dark:bg-purple-500/20 text-purple-600 dark:text-purple-300'
                                        };
                                        return (
                                            <div key={index} className="flex items-start">
                                                <div className="flex-shrink-0">
                                                    <div className={`w-8 h-8 sm:w-10 sm:h-10 ${colorClasses[method.color]} rounded-full flex items-center justify-center`}>
                                                        <svg
                                                            className="w-4 h-4 sm:w-5 sm:h-5"
                                                            fill="none"
                                                            stroke="currentColor"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            {getIconSVG(method.icon)}
                                                        </svg>
                                                    </div>
                                                </div>
                                                <div className="ml-3 sm:ml-4">
                                                    <h3 className="text-sm sm:text-base md:text-lg font-medium text-gray-900 dark:text-gray-100">
                                                        {method.label}
                                                    </h3>
                                                    <p className="text-xs sm:text-sm md:text-base text-gray-600 dark:text-gray-200 break-all">
                                                        {method.value}
                                                    </p>
                                                </div>
                                            </div>
                                        );
                                    })}
                                </div>
                            </div>
                        </div>

                        {/* Contact Form - Functional (React-based) */}
                        <div className="lg:col-span-2 order-1 lg:order-2">
                            <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:from-gray-700 dark:to-gray-700 rounded-lg shadow-lg border border-gray-200 dark:border-gray-600 p-4 sm:p-6 lg:p-8">
                                <h2 className="text-lg sm:text-xl md:text-2xl font-semibold text-gray-900 dark:text-gray-100 mb-4 sm:mb-6">
                                    Send us a Message
                                </h2>
                                <form onSubmit={handleSubmit} className="space-y-4 sm:space-y-6">
                                    <div className="grid grid-cols-1 sm:grid-cols-2 gap-4 sm:gap-6">
                                        <div>
                                            <label
                                                htmlFor="name"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-200 mb-2"
                                            >
                                                Full Name *
                                            </label>
                                            <input
                                                type="text"
                                                id="name"
                                                value={data.name}
                                                onChange={(e) => setData("name", e.target.value)}
                                                className={`w-full px-3 sm:px-4 py-2 sm:py-3 border rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors text-sm sm:text-base ${
                                                    errors.name ? "border-red-500 dark:border-red-400" : "border-gray-300 dark:border-gray-600"
                                                }`}
                                                placeholder="Enter your full name"
                                                required
                                            />
                                            {errors.name && (
                                                <p className="mt-1 text-xs sm:text-sm text-red-600 dark:text-red-400">
                                                    {errors.name}
                                                </p>
                                            )}
                                        </div>
                                        <div>
                                            <label
                                                htmlFor="email"
                                                className="block text-sm font-medium text-gray-700 dark:text-gray-200 mb-2"
                                            >
                                                Email Address *
                                            </label>
                                            <input
                                                type="email"
                                                id="email"
                                                value={data.email}
                                                onChange={(e) => setData("email", e.target.value)}
                                                className={`w-full px-3 sm:px-4 py-2 sm:py-3 border rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors text-sm sm:text-base ${
                                                    errors.email ? "border-red-500 dark:border-red-400" : "border-gray-300 dark:border-gray-600"
                                                }`}
                                                placeholder="Enter your email address"
                                                required
                                            />
                                            {errors.email && (
                                                <p className="mt-1 text-xs sm:text-sm text-red-600 dark:text-red-400">
                                                    {errors.email}
                                                </p>
                                            )}
                                        </div>
                                    </div>
                                    <div>
                                        <label
                                            htmlFor="phone"
                                            className="block text-sm font-medium text-gray-700 dark:text-gray-200 mb-2"
                                        >
                                            Phone Number (Optional)
                                        </label>
                                        <input
                                            type="tel"
                                            id="phone"
                                            value={data.phone}
                                            onChange={(e) => setData("phone", e.target.value)}
                                            className="w-full px-3 sm:px-4 py-2 sm:py-3 border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors text-sm sm:text-base"
                                            placeholder="Enter your phone number"
                                        />
                                    </div>
                                    <div>
                                        <label
                                            htmlFor="subject"
                                            className="block text-sm font-medium text-gray-700 dark:text-gray-200 mb-2"
                                        >
                                            Subject *
                                        </label>
                                        <input
                                            type="text"
                                            id="subject"
                                            value={data.subject}
                                            onChange={(e) => setData("subject", e.target.value)}
                                            className={`w-full px-3 sm:px-4 py-2 sm:py-3 border rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors text-sm sm:text-base ${
                                                errors.subject ? "border-red-500 dark:border-red-400" : "border-gray-300 dark:border-gray-600"
                                            }`}
                                            placeholder="Enter the subject of your message"
                                            required
                                        />
                                        {errors.subject && (
                                            <p className="mt-1 text-xs sm:text-sm text-red-600 dark:text-red-400">
                                                {errors.subject}
                                            </p>
                                        )}
                                    </div>
                                    <div>
                                        <label
                                            htmlFor="message"
                                            className="block text-sm font-medium text-gray-700 dark:text-gray-200 mb-2"
                                        >
                                            Message *
                                        </label>
                                        <textarea
                                            id="message"
                                            rows={6}
                                            value={data.message}
                                            onChange={(e) => setData("message", e.target.value)}
                                            className={`w-full px-3 sm:px-4 py-2 sm:py-3 border rounded-lg bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors resize-vertical text-sm sm:text-base ${
                                                errors.message ? "border-red-500 dark:border-red-400" : "border-gray-300 dark:border-gray-600"
                                            }`}
                                            placeholder="Enter your message here..."
                                            required
                                        />
                                        {errors.message && (
                                            <p className="mt-1 text-xs sm:text-sm text-red-600 dark:text-red-400">
                                                {errors.message}
                                            </p>
                                        )}
                                        <p className="mt-1 text-xs sm:text-sm text-gray-500 dark:text-gray-300">
                                            {data.message.length}/5000 characters
                                        </p>
                                    </div>
                                    <div>
                                        <button
                                            type="submit"
                                            disabled={isSubmitting || processing}
                                            className={`w-full py-2 sm:py-3 px-4 sm:px-6 rounded-lg font-medium text-white transition-all duration-200 text-sm sm:text-base ${
                                                isSubmitting || processing
                                                    ? "bg-gray-400 dark:bg-gray-600 cursor-not-allowed"
                                                    : "bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 focus:ring-2 focus:ring-blue-500 focus:ring-offset-2 dark:focus:ring-offset-gray-800 shadow-sm hover:shadow-md transform hover:scale-[1.02]"
                                            }`}
                                        >
                                            {isSubmitting || processing ? (
                                                <span className="flex items-center justify-center">
                                                    <svg
                                                        className="animate-spin -ml-1 mr-2 sm:mr-3 h-4 w-4 sm:h-5 sm:w-5 text-white"
                                                        xmlns="http://www.w3.org/2000/svg"
                                                        fill="none"
                                                        viewBox="0 0 24 24"
                                                    >
                                                        <circle
                                                            className="opacity-25"
                                                            cx="12"
                                                            cy="12"
                                                            r="10"
                                                            stroke="currentColor"
                                                            strokeWidth="4"
                                                        ></circle>
                                                        <path
                                                            className="opacity-75"
                                                            fill="currentColor"
                                                            d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                        ></path>
                                                    </svg>
                                                    Sending Message...
                                                </span>
                                            ) : (
                                                "Send Message"
                                            )}
                                        </button>
                                    </div>
                                </form>
                            </div>
                        </div>
                    </div>

                    {/* Why Choose Us Section - CMS Driven */}
                    <div className="mt-8 sm:mt-12 lg:mt-16">
                        <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:from-gray-700 dark:to-gray-700 rounded-lg shadow-lg border border-gray-200 dark:border-gray-600 p-4 sm:p-6 lg:p-8">
                            <h3 className="text-lg sm:text-xl md:text-2xl font-semibold text-gray-900 dark:text-gray-100 mb-4 sm:mb-6 text-center">
                                {whyChoose.title}
                            </h3>
                            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-3 gap-4 sm:gap-6 lg:gap-8">
                                {whyChoose.features.map((feature, index) => {
                                    const colorClasses = {
                                        blue: 'bg-blue-100 dark:bg-blue-500/20 text-blue-600 dark:text-blue-300',
                                        green: 'bg-green-100 dark:bg-green-500/20 text-green-600 dark:text-green-300',
                                        purple: 'bg-purple-100 dark:bg-purple-500/20 text-purple-600 dark:text-purple-300'
                                    };
                                    return (
                                        <div key={index} className="flex flex-col items-center text-center p-3 sm:p-4">
                                            <div className={`w-12 h-12 sm:w-16 sm:h-16 ${colorClasses[feature.color]} rounded-full flex items-center justify-center mb-3 sm:mb-4`}>
                                                <svg
                                                    className="w-6 h-6 sm:w-8 sm:h-8"
                                                    fill="none"
                                                    stroke="currentColor"
                                                    viewBox="0 0 24 24"
                                                >
                                                    {getIconSVG(feature.icon)}
                                                </svg>
                                            </div>
                                            <h4 className="text-sm sm:text-base md:text-lg font-medium text-gray-900 dark:text-gray-100 mb-2">
                                                {feature.title}
                                            </h4>
                                            <div 
                                                className="text-xs sm:text-sm md:text-base text-gray-600 dark:text-gray-200 content-area"
                                                dangerouslySetInnerHTML={{ __html: feature.description }}
                                            />
                                        </div>
                                    );
                                })}
                            </div>
                        </div>
                    </div>

                    {/* FAQ Section - CMS Driven */}
                    <div className="mt-8 sm:mt-12 lg:mt-16">
                        <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:from-gray-700 dark:to-gray-700 rounded-lg shadow-lg border border-gray-200 dark:border-gray-600 p-4 sm:p-6 lg:p-8">
                            <h3 className="text-lg sm:text-xl md:text-2xl font-semibold text-gray-900 dark:text-gray-100 mb-4 sm:mb-6 text-center">
                                {contactFaqs.title}
                            </h3>
                            <div className="grid grid-cols-1 md:grid-cols-2 gap-4 sm:gap-6">
                                {contactFaqs.faqs.map((faq, index) => (
                                    <div key={index} className="p-3 sm:p-4 bg-white/50 dark:bg-gray-600 rounded-lg">
                                        <h4 className="text-sm sm:text-base font-medium text-gray-900 dark:text-gray-100 mb-2">
                                            {faq.question}
                                        </h4>
                                        <div 
                                            className="text-xs sm:text-sm text-gray-600 dark:text-gray-200 content-area"
                                            dangerouslySetInnerHTML={{ __html: faq.answer }}
                                        />
                                    </div>
                                ))}
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </MainLayout>
    );
}

