import React, { useState } from "react";
import { Head } from "@inertiajs/react";
import MainLayout from "@/Layouts/MainLayout";

export default function CookiePolicy({ lastUpdated }) {
    const [activeTab, setActiveTab] = useState("overview");

    const cookieCategories = [
        {
            id: "essential",
            name: "Essential Cookies",
            description: "Required for basic website functionality",
            enabled: true,
            locked: true,
            cookies: [
                {
                    name: "session_token",
                    purpose: "User authentication and session management",
                    duration: "Session",
                },
                {
                    name: "csrf_token",
                    purpose: "Security protection against cross-site attacks",
                    duration: "2 hours",
                },
                {
                    name: "cookie_consent",
                    purpose: "Remember your cookie preferences",
                    duration: "1 year",
                },
                {
                    name: "language_preference",
                    purpose: "Remember your language selection",
                    duration: "1 year",
                },
            ],
        },
        {
            id: "analytics",
            name: "Analytics Cookies",
            description: "Help us understand how visitors use our website",
            enabled: false,
            locked: false,
            cookies: [
                {
                    name: "_ga",
                    purpose: "Google Analytics - distinguish users",
                    duration: "2 years",
                },
                {
                    name: "_gid",
                    purpose: "Google Analytics - distinguish users",
                    duration: "24 hours",
                },
                {
                    name: "_gat",
                    purpose: "Google Analytics - throttle request rate",
                    duration: "1 minute",
                },
                {
                    name: "hotjar_session",
                    purpose: "Hotjar - user behavior analysis",
                    duration: "30 minutes",
                },
            ],
        },
        {
            id: "marketing",
            name: "Marketing Cookies",
            description: "Used to deliver relevant advertisements",
            enabled: false,
            locked: false,
            cookies: [
                {
                    name: "facebook_pixel",
                    purpose: "Facebook advertising and retargeting",
                    duration: "90 days",
                },
                {
                    name: "google_ads",
                    purpose: "Google Ads conversion tracking",
                    duration: "90 days",
                },
                {
                    name: "linkedin_insight",
                    purpose: "LinkedIn advertising analytics",
                    duration: "90 days",
                },
            ],
        },
        {
            id: "preferences",
            name: "Preference Cookies",
            description: "Remember your settings and preferences",
            enabled: false,
            locked: false,
            cookies: [
                {
                    name: "theme_preference",
                    purpose: "Remember dark/light mode selection",
                    duration: "1 year",
                },
                {
                    name: "notification_settings",
                    purpose: "Remember notification preferences",
                    duration: "1 year",
                },
                {
                    name: "dashboard_layout",
                    purpose: "Remember dashboard customization",
                    duration: "6 months",
                },
            ],
        },
    ];

    return (
        <MainLayout>
            <Head>
                <title>Cookie Policy | Academic Scribe</title>
                <meta
                    name="description"
                    content="Cookie Policy for Academic Scribe. Learn about the cookies we use, how they work, and how to manage your cookie preferences."
                />
                <meta name="robots" content="index, follow" />
                <link rel="canonical" href={route("cookie-policy")} />
            </Head>

            <div className="min-h-screen bg-gray-50 py-12">
                <div className="max-w-4xl mx-auto px-4 sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="bg-white rounded-lg shadow-sm p-8 mb-8">
                        <h1 className="text-4xl font-bold text-gray-900 mb-4">
                            Cookie Policy
                        </h1>
                        <p className="text-gray-600 text-lg">
                            Last updated: {lastUpdated || "January 1, 2024"}
                        </p>
                        <p className="text-gray-600 mt-4">
                            This Cookie Policy explains how Academic Scribe uses
                            cookies and similar technologies to enhance your
                            browsing experience and improve our services.
                        </p>
                    </div>

                    {/* Tab Navigation */}
                    <div className="bg-white rounded-lg shadow-sm mb-8">
                        <div className="border-b border-gray-200">
                            <nav className="flex space-x-8 px-8">
                                {[
                                    {
                                        id: "overview",
                                        label: "Overview",
                                        icon: "📋",
                                    },
                                    {
                                        id: "categories",
                                        label: "Cookie Categories",
                                        icon: "🍪",
                                    },
                                    {
                                        id: "manage",
                                        label: "Manage Cookies",
                                        icon: "⚙️",
                                    },
                                    {
                                        id: "third-party",
                                        label: "Third-Party",
                                        icon: "🔗",
                                    },
                                ].map((tab) => (
                                    <button
                                        key={tab.id}
                                        onClick={() => setActiveTab(tab.id)}
                                        className={`py-4 px-1 border-b-2 font-medium text-sm flex items-center ${
                                            activeTab === tab.id
                                                ? "border-blue-500 text-blue-600"
                                                : "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300"
                                        }`}
                                    >
                                        <span className="mr-2">{tab.icon}</span>
                                        {tab.label}
                                    </button>
                                ))}
                            </nav>
                        </div>

                        <div className="p-8">
                            {/* Overview Tab */}
                            {activeTab === "overview" && (
                                <div className="space-y-8">
                                    <section>
                                        <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                            What Are Cookies?
                                        </h2>
                                        <p className="text-gray-600 mb-4">
                                            Cookies are small text files that
                                            are stored on your device when you
                                            visit our website. They help us
                                            provide you with a better experience
                                            by remembering your preferences and
                                            understanding how you use our site.
                                        </p>
                                        <div className="bg-blue-50 rounded-lg p-6">
                                            <h3 className="text-lg font-semibold text-blue-900 mb-2">
                                                🍪 Cookie Basics
                                            </h3>
                                            <ul className="list-disc list-inside text-blue-800 space-y-1">
                                                <li>
                                                    Small data files stored on
                                                    your device
                                                </li>
                                                <li>
                                                    Help websites remember your
                                                    preferences
                                                </li>
                                                <li>
                                                    Enable personalized
                                                    experiences
                                                </li>
                                                <li>
                                                    Can be deleted or blocked
                                                    through browser settings
                                                </li>
                                            </ul>
                                        </div>
                                    </section>

                                    <section>
                                        <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                            How We Use Cookies
                                        </h2>
                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                            <div className="bg-gray-50 rounded-lg p-6">
                                                <h3 className="text-lg font-semibold text-gray-900 mb-3">
                                                    🔐 Essential Functions
                                                </h3>
                                                <ul className="text-gray-600 space-y-2">
                                                    <li>
                                                        • User authentication
                                                        and login
                                                    </li>
                                                    <li>
                                                        • Shopping cart and
                                                        order processing
                                                    </li>
                                                    <li>
                                                        • Security and fraud
                                                        prevention
                                                    </li>
                                                    <li>
                                                        • Website functionality
                                                        and navigation
                                                    </li>
                                                </ul>
                                            </div>
                                            <div className="bg-gray-50 rounded-lg p-6">
                                                <h3 className="text-lg font-semibold text-gray-900 mb-3">
                                                    📊 Analytics & Improvement
                                                </h3>
                                                <ul className="text-gray-600 space-y-2">
                                                    <li>
                                                        • Understanding user
                                                        behavior
                                                    </li>
                                                    <li>
                                                        • Measuring website
                                                        performance
                                                    </li>
                                                    <li>
                                                        • Identifying popular
                                                        content
                                                    </li>
                                                    <li>
                                                        • Improving user
                                                        experience
                                                    </li>
                                                </ul>
                                            </div>
                                            <div className="bg-gray-50 rounded-lg p-6">
                                                <h3 className="text-lg font-semibold text-gray-900 mb-3">
                                                    🎯 Personalization
                                                </h3>
                                                <ul className="text-gray-600 space-y-2">
                                                    <li>
                                                        • Remembering your
                                                        preferences
                                                    </li>
                                                    <li>
                                                        • Customizing content
                                                        display
                                                    </li>
                                                    <li>
                                                        • Language and region
                                                        settings
                                                    </li>
                                                    <li>
                                                        • Theme and layout
                                                        preferences
                                                    </li>
                                                </ul>
                                            </div>
                                            <div className="bg-gray-50 rounded-lg p-6">
                                                <h3 className="text-lg font-semibold text-gray-900 mb-3">
                                                    📢 Marketing & Advertising
                                                </h3>
                                                <ul className="text-gray-600 space-y-2">
                                                    <li>
                                                        • Relevant advertisement
                                                        delivery
                                                    </li>
                                                    <li>
                                                        • Conversion tracking
                                                    </li>
                                                    <li>
                                                        • Retargeting campaigns
                                                    </li>
                                                    <li>
                                                        • Social media
                                                        integration
                                                    </li>
                                                </ul>
                                            </div>
                                        </div>
                                    </section>

                                    <section>
                                        <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                            Types of Cookies We Use
                                        </h2>
                                        <div className="space-y-4">
                                            <div className="border border-gray-200 rounded-lg p-4">
                                                <h3 className="font-semibold text-gray-900 mb-2">
                                                    Session Cookies
                                                </h3>
                                                <p className="text-gray-600 text-sm">
                                                    Temporary cookies that
                                                    expire when you close your
                                                    browser. Essential for
                                                    website functionality.
                                                </p>
                                            </div>
                                            <div className="border border-gray-200 rounded-lg p-4">
                                                <h3 className="font-semibold text-gray-900 mb-2">
                                                    Persistent Cookies
                                                </h3>
                                                <p className="text-gray-600 text-sm">
                                                    Remain on your device for a
                                                    set period. Help remember
                                                    your preferences between
                                                    visits.
                                                </p>
                                            </div>
                                            <div className="border border-gray-200 rounded-lg p-4">
                                                <h3 className="font-semibold text-gray-900 mb-2">
                                                    First-Party Cookies
                                                </h3>
                                                <p className="text-gray-600 text-sm">
                                                    Set directly by Academic
                                                    Scribe for website
                                                    functionality and user
                                                    experience.
                                                </p>
                                            </div>
                                            <div className="border border-gray-200 rounded-lg p-4">
                                                <h3 className="font-semibold text-gray-900 mb-2">
                                                    Third-Party Cookies
                                                </h3>
                                                <p className="text-gray-600 text-sm">
                                                    Set by external services
                                                    like analytics providers and
                                                    advertising networks.
                                                </p>
                                            </div>
                                        </div>
                                    </section>
                                </div>
                            )}

                            {/* Categories Tab */}
                            {activeTab === "categories" && (
                                <div className="space-y-6">
                                    <h2 className="text-2xl font-bold text-gray-900 mb-6">
                                        Cookie Categories
                                    </h2>
                                    {cookieCategories.map((category) => (
                                        <div
                                            key={category.id}
                                            className="border border-gray-200 rounded-lg p-6"
                                        >
                                            <div className="flex items-center justify-between mb-4">
                                                <div>
                                                    <h3 className="text-xl font-semibold text-gray-900">
                                                        {category.name}
                                                    </h3>
                                                    <p className="text-gray-600 mt-1">
                                                        {category.description}
                                                    </p>
                                                </div>
                                                <div className="flex items-center">
                                                    {category.locked ? (
                                                        <span className="text-sm text-gray-500 mr-3">
                                                            Always Active
                                                        </span>
                                                    ) : (
                                                        <label className="relative inline-flex items-center cursor-pointer">
                                                            <input
                                                                type="checkbox"
                                                                className="sr-only peer"
                                                                defaultChecked={
                                                                    category.enabled
                                                                }
                                                            />
                                                            <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                                        </label>
                                                    )}
                                                </div>
                                            </div>

                                            <div className="overflow-x-auto">
                                                <table className="min-w-full divide-y divide-gray-200">
                                                    <thead className="bg-gray-50">
                                                        <tr>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                                Cookie Name
                                                            </th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                                Purpose
                                                            </th>
                                                            <th className="px-4 py-3 text-left text-xs font-medium text-gray-500 uppercase tracking-wider">
                                                                Duration
                                                            </th>
                                                        </tr>
                                                    </thead>
                                                    <tbody className="bg-white divide-y divide-gray-200">
                                                        {category.cookies.map(
                                                            (cookie, index) => (
                                                                <tr key={index}>
                                                                    <td className="px-4 py-4 whitespace-nowrap text-sm font-medium text-gray-900">
                                                                        {
                                                                            cookie.name
                                                                        }
                                                                    </td>
                                                                    <td className="px-4 py-4 text-sm text-gray-600">
                                                                        {
                                                                            cookie.purpose
                                                                        }{" "}
                                                                    </td>
                                                                    <td className="px-4 py-4 whitespace-nowrap text-sm text-gray-600">
                                                                        {
                                                                            cookie.duration
                                                                        }
                                                                    </td>
                                                                </tr>
                                                            )
                                                        )}
                                                    </tbody>
                                                </table>
                                            </div>
                                        </div>
                                    ))}
                                </div>
                            )}

                            {/* Manage Cookies Tab */}
                            {activeTab === "manage" && (
                                <div className="space-y-8">
                                    <section>
                                        <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                            Manage Your Cookie Preferences
                                        </h2>
                                        <p className="text-gray-600 mb-6">
                                            You have control over which cookies
                                            are stored on your device. Use the
                                            options below to customize your
                                            cookie preferences.
                                        </p>

                                        {/* Cookie Preference Controls */}
                                        <div className="bg-gray-50 rounded-lg p-6 mb-6">
                                            <h3 className="text-lg font-semibold text-gray-900 mb-4">
                                                Cookie Preferences
                                            </h3>
                                            <div className="space-y-4">
                                                {cookieCategories.map(
                                                    (category) => (
                                                        <div
                                                            key={category.id}
                                                            className="flex items-center justify-between p-4 bg-white rounded-lg border"
                                                        >
                                                            <div>
                                                                <h4 className="font-medium text-gray-900">
                                                                    {
                                                                        category.name
                                                                    }
                                                                </h4>
                                                                <p className="text-sm text-gray-600">
                                                                    {
                                                                        category.description
                                                                    }
                                                                </p>
                                                            </div>
                                                            <div className="flex items-center">
                                                                {category.locked ? (
                                                                    <span className="text-sm text-gray-500 bg-gray-100 px-3 py-1 rounded-full">
                                                                        Always
                                                                        Active
                                                                    </span>
                                                                ) : (
                                                                    <label className="relative inline-flex items-center cursor-pointer">
                                                                        <input
                                                                            type="checkbox"
                                                                            className="sr-only peer"
                                                                            defaultChecked={
                                                                                category.enabled
                                                                            }
                                                                        />
                                                                        <div className="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-blue-300 rounded-full peer peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all peer-checked:bg-blue-600"></div>
                                                                    </label>
                                                                )}
                                                            </div>
                                                        </div>
                                                    )
                                                )}
                                            </div>

                                            <div className="flex space-x-4 mt-6">
                                                <button className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium transition-colors">
                                                    Save Preferences
                                                </button>
                                                <button className="bg-gray-200 hover:bg-gray-300 text-gray-800 px-6 py-2 rounded-lg font-medium transition-colors">
                                                    Accept All
                                                </button>
                                                <button className="bg-gray-200 hover:bg-gray-300 text-gray-800 px-6 py-2 rounded-lg font-medium transition-colors">
                                                    Reject All
                                                </button>
                                            </div>
                                        </div>
                                    </section>

                                    <section>
                                        <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                            Browser Cookie Settings
                                        </h2>
                                        <p className="text-gray-600 mb-6">
                                            You can also manage cookies directly
                                            through your browser settings.
                                            Here's how to do it in popular
                                            browsers:
                                        </p>

                                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <h3 className="font-semibold text-gray-900 mb-3 flex items-center">
                                                    <span className="mr-2">
                                                        🌐
                                                    </span>
                                                    Google Chrome
                                                </h3>
                                                <ol className="list-decimal list-inside text-sm text-gray-600 space-y-1">
                                                    <li>
                                                        Click the three dots
                                                        menu → Settings
                                                    </li>
                                                    <li>
                                                        Go to Privacy and
                                                        Security → Cookies
                                                    </li>
                                                    <li>
                                                        Choose your cookie
                                                        preferences
                                                    </li>
                                                    <li>
                                                        Manage exceptions for
                                                        specific sites
                                                    </li>
                                                </ol>
                                            </div>

                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <h3 className="font-semibold text-gray-900 mb-3 flex items-center">
                                                    <span className="mr-2">
                                                        🦊
                                                    </span>
                                                    Mozilla Firefox
                                                </h3>
                                                <ol className="list-decimal list-inside text-sm text-gray-600 space-y-1">
                                                    <li>
                                                        Click the menu button →
                                                        Settings
                                                    </li>
                                                    <li>
                                                        Select Privacy &
                                                        Security
                                                    </li>
                                                    <li>
                                                        Go to Cookies and Site
                                                        Data
                                                    </li>
                                                    <li>
                                                        Adjust your cookie
                                                        settings
                                                    </li>
                                                </ol>
                                            </div>

                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <h3 className="font-semibold text-gray-900 mb-3 flex items-center">
                                                    <span className="mr-2">
                                                        🧭
                                                    </span>
                                                    Safari
                                                </h3>
                                                <ol className="list-decimal list-inside text-sm text-gray-600 space-y-1">
                                                    <li>
                                                        Safari menu →
                                                        Preferences
                                                    </li>
                                                    <li>
                                                        Click the Privacy tab
                                                    </li>
                                                    <li>
                                                        Choose cookie blocking
                                                        options
                                                    </li>
                                                    <li>Manage website data</li>
                                                </ol>
                                            </div>

                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <h3 className="font-semibold text-gray-900 mb-3 flex items-center">
                                                    <span className="mr-2">
                                                        🔷
                                                    </span>
                                                    Microsoft Edge
                                                </h3>
                                                <ol className="list-decimal list-inside text-sm text-gray-600 space-y-1">
                                                    <li>
                                                        Click the three dots →
                                                        Settings
                                                    </li>
                                                    <li>
                                                        Go to Cookies and site
                                                        permissions
                                                    </li>
                                                    <li>
                                                        Select Cookies and site
                                                        data
                                                    </li>
                                                    <li>
                                                        Configure your
                                                        preferences
                                                    </li>
                                                </ol>
                                            </div>
                                        </div>

                                        <div className="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mt-6">
                                            <div className="flex">
                                                <div className="flex-shrink-0">
                                                    <span className="text-yellow-400 text-xl">
                                                        ⚠️
                                                    </span>
                                                </div>
                                                <div className="ml-3">
                                                    <h3 className="text-sm font-medium text-yellow-800">
                                                        Important Note
                                                    </h3>
                                                    <p className="mt-1 text-sm text-yellow-700">
                                                        Disabling certain
                                                        cookies may affect
                                                        website functionality.
                                                        Essential cookies are
                                                        required for basic site
                                                        operations.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </section>
                                </div>
                            )}

                            {/* Third-Party Tab */}
                            {activeTab === "third-party" && (
                                <div className="space-y-8">
                                    <section>
                                        <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                            Third-Party Cookies and Services
                                        </h2>
                                        <p className="text-gray-600 mb-6">
                                            We work with trusted third-party
                                            services that may set their own
                                            cookies. These services help us
                                            provide better functionality and
                                            understand how our website is used.
                                        </p>

                                        <div className="space-y-6">
                                            {/* Google Analytics */}
                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <div className="flex items-start justify-between mb-4">
                                                    <div>
                                                        <h3 className="text-lg font-semibold text-gray-900">
                                                            Google Analytics
                                                        </h3>
                                                        <p className="text-gray-600 mt-1">
                                                            Helps us understand
                                                            website usage and
                                                            improve user
                                                            experience
                                                        </p>
                                                    </div>
                                                    <span className="bg-blue-100 text-blue-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                                        Analytics
                                                    </span>
                                                </div>
                                                <div className="text-sm text-gray-600 space-y-2">
                                                    <p>
                                                        <strong>
                                                            Purpose:
                                                        </strong>{" "}
                                                        Website analytics and
                                                        performance tracking
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Data Collected:
                                                        </strong>{" "}
                                                        Page views, user
                                                        interactions, device
                                                        information
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Retention:
                                                        </strong>{" "}
                                                        26 months
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Opt-out:
                                                        </strong>
                                                        <a
                                                            href="https://tools.google.com/dlpage/gaoptout"
                                                            className="text-blue-600 hover:text-blue-800 ml-1"
                                                            target="_blank"
                                                            rel="noopener noreferrer"
                                                        >
                                                            Google Analytics
                                                            Opt-out
                                                        </a>
                                                    </p>
                                                </div>
                                            </div>

                                            {/* Hotjar */}
                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <div className="flex items-start justify-between mb-4">
                                                    <div>
                                                        <h3 className="text-lg font-semibold text-gray-900">
                                                            Hotjar
                                                        </h3>
                                                        <p className="text-gray-600 mt-1">
                                                            User behavior
                                                            analysis and website
                                                            optimization
                                                        </p>
                                                    </div>
                                                    <span className="bg-purple-100 text-purple-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                                        Analytics
                                                    </span>
                                                </div>
                                                <div className="text-sm text-gray-600 space-y-2">
                                                    <p>
                                                        <strong>
                                                            Purpose:
                                                        </strong>{" "}
                                                        Heatmaps, session
                                                        recordings, user
                                                        feedback
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Data Collected:
                                                        </strong>{" "}
                                                        Mouse movements, clicks,
                                                        form interactions
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Retention:
                                                        </strong>{" "}
                                                        365 days
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Opt-out:
                                                        </strong>
                                                        <a
                                                            href="https://www.hotjar.com/legal/compliance/opt-out"
                                                            className="text-blue-600 hover:text-blue-800 ml-1"
                                                            target="_blank"
                                                            rel="noopener noreferrer"
                                                        >
                                                            Hotjar Opt-out
                                                        </a>
                                                    </p>
                                                </div>
                                            </div>

                                            {/* Facebook Pixel */}
                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <div className="flex items-start justify-between mb-4">
                                                    <div>
                                                        <h3 className="text-lg font-semibold text-gray-900">
                                                            Facebook Pixel
                                                        </h3>
                                                        <p className="text-gray-600 mt-1">
                                                            Advertising
                                                            optimization and
                                                            conversion tracking
                                                        </p>
                                                    </div>
                                                    <span className="bg-green-100 text-green-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                                        Marketing
                                                    </span>
                                                </div>
                                                <div className="text-sm text-gray-600 space-y-2">
                                                    <p>
                                                        <strong>
                                                            Purpose:
                                                        </strong>{" "}
                                                        Ad targeting, conversion
                                                        tracking, retargeting
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Data Collected:
                                                        </strong>{" "}
                                                        Page visits, purchases,
                                                        custom events
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Retention:
                                                        </strong>{" "}
                                                        90 days
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Opt-out:
                                                        </strong>
                                                        <a
                                                            href="https://www.facebook.com/settings?tab=ads"
                                                            className="text-blue-600 hover:text-blue-800 ml-1"
                                                            target="_blank"
                                                            rel="noopener noreferrer"
                                                        >
                                                            Facebook Ad
                                                            Preferences
                                                        </a>
                                                    </p>
                                                </div>
                                            </div>

                                            {/* Google Ads */}
                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <div className="flex items-start justify-between mb-4">
                                                    <div>
                                                        <h3 className="text-lg font-semibold text-gray-900">
                                                            Google Ads
                                                        </h3>
                                                        <p className="text-gray-600 mt-1">
                                                            Conversion tracking
                                                            and remarketing
                                                        </p>
                                                    </div>
                                                    <span className="bg-green-100 text-green-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                                        Marketing
                                                    </span>
                                                </div>
                                                <div className="text-sm text-gray-600 space-y-2">
                                                    <p>
                                                        <strong>
                                                            Purpose:
                                                        </strong>{" "}
                                                        Ad performance
                                                        measurement, remarketing
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Data Collected:
                                                        </strong>{" "}
                                                        Conversion events , user
                                                        interactions
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Retention:
                                                        </strong>{" "}
                                                        90 days
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Opt-out:
                                                        </strong>
                                                        <a
                                                            href="https://adssettings.google.com/"
                                                            className="text-blue-600 hover:text-blue-800 ml-1"
                                                            target="_blank"
                                                            rel="noopener noreferrer"
                                                        >
                                                            Google Ads Settings
                                                        </a>
                                                    </p>
                                                </div>
                                            </div>

                                            {/* Stripe */}
                                            <div className="border border-gray-200 rounded-lg p-6">
                                                <div className="flex items-start justify-between mb-4">
                                                    <div>
                                                        <h3 className="text-lg font-semibold text-gray-900">
                                                            Stripe
                                                        </h3>
                                                        <p className="text-gray-600 mt-1">
                                                            Secure payment
                                                            processing
                                                        </p>
                                                    </div>
                                                    <span className="bg-red-100 text-red-800 text-xs font-medium px-2.5 py-0.5 rounded">
                                                        Essential
                                                    </span>
                                                </div>
                                                <div className="text-sm text-gray-600 space-y-2">
                                                    <p>
                                                        <strong>
                                                            Purpose:
                                                        </strong>{" "}
                                                        Payment processing,
                                                        fraud prevention
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Data Collected:
                                                        </strong>{" "}
                                                        Payment information,
                                                        transaction data
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Retention:
                                                        </strong>{" "}
                                                        As required by law
                                                    </p>
                                                    <p>
                                                        <strong>
                                                            Privacy Policy:
                                                        </strong>
                                                        <a
                                                            href="https://stripe.com/privacy"
                                                            className="text-blue-600 hover:text-blue-800 ml-1"
                                                            target="_blank"
                                                            rel="noopener noreferrer"
                                                        >
                                                            Stripe Privacy
                                                            Policy
                                                        </a>
                                                    </p>
                                                </div>
                                            </div>
                                        </div>

                                        <div className="bg-blue-50 border border-blue-200 rounded-lg p-6 mt-8">
                                            <h3 className="text-lg font-semibold text-blue-900 mb-3">
                                                Managing Third-Party Cookies
                                            </h3>
                                            <div className="text-blue-800 space-y-2">
                                                <p>
                                                    • Visit each service's
                                                    privacy settings to manage
                                                    preferences
                                                </p>
                                                <p>
                                                    • Use browser settings to
                                                    block third-party cookies
                                                    entirely
                                                </p>
                                                <p>
                                                    • Consider using
                                                    privacy-focused browser
                                                    extensions
                                                </p>
                                                <p>
                                                    • Review and update your
                                                    preferences regularly
                                                </p>
                                            </div>
                                        </div>
                                    </section>

                                    <section>
                                        <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                            Data Sharing and Privacy
                                        </h2>
                                        <p className="text-gray-600 mb-4">
                                            When you interact with third-party
                                            services through our website, they
                                            may collect information according to
                                            their own privacy policies:
                                        </p>

                                        <div className="bg-gray-50 rounded-lg p-6">
                                            <ul className="space-y-3 text-gray-600">
                                                <li className="flex items-start">
                                                    <span className="text-green-500 mr-2 mt-1">
                                                        ✓
                                                    </span>
                                                    We only work with reputable,
                                                    privacy-compliant services
                                                </li>
                                                <li className="flex items-start">
                                                    <span className="text-green-500 mr-2 mt-1">
                                                        ✓
                                                    </span>
                                                    Data sharing is limited to
                                                    what's necessary for
                                                    functionality
                                                </li>
                                                <li className="flex items-start">
                                                    <span className="text-green-500 mr-2 mt-1">
                                                        ✓
                                                    </span>
                                                    You can opt out of
                                                    non-essential third-party
                                                    cookies
                                                </li>
                                                <li className="flex items-start">
                                                    <span className="text-green-500 mr-2 mt-1">
                                                        ✓
                                                    </span>
                                                    We regularly review our
                                                    third-party integrations
                                                </li>
                                            </ul>
                                        </div>
                                    </section>
                                </div>
                            )}
                        </div>
                    </div>

                    {/* Additional Information */}
                    <div className="bg-white rounded-lg shadow-sm p-8 space-y-8">
                        {/* Your Rights */}
                        <section>
                            <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                Your Cookie Rights
                            </h2>
                            <p className="text-gray-600 mb-6">
                                You have several rights regarding cookies and
                                how your data is processed:
                            </p>

                            <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <div className="space-y-4">
                                    <div className="flex items-start">
                                        <span className="text-blue-500 mr-3 mt-1">
                                            🔧
                                        </span>
                                        <div>
                                            <h3 className="font-semibold text-gray-900">
                                                Control
                                            </h3>
                                            <p className="text-gray-600 text-sm">
                                                Choose which cookies to accept
                                                or reject
                                            </p>
                                        </div>
                                    </div>
                                    <div className="flex items-start">
                                        <span className="text-blue-500 mr-3 mt-1">
                                            👁️
                                        </span>
                                        <div>
                                            <h3 className="font-semibold text-gray-900">
                                                Transparency
                                            </h3>
                                            <p className="text-gray-600 text-sm">
                                                Know what cookies are being used
                                                and why
                                            </p>
                                        </div>
                                    </div>
                                </div>
                                <div className="space-y-4">
                                    <div className="flex items-start">
                                        <span className="text-blue-500 mr-3 mt-1">
                                            🗑️
                                        </span>
                                        <div>
                                            <h3 className="font-semibold text-gray-900">
                                                Deletion
                                            </h3>
                                            <p className="text-gray-600 text-sm">
                                                Delete existing cookies from
                                                your device
                                            </p>
                                        </div>
                                    </div>
                                    <div className="flex items-start">
                                        <span className="text-blue-500 mr-3 mt-1">
                                            🔄
                                        </span>
                                        <div>
                                            <h3 className="font-semibold text-gray-900">
                                                Change
                                            </h3>
                                            <p className="text-gray-600 text-sm">
                                                Modify your preferences at any
                                                time
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {/* Updates */}
                        <section>
                            <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                Policy Updates
                            </h2>
                            <p className="text-gray-600 mb-4">
                                We may update this Cookie Policy from time to
                                time to reflect changes in our practices or for
                                legal and regulatory reasons. When we make
                                changes:
                            </p>

                            <ul className="list-disc list-inside text-gray-600 space-y-2">
                                <li>
                                    We'll update the "Last updated" date at the
                                    top of this page
                                </li>
                                <li>
                                    Significant changes will be communicated via
                                    email or website notice
                                </li>
                                <li>
                                    You'll be asked to review and accept updated
                                    cookie preferences
                                </li>
                                <li>
                                    Continued use of our website constitutes
                                    acceptance of changes
                                </li>
                            </ul>
                        </section>

                        {/* Contact Information */}
                        <section>
                            <h2 className="text-2xl font-bold text-gray-900 mb-4">
                                Questions About Cookies?
                            </h2>
                            <p className="text-gray-600 mb-6">
                                If you have questions about our use of cookies
                                or this Cookie Policy, please don't hesitate to
                                contact us:
                            </p>

                            <div className="bg-gray-50 rounded-lg p-6">
                                <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div>
                                        <h3 className="font-semibold text-gray-900 mb-3">
                                            Contact Information
                                        </h3>
                                        <div className="space-y-2 text-gray-600">
                                            <div>
                                                <strong>Email:</strong>{" "}
                                                privacy@academicscribe.com
                                            </div>
                                            <div>
                                                <strong>Support:</strong>{" "}
                                                support@academicscribe.com
                                            </div>
                                            <div>
                                                <strong>Phone:</strong> +1 (800)
                                                123-4567
                                            </div>
                                        </div>
                                    </div>
                                    <div>
                                        <h3 className="font-semibold text-gray-900 mb-3">
                                            Business Address
                                        </h3>
                                        <div className="text-gray-600">
                                            Academic Scribe
                                            <br />
                                            123 Academic Street
                                            <br />
                                            Education City, EC 12345
                                            <br />
                                            United States
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </section>

                        {/* Quick Actions */}
                        <section className="bg-blue-50 rounded-lg p-6">
                            <h2 className="text-xl font-bold text-blue-900 mb-4">
                                Quick Cookie Actions
                            </h2>
                            <div className="flex flex-wrap gap-4">
                                <button className="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium transition-colors">
                                    Manage Preferences
                                </button>
                                <button className="bg-white hover:bg-gray-50 text-blue-600 border border-blue-600 px-6 py-2 rounded-lg font-medium transition-colors">
                                    Clear All Cookies
                                </button>
                                <button className="bg-white hover:bg-gray-50 text-blue-600 border border-blue-600 px-6 py-2 rounded-lg font-medium transition-colors">
                                    View Browser Settings
                                </button>
                            </div>
                        </section>
                    </div>

                    {/* Footer Navigation */}
                    <div className="mt-8 text-center">
                        <div className="flex flex-wrap justify-center gap-6 text-sm">
                            <a
                                href="/privacy-policy"
                                className="text-blue-600 hover:text-blue-800"
                            >
                                Privacy Policy
                            </a>
                            <a
                                href="/terms-of-service"
                                className="text-blue-600 hover:text-blue-800"
                            >
                                Terms of Service
                            </a>

                            <a
                                href="/contact"
                                className="text-blue-600 hover:text-blue-800"
                            >
                                Contact Us
                            </a>
                        </div>
                    </div>
                </div>
            </div>
        </MainLayout>
    );
}
