import { useState, useEffect } from "react";
import { Head, Link, usePage } from "@inertiajs/react";
import ClientLayout from "@/Layouts/ClientLayout";
import SectionContainer from "@/Components/SectionContainer";
import StatsCard from "@/Components/StatsCard";
import OrderStatusBadge from "@/Components/OrderStatusBadge";
import PaymentStatusBadge from "@/Components/PaymentStatusBadge";
import DeadlineCountdown from "@/Components/DeadlineCountdown";
import SimpleBarChart from "@/Components/SimpleBarChart";
import DashboardDeadlineCountdown from "@/Components/DashboardDeadlineCountdown";
import {
    PlusCircle,
    MessageCircle,
    HelpCircle,
    Clock,
    CreditCard,
    ListOrdered,
    Upload,
    ChevronRight,
    AlertCircle,
    FileText,
    BarChart,
    ExternalLink,
    Calendar,
    DollarSign,
    User,
    Wallet,
    ArrowRight,
    CreditCard as CreditCardIcon,
    Star,
    Award,
    TrendingUp,
    CheckCircle,
} from "lucide-react";

export default function Dashboard({
    orderStats,
    recentOrders,
    availableCoupons,
    loyaltyInfo,
    preferredWriters,
    pendingBids,
    financialSummary,
    orderActivityData,
    error,
}) {
    const [showError, setShowError] = useState(!!error);
    const { auth } = usePage().props;
    const [greeting, setGreeting] = useState("Hello");
    const [showAllCoupons, setShowAllCoupons] = useState(false);
    const [expandedOrderId, setExpandedOrderId] = useState(null);
    const [orderActivityExpanded, setOrderActivityExpanded] = useState(false);

    // Set greeting based on time of day
    useEffect(() => {
        const hours = new Date().getHours();
        let newGreeting = "Hello";

        if (hours < 12) {
            newGreeting = "Good morning";
        } else if (hours < 18) {
            newGreeting = "Good afternoon";
        } else {
            newGreeting = "Good evening";
        }

        setGreeting(newGreeting);
    }, []);

    // Process order activity data for chart visualization
    const processOrderActivityData = () => {
        // Check if we have real order activity data
        if (
            orderActivityData &&
            Array.isArray(orderActivityData) &&
            orderActivityData.length > 0
        ) {
            // Process the data from the server
            return orderActivityData.map((item) => ({
                label: item.month,
                value: item.count,
                completed: item.completed,
                amount: item.amount,
            }));
        }

        // Fallback to sample data if no real data is available
        return [
            { label: "Jan", value: 2, completed: 1, amount: 0 },
            { label: "Feb", value: 4, completed: 3, amount: 0 },
            { label: "Mar", value: 3, completed: 2, amount: 0 },
            { label: "Apr", value: 5, completed: 4, amount: 0 },
            { label: "May", value: 7, completed: 5, amount: 0 },
            { label: "Jun", value: 6, completed: 4, amount: 0 },
        ];
    };

    const chartData = processOrderActivityData();
    const maxOrderActivity = Math.max(...chartData.map((item) => item.value));

    // Calculate completion rate for the chart period
    const calculateCompletionRate = () => {
        if (!chartData || chartData.length === 0) return 0;

        const totalOrders = chartData.reduce(
            (sum, item) => sum + item.value,
            0
        );
        const completedOrders = chartData.reduce(
            (sum, item) => sum + (item.completed || 0),
            0
        );

        return totalOrders > 0
            ? Math.round((completedOrders / totalOrders) * 100)
            : 0;
    };

    // Calculate average order value for the chart period
    const calculateAvgOrderValue = () => {
        if (!chartData || chartData.length === 0) return 0;

        const totalAmount = chartData.reduce(
            (sum, item) => sum + (item.amount || 0),
            0
        );
        const completedOrders = chartData.reduce(
            (sum, item) => sum + (item.completed || 0),
            0
        );

        return completedOrders > 0
            ? (totalAmount / completedOrders).toFixed(2)
            : 0;
    };
    return (
        <ClientLayout>
            <Head title="Dashboard" />

            <div className="py-2">
                <div className="max-w-7xl mx-auto sm:px-4 lg:px-4">
                    {/* Page Header */}
                    <div className="mb-6 flex flex-col md:flex-row md:items-center md:justify-between gap-4">
                        <h1 className="text-2xl font-bold dark:text-white">
                            Dashboard
                        </h1>

                        {/* Primary Action Button */}
                        <Link
                            href="/place-order"
                            className="inline-flex items-center px-4 py-2 bg-gradient-to-r from-indigo-600 to-blue-600 hover:from-indigo-700 hover:to-blue-700 text-white font-medium rounded-lg transition-all duration-200 shadow hover:shadow-lg gap-2"
                        >
                            <PlusCircle size={18} />
                            <span>Place New Order</span>
                        </Link>
                    </div>

                    {/* Compact User Salutation */}
                    <div className="mb-4">
                        <div className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600 rounded-lg px-4 py-3 shadow-sm flex items-center justify-between gap-4 flex-wrap">
                            <div className="flex items-center gap-3 flex-1 min-w-0">
                                <h2 className="text-lg font-medium text-gray-900 dark:text-white truncate">
                                    {greeting},{" "}
                                    <span className="font-semibold">
                                        {auth.user?.name?.split(" ")[0] || "there"}
                                    </span>
                                    !
                                </h2>
                                {pendingBids &&
                                    pendingBids.total_orders_with_bids > 0 && (
                                        <Link
                                            href="/client/bids"
                                            className="inline-flex items-center gap-1.5 px-3 py-1.5 bg-indigo-600 hover:bg-indigo-700 text-white text-sm font-medium rounded-md transition-colors whitespace-nowrap"
                                        >
                                            <AlertCircle size={14} />
                                            <span>
                                                {pendingBids.total_bids} New Bid
                                                {pendingBids.total_bids !== 1 ? "s" : ""}
                                            </span>
                                        </Link>
                                    )}
                            </div>
                            <Link
                                href="/place-order"
                                className="inline-flex items-center gap-2 px-4 py-2 bg-indigo-600 hover:bg-indigo-700 text-white text-sm font-medium rounded-lg transition-colors whitespace-nowrap"
                            >
                                <PlusCircle size={16} />
                                <span>Place Order</span>
                            </Link>
                        </div>
                    </div>

                    {/* Error notification if applicable */}
                    {showError && (
                        <div className="mb-6 bg-red-100 border border-red-200 text-red-700 px-4 py-3 rounded-lg dark:bg-red-900/30 dark:border-red-800 dark:text-red-400 flex items-start justify-between">
                            <div className="flex items-center gap-2">
                                <AlertCircle className="h-5 w-5 text-red-500" />
                                <p>{error}</p>
                            </div>
                            <button
                                onClick={() => setShowError(false)}
                                className="text-red-500 hover:text-red-700 dark:hover:text-red-300"
                            >
                                &times;
                            </button>
                        </div>
                    )}

                    {/* Two-column layout for desktop */}
                    <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-6">
                        {/* Main content column */}
                        <div className="lg:col-span-2 space-y-6">
                            {/* Compact Order Statistics */}
                            <SectionContainer
                                title="Order Statistics"
                                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                            >
                                <div className="grid grid-cols-2 md:grid-cols-4 lg:grid-cols-5 gap-3">
                                    <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-3 text-center hover:shadow-md transition-shadow cursor-pointer" onClick={() => window.location.href = '/orders'}>
                                        <div className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                                            {orderStats?.total ?? 0}
                                        </div>
                                        <div className="text-xs text-gray-600 dark:text-gray-400 mt-1">Total Orders</div>
                                    </div>

                                    <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-3 text-center hover:shadow-md transition-shadow cursor-pointer" onClick={() => window.location.href = '/orders?status=completed'}>
                                        <div className="text-2xl font-bold text-green-600 dark:text-green-400">
                                            {orderStats?.completed ?? 0}
                                        </div>
                                        <div className="text-xs text-gray-600 dark:text-gray-400 mt-1">Completed</div>
                                    </div>

                                    <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-3 text-center hover:shadow-md transition-shadow cursor-pointer" onClick={() => window.location.href = '/orders?status=pending'}>
                                        <div className="text-2xl font-bold text-orange-600 dark:text-orange-400">
                                            {orderStats?.pending ?? 0}
                                        </div>
                                        <div className="text-xs text-gray-600 dark:text-gray-400 mt-1">Pending</div>
                                    </div>

                                    <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-3 text-center hover:shadow-md transition-shadow cursor-pointer" onClick={() => window.location.href = '/billing'}>
                                        <div className="text-xl font-bold text-purple-600 dark:text-purple-400">
                                            ${parseFloat(orderStats?.totalSpent ?? 0).toFixed(0)}
                                        </div>
                                        <div className="text-xs text-gray-600 dark:text-gray-400 mt-1">Total Spent</div>
                                    </div>

                                    {pendingBids && pendingBids.total_bids > 0 && (
                                        <div className="bg-white dark:bg-gray-700 rounded-lg border border-orange-200 dark:border-orange-800 p-3 text-center hover:shadow-md transition-shadow cursor-pointer" onClick={() => window.location.href = '/client/bids'}>
                                            <div className="text-2xl font-bold text-orange-600 dark:text-orange-400">
                                                {pendingBids.total_bids}
                                            </div>
                                            <div className="text-xs text-gray-600 dark:text-gray-400 mt-1">Pending Bids</div>
                                        </div>
                                    )}
                                </div>
                            </SectionContainer>

                            {/* Order Activity Chart - NEW
                            <SectionContainer className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600">
                                <div className="flex items-center justify-between mb-4">
                                    <h2 className="text-lg font-bold text-gray-800 dark:text-gray-300">
                                        Order Activity
                                    </h2>
                                    <div className="bg-indigo-100 dark:bg-indigo-900/30 text-indigo-700 dark:text-indigo-300 px-2 py-1 rounded text-xs font-medium">
                                        Last 6 Months
                                    </div>
                                </div>

                                <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-4">
                                    <SimpleBarChart
                                        data={orderActivityData}
                                        maxValue={maxOrderActivity}
                                        barColor="bg-indigo-500"
                                    />

                                    <div className="mt-4 grid grid-cols-2 gap-2">
                                        <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-3">
                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                Completion Rate
                                            </div>
                                            <div className="flex items-center gap-2 mt-1">
                                                <CheckCircle
                                                    size={16}
                                                    className="text-green-500"
                                                />
                                                <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                    {orderStats?.completed &&
                                                    orderStats?.total
                                                        ? `${Math.round(
                                                              (orderStats.completed /
                                                                  orderStats.total) *
                                                                  100
                                                          )}%`
                                                        : "N/A"}
                                                </span>
                                            </div>
                                        </div>

                                        <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-3">
                                            <div className="text-xs text-gray-500 dark:text-gray-400">
                                                Avg. Order Value
                                            </div>
                                            <div className="flex items-center gap-2 mt-1">
                                                <TrendingUp
                                                    size={16}
                                                    className="text-indigo-500"
                                                />
                                                <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                    {orderStats?.totalSpent &&
                                                    orderStats?.total
                                                        ? `$${(
                                                              orderStats.totalSpent /
                                                              orderStats.total
                                                          ).toFixed(2)}`
                                                        : "$0.00"}
                                                </span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </SectionContainer> */}

                            {/* Order Activity Chart - Collapsible */}
                            <SectionContainer 
                                title="Order Activity"
                                defaultExpanded={orderActivityExpanded}
                                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                            >
                                {orderActivityExpanded && (
                                    <>
                                        <div className="flex items-center justify-end mb-4">
                                            <div className="bg-indigo-100 dark:bg-indigo-900/30 text-indigo-700 dark:text-indigo-300 px-2 py-1 rounded text-xs font-medium">
                                                Last 6 Months
                                            </div>
                                        </div>

                                        <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-4">
                                            {chartData && chartData.length > 0 ? (
                                                <SimpleBarChart
                                                    data={chartData}
                                                    maxValue={maxOrderActivity || 1}
                                                    barColor="bg-indigo-500"
                                                />
                                            ) : (
                                                <div className="flex justify-center items-center h-24 bg-gray-50 dark:bg-gray-800 rounded-md">
                                                    <p className="text-sm text-gray-500 dark:text-gray-400">
                                                        No order activity data available
                                                    </p>
                                                </div>
                                            )}

                                            <div className="mt-4 grid grid-cols-2 gap-2">
                                                <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-3">
                                                    <div className="text-xs text-gray-500 dark:text-gray-400">
                                                        Completion Rate
                                                    </div>
                                                    <div className="flex items-center gap-2 mt-1">
                                                        <CheckCircle
                                                            size={16}
                                                            className="text-green-500"
                                                        />
                                                        <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                            {calculateCompletionRate()}%
                                                        </span>
                                                    </div>
                                                </div>

                                                <div className="bg-gray-50 dark:bg-gray-800 rounded-lg p-3">
                                                    <div className="text-xs text-gray-500 dark:text-gray-400">
                                                        Avg. Order Value
                                                    </div>
                                                    <div className="flex items-center gap-2 mt-1">
                                                        <TrendingUp
                                                            size={16}
                                                            className="text-indigo-500"
                                                        />
                                                        <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                            ${calculateAvgOrderValue()}
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </>
                                )}
                                {!orderActivityExpanded && (
                                    <div className="text-center py-4">
                                        <button
                                            onClick={() => setOrderActivityExpanded(true)}
                                            className="text-sm text-indigo-600 hover:text-indigo-800 dark:text-indigo-400 dark:hover:text-indigo-300 font-medium"
                                        >
                                            Click to view order activity chart
                                        </button>
                                    </div>
                                )}
                            </SectionContainer>

                            {/* Recent Orders - Compact List */}
                            <SectionContainer
                                title="Recent Orders"
                                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                            >
                                <div className="flex items-center justify-end mb-3">
                                    <Link
                                        href="/orders"
                                        className="text-sm text-indigo-600 hover:text-indigo-800 dark:text-indigo-400 dark:hover:text-indigo-300 flex items-center gap-1"
                                    >
                                        View all
                                        <ChevronRight size={14} />
                                    </Link>
                                </div>

                                {recentOrders && recentOrders.length > 0 ? (
                                    <div className="space-y-2">
                                        {recentOrders.map((order) => (
                                            <div
                                                key={order.id}
                                                className={`bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-3 hover:shadow-md transition-all duration-200 ${expandedOrderId === order.id ? 'shadow-md' : ''}`}
                                            >
                                                <div className="flex items-center justify-between gap-3">
                                                    <div className="flex-1 min-w-0">
                                                        <div className="flex items-center gap-2 mb-1.5 flex-wrap">
                                                            <Link
                                                                href={`/orders/${order.id}`}
                                                                className="font-medium text-gray-900 dark:text-white hover:text-indigo-600 dark:hover:text-indigo-400 truncate"
                                                            >
                                                                {order.title || `Order #${order.order_number}`}
                                                            </Link>
                                                            <OrderStatusBadge
                                                                status={order.status}
                                                            />
                                                            <PaymentStatusBadge
                                                                status={order.payment_status}
                                                            />
                                                        </div>
                                                        
                                                        {expandedOrderId === order.id ? (
                                                            <div className="mt-2 pt-2 border-t border-gray-200 dark:border-gray-600 space-y-1.5">
                                                                <div className="flex flex-wrap gap-x-4 gap-y-1 text-xs text-gray-600 dark:text-gray-400">
                                                                    <div className="flex items-center gap-1">
                                                                        <FileText size={12} />
                                                                        <span>{order.type_of_paper || "Paper"}</span>
                                                                    </div>
                                                                    <div className="flex items-center gap-1">
                                                                        <Calendar size={12} />
                                                                        <span>{order.deadline || "No deadline"}</span>
                                                                    </div>
                                                                    <div className="flex items-center gap-1">
                                                                        <DollarSign size={12} />
                                                                        <span>${order.net_amount || 0}</span>
                                                                    </div>
                                                                </div>
                                                                <div className="flex items-center gap-2">
                                                                    <DashboardDeadlineCountdown
                                                                        deadline={
                                                                            order.deadline_timestamp
                                                                                ? new Date(
                                                                                      order.deadline_timestamp * 1000
                                                                                  ).toISOString()
                                                                                : null
                                                                        }
                                                                        isLate={order.is_late}
                                                                    />
                                                                </div>
                                                                <div className="flex flex-wrap gap-2 pt-1">
                                                                    <Link
                                                                        href={`/orders/${order.id}`}
                                                                        className="inline-flex items-center px-2.5 py-1 bg-indigo-100 hover:bg-indigo-200 dark:bg-indigo-900/30 dark:hover:bg-indigo-800/50 text-indigo-700 dark:text-indigo-300 text-xs font-medium rounded transition-colors"
                                                                    >
                                                                        View Details
                                                                    </Link>
                                                                    {order.payment_status === "unpaid" && (
                                                                        <Link
                                                                            href={`/orders/${order.id}/payment`}
                                                                            className="inline-flex items-center px-2.5 py-1 bg-green-100 hover:bg-green-200 dark:bg-green-900/30 dark:hover:bg-green-800/50 text-green-700 dark:text-green-300 text-xs font-medium rounded transition-colors"
                                                                        >
                                                                            Pay Now
                                                                        </Link>
                                                                    )}
                                                                    {order.status === "bidding" && order.bids_count > 0 && (
                                                                        <Link
                                                                            href={`/orders/${order.id}#bids`}
                                                                            className="inline-flex items-center px-2.5 py-1 bg-purple-100 hover:bg-purple-200 dark:bg-purple-900/30 dark:hover:bg-purple-800/50 text-purple-700 dark:text-purple-300 text-xs font-medium rounded transition-colors"
                                                                        >
                                                                            View Bids ({order.bids_count})
                                                                        </Link>
                                                                    )}
                                                                </div>
                                                            </div>
                                                        ) : (
                                                            <div className="flex items-center gap-3 text-xs text-gray-600 dark:text-gray-400">
                                                                <span>{order.type_of_paper || "Paper"}</span>
                                                                <span>•</span>
                                                                <span>${order.net_amount || 0}</span>
                                                                {order.deadline && (
                                                    <>
                                                                        <span>•</span>
                                                                        <DashboardDeadlineCountdown
                                                                            deadline={
                                                                                order.deadline_timestamp
                                                                                    ? new Date(
                                                                                          order.deadline_timestamp * 1000
                                                                                      ).toISOString()
                                                                                    : null
                                                                            }
                                                                            isLate={order.is_late}
                                                                        />
                                                                    </>
                                                                )}
                                                            </div>
                                                        )}
                                                    </div>
                                                    <div className="flex items-center gap-2 flex-shrink-0">
                                                        {expandedOrderId === order.id ? (
                                                            <button
                                                                onClick={() => setExpandedOrderId(null)}
                                                                className="p-1.5 text-gray-500 hover:text-gray-700 dark:hover:text-gray-300 rounded transition-colors"
                                                                title="Collapse"
                                                            >
                                                                <ChevronRight size={16} className="rotate-90" />
                                                            </button>
                                                        ) : (
                                                            <button
                                                                onClick={() => setExpandedOrderId(order.id)}
                                                                className="p-1.5 text-gray-500 hover:text-gray-700 dark:hover:text-gray-300 rounded transition-colors"
                                                                title="Expand"
                                                            >
                                                                <ChevronRight size={16} />
                                                            </button>
                                                        )}
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                ) : (
                                    <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-6 text-center">
                                        <div className="flex justify-center mb-4">
                                            <div className="bg-gray-100 dark:bg-gray-600 p-3 rounded-full">
                                                <FileText
                                                    size={24}
                                                    className="text-gray-500 dark:text-gray-400"
                                                />
                                            </div>
                                        </div>
                                        <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                                            No orders yet
                                        </h3>
                                        <p className="text-gray-500 dark:text-gray-400 mb-4">
                                            You haven't placed any orders yet.
                                            Start by creating your first order.
                                        </p>
                                        <Link
                                            href="/place-order"
                                            className="inline-flex items-center px-4 py-2 bg-indigo-600 hover:bg-indigo-700 text-white font-medium rounded-lg transition-colors duration-200 gap-2"
                                        >
                                            <PlusCircle size={18} />
                                            <span>Place Your First Order</span>
                                        </Link>
                                    </div>
                                )}
                            </SectionContainer>

                        </div>

                        {/* Sidebar column - Optimized */}
                        <div className="space-y-4">
                            {/* Compact Financial Summary */}
                            <SectionContainer 
                                title="Financial Summary"
                                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                            >
                                <div className="flex items-center justify-end mb-3">
                                    <Link
                                        href="/billing"
                                        className="text-xs text-indigo-600 hover:text-indigo-800 dark:text-indigo-400 dark:hover:text-indigo-300 flex items-center gap-1"
                                    >
                                        Manage
                                        <ChevronRight size={12} />
                                    </Link>
                                </div>

                                <div className="space-y-2">
                                    <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-3">
                                        <div className="flex items-center justify-between">
                                            <div>
                                                <div className="text-xs text-gray-500 dark:text-gray-400">Total Spent</div>
                                                <div className="text-lg font-bold text-gray-900 dark:text-white mt-0.5">
                                                    ${parseFloat(financialSummary?.recent_payments?.total_spent || orderStats?.totalSpent || 0).toFixed(2)}
                                                </div>
                                            </div>
                                            <Wallet size={20} className="text-indigo-600 dark:text-indigo-400" />
                                        </div>
                                    </div>

                                    {financialSummary?.recent_payments?.last_transaction && (
                                        <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-2.5">
                                            <div className="text-xs text-gray-500 dark:text-gray-400 mb-1">Last Transaction</div>
                                            <div className="flex items-center justify-between">
                                                <div className="text-sm font-medium text-gray-900 dark:text-white">
                                                    ${financialSummary.recent_payments.last_transaction.amount}
                                                </div>
                                                <div className="text-xs text-gray-500 dark:text-gray-400">
                                                    {financialSummary.recent_payments.last_transaction.date}
                                                </div>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </SectionContainer>

                            {/* Compact Quick Actions */}
                            <SectionContainer 
                                title="Quick Actions"
                                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                            >
                                <div className="grid grid-cols-2 gap-2">
                                    <Link
                                        href="/place-order"
                                        className="bg-white dark:bg-gray-700 rounded-lg shadow-sm p-2.5 border border-gray-200 dark:border-gray-600 hover:shadow-md transition-all duration-200 flex flex-col items-center justify-center text-center gap-1.5"
                                        title="Place New Order"
                                    >
                                        <PlusCircle size={18} className="text-indigo-600 dark:text-indigo-400" />
                                        <span className="font-medium text-xs text-gray-900 dark:text-white">New Order</span>
                                    </Link>
                                    <Link
                                        href="/orders"
                                        className="bg-white dark:bg-gray-700 rounded-lg shadow-sm p-2.5 border border-gray-200 dark:border-gray-600 hover:shadow-md transition-all duration-200 flex flex-col items-center justify-center text-center gap-1.5"
                                        title="My Orders"
                                    >
                                        <ListOrdered size={18} className="text-blue-600 dark:text-blue-400" />
                                        <span className="font-medium text-xs text-gray-900 dark:text-white">My Orders</span>
                                    </Link>
                                    <Link
                                        href="/orders"
                                        className="bg-white dark:bg-gray-700 rounded-lg shadow-sm p-2.5 border border-gray-200 dark:border-gray-600 hover:shadow-md transition-all duration-200 flex flex-col items-center justify-center text-center gap-1.5"
                                        title="Upload Files"
                                    >
                                        <Upload size={18} className="text-green-600 dark:text-green-400" />
                                        <span className="font-medium text-xs text-gray-900 dark:text-white">Upload Files</span>
                                    </Link>
                                    <Link
                                        href="/support"
                                        className="bg-white dark:bg-gray-700 rounded-lg shadow-sm p-2.5 border border-gray-200 dark:border-gray-600 hover:shadow-md transition-all duration-200 flex flex-col items-center justify-center text-center gap-1.5"
                                        title="Get Help"
                                    >
                                        <HelpCircle size={18} className="text-purple-600 dark:text-purple-400" />
                                        <span className="font-medium text-xs text-gray-900 dark:text-white">Get Help</span>
                                    </Link>
                                </div>
                            </SectionContainer>

                            {/* Messages - Compact */}
                            <SectionContainer 
                                title="Messages"
                                className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                            >
                                <div className="flex items-center justify-end mb-3">
                                    <Link
                                        href="/messages"
                                        className="text-xs text-indigo-600 hover:text-indigo-800 dark:text-indigo-400 dark:hover:text-indigo-300 flex items-center gap-1"
                                    >
                                        View all
                                        <ChevronRight size={12} />
                                    </Link>
                                </div>
                                <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-4 text-center">
                                    <MessageCircle size={32} className="mx-auto mb-2 text-gray-400 dark:text-gray-500" />
                                    <p className="text-sm text-gray-600 dark:text-gray-400">No new messages</p>
                                </div>
                            </SectionContainer>

                            {/* Available Coupons */}
                            {availableCoupons &&
                                availableCoupons.length > 0 && (
                                    <SectionContainer 
                                        title="Available Coupons"
                                        className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                                    >
                                        <div className="flex items-center justify-end mb-3">
                                            <span className="bg-indigo-600 text-white text-xs font-bold px-2 py-1 rounded-full">
                                                {availableCoupons.length} Available
                                            </span>
                                        </div>

                                        <div className="space-y-2">
                                            {(showAllCoupons ? availableCoupons : availableCoupons.slice(0, 2))
                                                .map((coupon) => (
                                                    <div
                                                        key={coupon.id || coupon.code}
                                                        className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-2.5 shadow-sm hover:shadow-md transition-shadow duration-200"
                                                    >
                                                        <div className="flex justify-between items-start">
                                                            <div className="flex-1 min-w-0">
                                                                <div className="font-medium text-sm text-gray-900 dark:text-white mb-0.5 truncate">
                                                                    {coupon.code}
                                                                </div>
                                                                <div className="text-xs text-gray-600 dark:text-gray-400 line-clamp-2">
                                                                    {coupon.description}
                                                                </div>
                                                            </div>
                                                            <div className="bg-indigo-100 dark:bg-indigo-900/40 text-indigo-700 dark:text-indigo-300 px-2 py-0.5 rounded text-xs font-medium ml-2 flex-shrink-0">
                                                                {coupon.discount_type === "percentage"
                                                                    ? `${coupon.discount_amount}%`
                                                                    : `$${coupon.discount_amount}`}
                                                            </div>
                                                        </div>
                                                    </div>
                                                ))}

                                            {availableCoupons.length > 2 && (
                                                <button
                                                    onClick={() => setShowAllCoupons(!showAllCoupons)}
                                                    className="w-full text-center text-xs text-indigo-600 dark:text-indigo-400 hover:text-indigo-800 dark:hover:text-indigo-300 font-medium pt-1 pb-0.5 transition-colors cursor-pointer"
                                                >
                                                    {showAllCoupons ? (
                                                        <span>Show less</span>
                                                    ) : (
                                                        <span>
                                                            + {availableCoupons.length - 2} more coupon(s) available
                                                        </span>
                                                    )}
                                                </button>
                                            )}

                                            <div className="flex justify-end mt-2">
                                                <Link
                                                    href="/place-order"
                                                    className="inline-flex items-center px-3 py-1.5 bg-indigo-100 hover:bg-indigo-200 dark:bg-indigo-900/30 dark:hover:bg-indigo-800/50 text-indigo-700 dark:text-indigo-300 text-xs font-medium rounded-lg transition-all duration-200 gap-1"
                                                >
                                                    <span>Use a Coupon</span>
                                                    <ChevronRight size={14} />
                                                </Link>
                                            </div>
                                        </div>
                                    </SectionContainer>
                                )}

                            {/* Preferred Writers - Compact */}
                            {preferredWriters &&
                                preferredWriters.length > 0 && (
                                    <SectionContainer 
                                        title="Preferred Writers"
                                        className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                                    >
                                        <div className="flex items-center justify-end mb-3">
                                            <Link
                                                href="/writers"
                                                className="text-xs text-indigo-600 hover:text-indigo-800 dark:text-indigo-400 dark:hover:text-indigo-300 flex items-center gap-1"
                                            >
                                                View all
                                                <ChevronRight size={12} />
                                            </Link>
                                        </div>

                                        <div className="space-y-2">
                                            {preferredWriters.slice(0, 2).map((writer) => (
                                                <div
                                                    key={writer.id}
                                                    className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-2.5 shadow-sm hover:shadow-md transition-shadow duration-200"
                                                >
                                                    <div className="flex items-start gap-2">
                                                        <div className="bg-indigo-100 dark:bg-indigo-900/30 p-1.5 rounded-full flex-shrink-0">
                                                            <User size={16} className="text-indigo-600 dark:text-indigo-400" />
                                                        </div>
                                                        <div className="flex-1 min-w-0">
                                                            <div className="font-medium text-sm text-gray-900 dark:text-white truncate">
                                                                {writer.nickname}
                                                            </div>
                                                            <div className="text-xs bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 px-1.5 py-0.5 rounded-full inline-block mt-1">
                                                                {writer.completedOrders} Orders
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            ))}
                                        </div>
                                    </SectionContainer>
                                )}

                            {/* Loyalty Status - Compact */}
                            {loyaltyInfo && (
                                <SectionContainer 
                                    title="Loyalty Status"
                                    className="bg-gradient-to-r from-blue-100 to-cyan-100 dark:bg-none dark:bg-gray-800 border border-gray-300 dark:border-gray-600"
                                >
                                    <div className="bg-white dark:bg-gray-700 rounded-lg border border-gray-200 dark:border-gray-600 p-3">
                                        <div className="flex items-center justify-between mb-2">
                                            <div className="flex items-center gap-1.5 text-amber-500">
                                                <Award size={16} />
                                                <span className="text-sm font-medium">{loyaltyInfo.tier}</span>
                                            </div>
                                        </div>
                                        {loyaltyInfo.next_tier && (
                                            <div className="mb-2">
                                                <div className="flex justify-between items-center mb-1">
                                                    <span className="text-xs text-gray-600 dark:text-gray-400">Progress</span>
                                                    <span className="text-xs font-medium text-gray-900 dark:text-white">
                                                        {loyaltyInfo.completedOrders}/{loyaltyInfo.nextTierOrders}
                                                    </span>
                                                </div>
                                                <div className="w-full bg-gray-200 dark:bg-gray-600 rounded-full h-1.5">
                                                    <div
                                                        className="bg-indigo-500 h-1.5 rounded-full"
                                                        style={{
                                                            width: `${Math.min(100, (loyaltyInfo.completedOrders / loyaltyInfo.nextTierOrders) * 100)}%`,
                                                        }}
                                                    ></div>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                </SectionContainer>
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
}
