import { Head } from "@inertiajs/react";
import { useMemo } from "react";
import MainLayout from "@/Layouts/MainLayout";
import GuaranteeHero from "@/Components/Guarantees/GuaranteeHero";
import GuaranteesGrid from "@/Components/Guarantees/GuaranteesGrid";
import QualityStandards from "@/Components/Guarantees/QualityStandards";
import GuaranteeProcess from "@/Components/Guarantees/GuaranteeProcess";
import ServiceGuarantees from "@/Components/Guarantees/ServiceGuarantees";
import TrustIndicators from "@/Components/Guarantees/TrustIndicators";
import GuaranteeComparison from "@/Components/Guarantees/GuaranteeComparison";
import ContactSupport from "@/Components/Guarantees/ContactSupport";
import ServiceLinksFooter from "@/Components/Guarantees/ServiceLinksFooter";
import ServiceTestimonials from "@/Components/Services/ServiceTestimonials";
import ServiceFAQs from "@/Components/Services/ServiceFAQs";
import ServiceCTA from "@/Components/Services/ServiceCTA";
import PriceCalculatorSection from "@/Components/CMS/PriceCalculatorSection";

export default function Guarantees(props) {
    const { seoData, contentBlocks, structuredData, isCMS } = props;

    // Sort content blocks by order field for dynamic rendering
    const sortedBlocks = useMemo(() => {
        if (!isCMS || !contentBlocks) return [];
        return [...contentBlocks].sort((a, b) => (a.order || 0) - (b.order || 0));
    }, [contentBlocks, isCMS]);

    // Render individual block based on block_type
    const renderBlock = (block) => {
        const content = block.content;
        
        switch (block.block_type) {
            case 'guarantees_hero':
                return (
                    <section key={block.id} className="bg-gradient-to-r from-green-600 to-blue-600 py-16 text-white">
                        <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                            <h1 className="text-4xl md:text-5xl font-bold mb-6">
                                {content?.title || fallbackHero.title}
                            </h1>
                            <p className="text-xl md:text-2xl mb-8 max-w-3xl mx-auto">
                                {content?.description || fallbackHero.description}
                            </p>
                            <div className="flex flex-col sm:flex-row gap-4 justify-center">
                                <a
                                    href={content?.primaryButtonLink || fallbackHero.primaryButtonLink}
                                    className="bg-white text-green-600 hover:bg-gray-100 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                                >
                                    {content?.primaryButtonText || fallbackHero.primaryButtonText}
                                </a>
                                <a
                                    href={content?.secondaryButtonLink || fallbackHero.secondaryButtonLink}
                                    className="border-2 border-white text-white hover:bg-white hover:text-green-600 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                                >
                                    {content?.secondaryButtonText || fallbackHero.secondaryButtonText}
                                </a>
                            </div>
                        </div>
                    </section>
                );
            case 'main_guarantees':
                return <GuaranteesGrid key={block.id} title={content?.title || fallbackGuarantees.title} description={content?.description || fallbackGuarantees.description} guarantees={content?.guarantees || fallbackGuarantees.guarantees} />;
            case 'quality_standards':
                return <QualityStandards key={block.id} title={content?.title} description={content?.description} standards={content?.standards || []} />;
            case 'guarantee_process':
                return <GuaranteeProcess key={block.id} title={content?.title} description={content?.description} steps={content?.steps || []} />;
            case 'service_guarantees':
                return <ServiceGuarantees key={block.id} title={content?.title} description={content?.description} services={content?.services || []} />;
            case 'guarantees_testimonials':
                return <ServiceTestimonials key={block.id} title={content?.title} description={content?.description} testimonials={content?.testimonials || []} />;
            case 'guarantees_faqs':
                return <ServiceFAQs key={block.id} title={content?.title} description={content?.description} faqs={content?.faqs || []} />;
            case 'trust_indicators':
                return <TrustIndicators key={block.id} title={content?.title} description={content?.description} indicators={content?.indicators || []} />;
            case 'guarantee_comparison':
                return <GuaranteeComparison key={block.id} title={content?.title} description={content?.description} features={content?.features || []} ctaText={content?.ctaText} ctaLink={content?.ctaLink} />;
            case 'guarantees_cta':
                return <ServiceCTA key={block.id} title={content?.title} description={content?.description} primaryButtonText={content?.primaryButtonText} primaryButtonLink={content?.primaryButtonLink} secondaryButtonText={content?.secondaryButtonText} secondaryButtonLink={content?.secondaryButtonLink} bgColor="gradient" />;
            case 'contact_support':
                return <ContactSupport key={block.id} title={content?.title} description={content?.description} contacts={content?.contacts || []} notice={content?.notice} />;
            case 'service_links':
                return <ServiceLinksFooter key={block.id} title={content?.title} description={content?.description} services={content?.services || []} />;
            case 'price_calculator':
                return <PriceCalculatorSection key={block.id} content={content} />;
            default:
                return null;
        }
    };

    // Extract content from CMS blocks (for fallback compatibility)
    const getContentBlock = (blockKey) => {
        if (!isCMS || !contentBlocks) return null;
        const block = contentBlocks.find((b) => b.block_key === blockKey);
        return block ? block.content : null;
    };

    // Get all content blocks
    const hero = getContentBlock('guarantees_hero');
    const mainGuarantees = getContentBlock('main_guarantees');
    const qualityStandards = getContentBlock('quality_standards');
    const guaranteeProcess = getContentBlock('guarantee_process');
    const serviceGuarantees = getContentBlock('service_guarantees');
    const testimonials = getContentBlock('guarantees_testimonials');
    const faqs = getContentBlock('guarantees_faqs');
    const trustIndicators = getContentBlock('trust_indicators');
    const comparison = getContentBlock('guarantee_comparison');
    const contactSupport = getContentBlock('contact_support');
    const serviceLinks = getContentBlock('service_links');
    const cta = getContentBlock('guarantees_cta');

    // FALLBACK DATA (exact copy from original for SEO if CMS not available)
    const fallbackHero = {
        title: 'Quality Guarantees You Can Trust',
        subtitle: '',
        description: '100% plagiarism-free guarantee, money-back guarantee, free unlimited revisions, and on-time delivery for all essay writing, research papers, and dissertation services.',
        primaryButtonText: 'Order Risk-Free Now',
        primaryButtonLink: '/place-order',
        secondaryButtonText: 'View All Services',
        secondaryButtonLink: '/services',
    };

    const fallbackGuarantees = {
        title: 'Our Comprehensive Quality Guarantees',
        description: 'Risk-free academic writing services with multiple guarantees for your peace of mind',
        guarantees: [
            {
                icon: "🔒",
                title: "100% Plagiarism-Free Guarantee",
                description: "Every paper is written from scratch with original content. We provide free plagiarism reports with every order.",
                features: [
                    "Original content written from scratch",
                    "Free Turnitin plagiarism report",
                    "Multiple plagiarism detection tools",
                    "Zero tolerance for copied content",
                ],
                link: "/services/essay-writing",
                linkText: "Get Original Essays",
            },
            {
                icon: "💰",
                title: "Money-Back Guarantee",
                description: "Not satisfied with your paper? Get your money back. We offer full refunds for unsatisfactory work.",
                features: [
                    "Full refund if not satisfied",
                    "Partial refunds for late delivery",
                    "No questions asked policy",
                    "Quick refund processing",
                ],
                link: "/services/research-papers",
                linkText: "Risk-Free Research Papers",
            },
            {
                icon: "🔄",
                title: "Free Unlimited Revisions",
                description: "We revise your paper until you're completely satisfied. Free revisions within 30 days of delivery.",
                features: [
                    "Unlimited revisions for 30 days",
                    "Quick revision turnaround",
                    "No additional charges",
                    "Direct communication with writer",
                ],
                link: "/services/assignments",
                linkText: "Get Perfect Assignments",
            },
            {
                icon: "⏰",
                title: "On-Time Delivery Guarantee",
                description: "We deliver your paper on time, every time. Late delivery results in automatic compensation.",
                features: [
                    "Guaranteed on-time delivery",
                    "Compensation for late delivery",
                    "Progress tracking available",
                    "Rush orders accepted",
                ],
                link: "/services/dissertation-writing",
                linkText: "Timely Dissertation Help",
            },
            {
                icon: "🎓",
                title: "Expert Writer Guarantee",
                description: "Your paper is written by qualified experts in your field with advanced degrees and experience.",
                features: [
                    "PhD and Masters level writers",
                    "Subject matter experts",
                    "Native English speakers",
                    "Verified academic credentials",
                ],
                link: "/services/thesis-proposals",
                linkText: "Expert Thesis Writing",
            },
            {
                icon: "🔐",
                title: "Privacy & Confidentiality",
                description: "Your personal information and order details are kept strictly confidential and secure.",
                features: [
                    "SSL encrypted transactions",
                    "Anonymous communication",
                    "No data sharing with third parties",
                    "Secure payment processing",
                ],
                link: "/services/case-studies",
                linkText: "Confidential Case Studies",
            },
        ]
    };

    return (
        <MainLayout>
            <Head>
                <title>{seoData?.title || 'Quality Guarantees | 100% Plagiarism-Free, Money-Back, Free Revisions | Academic Scribe'}</title>
                <meta name="description" content={seoData?.description || 'Quality guarantees for essay writing, research papers, dissertations. 100% plagiarism-free guarantee, money-back guarantee, free unlimited revisions, on-time delivery, expert writers, privacy protection.'} />
                <meta name="keywords" content={seoData?.meta_keywords || 'plagiarism-free guarantee, money-back guarantee, free revisions guarantee, on-time delivery guarantee, expert writer guarantee, privacy guarantee, quality assurance, academic writing guarantees, essay writing guarantee, research paper guarantee, dissertation guarantee'} />
                <link rel="canonical" href={seoData?.canonicalUrl || 'https://academicscribe.com/guarantees'} />
                <meta property="og:title" content={seoData?.ogTitle || 'Quality Guarantees | 100% Plagiarism-Free, Money-Back, Free Revisions'} />
                <meta property="og:description" content={seoData?.ogDescription || 'Comprehensive quality guarantees for all academic writing services. Plagiarism-free, money-back, free revisions, on-time delivery guaranteed.'} />
                <meta property="og:type" content="website" />
                <meta property="og:url" content={seoData?.ogUrl || 'https://academicscribe.com/guarantees'} />
                <meta property="og:image" content="/images/quality-guarantees.jpg" />
                <meta name="twitter:card" content="summary_large_image" />
                <meta name="twitter:title" content={seoData?.twitter_title || 'Quality Guarantees | 100% Plagiarism-Free, Money-Back, Free Revisions'} />
                <meta name="twitter:description" content={seoData?.twitter_description || 'Comprehensive quality guarantees for academic writing services. Risk-free ordering with multiple guarantees.'} />
                <meta name="twitter:image" content="/images/quality-guarantees.jpg" />
                
                {/* Structured Data */}
                {structuredData && (
                    <script type="application/ld+json">
                        {JSON.stringify(structuredData)}
                    </script>
                )}
            </Head>

            <div className="guarantees-page">
                {isCMS && sortedBlocks.length > 0 ? (
                    sortedBlocks.map((block) => renderBlock(block))
                ) : (
                    <>
                        {/* Fallback: Static hero section */}
                        <section className="bg-gradient-to-r from-green-600 to-blue-600 py-16 text-white">
                            <div className="max-w-6xl mx-auto px-4 sm:px-6 lg:px-8 text-center">
                                <h1 className="text-4xl md:text-5xl font-bold mb-6">
                                    {fallbackHero.title}
                                </h1>
                                <p className="text-xl md:text-2xl mb-8 max-w-3xl mx-auto">
                                    {fallbackHero.description}
                                </p>
                                <div className="flex flex-col sm:flex-row gap-4 justify-center">
                                    <a
                                        href={fallbackHero.primaryButtonLink}
                                        className="bg-white text-green-600 hover:bg-gray-100 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                                    >
                                        {fallbackHero.primaryButtonText}
                                    </a>
                                    <a
                                        href={fallbackHero.secondaryButtonLink}
                                        className="border-2 border-white text-white hover:bg-white hover:text-green-600 px-8 py-4 rounded-lg font-bold text-lg transition-colors"
                                    >
                                        {fallbackHero.secondaryButtonText}
                                    </a>
                                </div>
                            </div>
                        </section>

                        {/* Fallback: Main Guarantees */}
                        <GuaranteesGrid 
                            title={fallbackGuarantees.title}
                            description={fallbackGuarantees.description}
                            guarantees={fallbackGuarantees.guarantees}
                        />
                    </>
                )}
            </div>

            {/* 13. Mobile Bottom CTA (kept from original for UX) */}
            <div className="fixed bottom-0 left-0 right-0 bg-white border-t border-gray-200 p-4 lg:hidden z-50">
                <div className="flex justify-between items-center">
                    <div className="text-center">
                        <div className="text-sm text-gray-600">All Guarantees</div>
                        <div className="text-lg font-bold text-green-600">Included Free</div>
                    </div>
                    <div className="flex space-x-2">
                        <a
                            href="/services"
                            className="border-green-600 text-green-600 hover:bg-green-600 hover:text-white px-4 py-2 rounded-lg font-semibold transition-colors border"
                        >
                            View Services
                        </a>
                        <a
                            href="/place-order"
                            className="bg-green-600 hover:bg-green-700 text-white px-6 py-2 rounded-lg font-semibold transition-colors"
                        >
                            Order Now
                        </a>
                    </div>
                </div>
            </div>
        </MainLayout>
    );
}
