import React from "react";
import { Head, Link } from "@inertiajs/react";
import { formatDistanceToNow } from "date-fns";
import ClientLayout from "@/Layouts/ClientLayout";
import WriterLayout from "@/Layouts/WriterLayout";
import AdminLayout from "@/Layouts/AdminLayout";
import {
    Bell,
    ArrowLeft,
    ExternalLink,
    Clock,
    FileText,
    DollarSign,
    User,
    CheckCircle,
    AlertCircle,
    Info,
} from "lucide-react";

export default function Show({ auth, notification }) {
    // Parse notification data
    const data =
        typeof notification.data === "string"
            ? JSON.parse(notification.data)
            : notification.data;

    // Format the notification time
    const getFormattedTime = () => {
        try {
            return formatDistanceToNow(new Date(notification.created_at), {
                addSuffix: true,
            });
        } catch (error) {
            return "recently";
        }
    };

    // Get icon based on notification type
    const getIcon = () => {
        const iconName = data?.icon || "bell";

        switch (iconName) {
            case "check-circle":
                return <CheckCircle className="h-6 w-6 text-green-500" />;
            case "alert-circle":
                return <AlertCircle className="h-6 w-6 text-red-500" />;
            case "info":
                return <Info className="h-6 w-6 text-blue-500" />;
            case "file-text":
                return <FileText className="h-6 w-6 text-gray-500" />;
            case "dollar-sign":
                return <DollarSign className="h-6 w-6 text-green-500" />;
            default:
                return <Bell className="h-6 w-6 text-blue-500" />;
        }
    };

    // Choose the appropriate layout based on user type
    const getLayout = () => {
        switch (auth.user.user_type) {
            case "admin":
                return AdminLayout;
            case "writer":
                return WriterLayout;
            default:
                return ClientLayout;
        }
    };

    const getRouteUrl = (routeName, params = {}) => {
        try {
            if (route().has(routeName)) {
                const baseUrl = window.location.origin;
                const routePath = route(routeName, params).replace(
                    /^https?:\/\/[^/]+/,
                    ""
                );
                return `${baseUrl}${routePath}`;
            }
            return route(routeName, params);
        } catch (error) {
            console.error(`Error getting route URL for ${routeName}:`, error);
            return "#";
        }
    };

    const Layout = getLayout();

    return (
        <Layout>
            <Head title={data?.title || "Notification"} />

            <div className="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                {/* Header */}
                <div className="bg-gray-50 dark:bg-gray-900 px-4 py-5 border-b border-gray-200 dark:border-gray-700 sm:px-6">
                    <div className="flex items-center justify-between">
                        <div className="flex items-center gap-3">
                            <Link
                                href={route("notifications.index")}
                                className="text-gray-500 dark:text-gray-400 hover:text-gray-700 dark:hover:text-gray-300"
                            >
                                <ArrowLeft className="h-5 w-5" />
                            </Link>

                            <h3 className="text-lg leading-6 font-medium text-gray-900 dark:text-white">
                                {data?.title || "Notification"}
                            </h3>
                        </div>

                        <div className="text-sm text-gray-500 dark:text-gray-400 flex items-center gap-1">
                            <Clock className="h-4 w-4" />
                            <span>{getFormattedTime()}</span>
                        </div>
                    </div>
                </div>

                {/* Notification Content */}
                <div className="px-4 py-5 sm:p-6">
                    <div className="flex items-start gap-4">
                        <div className="flex-shrink-0">{getIcon()}</div>

                        <div className="flex-1">
                            <div className="text-sm text-gray-900 dark:text-gray-100">
                                <p className="mb-4 text-lg">
                                    {data?.message || "No message content"}
                                </p>

                                {/* Additional notification details */}
                                {data?.details && (
                                    <div className="mt-4 p-4 bg-gray-50 dark:bg-gray-750 rounded-md">
                                        <h4 className="font-medium mb-2 text-gray-900 dark:text-white">
                                            Details
                                        </h4>
                                        <p className="text-gray-600 dark:text-gray-300">
                                            {data.details}
                                        </p>
                                    </div>
                                )}

                                {/* Related Order Information */}
                                {notification.related_data?.order && (
                                    <div className="mt-6 p-4 border border-gray-200 dark:border-gray-700 rounded-md">
                                        <h4 className="font-medium mb-2 text-gray-900 dark:text-white flex items-center gap-2">
                                            <FileText className="h-4 w-4" />
                                            <span>Related Order</span>
                                        </h4>
                                        <div className="grid grid-cols-2 gap-3">
                                            <div>
                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                    Order Number
                                                </p>
                                                <p className="font-medium text-gray-900 dark:text-white">
                                                    #
                                                    {
                                                        notification
                                                            .related_data.order
                                                            .order_number
                                                    }
                                                </p>
                                            </div>
                                            <div>
                                                <p className="text-sm text-gray-500 dark:text-gray-400">
                                                    Status
                                                </p>
                                                <p className="font-medium text-gray-900 dark:text-white">
                                                    {
                                                        notification
                                                            .related_data.order
                                                            .status
                                                    }
                                                </p>
                                            </div>
                                        </div>
                                        <div className="mt-3 flex gap-4">
                                            {/* View Order */}
                                            <Link
                                                href={getRouteUrl(
                                                    notification.related_data
                                                        .order.routes.view,
                                                    notification.related_data
                                                        .order.id
                                                )}
                                                className="inline-flex items-center gap-1 text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                            >
                                                <span>View Order</span>
                                                <ExternalLink className="h-3.5 w-3.5" />
                                            </Link>

                                            {/* Payment */}
                                            {notification.related_data.order
                                                .payment_status !== "paid" && (
                                                <Link
                                                    href={getRouteUrl(
                                                        notification
                                                            .related_data.order
                                                            .routes.payment,
                                                        notification
                                                            .related_data.order
                                                            .id
                                                    )}
                                                    className="inline-flex items-center gap-1 text-green-600 dark:text-green-400 hover:text-green-800 dark:hover:text-green-300"
                                                >
                                                    <span>Make Payment</span>
                                                    <DollarSign className="h-3.5 w-3.5" />
                                                </Link>
                                            )}

                                            {/* Payment History */}
                                            <Link
                                                href={route(
                                                    notification.related_data
                                                        .order.routes
                                                        .payment_history,
                                                    notification.related_data
                                                        .order.id
                                                )}
                                                className="inline-flex items-center gap-1 text-purple-600 dark:text-purple-400 hover:text-purple-800 dark:hover:text-purple-300"
                                            >
                                                <span>Payment History</span>
                                                <Clock className="h-3.5 w-3.5" />
                                            </Link>
                                        </div>
                                    </div>
                                )}
                                {/* Related Payment Information */}
                                {notification.related_data?.payment && (
                                    <div className="mt-4 p-4 border border-gray-200 dark:border-gray-700 rounded-md">
                                        {/* Payment content as before */}
                                        {/* Add link to payment status */}
                                        <div className="mt-3">
                                            <Link
                                                href={route(
                                                    notification.related_data
                                                        .payment.routes.status,
                                                    notification.related_data
                                                        .payment.id
                                                )}
                                                className="inline-flex items-center gap-1 text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                            >
                                                <span>Check Status</span>
                                                <ExternalLink className="h-3.5 w-3.5" />
                                            </Link>

                                            {/* Only show if there's an order_id */}
                                            {notification.related_data.payment
                                                .order_id && (
                                                <Link
                                                    href={route(
                                                        "orders.show",
                                                        notification
                                                            .related_data
                                                            .payment.order_id
                                                    )}
                                                    className="inline-flex items-center gap-1 text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 ml-4"
                                                >
                                                    <span>View Order</span>
                                                    <FileText className="h-3.5 w-3.5" />
                                                </Link>
                                            )}
                                        </div>
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </div>

                {/* Actions */}
                {data?.actions && data.actions.length > 0 && (
                    <div className="px-4 py-5 bg-gray-50 dark:bg-gray-900 border-t border-gray-200 dark:border-gray-700 sm:px-6">
                        <h4 className="text-sm font-medium text-gray-500 dark:text-gray-400 mb-3">
                            Actions
                        </h4>
                        <div className="flex flex-wrap gap-3">
                            {data.actions.map((action, index) => (
                                <a
                                    key={index}
                                    href={action.url}
                                    className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md shadow-sm text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                >
                                    {action.text || "View Details"}
                                    <ExternalLink className="ml-1.5 h-3.5 w-3.5" />
                                </a>
                            ))}
                        </div>
                    </div>
                )}
            </div>
        </Layout>
    );
}
