import React, { useMemo, useState, useEffect, useRef } from "react";
import { Head } from "@inertiajs/react";
import { useForm } from "@inertiajs/react";
import { Button } from "@/Components/ui/Button";
// import { OrderPreview } from "@/Components/OrderPreview";

import StepIndicator from "@/Components/OrderForm/StepIndicator";
import BasicInformation from "@/Components/OrderForm/BasicInformation";
import ProjectSpecifications from "@/Components/OrderForm/ProjectSpecifications";
import WritingPreferences from "@/Components/OrderForm/WritingPreferences";
import OrderSummaryOld from "@/Components/OrderForm/OrderSummaryOld";
import ClientLayout from "@/Layouts/ClientLayout";
import MainLayout from "@/Layouts/MainLayout";
import ValidationErrors from "@/Components/OrderForm/ValidationErrors";
import { router } from "@inertiajs/react";
import axios from "axios";
import { OrderStateManager } from "@/utils/orderStateManager";
import { URLStateManager } from "@/utils/urlStateManager";
import { PerformanceMonitor } from "@/utils/performanceMonitor";

export default function Create({
    academicLevels,
    deadlines,
    subjects,
    paperTypes,
    couponEnabled,
    couponCode,
    currencies,
    singleSlidePrice,
    suggestedCoupons,
    // New props for handling anonymous vs authenticated users
    isAuthenticated,
    user,
    discountInfo,
    // Order context restoration props
    restoredOrderData,
    restoredStep,
    welcomeMessage,
    needsTokenRefresh,
}) {
    // Multi-phase step system for anonymous and authenticated users
    const [currentStep, setCurrentStep] = useState(1);

    // Authentication state management for dynamic layout switching
    const [authState, setAuthState] = useState({
        isAuthenticated: isAuthenticated,
        user: user,
        showAuthModal: false,
        authRequired: false, // Flag for when authentication becomes required
    });

    // Discount states (declared early to avoid initialization issues)
    const [appliedDiscount, setAppliedDiscount] = useState({
        discountType: null,
        discountAmount: 0,
        calculatedDiscount: 0,
        couponCode: null,
        error: null,
    });

    // Backend-qualified discounts 
    const [qualifiedDiscounts, setQualifiedDiscounts] = useState([]);

    // Phase configuration for anonymous vs authenticated flow
    const stepConfig = {
        anonymous: {
            totalSteps: 3,
            steps: [
                { id: 1, title: "Service & Requirements", phase: "anonymous" },
                { id: 2, title: "Project Details", phase: "anonymous" },
                { id: 3, title: "Pricing Preview", phase: "anonymous" },
            ],
        },
        authenticated: {
            totalSteps: 5,
            steps: [
                { id: 1, title: "Service & Requirements", phase: "anonymous" },
                { id: 2, title: "Project Details", phase: "anonymous" },
                { id: 3, title: "Pricing Preview", phase: "anonymous" },
                { id: 4, title: "Discount Selection", phase: "authenticated" },
                { id: 5, title: "Review & Submit", phase: "authenticated" },
            ],
        },
    };

    // Dynamic step configuration based on authentication status
    const currentConfig = authState.isAuthenticated
        ? stepConfig.authenticated
        : stepConfig.anonymous;

    // Always show the full step count to anonymous users so they know there are more steps ahead
    // but the authentication gate will stop them at step 3
    const totalSteps = stepConfig.authenticated.totalSteps;

    // state to delay showing the submit button to prevent auto-submission
    const [showSubmitButton, setShowSubmitButton] = useState(false);

    // ref to track if we're processing a step change
    const processingStepRef = useRef(null);

    // Layout switching state with enhanced transition control
    const [layoutTransition, setLayoutTransition] = useState({
        isTransitioning: false,
        phase: "idle", // 'idle', 'fade-out', 'switching', 'fade-in'
        previousLayout: null,
        currentLayout: null,
    });

    // Responsive helper to adjust transition behavior based on screen size
    const [isMobile, setIsMobile] = useState(false);

    useEffect(() => {
        const checkScreenSize = () => {
            const newIsMobile = window.innerWidth < 640; // sm breakpoint
            setIsMobile(newIsMobile);
        };

        checkScreenSize();
        window.addEventListener("resize", checkScreenSize);

        return () => window.removeEventListener("resize", checkScreenSize);
    }, []);

    const { data, setData, post, processing, errors, progress } = useForm({
        title: "",
        instructions: "",
        pages: 1,
        subject: "",
        aclevel: "",
        papertype: "",
        ppslides: 0,
        deadline: "",
        clientTimezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
        spacing: "",
        totalAmount: 0,
        styles: "",
        sources: 0,
        language: "",
        files: [],
        urgency: "",
        couponCode: "",
        discountType: null,
        discountAmount: 0,
        netAmount: 0,
        discount: 0,
        currency: "",
        currencyRate: 1,
    });

    // Effect to continuously save form data using State Manager and update URL
    useEffect(() => {
        const formDataForStorage = {
            ...data,
            currentStep,
            authState: {
                isAuthenticated: authState.isAuthenticated,
                user: authState.user,
            },
            appliedDiscount,
            qualifiedDiscounts,
        };

        // Monitor save operations
        PerformanceMonitor.monitor('save', () => {
            OrderStateManager.saveOrderState(formDataForStorage);
        });

        // Monitor URL operations
        if (currentStep > 1 && (data.title || data.subject || data.aclevel)) {
            PerformanceMonitor.monitor('url', () => {
                URLStateManager.updateURL(data, currentStep);
            });
        }
    }, [
        data,
        currentStep,
        authState.isAuthenticated,
        authState.user,
        appliedDiscount,
        qualifiedDiscounts,
    ]);

    // Effect to fetch qualified discounts when authentication status changes
    useEffect(() => {
        if (authState.isAuthenticated) {
            fetchQualifiedDiscounts();
        } else {
            setQualifiedDiscounts([]);
        }
    }, [authState.isAuthenticated]);

    // Function to fetch backend-qualified discounts
    const fetchQualifiedDiscounts = async () => {
        try {
            // Calculate total amount on the fly to avoid initialization issues
            const pages = data.pages || 1;
            const slides = data.ppslides || 0;
            const pricePerSlide = singleSlidePrice || 4;
            const currencyRate = data.currencyRate || 1;

            // Parse values from form data strings
            const urgency = parseValue(data.deadline);
            const aclevel = parseValue(data.aclevel);
            const spacing = parseValue(data.spacing);
            const subject = parseValue(data.subject);
            const paperType = parseValue(data.papertype);

            // Only proceed if we have all required values
            if (!urgency || !aclevel || !spacing || !subject || !paperType) {
                return;
            }

            // Calculate cost per page
            const costPerPage =
                aclevel * urgency * spacing * subject * paperType;
            const basePrice = costPerPage * pages;

            // Apply currency conversion to slide price if not using USD
            const convertedPricePerSlide =
                data.currency && data.currency !== "USD"
                    ? pricePerSlide * currencyRate
                    : pricePerSlide;

            const slidePrice = slides * convertedPricePerSlide;
            const calculatedTotal = basePrice + slidePrice;

            const response = await axios.get("/qualified-discounts", {
                params: {
                    pages: pages,
                    order_total: calculatedTotal,
                },
            });

            if (response.data.success) {
                setQualifiedDiscounts(response.data.discounts || []);
                console.log('Qualified discounts loaded:', response.data.discounts?.length || 0);
            } else {
                console.log('No qualified discounts available for this order');
                setQualifiedDiscounts([]);
            }
        } catch (error) {
            console.error("Error fetching qualified discounts:", error);
            setQualifiedDiscounts([]);
        }
    };

    // Effect to refetch qualified discounts when order details change
    useEffect(() => {
        if (
            authState.isAuthenticated &&
            data.pages &&
            data.deadline &&
            data.aclevel
        ) {
            const delayedFetch = setTimeout(() => {
                fetchQualifiedDiscounts();
            }, 500); // Debounce API calls

            return () => clearTimeout(delayedFetch);
        }
    }, [
        data.pages,
        data.deadline,
        data.aclevel,
        data.subject,
        data.papertype,
        authState.isAuthenticated,
    ]);

    // Effect to restore form data on component mount if available
    useEffect(() => {
        // Component mounted successfully

        // Check for restored order context from authentication first
        if (restoredOrderData && Object.keys(restoredOrderData).length > 0) {
            // Restore form data from authentication context
            setData((prevData) => ({
                ...prevData,
                ...restoredOrderData,
            }));

            // Restore step from authentication context
            if (restoredStep && restoredStep > 1) {
                setCurrentStep(restoredStep);
            }

            // Show welcome message if available
            if (welcomeMessage) {
                // Display welcome message as a toast notification
                setTimeout(() => {
                    // You can implement a proper toast notification system here
                    // For now, we'll use a simple alert
                    alert(welcomeMessage);
                }, 500);
            }

            return; // Don't process localStorage if we have restored context
        }

        // Check URL state first (for direct access and deep linking)
        const urlState = PerformanceMonitor.monitor('load', () => 
            URLStateManager.decodeOrderState()
        );
        
        if (urlState && URLStateManager.validateURLParams()) {
            try {
                // Restore form data from URL
                setData((prevData) => ({
                    ...prevData,
                    ...urlState.orderData,
                }));

                // Restore step from URL
                if (urlState.currentStep && urlState.currentStep > 1) {
                    setCurrentStep(urlState.currentStep);
                }

                console.log('Order state restored from URL', {
                    step: urlState.currentStep,
                    timestamp: urlState.timestamp
                });
                return; // Don't process localStorage if we have valid URL state
            } catch (error) {
                // Silently handle URL state errors - localStorage fallback will handle restoration
                console.debug("Error restoring form data from URL (falling back to localStorage):", error);
                URLStateManager.clearURLState();
            }
        }

        // Fallback to localStorage state with better error handling
        try {
            const storedData = PerformanceMonitor.monitor('load', () => 
                OrderStateManager.loadOrderState()
            );
            
            if (storedData) {
                // Handle both wrapped (data property) and unwrapped formats
                const orderData = storedData.data || storedData;
                
                // Only restore if we have meaningful data
                if (orderData && (Object.keys(orderData).length > 0 || storedData.currentStep || storedData.step)) {
                    // Restore form data from localStorage
                    setData((prevData) => ({
                        ...prevData,
                        ...orderData,
                    }));

                    // Restore step from localStorage (check both formats)
                    const stepToRestore = storedData.currentStep || storedData.step || orderData.currentStep || orderData.step;
                    if (stepToRestore && stepToRestore > 1) {
                        setCurrentStep(stepToRestore);
                    }

                    console.log('Order state restored from localStorage', {
                        step: stepToRestore,
                        timestamp: storedData.timestamp,
                        hasData: !!orderData
                    });
                }
            }
        } catch (error) {
            console.error("Error restoring form data from localStorage:", error);
            // Clear corrupted state only if it's truly corrupted
            try {
                OrderStateManager.clearOrderState();
            } catch (clearError) {
                console.error("Error clearing corrupted state:", clearError);
            }
        }
    }, [restoredOrderData, restoredStep, welcomeMessage]); // Include new props in dependencies

    // Handle browser back/forward navigation
    useEffect(() => {
        const cleanup = URLStateManager.handleNavigation((urlState) => {
            if (urlState) {
                setData((prevData) => ({
                    ...prevData,
                    ...urlState.orderData,
                }));
                setCurrentStep(urlState.currentStep);
            }
        });

        return cleanup;
    }, []);

    const validateCoupon = async (code) => {
        if (!code) {
            setAppliedDiscount({
                discountType: null,
                discountAmount: 0,
                calculatedDiscount: 0,
                couponCode: null,
                error: null,
            });
            return;
        }

        try {
            // Axios automatically handles CSRF via XSRF-TOKEN cookie
            const response = await axios.post("/validate-coupon", {
                code: code,
                order_total: totalAmount,
                pages: data.pages || 1,
            });

            if (response.data.valid) {
                // Calculate discount - use calculated_discount from backend if available,
                // otherwise calculate it, ensuring fixed discounts don't exceed order total
                const discountValue = response.data.calculated_discount !== undefined
                    ? parseFloat(response.data.calculated_discount)
                    : response.data.type === "percentage"
                        ? (parseFloat(totalAmount) * response.data.amount) / 100
                        : Math.min(parseFloat(response.data.amount), parseFloat(totalAmount));

                setAppliedDiscount({
                    discountType: response.data.type,
                    discountAmount: response.data.amount, // Raw coupon amount
                    calculatedDiscount: discountValue, // Actual discount applied (capped)
                    couponCode: code,
                    error: null,
                });

                // Update form data with coupon information
                setData((data) => ({
                    ...data,
                    couponCode: code,
                    discountType: response.data.type,
                    discountAmount: response.data.amount,
                }));

                setData("discount", discountValue);
                setData("netAmount", parseFloat(totalAmount) - discountValue);

                return true;
            } else {
                setAppliedDiscount({
                    discountType: null,
                    discountAmount: 0,
                    calculatedDiscount: 0,
                    couponCode: null,
                    error: response.data.message || "Invalid coupon code",
                });

                // Reset coupon data
                setData((data) => ({
                    ...data,
                    couponCode: "",
                    discountType: null,
                    discountAmount: 0,
                    discount: 0,
                    netAmount: parseFloat(totalAmount),
                }));

                return false;
            }
        } catch (error) {
            setAppliedDiscount({
                discountType: null,
                discountAmount: 0,
                calculatedDiscount: 0,
                couponCode: null,
                error: "Error validating coupon",
            });
            console.error("Coupon validation error:", error);
            return false;
        }
    };

    const parseValue = (str) => {
        if (!str) return 0;
        return parseFloat(str.split("#")[0]) || 0;
    };

    // State for backend pricing calculation
    const [backendPricing, setBackendPricing] = useState({
        basePrice: 0,
        totalAmount: 0,
        costPerPage: 0,
        netAmount: 0,
        discount: 0,
        costPerSlide: 0,
        isLoading: false,
        error: null,
    });

    // Function to calculate pricing securely on the backend
    const calculateBackendPricing = async () => {
        // Check if we have all required data for calculation
        if (!data.aclevel || !data.subject || !data.papertype || !data.deadline || !data.spacing) {
            setBackendPricing(prev => ({
                ...prev,
                basePrice: 0,
                totalAmount: 0,
                costPerPage: 0,
                netAmount: 0,
                discount: 0,
                costPerSlide: 0,
            }));
            return;
        }

        setBackendPricing(prev => ({ ...prev, isLoading: true, error: null }));

        try {
            // Parse values for backend - extract value and look up ID from props
            const parseIdFromValue = (str, items, fieldName) => {
                if (!str || typeof str !== 'string') return null;
                
                // Split by '#' to get value and name
                const parts = str.split('#');
                if (parts.length < 2) return null;
                
                const value = parseFloat(parts[0]);
                const name = parts[1];
                
                if (isNaN(value)) return null;
                
                // Find the item by matching both value and name
                let item = items?.find(item => 
                    Math.abs(item.value - value) < 0.01 && item.name === name
                );
                
                // Fallback: If no match found, try matching by name only
                if (!item) {
                    item = items?.find(item => item.name === name);
                }
                
                return item ? item.id : null;
            };

            // Special parsing for deadline which has value#urgency#duration format
            const parseDeadlineId = (str, items) => {
                if (!str || typeof str !== 'string') return null;
                
                const parts = str.split('#');
                if (parts.length < 3) return null;
                
                const value = parseFloat(parts[0]);
                const urgency = parseInt(parts[1]);
                const duration = parts[2];
                
                if (isNaN(value) || isNaN(urgency)) return null;
                
                // Find the item by matching value, urgency, and duration
                let item = items?.find(item => 
                    Math.abs(item.value - value) < 0.01 && 
                    item.urgency === urgency && 
                    item.duration === duration
                );
                
                // Enhanced fallback: Try matching by value and duration only (ignore urgency)
                if (!item) {
                    item = items?.find(item => 
                        Math.abs(item.value - value) < 0.01 && 
                        item.duration === duration
                    );
                }
                
                // Fallback: If no exact match found, try matching by urgency and duration only
                if (!item) {
                    item = items?.find(item => item.urgency === urgency && item.duration === duration);
                }
                
                return item ? item.id : null;
            };

            const academicLevelId = parseIdFromValue(data.aclevel, academicLevels, 'academicLevels');
            const subjectId = parseIdFromValue(data.subject, subjects, 'subjects');
            const paperTypeId = parseIdFromValue(data.papertype, paperTypes, 'paperTypes');
            const deadlineId = parseDeadlineId(data.deadline, deadlines);

            // Validate that all IDs are valid numbers
            if (!academicLevelId || !subjectId || !paperTypeId || !deadlineId) {
                // Fallback to frontend calculation if backend parsing fails
                const fallbackCalculation = calculateFrontendPricing();
                setBackendPricing({
                    basePrice: fallbackCalculation.basePrice,
                    totalAmount: fallbackCalculation.totalAmount,
                    costPerPage: fallbackCalculation.costPerPage,
                    netAmount: fallbackCalculation.netAmount,
                    discount: fallbackCalculation.discount,
                    costPerSlide: fallbackCalculation.costPerSlide,
                    isLoading: false,
                    error: 'Using frontend calculation due to backend parsing issues.',
                });
                return;
            }

            const spacingValue = data.spacing?.split('#')[1]?.toLowerCase().replace(' spaced', '') || 'double';
            
            const requestData = {
                academic_level_id: academicLevelId,
                paper_type_id: paperTypeId,
                deadline_id: deadlineId,
                pages: data.pages || 1,
                slides: data.ppslides || 0,
                subject_id: subjectId,
                sources: data.sources || 0,
                currency_code: data.currency || 'USD',
                spacing: spacingValue,
                coupon_code: appliedDiscount.discountType ? appliedDiscount.couponCode : null,
            };

            const response = await axios.post('/calculate-order-pricing', requestData);

            if (response.data.success) {
                const pricing = response.data.pricing;
                setBackendPricing({
                    basePrice: pricing.base_price,
                    totalAmount: pricing.total_before_discount,
                    costPerPage: pricing.breakdown.cost_per_page || pricing.breakdown.pages_cost / (data.pages || 1),
                    netAmount: pricing.final_price,
                    discount: pricing.discount_amount,
                    costPerSlide: pricing.breakdown.slides_cost,
                    isLoading: false,
                    error: null,
                });

                // Update form data with backend-calculated values
                setData("totalAmount", pricing.total_before_discount);
                setData("netAmount", pricing.final_price);
                setData("discount", pricing.discount_amount);
            } else {
                // Fallback to frontend calculation if backend calculation fails
                const fallbackCalculation = calculateFrontendPricing();
                setBackendPricing({
                    basePrice: fallbackCalculation.basePrice,
                    totalAmount: fallbackCalculation.totalAmount,
                    costPerPage: fallbackCalculation.costPerPage,
                    netAmount: fallbackCalculation.netAmount,
                    discount: fallbackCalculation.discount,
                    costPerSlide: fallbackCalculation.costPerSlide,
                    isLoading: false,
                    error: 'Using frontend calculation due to backend calculation issues.',
                });
            }
        } catch (error) {
            // Fallback to frontend calculation if backend fails
            const fallbackCalculation = calculateFrontendPricing();
            setBackendPricing({
                basePrice: fallbackCalculation.basePrice,
                totalAmount: fallbackCalculation.totalAmount,
                costPerPage: fallbackCalculation.costPerPage,
                netAmount: fallbackCalculation.netAmount,
                discount: fallbackCalculation.discount,
                costPerSlide: fallbackCalculation.costPerSlide,
                isLoading: false,
                error: 'Using frontend calculation due to backend errors.',
            });
        }
    };

    // Fallback frontend calculation function
    const calculateFrontendPricing = () => {
        try {
            // Parse values from form data strings
            const urgency = parseValue(data.deadline);
            const aclevel = parseValue(data.aclevel);
            const spacing = parseValue(data.spacing);
            const subject = parseValue(data.subject);
            const paperType = parseValue(data.papertype);

            // Calculate cost per page
            const costPerPage = aclevel * urgency * spacing * subject * paperType;
            const basePrice = costPerPage * (data.pages || 1);

            // Calculate slide cost
            const slides = data.ppslides || 0;
            const pricePerSlide = singleSlidePrice || 4;
            const costPerSlide = slides * pricePerSlide;

            const totalAmount = basePrice + costPerSlide;
            const discount = appliedDiscount.discountType === "percentage" 
                ? (totalAmount * appliedDiscount.discountAmount) / 100 
                : Math.min(parseFloat(appliedDiscount.discountAmount) || 0, totalAmount);
            const netAmount = totalAmount - discount;

            return {
                basePrice,
                totalAmount,
                costPerPage,
                netAmount,
                discount,
                costPerSlide,
            };
        } catch (error) {
            console.error('Frontend calculation error:', error);
            return {
                basePrice: 0,
                totalAmount: 0,
                costPerPage: 0,
                netAmount: 0,
                discount: 0,
                costPerSlide: 0,
            };
        }
    };

    // Use backend pricing instead of frontend calculation
    const {
        basePrice,
        totalAmount,
        costPerPage,
        costPerSlide,
        discount,
        netAmount,
    } = backendPricing;

    // Trigger backend calculation when relevant data changes
    useEffect(() => {
        const timeoutId = setTimeout(() => {
            calculateBackendPricing();
        }, 500); // Debounce to avoid too many API calls

        return () => clearTimeout(timeoutId);
    }, [
        data.pages,
        data.ppslides,
        data.deadline,
        data.aclevel,
        data.spacing,
        data.subject,
        data.papertype,
        data.currency,
        data.currencyRate,
        data.sources,
        appliedDiscount,
        authState.isAuthenticated,
    ]);

    // Backend pricing calculation now handles all form data updates

    // This effect is now handled in the previous useEffect

    // Extract urgency value from deadline string
    useEffect(() => {
        if (data.deadline) {
            const parts = data.deadline.split("#");
            const urgencyValue = `${parts[1]} ${parts[2]}`;
            setData("urgency", urgencyValue);
        }
    }, [data.deadline]);

    const [validationErrors, setValidationErrors] = useState({});

    const validateStep = () => {
        let isValid = true;
        // Create a new errors object
        const newErrors = {};

        // Validate based on current step
        if (currentStep === 1) {
            // Basic Information validation
            if (!data.title) {
                newErrors.title = "Title is required";
                isValid = false;
            }
            if (!data.aclevel) {
                newErrors.aclevel = "Academic level is required";
                isValid = false;
            }
            if (!data.subject) {
                newErrors.subject = "Subject is required";
                isValid = false;
            }
            if (!data.papertype) {
                newErrors.papertype = "Paper type is required";
                isValid = false;
            }
            if (!data.currency) {
                newErrors.currency = "Please select a currency";
                isValid = false;
            }
        } else if (currentStep === 2) {
            // Project Specifications validation
            if (!data.deadline) {
                newErrors.deadline = "Deadline is required";
                isValid = false;
            }
            if (!data.spacing) {
                newErrors.spacing = "Spacing is required";
                isValid = false;
            }
            if (!data.pages || data.pages < 1) {
                newErrors.pages = "At least 1 page is required";
                isValid = false;
            }
        } else if (currentStep === 3) {
            // Writing Preferences validation
            if (!data.instructions) {
                newErrors.instructions = "Instructions are required";
                isValid = false;
            }
            if (!data.styles) {
                newErrors.styles = "Citation style is required";
                isValid = false;
            }
            if (!data.language) {
                newErrors.language = "Language preference is required";
                isValid = false;
            }

            if (data.sources === undefined) {
                newErrors.sources = "Number of sources is required";
                isValid = false;
            }
        }

        // Validate step progression based on authentication status
        if (authState.isAuthenticated && currentStep > 3) {
            // For authenticated users, steps 4+ are valid
            // Allow step 4 even if no discounts are available
            // The user can still proceed to the next step
            console.log('Step 4 validation - Auth:', authState.isAuthenticated, 'Discounts:', qualifiedDiscounts.length);
        } else if (!authState.isAuthenticated && currentStep > 3) {
            // Anonymous users shouldn't be on steps 4+
            newErrors.general = "Authentication required for this step.";
            isValid = false;
        }

        // Instead of using clearErrors and setError, we'll use a state variable
        if (!isValid) {
            // Create a separate state for validation errors
            setValidationErrors(newErrors);
            console.error("Validation errors:", newErrors);
        } else {
            setValidationErrors({});
        }

        return isValid;
    };

    // Add effect to delay showing the submit button
    useEffect(() => {
        if (currentStep === totalSteps) {
            // Reset submit button visibility first
            setShowSubmitButton(false);
            console.log('Final step reached - submit button will show in 100ms');

            // Delay showing the submit button to ensure OrderSummary renders first
            const timer = setTimeout(() => {
                setShowSubmitButton(true);
                console.log('Submit button now visible');
            }, 100); // Reduced from 300ms to 100ms for better responsiveness

            return () => clearTimeout(timer);
        } else {
            setShowSubmitButton(false);
        }
    }, [currentStep, totalSteps]);

    // Error recovery system
    const handleStateRecovery = () => {
        try {
            // Clear current state
            OrderStateManager.clearOrderState();
            URLStateManager.clearURLState();
            
            // Reset form data
            setData({
                title: "",
                instructions: "",
                pages: 1,
                subject: "",
                aclevel: "",
                papertype: "",
                ppslides: 0,
                deadline: "",
                clientTimezone: Intl.DateTimeFormat().resolvedOptions().timeZone,
                spacing: "",
                totalAmount: 0,
                styles: "",
                sources: 0,
                language: "",
                files: [],
                urgency: "",
                couponCode: "",
                discountType: null,
                discountAmount: 0,
                netAmount: 0,
                discount: 0,
                currency: "",
                currencyRate: 1,
            });

            // Reset step
            setCurrentStep(1);

            // Reset discount states
            setAppliedDiscount({
                discountType: null,
                discountAmount: 0,
                calculatedDiscount: 0,
                couponCode: null,
                error: null,
            });
            setQualifiedDiscounts([]);

            // Clear validation errors
            setValidationErrors({});

            console.log('Order state recovered successfully');
        } catch (error) {
            console.error('Failed to recover order state:', error);
        }
    };

    // Create shareable link
    const createShareableLink = () => {
        try {
            const shareableLink = URLStateManager.createShareableLink(data, currentStep);
            
            // Copy to clipboard
            navigator.clipboard.writeText(shareableLink).then(() => {
                // You can implement a toast notification here
                alert('Shareable link copied to clipboard!');
            }).catch(() => {
                // Fallback for older browsers
                const textArea = document.createElement('textarea');
                textArea.value = shareableLink;
                document.body.appendChild(textArea);
                textArea.select();
                document.execCommand('copy');
                document.body.removeChild(textArea);
                alert('Shareable link copied to clipboard!');
            });
        } catch (error) {
            console.error('Failed to create shareable link:', error);
        }
    };

    // Update handleNext to prevent double-processing
    const handleNext = () => {
        console.log(`handleNext called. Current step: ${currentStep}`);

        // Prevent double processing
        if (processingStepRef.current === currentStep) {
            console.log(
                "Already processing this step - preventing double execution"
            );
            return;
        }

        processingStepRef.current = currentStep;

        if (validateStep()) {
            // If validation passes, move to next step
            const nextStep = Math.min(currentStep + 1, totalSteps);
            console.log(`Validation passed - advancing to step ${nextStep}`);
            setCurrentStep(nextStep);
            // Scroll to top for better UX
            window.scrollTo(0, 0);

            // Reset the processing flag after a short delay
            setTimeout(() => {
                processingStepRef.current = null;
            }, 300);
        } else {
            console.log("Validation failed - staying on current step");
            processingStepRef.current = null;
        }
    };

    // Handle "Previous" button click
    const handlePrevious = () => {
        setCurrentStep((prev) => Math.max(prev - 1, 1));
        window.scrollTo(0, 0);
    };

    // REMOVED: refreshCSRFToken function - Laravel + Inertia handle CSRF automatically via cookies

    const handleSubmit = async (e) => {
        e.preventDefault();
        e.stopPropagation();

        console.log(
            `handleSubmit called. currentStep: ${currentStep}, totalSteps: ${totalSteps}`
        );

        // Only proceed with submission if we're on the final review step
        // Allow submission even if submit button hasn't shown yet (fallback for immediate clicks)
        if (currentStep === totalSteps) {
            console.log("Final step - submitting form");

            // Verify the order amounts before submission
            console.log("Order amounts for submission:", {
                totalAmount: data.totalAmount,
                discount: data.discount,
                netAmount: data.netAmount,
                couponCode: data.couponCode,
                couponDiscount: data.discountAmount,
                couponType: data.discountType,
            });

            const formData = new FormData();

            // Append all form fields
            Object.keys(data).forEach((key) => {
                if (key !== "files") {
                    formData.append(key, data[key]);
                }
            });

            // Append multiple files
            console.log('[FILE UPLOAD] Checking files before submission:', {
                hasFiles: data.files && data.files.length > 0,
                filesCount: data.files ? data.files.length : 0,
                files: data.files ? data.files.map(f => ({
                    name: f.name,
                    size: f.size,
                    type: f.type
                })) : []
            });

            if (data.files && data.files.length > 0) {
                data.files.forEach((file, index) => {
                    console.log(`[FILE UPLOAD] Appending file ${index + 1}:`, {
                        name: file.name,
                        size: file.size,
                        type: file.type
                    });
                    formData.append("files[]", file);
                });
                console.log('[FILE UPLOAD] All files appended to FormData');
            } else {
                console.log('[FILE UPLOAD] No files to append');
            }

            // Submit the form data to the server
            console.log("Submitting order to /orders endpoint...");
            console.log("FormData contents:", Object.fromEntries(formData));
            
            // Let Inertia.js handle CSRF tokens automatically
            post("/orders", formData, {
                forceFormData: true,
                preserveScroll: true,
                onSuccess: (response) => {
                    console.log("Order submitted successfully!", response);
                    // Clear stored form data after successful submission
                    localStorage.removeItem("orderFormData");
                },
                onError: (errors) => {
                    console.error("Order submission failed:", errors);
                    console.error("Error details:", JSON.stringify(errors, null, 2));
                },
                onFinish: () => {
                    console.log("Order submission request finished");
                },
            });
        } else {
            // If not on the final step, check auth and go to next step
            console.log(
                `Not final step (current: ${currentStep}, total: ${totalSteps}) - checking auth and advancing to next step`
            );
            handleNextWithAuthCheck(e);
        }
    };

    const handleCurrencyChange = (currencyCode, rate) => {
        setData((prevData) => ({
            ...prevData,
            currency: currencyCode,
            currencyRate: rate,
        }));
    };

    const getCurrencySymbol = () => {
        return data.currency === "EUR"
            ? "€"
            : data.currency === "GBP"
            ? "£"
            : data.currency === "CAD"
            ? "C$"
            : data.currency === "AUD"
            ? "A$"
            : "$";
    };

    // Authentication and layout switching handlers
    const handleAuthenticationSuccess = async (authenticatedUser) => {
        // Laravel automatically rotates CSRF token on login and updates the XSRF-TOKEN cookie
        // Inertia/Axios automatically use the updated cookie - no manual intervention needed

        // Adjust timing based on device capabilities
        const timings = isMobile
            ? { fadeOut: 100, switching: 150, fadeIn: 250, total: 300 }
            : { fadeOut: 150, switching: 200, fadeIn: 350, total: 450 };

        // Start smooth transition sequence
        setLayoutTransition({
            isTransitioning: true,
            phase: "fade-out",
            previousLayout: "MainLayout",
            currentLayout: "ClientLayout",
        });

        // Phase 1: Fade out current content
        setTimeout(() => {
            setLayoutTransition((prev) => ({
                ...prev,
                phase: "switching",
            }));

            // Update authentication state during fade-out
            setAuthState((prev) => ({
                ...prev,
                isAuthenticated: true,
                user: authenticatedUser,
                showAuthModal: false,
                authRequired: false,
            }));
        }, timings.fadeOut);

        // Phase 2: Switch layout and fade in
        setTimeout(() => {
            setLayoutTransition((prev) => ({
                ...prev,
                phase: "fade-in",
            }));
        }, timings.switching);

        // Phase 3: Complete transition
        setTimeout(() => {
            setLayoutTransition({
                isTransitioning: false,
                phase: "idle",
                previousLayout: null,
                currentLayout: null,
            });
        }, timings.total);
    };

    // Store order context before authentication
    const storeOrderContext = async () => {
        return PerformanceMonitor.monitorAsync('auth', async () => {
            try {
                const orderData = {
                    title: data.title,
                    instructions: data.instructions,
                    pages: data.pages,
                    subject: data.subject,
                    aclevel: data.aclevel,
                    papertype: data.papertype,
                    ppslides: data.ppslides,
                    deadline: data.deadline,
                    spacing: data.spacing,
                    styles: data.styles,
                    sources: data.sources,
                    language: data.language,
                    currency: data.currency,
                    currencyRate: data.currencyRate,
                    appliedDiscount: appliedDiscount,
                    qualifiedDiscounts: qualifiedDiscounts,
                };

                const response = await axios.post('/store-order-context', {
                    orderData: orderData,
                    currentStep: currentStep
                });

                if (!response.data.success) {
                    console.warn('Order context storage warning:', response.data.message);
                }
            } catch (error) {
                console.error('Failed to store order context:', error);
                // Don't block authentication if context storage fails
                // The user can still proceed with auth, just without context preservation
            }
        });
    };

    const handleAuthenticationRequired = async () => {
        // Store order context before showing auth modal
        await storeOrderContext();
        
        setAuthState((prev) => ({
            ...prev,
            authRequired: true,
            showAuthModal: true,
        }));
    };

    const handleAuthModalClose = () => {
        setAuthState((prev) => ({
            ...prev,
            showAuthModal: false,
        }));
    };

    // Check if authentication is required based on current step and phase
    const checkAuthRequirement = () => {
        // For anonymous users, authentication is required when:
        // 1. They try to proceed beyond step 3 (pricing preview)
        // 2. They want to access personalized discounts
        // 3. They want to submit an actual order

        if (!authState.isAuthenticated) {
            // Check if user is trying to proceed beyond anonymous phase
            if (currentStep >= stepConfig.anonymous.totalSteps) {
                return true;
            }

            // Check if user wants to access authenticated features
            const currentStepConfig = stepConfig.anonymous.steps.find(
                (s) => s.id === currentStep + 1
            );
            if (
                currentStepConfig &&
                currentStepConfig.phase === "authenticated"
            ) {
                return true;
            }
        }
        return false;
    };

    // Handle step progression with authentication check
    const handleNextWithAuthCheck = (e) => {
        // Prevent any default form submission behavior
        if (e) {
            e.preventDefault();
            e.stopPropagation();
        }

        if (checkAuthRequirement()) {
            handleAuthenticationRequired();
            return;
        }

        handleNext();
    };

    // Render the current step content based on phase and authentication status
    const renderStepContent = () => {
        // Get current step configuration
        const currentStepInfo = currentConfig.steps.find(
            (s) => s.id === currentStep
        );

        switch (currentStep) {
            // Phase 1: Service & Requirements (Anonymous accessible)
            case 1:
                return (
                    <BasicInformation
                        data={data}
                        setData={setData}
                        errors={validationErrors}
                        academicLevels={academicLevels}
                        subjects={subjects}
                        currencies={currencies}
                        paperTypes={paperTypes}
                        orderCalculation={{
                            currencySymbol: getCurrencySymbol(),
                            costPerPage: costPerPage,
                            totalAmount: totalAmount,
                            basePrice: basePrice,
                            netAmount: netAmount,
                            discount: discount,
                        }}
                        onCurrencyChange={handleCurrencyChange}
                    />
                );

            // Phase 2: Project Details (Anonymous accessible)
            case 2:
                return (
                    <ProjectSpecifications
                        data={data}
                        setData={setData}
                        errors={validationErrors}
                        deadlines={deadlines}
                        singleSlidePrice={singleSlidePrice}
                        basePrice={basePrice}
                        costPerPage={costPerPage}
                        totalAmount={totalAmount}
                        orderCalculation={{
                            currencySymbol: getCurrencySymbol(),
                            costPerPage: costPerPage,
                            totalAmount: totalAmount,
                            basePrice: basePrice,
                            netAmount: netAmount,
                            discount: discount,
                            costPerSlide: costPerSlide,
                        }}
                        onCurrencyChange={handleCurrencyChange}
                    />
                );
            // Phase 3: Pricing Preview (Anonymous accessible) OR Writing Preferences (Authenticated)
            case 3:
                if (!authState.isAuthenticated) {
                    // Anonymous users see pricing preview with discount teasers
                    return (
                        <div className="space-y-6">
                            <div className="text-center">
                                <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-4">
                                    Your Project Pricing
                                </h3>
                                <p className="text-gray-600 dark:text-gray-400 mb-6">
                                    Based on your requirements, here's your
                                    project cost:
                                </p>
                            </div>

                            {/* Pricing Breakdown for Anonymous Users */}
                            <div className="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg p-6">
                                {backendPricing.error && (
                                    <div className="mb-4 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                                        <div className="flex items-center">
                                            <svg className="w-5 h-5 text-red-500 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                                            </svg>
                                            <span className="text-sm text-red-700 dark:text-red-300">
                                                {backendPricing.error}
                                            </span>
                                        </div>
                                    </div>
                                )}
                                <div className="space-y-4">
                                    <div className="flex justify-between items-center">
                                        <span className="font-medium text-gray-900 dark:text-white">
                                            Base Price:
                                        </span>
                                        <span className="text-lg font-bold text-gray-900 dark:text-white">
                                            {getCurrencySymbol()}
                                            {backendPricing.isLoading ? (
                                                <span className="inline-flex items-center">
                                                    <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-gray-500" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                    </svg>
                                                    Calculating...
                                                </span>
                                            ) : backendPricing.error ? (
                                                <span className="text-red-600 dark:text-red-400">
                                                    Error
                                                </span>
                                            ) : (
                                                parseFloat(totalAmount).toFixed(2)
                                            )}
                                        </span>
                                    </div>

                                    {/* Discount Teaser for Anonymous Users */}
                                    <div className="border-t pt-4 mt-4">
                                        <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                                            <div className="flex items-center mb-2">
                                                <div className="w-2 h-2 bg-blue-500 rounded-full mr-2"></div>
                                                <h4 className="font-semibold text-blue-900 dark:text-blue-100">
                                                    💰 Great news! You qualify
                                                    for discounts!
                                                </h4>
                                            </div>
                                            <div className="text-sm text-blue-800 dark:text-blue-200 space-y-2">
                                                <p>
                                                    <strong>
                                                        🎉 New Customer:
                                                    </strong>{" "}
                                                    Save up to 20% on your first
                                                    order
                                                </p>
                                                <p>
                                                    <strong>
                                                        💡 Volume Bonus:
                                                    </strong>{" "}
                                                    {data.pages >= 10
                                                        ? "You qualify for volume discount!"
                                                        : `Order ${
                                                              10 -
                                                              (data.pages || 1)
                                                          } more pages for volume discount`}
                                                </p>
                                                <p>
                                                    <strong>
                                                        ⚡ Flash Sales:
                                                    </strong>{" "}
                                                    {discountInfo?.flash_discounts_available
                                                        ? "Active now!"
                                                        : "Check back for special offers"}
                                                </p>
                                                <p>
                                                    <strong>
                                                        🔒 Minimum Guaranteed:
                                                    </strong>{" "}
                                                    Everyone saves at least 10%
                                                </p>
                                            </div>

                                            <div className="mt-4 pt-3 border-t border-blue-200 dark:border-blue-700">
                                                <div className="flex justify-between items-center">
                                                    <span className="text-sm font-medium text-blue-900 dark:text-blue-100">
                                                        Your estimated final
                                                        price:
                                                    </span>
                                                    <div className="text-right">
                                                        <div className="text-lg font-bold text-green-600 dark:text-green-400">
                                                            {getCurrencySymbol()}
                                                            {Math.max(
                                                                parseFloat(
                                                                    totalAmount
                                                                ) * 0.8, // 20% off (best case)
                                                                parseFloat(
                                                                    totalAmount
                                                                ) * 0.9 // 10% off (minimum)
                                                            ).toFixed(2)}{" "}
                                                            -{" "}
                                                            {getCurrencySymbol()}
                                                            {(
                                                                parseFloat(
                                                                    totalAmount
                                                                ) * 0.9
                                                            ).toFixed(2)}
                                                        </div>
                                                        <div className="text-xs text-gray-600 dark:text-gray-400">
                                                            You save:{" "}
                                                            {getCurrencySymbol()}
                                                            {(
                                                                parseFloat(
                                                                    totalAmount
                                                                ) * 0.1
                                                            ).toFixed(2)}{" "}
                                                            -{" "}
                                                            {getCurrencySymbol()}
                                                            {(
                                                                parseFloat(
                                                                    totalAmount
                                                                ) * 0.2
                                                            ).toFixed(2)}
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    {/* Continue CTA */}
                                    <div className="bg-gradient-to-r from-blue-50 to-purple-50 dark:from-blue-900/20 dark:to-purple-900/20 border border-blue-200 dark:border-blue-700 rounded-lg p-4 mt-6">
                                        <div className="text-center">
                                            <h4 className="font-semibold text-gray-900 dark:text-white mb-2">
                                                Ready to unlock your savings?
                                            </h4>
                                            <p className="text-sm text-gray-600 dark:text-gray-400 mb-3">
                                                Create an account or sign in to
                                                access your personalized
                                                discounts and place your order.
                                            </p>
                                            <div className="text-xs text-gray-500 dark:text-gray-500">
                                                ✅ Free account • ✅ Secure
                                                payment • ✅ Order tracking • ✅
                                                24/7 support
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    );
                } else {
                    // Authenticated users see writing preferences (requirements step)
                    return (
                        <WritingPreferences
                            data={data}
                            setData={setData}
                            errors={validationErrors}
                            orderCalculation={{
                                currencySymbol: getCurrencySymbol(),
                                costPerPage: costPerPage,
                                totalAmount: totalAmount,
                                basePrice: basePrice,
                                netAmount: netAmount,
                                discount: discount,
                            }}
                            onCurrencyChange={handleCurrencyChange}
                        />
                    );
                }

            // Phase 4: Discount Selection (Authenticated only)
            case 4:
                return (
                    <div className="space-y-6">
                        <div className="text-center">
                            <h3 className="text-xl font-semibold text-gray-900 dark:text-white mb-4">
                                💰 Choose Your Discount
                            </h3>
                            <p className="text-gray-600 dark:text-gray-400 mb-6">
                                Great news! You qualify for multiple discounts.
                                Select the best one for your order:
                            </p>
                        </div>

                        <div className="space-y-4">
                            {/* Backend-Qualified Discounts Section */}
                            {qualifiedDiscounts.length > 0 && (
                                <div className="bg-gradient-to-r from-green-50 to-blue-50 dark:from-green-900/20 dark:to-blue-900/20 border border-green-200 dark:border-green-700 rounded-lg p-6">
                                    <div className="flex items-center mb-4">
                                        <div className="w-3 h-3 bg-green-500 rounded-full mr-3"></div>
                                        <h4 className="text-lg font-semibold text-gray-900 dark:text-white">
                                            🎉 Your Available Discounts
                                        </h4>
                                    </div>

                                    <div className="space-y-3">
                                        {qualifiedDiscounts.map(
                                            (discount, index) => (
                                                <div
                                                    key={index}
                                                    className="flex items-center justify-between p-3 bg-white dark:bg-gray-800 rounded-lg border border-green-200 dark:border-green-600"
                                                >
                                                    <div className="flex items-center">
                                                        <div className="w-2 h-2 bg-green-500 rounded-full mr-3"></div>
                                                        <div>
                                                            <p className="font-medium text-gray-900 dark:text-white">
                                                                {discount.description ||
                                                                    `${discount.type.replace(
                                                                        "_",
                                                                        " "
                                                                    )} Discount`}
                                                            </p>
                                                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                                                Code:{" "}
                                                                {discount.code}
                                                                {discount.is_automatic && (
                                                                    <span className="ml-2 text-xs bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 px-2 py-0.5 rounded">
                                                                        Auto Applied
                                                                    </span>
                                                                )}
                                                            </p>
                                                        </div>
                                                    </div>
                                                    <div className="text-right">
                                                        <p className="font-bold text-green-600 dark:text-green-400">
                                                            {discount.discount_type ===
                                                            "percentage"
                                                                ? `${discount.discount_amount}% OFF`
                                                                : `${getCurrencySymbol()}${
                                                                      discount.discount_amount
                                                                  } OFF`}
                                                        </p>
                                                        <p className="text-xs text-gray-500">
                                                            -
                                                            {getCurrencySymbol()}
                                                            {discount.discount_type ===
                                                            "percentage"
                                                                ? (
                                                                      (parseFloat(
                                                                          totalAmount
                                                                      ) *
                                                                          parseFloat(
                                                                              discount.discount_amount
                                                                          )) /
                                                                      100
                                                                  ).toFixed(2)
                                                                : parseFloat(
                                                                      discount.discount_amount
                                                                  ).toFixed(2)}
                                                        </p>
                                                    </div>
                                                </div>
                                            )
                                        )}
                                    </div>
                                </div>
                            )}

                            {/* Optional Coupon Section */}
                            <div className="bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg p-6">
                                <div className="flex items-center mb-4">
                                    <div className="w-3 h-3 bg-blue-500 rounded-full mr-3"></div>
                                    <h4 className="text-lg font-semibold text-gray-900 dark:text-white">
                                        🎫 Additional Coupon (Optional)
                                    </h4>
                                </div>

                                <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                                    Have a coupon code? Enter it below to stack
                                    with your automatic discounts:
                                </p>

                                <div className="space-y-4">
                                    <div className="flex gap-3">
                                        <input
                                            type="text"
                                            placeholder="Enter coupon code (e.g., SAVE15)"
                                            value={data.couponCode || ""}
                                            onChange={(e) =>
                                                setData(
                                                    "couponCode",
                                                    e.target.value
                                                )
                                            }
                                            className="flex-1 px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 dark:bg-gray-700 dark:text-white"
                                        />
                                        <button
                                            type="button"
                                            onClick={() =>
                                                validateCoupon(data.couponCode)
                                            }
                                            disabled={!data.couponCode}
                                            className="px-4 py-2 bg-blue-600 hover:bg-blue-700 disabled:bg-gray-400 text-white rounded-lg font-medium transition-colors"
                                        >
                                            Apply
                                        </button>
                                    </div>

                                    {/* Coupon validation feedback */}
                                    {appliedDiscount.error && (
                                        <div className="p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                                            <p className="text-sm text-red-600 dark:text-red-400">
                                                {appliedDiscount.error}
                                            </p>
                                        </div>
                                    )}

                                    {appliedDiscount.discountType &&
                                        !appliedDiscount.error && (
                                            <div className="p-3 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg">
                                                <div className="flex items-center justify-between">
                                                    <div>
                                                        <p className="font-medium text-green-800 dark:text-green-200">
                                                            Coupon Applied!
                                                        </p>
                                                        <p className="text-sm text-green-600 dark:text-green-400">
                                                            {appliedDiscount.discountType ===
                                                            "percentage"
                                                                ? `${appliedDiscount.discountAmount}% additional discount`
                                                                : `${getCurrencySymbol()}${
                                                                      appliedDiscount.calculatedDiscount !== undefined
                                                                          ? appliedDiscount.calculatedDiscount.toFixed(2)
                                                                          : appliedDiscount.discountAmount
                                                                  } off`}
                                                        </p>
                                                    </div>
                                                    <p className="font-bold text-green-600 dark:text-green-400">
                                                        -{getCurrencySymbol()}
                                                        {discount}
                                                    </p>
                                                </div>
                                            </div>
                                        )}

                                    {/* Suggested coupons */}
                                    {suggestedCoupons &&
                                        suggestedCoupons.length > 0 && (
                                            <div className="mt-4">
                                                <p className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                                                    💡 Try these popular codes:
                                                </p>
                                                <div className="flex flex-wrap gap-2">
                                                    {suggestedCoupons.map(
                                                        (coupon) => (
                                                            <button
                                                                key={
                                                                    coupon.code
                                                                }
                                                                type="button"
                                                                onClick={() => {
                                                                    setData(
                                                                        "couponCode",
                                                                        coupon.code
                                                                    );
                                                                    validateCoupon(
                                                                        coupon.code
                                                                    );
                                                                }}
                                                                className="px-3 py-1 text-xs bg-blue-100 hover:bg-blue-200 dark:bg-blue-900/30 dark:hover:bg-blue-900/50 text-blue-800 dark:text-blue-200 rounded-full border border-blue-300 dark:border-blue-700 transition-colors"
                                                            >
                                                                {coupon.code}
                                                            </button>
                                                        )
                                                    )}
                                                </div>
                                            </div>
                                        )}
                                </div>
                            </div>

                            {/* Total Summary */}
                            <div className="bg-gradient-to-r from-gray-50 to-blue-50 dark:from-gray-800 dark:to-blue-900/20 border border-gray-200 dark:border-gray-700 rounded-lg p-6">
                                {backendPricing.error && (
                                    <div className="mb-4 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                                        <div className="flex items-center">
                                            <svg className="w-5 h-5 text-red-500 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                                            </svg>
                                            <span className="text-sm text-red-700 dark:text-red-300">
                                                {backendPricing.error}
                                            </span>
                                        </div>
                                    </div>
                                )}
                                <h4 className="text-lg font-semibold text-gray-900 dark:text-white mb-4">
                                    📊 Your Order Summary
                                </h4>

                                <div className="space-y-3">
                                    <div className="flex justify-between">
                                        <span className="text-gray-600 dark:text-gray-400">
                                            Base Price:
                                        </span>
                                        <span className="font-medium text-gray-900 dark:text-white">
                                            {getCurrencySymbol()}
                                            {backendPricing.isLoading ? (
                                                <span className="inline-flex items-center">
                                                    <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-gray-500" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                    </svg>
                                                    Calculating...
                                                </span>
                                            ) : backendPricing.error ? (
                                                <span className="text-red-600 dark:text-red-400">
                                                    Error
                                                </span>
                                            ) : (
                                                parseFloat(totalAmount).toFixed(2)
                                            )}
                                        </span>
                                    </div>

                                    <div className="flex justify-between text-green-600 dark:text-green-400">
                                        <span>Total Savings:</span>
                                        <span className="font-medium">
                                            -{getCurrencySymbol()}
                                            {backendPricing.isLoading ? (
                                                <span className="inline-flex items-center">
                                                    <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-gray-500" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                    </svg>
                                                    Calculating...
                                                </span>
                                            ) : backendPricing.error ? (
                                                <span className="text-red-600 dark:text-red-400">
                                                    Error
                                                </span>
                                            ) : (
                                                parseFloat(discount).toFixed(2)
                                            )}
                                        </span>
                                    </div>

                                    <div className="border-t pt-3 mt-3">
                                        <div className="flex justify-between items-center">
                                            <span className="text-lg font-semibold text-gray-900 dark:text-white">
                                                Final Price:
                                            </span>
                                            <span className="text-xl font-bold text-blue-600 dark:text-blue-400">
                                                {getCurrencySymbol()}
                                                {backendPricing.isLoading ? (
                                                    <span className="inline-flex items-center">
                                                        <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-blue-500" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                            <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                            <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                        </svg>
                                                        Calculating...
                                                    </span>
                                                ) : backendPricing.error ? (
                                                    <span className="text-red-600 dark:text-red-400">
                                                        Error
                                                    </span>
                                                ) : (
                                                    parseFloat(netAmount).toFixed(2)
                                                )}
                                            </span>
                                        </div>
                                        <p className="text-sm text-gray-500 dark:text-gray-400 mt-1">
                                            {backendPricing.isLoading ? (
                                                <span className="inline-flex items-center">
                                                    <svg className="animate-spin -ml-1 mr-2 h-4 w-4 text-gray-500" xmlns="http://www.w3.org/2000/svg" fill="none" viewBox="0 0 24 24">
                                                        <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                                        <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                                    </svg>
                                                    Calculating savings...
                                                </span>
                                            ) : backendPricing.error ? (
                                                <span className="text-red-600 dark:text-red-400">
                                                    Error calculating savings
                                                </span>
                                            ) : (
                                                <>
                                            You save {getCurrencySymbol()}
                                            {(
                                                parseFloat(totalAmount) -
                                                parseFloat(netAmount)
                                            ).toFixed(2)}
                                            (
                                            {(
                                                ((parseFloat(totalAmount) -
                                                    parseFloat(netAmount)) /
                                                    parseFloat(totalAmount)) *
                                                100
                                            ).toFixed(0)}
                                            %)
                                                </>
                                            )}
                                        </p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                );

            // Phase 5: Review & Submit (Authenticated only)
            case 5:
                return (
                    <OrderSummaryOld
                        data={data}
                        orderCalculation={{
                            basePrice,
                            totalAmount,
                            costPerPage,
                            costPerSlide,
                            discount,
                            netAmount,
                            currencySymbol: getCurrencySymbol(),
                        }}
                        academicLevels={academicLevels}
                        subjects={subjects}
                        paperTypes={paperTypes}
                        processing={processing}
                    />
                );

            default:
                return null;
        }
    };

    // Enhanced layout selection with transition awareness
    const LayoutComponent = authState.isAuthenticated
        ? ClientLayout
        : MainLayout;

    // Conditional header based on layout
    const layoutProps = authState.isAuthenticated
        ? {
              header: (
                  <h2 className="text-xl font-semibold leading-tight text-gray-800 dark:text-gray-200">
                      Create an Order
                  </h2>
              ),
          }
        : {};

    // Enhanced responsive transition classes with phase-based animations
    const getTransitionClasses = () => {
        const baseClasses =
            "py-6 sm:py-8 lg:py-12 transition-all duration-300 ease-in-out";

        switch (layoutTransition.phase) {
            case "fade-out":
                return `${baseClasses} opacity-30 transform scale-98 sm:scale-99`;
            case "switching":
                return `${baseClasses} opacity-0 transform scale-95 sm:scale-96`;
            case "fade-in":
                return `${baseClasses} opacity-70 transform scale-99 sm:scale-99`;
            case "idle":
            default:
                return `${baseClasses} opacity-100 transform scale-100`;
        }
    };

    const containerClasses = getTransitionClasses();

    // Authentication Modal Component
    const AuthenticationModal = () => {
        const [authMode, setAuthMode] = useState("login"); // 'login' or 'register'
        const [authLoading, setAuthLoading] = useState(false);
        const [authForm, setAuthForm] = useState({
            name: "",
            email: "",
            password: "",
            password_confirmation: "",
        });
        const [authErrors, setAuthErrors] = useState({});

        const handleAuthInputChange = (e) => {
            const { name, value } = e.target;
            setAuthForm((prev) => ({
                ...prev,
                [name]: value,
            }));

            // Clear specific field error when user starts typing
            if (authErrors[name]) {
                setAuthErrors((prev) => ({
                    ...prev,
                    [name]: null,
                }));
            }
        };

        const validateAuthForm = () => {
            const errors = {};

            if (!authForm.email) {
                errors.email = "Email is required";
            } else if (!/\S+@\S+\.\S+/.test(authForm.email)) {
                errors.email = "Please enter a valid email address";
            }

            if (!authForm.password) {
                errors.password = "Password is required";
            } else if (authForm.password.length < 8) {
                errors.password = "Password must be at least 8 characters long";
            }

            if (authMode === "register") {
                if (!authForm.name) {
                    errors.name = "Name is required";
                }
                if (!authForm.password_confirmation) {
                    errors.password_confirmation =
                        "Please confirm your password";
                } else if (
                    authForm.password !== authForm.password_confirmation
                ) {
                    errors.password_confirmation = "Passwords do not match";
                }
            }

            return errors;
        };

        const handleAuthSubmit = async (e) => {
            e.preventDefault();

            const validationErrors = validateAuthForm();
            if (Object.keys(validationErrors).length > 0) {
                setAuthErrors(validationErrors);
                return;
            }

            setAuthLoading(true);
            setAuthErrors({});

            try {
                const endpoint = authMode === "login" ? "/order/login" : "/order/register";
                const formData =
                    authMode === "login"
                        ? {
                              email: authForm.email,
                              password: authForm.password,
                              remember: true, // Keep user logged in
                          }
                        : {
                              name: authForm.name,
                              email: authForm.email,
                              password: authForm.password,
                              password_confirmation:
                                  authForm.password_confirmation,
                          };

                // Use Inertia for proper Laravel authentication handling
                router.post(endpoint, formData, {
                    onSuccess: async (page) => {
                        // Extract user data from successful response
                        const authenticatedUser = page.props.auth?.user || {
                            name: authForm.name || authForm.email.split("@")[0],
                            email: authForm.email,
                        };

                        // Laravel automatically rotates CSRF token on login - no manual refresh needed

                        // Handle successful authentication with smooth transition
                        handleAuthenticationSuccess(authenticatedUser);
                    },
                    onError: (errors) => {
                        // Enhanced error handling with specific messages
                        const enhancedErrors = {};
                        
                        Object.keys(errors).forEach(key => {
                            if (key === 'email' && errors[key].includes('credentials')) {
                                enhancedErrors[key] = 'Invalid email or password. Please try again.';
                            } else if (key === 'email' && errors[key].includes('unique')) {
                                enhancedErrors[key] = 'This email is already registered. Please sign in instead.';
                            } else if (key === 'password') {
                                enhancedErrors[key] = errors[key];
                            } else if (key === 'name') {
                                enhancedErrors[key] = errors[key];
                            } else if (key === 'password_confirmation') {
                                enhancedErrors[key] = errors[key];
                            } else {
                                enhancedErrors[key] = errors[key];
                            }
                        });

                        // Add general error if no specific field errors
                        if (Object.keys(enhancedErrors).length === 0) {
                            enhancedErrors.general = 'Authentication failed. Please try again.';
                        }

                        setAuthErrors(enhancedErrors);
                        setAuthLoading(false);
                    },
                    onFinish: () => {
                        // Only set loading false if there were errors
                        // Success case is handled in onSuccess
                    },
                });
            } catch (error) {
                console.error("Authentication error:", error);
                setAuthErrors({
                    general:
                        "Network error. Please check your connection and try again.",
                });
                setAuthLoading(false);
            }
        };

        if (!authState.showAuthModal) return null;

        return (
            <div className="fixed inset-0 bg-black/50 backdrop-blur-sm z-50 flex items-center justify-center p-4">
                <div className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full max-h-[90vh] overflow-y-auto">
                    {/* Modal Header */}
                    <div className="flex items-center justify-between p-6 border-b border-gray-200 dark:border-gray-700">
                        <div>
                            <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                                {authMode === "login"
                                    ? "Sign In to Continue"
                                    : "Create Your Account"}
                            </h3>
                            <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                                {authMode === "login"
                                    ? "Access your personalized discounts and place your order"
                                    : "Join thousands of satisfied customers and save on your first order"}
                            </p>
                        </div>
                        <button
                            onClick={handleAuthModalClose}
                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300 transition-colors"
                        >
                            <svg
                                className="w-6 h-6"
                                fill="none"
                                stroke="currentColor"
                                viewBox="0 0 24 24"
                            >
                                <path
                                    strokeLinecap="round"
                                    strokeLinejoin="round"
                                    strokeWidth={2}
                                    d="M6 18L18 6M6 6l12 12"
                                />
                            </svg>
                        </button>
                    </div>

                    {/* Modal Body */}
                    <div className="p-6">
                        {/* Benefits Banner */}
                        <div className="bg-gradient-to-r from-blue-50 to-green-50 dark:from-blue-900/20 dark:to-green-900/20 border border-blue-200 dark:border-blue-700 rounded-lg p-4 mb-6">
                            <div className="flex items-center mb-2">
                                <div className="w-2 h-2 bg-green-500 rounded-full mr-2"></div>
                                <h4 className="font-semibold text-gray-900 dark:text-white">
                                    🎉 Your savings are waiting!
                                </h4>
                            </div>
                            <div className="text-sm text-gray-700 dark:text-gray-300 space-y-1">
                                <p>
                                    💰 <strong>Guaranteed savings:</strong>{" "}
                                    Minimum 10% off your order
                                </p>
                                <p>
                                    🎁 <strong>New customer bonus:</strong> Up
                                    to 20% off your first order
                                </p>
                                <p>
                                    📱 <strong>Order tracking:</strong>{" "}
                                    Real-time updates on your project
                                </p>
                                <p>
                                    🔒 <strong>Secure payment:</strong> Multiple
                                    payment options available
                                </p>
                            </div>
                        </div>

                        {/* Auth Form */}
                        <form onSubmit={handleAuthSubmit} className="space-y-4">
                            {authErrors.general && (
                                <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-3">
                                    <p className="text-sm text-red-600 dark:text-red-400">
                                        {authErrors.general}
                                    </p>
                                </div>
                            )}

                            {authMode === "register" && (
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                        Full Name *
                                    </label>
                                    <input
                                        type="text"
                                        name="name"
                                        value={authForm.name}
                                        onChange={handleAuthInputChange}
                                        className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors ${
                                            authErrors.name
                                                ? "border-red-300 dark:border-red-600"
                                                : "border-gray-300 dark:border-gray-600"
                                        }`}
                                        placeholder="Enter your full name"
                                        disabled={authLoading}
                                        required
                                    />
                                    {authErrors.name && (
                                        <p className="text-sm text-red-600 dark:text-red-400 mt-1">
                                            {authErrors.name}
                                        </p>
                                    )}
                                </div>
                            )}

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Email Address *
                                </label>
                                <input
                                    type="email"
                                    name="email"
                                    value={authForm.email}
                                    onChange={handleAuthInputChange}
                                    className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors ${
                                        authErrors.email
                                            ? "border-red-300 dark:border-red-600"
                                            : "border-gray-300 dark:border-gray-600"
                                    }`}
                                    placeholder="Enter your email address"
                                    disabled={authLoading}
                                    required
                                />
                                {authErrors.email && (
                                    <p className="text-sm text-red-600 dark:text-red-400 mt-1">
                                        {authErrors.email}
                                    </p>
                                )}
                            </div>

                            <div>
                                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                    Password *
                                </label>
                                <input
                                    type="password"
                                    name="password"
                                    value={authForm.password}
                                    onChange={handleAuthInputChange}
                                    className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors ${
                                        authErrors.password
                                            ? "border-red-300 dark:border-red-600"
                                            : "border-gray-300 dark:border-gray-600"
                                    }`}
                                    placeholder={
                                        authMode === "login"
                                            ? "Enter your password"
                                            : "Create a secure password"
                                    }
                                    disabled={authLoading}
                                    required
                                />
                                {authErrors.password && (
                                    <p className="text-sm text-red-600 dark:text-red-400 mt-1">
                                        {authErrors.password}
                                    </p>
                                )}
                            </div>

                            {authMode === "register" && (
                                <div>
                                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                                        Confirm Password *
                                    </label>
                                    <input
                                        type="password"
                                        name="password_confirmation"
                                        value={authForm.password_confirmation}
                                        onChange={handleAuthInputChange}
                                        className={`w-full px-3 py-2 border rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent dark:bg-gray-700 dark:text-white transition-colors ${
                                            authErrors.password_confirmation
                                                ? "border-red-300 dark:border-red-600"
                                                : "border-gray-300 dark:border-gray-600"
                                        }`}
                                        placeholder="Confirm your password"
                                        disabled={authLoading}
                                        required
                                    />
                                    {authErrors.password_confirmation && (
                                        <p className="text-sm text-red-600 dark:text-red-400 mt-1">
                                            {authErrors.password_confirmation}
                                        </p>
                                    )}
                                </div>
                            )}

                            <button
                                type="submit"
                                disabled={authLoading}
                                className="w-full bg-blue-600 hover:bg-blue-700 disabled:bg-blue-400 text-white font-medium py-2 px-4 rounded-md transition-colors duration-200 flex items-center justify-center"
                            >
                                {authLoading ? (
                                    <>
                                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                        {authMode === "login"
                                            ? "Signing In..."
                                            : "Creating Account..."}
                                    </>
                                ) : (
                                    <>
                                        {authMode === "login"
                                            ? "Sign In & Continue Order"
                                            : "Create Account & Continue Order"}
                                    </>
                                )}
                            </button>
                        </form>

                        {/* Progress indicator for order context */}
                        {authLoading && (
                            <div className="mt-4 p-3 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-700 rounded-lg">
                                <div className="flex items-center">
                                    <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-blue-600 mr-2"></div>
                                    <span className="text-sm text-blue-700 dark:text-blue-300">
                                        Preserving your order progress...
                                    </span>
                                </div>
                            </div>
                        )}

                        {/* Toggle Auth Mode */}
                        <div className="mt-6 text-center">
                            <p className="text-sm text-gray-600 dark:text-gray-400">
                                {authMode === "login"
                                    ? "Don't have an account?"
                                    : "Already have an account?"}{" "}
                                <button
                                    type="button"
                                    onClick={() => {
                                        setAuthMode(
                                            authMode === "login"
                                                ? "register"
                                                : "login"
                                        );
                                        setAuthErrors({});
                                        setAuthForm({
                                            name: "",
                                            email: "",
                                            password: "",
                                            password_confirmation: "",
                                        });
                                    }}
                                    className="text-blue-600 dark:text-blue-400 hover:underline font-medium"
                                >
                                    {authMode === "login"
                                        ? "Create one now"
                                        : "Sign in instead"}
                                </button>
                            </p>
                        </div>

                        {/* Trust Indicators */}
                        <div className="mt-6 pt-4 border-t border-gray-200 dark:border-gray-700">
                            <div className="flex items-center justify-center space-x-6 text-xs text-gray-500 dark:text-gray-400">
                                <div className="flex items-center">
                                    <svg
                                        className="w-4 h-4 mr-1"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M5 9V7a5 5 0 0110 0v2a2 2 0 012 2v5a2 2 0 01-2 2H5a2 2 0 01-2-2v-5a2 2 0 012-2zm8-2v2H7V7a3 3 0 016 0z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                    Secure & Safe
                                </div>
                                <div className="flex items-center">
                                    <svg
                                        className="w-4 h-4 mr-1"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path
                                            fillRule="evenodd"
                                            d="M6.267 3.455a3.066 3.066 0 001.745-.723 3.066 3.066 0 013.976 0 3.066 3.066 0 001.745.723 3.066 3.066 0 012.812 2.812c.051.643.304 1.254.723 1.745a3.066 3.066 0 010 3.976 3.066 3.066 0 00-.723 1.745 3.066 3.066 0 01-2.812 2.812 3.066 3.066 0 00-1.745.723 3.066 3.066 0 01-3.976 0 3.066 3.066 0 00-1.745-.723 3.066 3.066 0 01-2.812-2.812 3.066 3.066 0 00-.723-1.745 3.066 3.066 0 010-3.976 3.066 3.066 0 00.723-1.745 3.066 3.066 0 012.812-2.812zm7.44 5.252a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z"
                                            clipRule="evenodd"
                                        />
                                    </svg>
                                    No Spam
                                </div>
                                <div className="flex items-center">
                                    <svg
                                        className="w-4 h-4 mr-1"
                                        fill="currentColor"
                                        viewBox="0 0 20 20"
                                    >
                                        <path d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                    </svg>
                                    24/7 Support
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        );
    };


    // REMOVED: All manual CSRF token refresh useEffect hooks
    // Laravel + Inertia handle CSRF tokens automatically via XSRF-TOKEN cookie
    // No manual token management needed

    return (
        <LayoutComponent {...layoutProps}>
            <Head>
                {/* Primary Meta Tags */}
                <title>
                    {authState.isAuthenticated
                        ? "Order Essay Online - Professional Academic Writing Service | Academic Scribe"
                        : "Order Essay Online - Get Instant Quote | Academic Scribe"}
                </title>
                <meta
                    name="description"
                    content="Order essay online with Academic Scribe. Professional academic writing service offering custom essays, research papers, dissertations. Expert writers, plagiarism-free content, 24/7 support. Get your essay written today!"
                />
                <meta
                    name="keywords"
                    content="order essay online, buy essay online, custom essay writing, academic writing service, essay help, research paper writing, dissertation writing, paper writing service, essay writers, academic assistance"
                />
                <meta name="robots" content="index, follow" />
                <meta name="author" content="Academic Scribe" />
                <link rel="canonical" href={window.location.href} />

                {/* Open Graph Meta Tags */}
                <meta property="og:type" content="website" />
                <meta
                    property="og:title"
                    content="Order Essay Online - Professional Academic Writing Service | Academic Scribe"
                />
                <meta
                    property="og:description"
                    content="Order essay online with Academic Scribe. Get high-quality custom essays, research papers, and dissertations written by expert writers. Fast delivery, plagiarism-free content, 24/7 support."
                />
                <meta property="og:url" content={window.location.href} />
                <meta property="og:site_name" content="Academic Scribe" />
                <meta
                    property="og:image"
                    content={`${window.location.origin}/images/order-essay-online-og.jpg`}
                />
                <meta property="og:image:width" content="1200" />
                <meta property="og:image:height" content="630" />
                <meta
                    property="og:image:alt"
                    content="Order Essay Online - Professional Academic Writing Service"
                />
                <meta property="og:locale" content="en_US" />

                {/* Twitter Card Meta Tags */}
                <meta name="twitter:card" content="summary_large_image" />
                <meta name="twitter:site" content="@SupremeFreelancers" />
                <meta
                    name="twitter:title"
                    content="Order Essay Online - Professional Academic Writing Service"
                />
                <meta
                    name="twitter:description"
                    content="Order essay online with Academic Scribe. Expert writers, plagiarism-free content, 24/7 support. Get your custom essay today!"
                />
                <meta
                    name="twitter:image"
                    content={`${window.location.origin}/images/order-essay-online-twitter.jpg`}
                />

                {/* Additional SEO Meta Tags */}
                <meta
                    name="viewport"
                    content="width=device-width, initial-scale=1"
                />
                <meta
                    httpEquiv="Content-Type"
                    content="text/html; charset=utf-8"
                />
                <meta name="language" content="English" />
                <meta name="revisit-after" content="7 days" />
                <meta name="distribution" content="global" />
                <meta name="rating" content="general" />

                {/* Schema.org Structured Data */}
                <script type="application/ld+json">
                    {JSON.stringify({
                        "@context": "https://schema.org",
                        "@type": "Service",
                        name: "Order Essay Online",
                        description:
                            "Professional academic writing service offering custom essays, research papers, dissertations and other academic content with expert writers and 24/7 support.",
                        provider: {
                            "@type": "Organization",
                            name: "Academic Scribe",
                            url: window.location.origin,
                            logo: `${window.location.origin}/images/logo.png`,
                            sameAs: [
                                "https://facebook.com/supremefreelancers",
                                "https://twitter.com/supremefreelancers",
                                "https://linkedin.com/company/supremefreelancers",
                            ],
                        },
                        serviceType: "Academic Writing Service",
                        category: "Education",
                        offers: {
                            "@type": "Offer",
                            description: "Custom essay writing service",
                            priceRange: "$$$",
                            availability: "https://schema.org/InStock",
                            validFrom: new Date().toISOString(),
                            acceptedPaymentMethod: [
                                "https://schema.org/CreditCard",
                                "https://schema.org/PaymentCard",
                            ],
                        },
                        hasOfferCatalog: {
                            "@type": "OfferCatalog",
                            name: "Academic Writing Services",
                            itemListElement: [
                                {
                                    "@type": "Offer",
                                    itemOffered: {
                                        "@type": "Service",
                                        name: "Custom Essay Writing",
                                        description:
                                            "Professional custom essay writing service",
                                    },
                                },
                                {
                                    "@type": "Offer",
                                    itemOffered: {
                                        "@type": "Service",
                                        name: "Research Paper Writing",
                                        description:
                                            "Expert research paper writing assistance",
                                    },
                                },
                                {
                                    "@type": "Offer",
                                    itemOffered: {
                                        "@type": "Service",
                                        name: "Dissertation Writing",
                                        description:
                                            "Professional dissertation writing service",
                                    },
                                },
                            ],
                        },
                        areaServed: "Worldwide",
                        availableLanguage: "English",
                        hoursAvailable: {
                            "@type": "OpeningHoursSpecification",
                            dayOfWeek: [
                                "Monday",
                                "Tuesday",
                                "Wednesday",
                                "Thursday",
                                "Friday",
                                "Saturday",
                                "Sunday",
                            ],
                            opens: "00:00",
                            closes: "23:59",
                        },
                    })}
                </script>

                {/* Order Form Schema */}
                <script type="application/ld+json">
                    {JSON.stringify({
                        "@context": "https://schema.org",
                        "@type": "WebPage",
                        name: "Order Essay Online - Place Your Order",
                        description:
                            "Order your custom essay online through our secure order form. Choose your academic level, subject, deadline and get matched with expert writers.",
                        url: window.location.href,
                        mainEntity: {
                            "@type": "OrderAction",
                            name: "Order Essay Online",
                            description:
                                "Place an order for custom academic writing services",
                            target: {
                                "@type": "EntryPoint",
                                urlTemplate: window.location.href,
                                inLanguage: "en-US",
                                actionPlatform: [
                                    "https://schema.org/DesktopWebPlatform",
                                    "https://schema.org/MobileWebPlatform",
                                ],
                            },
                            result: {
                                "@type": "Order",
                                description: "Custom academic writing order",
                            },
                        },
                        breadcrumb: {
                            "@type": "BreadcrumbList",
                            itemListElement: [
                                {
                                    "@type": "ListItem",
                                    position: 1,
                                    name: "Home",
                                    item: window.location.origin,
                                },
                                {
                                    "@type": "ListItem",
                                    position: 2,
                                    name: "Order Essay Online",
                                    item: window.location.href,
                                },
                            ],
                        },
                    })}
                </script>

                {/* FAQ Schema (if applicable) */}
                <script type="application/ld+json">
                    {JSON.stringify({
                        "@context": "https://schema.org",
                        "@type": "FAQPage",
                        mainEntity: [
                            {
                                "@type": "Question",
                                name: "How can I order an essay online?",
                                acceptedAnswer: {
                                    "@type": "Answer",
                                    text: "To order an essay online, simply fill out our order form with your requirements including academic level, subject, deadline, and instructions. Our expert writers will then work on your custom essay.",
                                },
                            },
                            {
                                "@type": "Question",
                                name: "Is it safe to order essays online?",
                                acceptedAnswer: {
                                    "@type": "Answer",
                                    text: "Yes, ordering essays online through Academic Scribe is completely safe and secure. We use encrypted connections, protect your personal information, and guarantee original, plagiarism-free content.",
                                },
                            },
                            {
                                "@type": "Question",
                                name: "How much does it cost to order an essay online?",
                                acceptedAnswer: {
                                    "@type": "Answer",
                                    text: "Essay prices vary based on academic level, deadline, and length. Use our order form to get an instant quote for your specific requirements. We offer competitive pricing and regular discounts.",
                                },
                            },
                        ],
                    })}
                </script>
            </Head>



            {/* Responsive Transition Loading Overlay */}
            {layoutTransition.isTransitioning && (
                <div className="fixed inset-0 bg-white/80 dark:bg-gray-900/80 backdrop-blur-sm z-50 flex items-center justify-center transition-opacity duration-200 px-4">
                    <div className="flex flex-col items-center space-y-3 sm:space-y-4 max-w-sm mx-auto text-center">
                        <div className="animate-spin rounded-full h-6 w-6 sm:h-8 sm:w-8 border-b-2 border-blue-600"></div>
                        <p className="text-xs sm:text-sm text-gray-600 dark:text-gray-400 font-medium px-2">
                            {layoutTransition.phase === "fade-out" &&
                                "Preparing your dashboard..."}
                            {layoutTransition.phase === "switching" &&
                                "Switching to authenticated view..."}
                            {layoutTransition.phase === "fade-in" &&
                                "Almost ready..."}
                        </p>

                        {/* Responsive Layout transition indicator */}
                        <div className="flex items-center space-x-2 sm:space-x-3 mt-2">
                            <div
                                className={`w-2 h-2 sm:w-3 sm:h-3 rounded-full transition-colors duration-300 ${
                                    layoutTransition.previousLayout ===
                                    "MainLayout"
                                        ? "bg-gray-400"
                                        : "bg-transparent border border-gray-400"
                                }`}
                            ></div>
                            <div className="w-6 sm:w-8 h-0.5 bg-blue-600 animate-pulse"></div>
                            <div
                                className={`w-2 h-2 sm:w-3 sm:h-3 rounded-full transition-colors duration-300 ${
                                    layoutTransition.currentLayout ===
                                    "ClientLayout"
                                        ? "bg-blue-600"
                                        : "bg-transparent border border-blue-600"
                                }`}
                            ></div>
                        </div>
                        <p className="text-xs text-gray-500 dark:text-gray-500 hidden sm:block">
                            {layoutTransition.previousLayout} →{" "}
                            {layoutTransition.currentLayout}
                        </p>
                    </div>
                </div>
            )}

            <div className={containerClasses}>
                <div className="mx-auto max-w-7xl px-4 sm:px-6 lg:px-8">
                    <div className="shadow-sm sm:rounded-lg">
                        <div
                            className={`p-2 sm:p-3 bg-white dark:bg-gray-800 rounded-t-lg transition-all duration-300 ${
                                layoutTransition.isTransitioning
                                    ? "transform translate-y-1 sm:translate-y-2 opacity-75"
                                    : "transform translate-y-0 opacity-100"
                            }`}
                        >
                            {/* Enhanced StepIndicator with phase information */}
                            <StepIndicator
                                currentStep={currentStep}
                                totalSteps={totalSteps}
                                steps={currentConfig.steps}
                                isAuthenticated={authState.isAuthenticated}
                            />
                        </div>

                        <div
                            className={`p-2 sm:p-3 bg-white dark:bg-gray-800 rounded-t-lg transition-all duration-300 delay-75 ${
                                layoutTransition.isTransitioning
                                    ? "transform translate-y-1 sm:translate-y-2 opacity-75"
                                    : "transform translate-y-0 opacity-100"
                            }`}
                        >
                            <ValidationErrors errors={validationErrors} />
                            
                            {/* State Recovery Button (only show if there are validation errors) */}
                            {Object.keys(validationErrors).length > 0 && (
                                <div className="mt-3 p-3 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-700 rounded-lg">
                                    <div className="flex items-center justify-between">
                                        <div className="flex items-center">
                                            <svg className="w-5 h-5 text-yellow-500 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                                <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                                            </svg>
                                            <span className="text-sm text-yellow-700 dark:text-yellow-300">
                                                Having issues with the form?
                                            </span>
                                        </div>
                                        <button
                                            onClick={handleStateRecovery}
                                            className="px-3 py-1 text-xs bg-yellow-100 hover:bg-yellow-200 dark:bg-yellow-900/30 dark:hover:bg-yellow-900/50 text-yellow-800 dark:text-yellow-200 rounded-md border border-yellow-300 dark:border-yellow-700 transition-colors"
                                        >
                                            Reset Form
                                        </button>
                                    </div>
                                </div>
                            )}

                            {/* Performance Report Button (development only) */}
                            {/* {process.env.NODE_ENV === 'development' && (
                                <div className="mt-3 p-3 bg-purple-50 dark:bg-purple-900/20 border border-purple-200 dark:border-purple-700 rounded-lg">
                                    <div className="flex items-center justify-between">
                                        <div className="flex items-center">
                                            <svg className="w-5 h-5 text-purple-500 mr-2" fill="currentColor" viewBox="0 0 20 20">
                                                <path d="M9 2a1 1 0 000 2h2a1 1 0 100-2H9z" />
                                                <path fillRule="evenodd" d="M4 5a2 2 0 012-2 3 3 0 003 3h2a3 3 0 003-3 2 2 0 012 2v11a2 2 0 01-2 2H6a2 2 0 01-2-2V5zm3 4a1 1 0 000 2h.01a1 1 0 100-2H7zm3 0a1 1 0 000 2h3a1 1 0 100-2h-3zm-3 4a1 1 0 100 2h.01a1 1 0 100-2H7zm3 0a1 1 0 100 2h3a1 1 0 100-2h-3z" clipRule="evenodd" />
                                            </svg>
                                            <span className="text-sm text-purple-700 dark:text-purple-300">
                                                Performance Report
                                            </span>
                                        </div>
                                        <button
                                            onClick={() => PerformanceMonitor.generateReport()}
                                            className="px-3 py-1 text-xs bg-purple-100 hover:bg-purple-200 dark:bg-purple-900/30 dark:hover:bg-purple-900/50 text-purple-800 dark:text-purple-200 rounded-md border border-purple-300 dark:border-purple-700 transition-colors"
                                        >
                                            Generate
                                        </button>
                                    </div>
                                </div>
                            )} */}
                        </div>

                        <div
                            className={`p-4 sm:p-6 transition-all duration-300 delay-150 ${
                                layoutTransition.isTransitioning
                                    ? "transform translate-y-2 sm:translate-y-4 opacity-50"
                                    : "transform translate-y-0 opacity-100"
                            }`}
                        >
                            {/* SEO-optimized main heading */}
                            <header className="mb-6 text-center">
                                <h1 className="text-2xl sm:text-3xl lg:text-4xl font-bold text-gray-900 dark:text-white mb-3">
                                    Order Essay Online - Professional Academic
                                    Writing Service
                                </h1>
                                <p className="text-base sm:text-lg text-gray-600 dark:text-gray-300 max-w-3xl mx-auto">
                                    Get high-quality custom essays, research
                                    papers, and dissertations written by expert
                                    writers. Fast delivery, plagiarism-free
                                    content, and 24/7 support guaranteed.
                                </p>
                            </header>

                            <div className="">
                                <form
                                    id="order-form"
                                    onSubmit={(e) => {
                                        console.log('Form onSubmit triggered!');
                                        handleSubmit(e);
                                    }}
                                    className={`space-y-4 sm:space-y-6 bg-gradient-to-r from-orange-300 to-cyan-300 dark:bg-none dark:bg-gray-800 p-4 sm:p-6 rounded-lg transition-all duration-300 ${
                                        layoutTransition.isTransitioning
                                            ? "transform scale-98 sm:scale-99 shadow-sm"
                                            : "transform scale-100 shadow-lg"
                                    }`}
                                >
                                    {/* Render only the current step content */}
                                    {renderStepContent()}

                                    {/* Navigation buttons */}
                                    <div className="flex justify-between pt-4 mt-8 border-t border-gray-700">
                                        {currentStep > 1 && (
                                            <Button
                                                type="button" // Important: type="button" to prevent form submission
                                                onClick={handlePrevious}
                                                className="bg-gray-600 hover:bg-gray-500 dark:bg-gray-700 dark:hover:bg-gray-600"
                                            >
                                                Previous
                                            </Button>
                                        )}
                                        {currentStep < totalSteps ? (
                                            <Button
                                                type="button" // Important: type="button" to prevent form submission
                                                onClick={
                                                    handleNextWithAuthCheck
                                                }
                                                className="ml-auto bg-blue-700 hover:bg-blue-600 dark:bg-blue-500 dark:hover:bg-blue-400"
                                            >
                                                Next
                                            </Button>
                                        ) : (
                                            // Show submit button on final step (with or without delay)
                                            (currentStep === totalSteps) && (
                                                <div className="ml-auto text-right">
                                                    <p className="mb-3 text-sm font-medium text-gray-700 dark:text-gray-300">
                                                        Do you want to place
                                                        this order for{" "}
                                                        {getCurrencySymbol()}
                                                        {data.netAmount > 0
                                                            ? parseFloat(
                                                                  data.netAmount
                                                              ).toFixed(2)
                                                            : parseFloat(
                                                                  totalAmount
                                                              ).toFixed(2)}
                                                        ?
                                                    </p>
                                                    <div className="flex space-x-3 justify-end">
                                                        <Button
                                                            type="button"
                                                            onClick={
                                                                handlePrevious
                                                            }
                                                            className="bg-gray-500 hover:bg-gray-400 dark:bg-gray-600 dark:hover:bg-gray-500"
                                                        >
                                                            Review Again
                                                        </Button>
                                                        <Button
                                                            type="button"
                                                            className="bg-green-700 hover:bg-green-600 dark:bg-green-500 dark:hover:bg-green-400"
                                                            disabled={
                                                                processing
                                                            }
                                                            onClick={(e) => {
                                                                console.log('Submit button clicked!');
                                                                handleSubmit(e);
                                                            }}
                                                        >
                                                            {processing
                                                                ? "Processing..."
                                                                : "Place Order"}
                                                        </Button>
                                                    </div>
                                                </div>
                                            )
                                        )}
                                    </div>
                                </form>
                                
                                {/* Fallback submit button for step 5 */}
                                {currentStep === totalSteps && (
                                    <div className="mt-4 text-center">
                                        <Button
                                            type="button"
                                            className="bg-red-700 hover:bg-red-600 dark:bg-red-500 dark:hover:bg-red-400"
                                            onClick={(e) => {
                                                console.log('Fallback submit button clicked!');
                                                handleSubmit(e);
                                            }}
                                        >
                                            {processing ? "Processing..." : "Place Order (Fallback)"}
                                        </Button>
                                    </div>
                                )}
                            </div>
                        </div>

                        {/* <OrderPreview
                            data={data}
                            totalAmount={totalAmount}
                            basePrice={basePrice}
                            costPerPage={costPerPage}
                            costPerSlide={costPerSlide}
                            netAmount={netAmount}
                            discount={discount}
                        /> */}
                    </div>
                </div>
            </div>

            {/* Authentication Modal */}
            <AuthenticationModal />
        </LayoutComponent>
    );
}

