import React from "react";
import { Head, Link } from "@inertiajs/react";
import ClientLayout from "@/Layouts/ClientLayout";
import {
    CreditCard,
    Repeat,
    ArrowLeft,
    Clock,
    CheckCircle,
    XCircle,
    AlertTriangle,
} from "lucide-react";

const PaymentHistory = ({ order, payments }) => {
    // Helper function to get currency symbol
    const getCurrencySymbol = (currencyCode) => {
        switch (currencyCode?.toUpperCase()) {
            case "EUR":
                return "€";
            case "GBP":
                return "£";
            case "CAD":
                return "C$";
            case "AUD":
                return "A$";
            default:
                return "$";
        }
    };

    // Helper function to format dates
    const formatDate = (dateString) => {
        if (!dateString) return "N/A";
        return new Date(dateString).toLocaleString("en-US", {
            year: "numeric",
            month: "short",
            day: "numeric",
            hour: "2-digit",
            minute: "2-digit",
        });
    };

    // Helper function for payment status badge
    const renderStatusBadge = (status) => {
        if (!status) return null;

        switch (status.toLowerCase()) {
            case "completed":
            case "succeeded":
                return (
                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200">
                        <CheckCircle className="w-3 h-3 mr-1" />
                        Completed
                    </span>
                );
            case "failed":
            case "cancelled":
                return (
                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200">
                        <XCircle className="w-3 h-3 mr-1" />
                        Failed
                    </span>
                );
            case "pending":
                return (
                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-yellow-100 text-yellow-800 dark:bg-yellow-900 dark:text-yellow-200">
                        <Clock className="w-3 h-3 mr-1" />
                        Pending
                    </span>
                );
            default:
                return (
                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-gray-100 text-gray-800 dark:bg-gray-800 dark:text-gray-200">
                        <AlertTriangle className="w-3 h-3 mr-1" />
                        {status}
                    </span>
                );
        }
    };

    // Helper function for payment method icon
    const renderPaymentMethodIcon = (method) => {
        if (!method) {
            return (
                <div className="bg-gray-500 p-2 rounded-full">
                    <Repeat className="w-4 h-4 text-white" />
                </div>
            );
        }

        switch (method.toLowerCase()) {
            case "paypal":
                return (
                    <div className="bg-blue-500 p-2 rounded-full">
                        <svg
                            className="w-4 h-4 text-white"
                            viewBox="0 0 24 24"
                            fill="currentColor"
                        >
                            <path d="M7.076 21h4.924v-1h-4.924c-1.392 0-2.076-.635-2.076-1.919v-12.013c0-1.167.615-1.919 1.5-2.059v-1.009c-1.720.081-3.5 1.118-3.5 3.068v12.013c0 2.118 1.578 3.919 4.076 3.919z" />
                            <path d="M20.924 21h-4.924v-1h4.924c1.392 0 2.076-.635 2.076-1.919v-12.013c0-1.167-.615-1.919-1.5-2.059v-1.009c1.720.081 3.5 1.118 3.5 3.068v12.013c0 2.118-1.578 3.919-4.076 3.919z" />
                        </svg>
                    </div>
                );
            case "stripe":
                return (
                    <div className="bg-purple-500 p-2 rounded-full">
                        <CreditCard className="w-4 h-4 text-white" />
                    </div>
                );
            default:
                return (
                    <div className="bg-gray-500 p-2 rounded-full">
                        <Repeat className="w-4 h-4 text-white" />
                    </div>
                );
        }
    };

    return (
        <ClientLayout>
            <Head title={`Payment History - Order #${order.order_number}`} />

            <div className="max-w-5xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                {/* Back button */}
                <div className="mb-6">
                    <Link
                        href={route("orders.show", order.id)}
                        className="inline-flex items-center px-4 py-2 text-sm text-blue-600 dark:text-blue-400 bg-blue-50 dark:bg-blue-900/30 hover:bg-blue-100 dark:hover:bg-blue-900/50 rounded-full transition-colors"
                    >
                        <ArrowLeft className="w-4 h-4 mr-2" />
                        Back to Order
                    </Link>
                </div>

                {/* Page title */}
                <div className="mb-6">
                    <h1 className="text-2xl font-bold text-gray-900 dark:text-white">
                        Payment History
                    </h1>
                    <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                        Order #{order.order_number}
                    </p>
                </div>

                {/* Order summary */}
                <div className="bg-white dark:bg-gray-800 rounded-lg shadow-sm p-5 mb-6">
                    <h2 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                        Order Summary
                    </h2>
                    <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Order Amount
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {getCurrencySymbol(order.currency)}
                                {order.order_amount}
                            </p>
                        </div>
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Payment Status
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {renderStatusBadge(order.payment_status)}
                            </p>
                        </div>
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Order Date
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white">
                                {formatDate(order.dateposted)}
                            </p>
                        </div>
                        <div>
                            <p className="text-sm text-gray-500 dark:text-gray-400">
                                Order Status
                            </p>
                            <p className="font-medium text-gray-900 dark:text-white capitalize">
                                {order.order_status}
                            </p>
                        </div>
                    </div>
                </div>

                {/* Payment transactions */}
                <div className="bg-white dark:bg-gray-800 rounded-lg shadow-sm overflow-hidden">
                    <div className="p-5 border-b border-gray-200 dark:border-gray-700">
                        <h2 className="text-lg font-medium text-gray-900 dark:text-white">
                            Payment Transactions
                        </h2>
                    </div>

                    {payments && payments.length > 0 ? (
                        <div className="divide-y divide-gray-200 dark:divide-gray-700">
                            {payments.map((payment) => (
                                <div
                                    key={payment.id}
                                    className="p-5 hover:bg-gray-50 dark:hover:bg-gray-750 transition-colors"
                                >
                                    <div className="flex items-start">
                                        <div className="flex-shrink-0 mr-4">
                                            {renderPaymentMethodIcon(
                                                payment.method
                                            )}
                                        </div>
                                        <div className="flex-1 min-w-0">
                                            <div className="flex items-center justify-between mb-1">
                                                <p className="text-sm font-medium text-gray-900 dark:text-white truncate">
                                                    Payment via {payment.method}
                                                </p>
                                                {renderStatusBadge(
                                                    payment.status
                                                )}
                                            </div>
                                            <div className="flex flex-col sm:flex-row sm:items-center text-sm text-gray-500 dark:text-gray-400">
                                                <p className="mb-1 sm:mb-0">
                                                    Amount:{" "}
                                                    <span className="font-medium">
                                                        {getCurrencySymbol(
                                                            order.currency
                                                        )}
                                                        {payment.amount}
                                                    </span>
                                                </p>
                                                <span className="hidden sm:inline mx-2">
                                                    •
                                                </span>
                                                <p>
                                                    Date:{" "}
                                                    <span className="font-medium">
                                                        {formatDate(
                                                            payment.date
                                                        )}
                                                    </span>
                                                </p>
                                            </div>
                                            <p className="mt-1 text-xs text-gray-500 dark:text-gray-400 truncate">
                                                Transaction ID:{" "}
                                                {payment.transaction_id ||
                                                    "N/A"}
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            ))}
                        </div>
                    ) : (
                        <div className="p-6 text-center">
                            <p className="text-gray-500 dark:text-gray-400">
                                No payment transactions found for this order.
                            </p>
                            {order.payment_status === "unpaid" && (
                                <Link
                                    href={route("orders.payment", order.id)}
                                    className="mt-4 inline-flex items-center px-4 py-2 border border-transparent rounded-md shadow-sm text-sm font-medium text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                >
                                    Make Payment
                                </Link>
                            )}
                        </div>
                    )}
                </div>
            </div>
        </ClientLayout>
    );
};

export default PaymentHistory;
