import React, { useState } from "react";
import { Head } from "@inertiajs/react";
import ClientLayout from "@/Layouts/ClientLayout";
import PaymentMethodSelector from "@/Components/PaymentMethodSelector";
import PayPalButton from "@/Components/PayPalButton";
import StripePaymentForm from "@/Components/StripePaymentForm";
import PayPalDirectForm from "@/Components/PayPalDirectForm";
import {
    ArrowLeft,
    CreditCard,
    DollarSign,
    FileText,
    Clock,
    AlertCircle,
    CheckCircle,
    Tag,
} from "lucide-react";

const PaymentOptions = ({ order, paymentMethods }) => {
    const [selectedMethod, setSelectedMethod] = useState(null);
    const [paymentError, setPaymentError] = useState(null);
    const [paymentSuccess, setPaymentSuccess] = useState(false);

    const handlePaymentMethodSelect = (methodName) => {
        setSelectedMethod(methodName);
        setPaymentError(null);
    };

    const handlePaymentSuccess = (data) => {
        setPaymentSuccess(true);

        // Redirect after a short delay to show the success message
        setTimeout(() => {
            window.location.href =
                data.redirect || route("orders.show", order.id);
        }, 1500);
    };

    const handlePaymentError = (error) => {
        setPaymentError(
            error.message ||
                "There was a problem processing your payment. Please try again."
        );

        // Scroll to the top to show the error message
        window.scrollTo({ top: 0, behavior: "smooth" });
    };

    const renderPaymentForm = () => {
        if (!selectedMethod) {
            return (
                <div className="p-6 bg-gray-50 dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 text-center">
                    <p className="text-gray-600 dark:text-gray-400">
                        Please select a payment method to continue.
                    </p>
                </div>
            );
        }

        switch (selectedMethod) {
            case "paypal":
                return (
                    <PayPalButton
                        order={order}
                        onSuccess={handlePaymentSuccess}
                        onError={handlePaymentError}
                    />
                );
            case "stripe":
                return (
                    <StripePaymentForm
                        order={order}
                        onSuccess={handlePaymentSuccess}
                        onError={handlePaymentError}
                    />
                );
            case "paypal_direct":
                return (
                    <PayPalDirectForm
                        order={order}
                        onSuccess={handlePaymentSuccess}
                        onError={handlePaymentError}
                    />
                );
            default:
                return (
                    <div className="p-6 bg-yellow-50 dark:bg-yellow-900 rounded-lg border border-yellow-200 dark:border-yellow-800">
                        <p className="text-yellow-700 dark:text-yellow-300">
                            The selected payment method is not available.
                        </p>
                    </div>
                );
        }
    };

    // Add this helper function inside your component
    const getCurrencySymbol = (currencyCode) => {
        switch (currencyCode) {
            case "EUR":
                return "€";
            case "GBP":
                return "£";
            case "CAD":
                return "C$";
            case "AUD":
                return "A$";
            default:
                return "$";
        }
    };

    return (
        <ClientLayout>
            <Head title={`Payment for Order #${order.order_number}`} />

            <div className="max-w-5xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
                {/* Back button */}
                <div className="mb-6">
                    <a
                        href={route("orders.show", order.id)}
                        className="inline-flex items-center px-4 py-2 text-sm text-blue-600 dark:text-blue-400 bg-blue-50 dark:bg-blue-900/30 hover:bg-blue-100 dark:hover:bg-blue-900/50 rounded-full transition-colors"
                    >
                        <ArrowLeft className="w-4 h-4 mr-2" />
                        Back to Order
                    </a>
                </div>

                {/* Page title */}
                <h1 className="text-2xl font-bold text-gray-900 dark:text-white mb-6">
                    Complete Your Payment
                </h1>

                {/* Payment success/error messages */}
                {paymentSuccess && (
                    <div className="mb-6 p-4 bg-green-100 dark:bg-green-900 text-green-700 dark:text-green-300 rounded-lg flex items-center">
                        <CheckCircle className="w-5 h-5 mr-2" />
                        Payment successful! Redirecting you to your order...
                    </div>
                )}

                {paymentError && (
                    <div className="mb-6 p-4 bg-red-100 dark:bg-red-900 text-red-700 dark:text-red-300 rounded-lg flex items-start">
                        <AlertCircle className="w-5 h-5 mt-0.5 mr-2 flex-shrink-0" />
                        <span>{paymentError}</span>
                    </div>
                )}

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    {/* Left column: Order summary */}
                    <div className="lg:col-span-1">
                        <div className="bg-white dark:bg-gray-800 rounded-lg shadow overflow-hidden">
                            <div className="bg-gradient-to-r from-indigo-500 to-purple-600 p-4">
                                <h2 className="text-lg font-semibold text-white">
                                    Order Summary
                                </h2>
                            </div>

                            <div className="p-4 space-y-4">
                                <div className="flex justify-between pb-3 border-b border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center space-x-2">
                                        <FileText className="w-4 h-4 text-gray-500 dark:text-gray-400" />
                                        <span className="text-gray-600 dark:text-gray-400">
                                            Order ID:
                                        </span>
                                    </div>
                                    <span className="font-medium text-gray-900 dark:text-white">
                                        #{order.order_number}
                                    </span>
                                </div>

                                <div className="flex justify-between pb-3 border-b border-gray-200 dark:border-gray-700">
                                    <div className="flex items-center space-x-2">
                                        <Clock className="w-4 h-4 text-gray-500 dark:text-gray-400" />
                                        <span className="text-gray-600 dark:text-gray-400">
                                            Deadline:
                                        </span>
                                    </div>
                                    <span className="font-medium text-gray-900 dark:text-white">
                                        {new Date(
                                            order.deadline
                                        ).toLocaleDateString()}
                                    </span>
                                </div>

                                <div className="pt-2">
                                    <div className="flex justify-between mb-2">
                                        <span className="text-gray-600 dark:text-gray-400">
                                            Subtotal:
                                        </span>
                                        <span className="text-gray-900 dark:text-white">
                                            {getCurrencySymbol(order.currency)}
                                            {order.order_amount}
                                        </span>
                                    </div>

                                    {order.discount > 0 && (
                                        <div className="flex justify-between mb-2">
                                            <span className="flex items-center text-green-600 dark:text-green-400">
                                                <Tag className="w-4 h-4 mr-1" />
                                                Coupon Discount:
                                            </span>
                                            <span className="text-green-600 dark:text-green-400">
                                                -
                                                {getCurrencySymbol(
                                                    order.currency
                                                )}
                                                {order.discount}
                                            </span>
                                        </div>
                                    )}

                                    <div className="flex justify-between font-bold text-lg pt-3 border-t border-gray-200 dark:border-gray-700">
                                        <span className="text-gray-800 dark:text-white">
                                            Total:
                                        </span>
                                        <span className="text-gray-800 dark:text-white">
                                            {getCurrencySymbol(order.currency)}
                                            {order.net_amount}
                                        </span>
                                    </div>

                                    {/* Add savings message if discount is applied */}
                                    {order.discount > 0 && (
                                        <div className="mt-3 p-2 bg-green-50 dark:bg-green-900/20 rounded text-sm text-green-700 dark:text-green-300">
                                            <div className="flex items-center">
                                                <DollarSign className="w-4 h-4 mr-1 flex-shrink-0" />
                                                <span>
                                                    You saved{" "}
                                                    {getCurrencySymbol(
                                                        order.currency
                                                    )}
                                                    {order.discount} with your
                                                    coupon!
                                                </span>
                                            </div>
                                        </div>
                                    )}
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Right column: Payment options */}
                    <div className="lg:col-span-2">
                        <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-6">
                            {/* Payment method selector */}
                            <PaymentMethodSelector
                                orderId={order.id}
                                onSelect={handlePaymentMethodSelect}
                                selectedMethod={selectedMethod}
                                paymentMethods={paymentMethods}
                            />

                            {/* Selected payment method form */}
                            <div className="mt-8">{renderPaymentForm()}</div>
                        </div>

                        {/* Secure payment message */}
                        <div className="mt-4 p-4 bg-gray-50 dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-lg">
                            <div className="flex items-center">
                                <svg
                                    className="w-5 h-5 text-gray-500 dark:text-gray-400 mr-2"
                                    fill="none"
                                    stroke="currentColor"
                                    viewBox="0 0 24 24"
                                    xmlns="http://www.w3.org/2000/svg"
                                >
                                    <path
                                        strokeLinecap="round"
                                        strokeLinejoin="round"
                                        strokeWidth={2}
                                        d="M12 15v2m-6 4h12a2 2 0 002-2v-6a2 2 0 00-2-2H6a2 2 0 00-2 2v6a2 2 0 002 2zm10-10V7a4 4 0 00-8 0v4h8z"
                                    />
                                </svg>
                                <span className="text-sm text-gray-600 dark:text-gray-400">
                                    Secure Payment Processing — Your payment
                                    information is protected with
                                    industry-standard encryption.
                                </span>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
};

export default PaymentOptions;
