import { Head } from "@inertiajs/react";
import { motion } from "framer-motion";
import {
    ArrowRight,
    FileText,
    Search,
    BookOpen,
    Edit,
    Users,
    Library,
    Target,
    CheckCircle,
    Award,
    Clock,
    Shield,
} from "lucide-react";
import MainLayout from "@/Layouts/MainLayout";
import PropTypes from "prop-types";

export default function ServicesIndex({ services }) {
    // Icon mapping
    const iconMap = {
        FileText: FileText,
        Search: Search,
        BookOpen: BookOpen,
        Edit: Edit,
        Users: Users,
        Library: Library,
        Target: Target,
    };

    // Animation variants
    const fadeIn = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.6 } },
    };

    const staggerContainer = {
        hidden: { opacity: 0 },
        visible: {
            opacity: 1,
            transition: {
                staggerChildren: 0.2,
            },
        },
    };

    return (
        <MainLayout>
            <Head>
                <title>
                    Professional Academic Writing Services | Expert Writers |
                    Academic Scribe
                </title>
                <meta
                    name="description"
                    content="Comprehensive academic writing services including essays, research papers, dissertations, and assignments. Expert PhD writers, plagiarism-free guarantee, 24/7 support, unlimited revisions."
                />
                <meta
                    name="keywords"
                    content="academic writing services, essay writing service, research paper help, dissertation writing, assignment assistance, professional writers, custom writing"
                />
                <link
                    rel="canonical"
                    href={`${window.location.origin}/services`}
                />

                {/* Open Graph */}
                <meta
                    property="og:title"
                    content="Professional Academic Writing Services | Academic Scribe"
                />
                <meta
                    property="og:description"
                    content="Expert academic writing services with PhD writers. Essays, research papers, dissertations and more. Plagiarism-free guarantee, 24/7 support."
                />
                <meta
                    property="og:url"
                    content={`${window.location.origin}/services`}
                />
                <meta property="og:type" content="website" />
                <meta
                    property="og:image"
                    content="/images/services/academic-writing-services.jpg"
                />

                {/* Structured Data - Service Catalog */}
                <script
                    type="application/ld+json"
                    dangerouslySetInnerHTML={{
                        __html: JSON.stringify({
                            "@context": "https://schema.org",
                            "@type": "ItemList",
                            name: "Academic Writing Services",
                            description:
                                "Comprehensive academic writing services for students",
                            itemListElement: services.map((service, index) => ({
                                "@type": "Service",
                                position: index + 1,
                                name: service.title,
                                description: service.description,
                                url: `${window.location.origin}${service.url}`,
                                provider: {
                                    "@type": "Organization",
                                    name: "Academic Scribe",
                                },
                            })),
                        }),
                    }}
                />
            </Head>

            {/* Hero Section */}
            <section className="w-full bg-gradient-to-r from-blue-900 to-cyan-500 text-white pt-32 pb-20">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <motion.div
                        initial="hidden"
                        animate="visible"
                        variants={fadeIn}
                        className="max-w-4xl mx-auto text-center"
                    >
                        <h1 className="text-4xl md:text-5xl font-bold leading-tight mb-6">
                            Professional Academic Writing Services
                        </h1>
                        <p className="text-xl mb-8 text-blue-100">
                            Expert assistance with all your academic writing
                            needs. Our team of PhD-level writers delivers
                            excellence across every subject and academic level.
                        </p>

                        <div className="grid md:grid-cols-4 gap-6 mb-10 max-w-4xl mx-auto">
                            <div className="flex items-center justify-center space-x-2">
                                <Award className="h-6 w-6 text-yellow-400" />
                                <span className="text-blue-100">
                                    PhD Writers
                                </span>
                            </div>
                            <div className="flex items-center justify-center space-x-2">
                                <Shield className="h-6 w-6 text-green-400" />
                                <span className="text-blue-100">
                                    Plagiarism-Free
                                </span>
                            </div>
                            <div className="flex items-center justify-center space-x-2">
                                <Clock className="h-6 w-6 text-orange-400" />
                                <span className="text-blue-100">
                                    On-Time Delivery
                                </span>
                            </div>
                            <div className="flex items-center justify-center space-x-2">
                                <CheckCircle className="h-6 w-6 text-green-400" />
                                <span className="text-blue-100">
                                    24/7 Support
                                </span>
                            </div>
                        </div>

                        <motion.div
                            className="flex flex-col sm:flex-row justify-center space-y-4 sm:space-y-0 sm:space-x-4"
                            initial={{ opacity: 0 }}
                            animate={{ opacity: 1 }}
                            transition={{ delay: 0.4, duration: 0.6 }}
                        >
                            <a
                                href="/place-order"
                                className="bg-white text-blue-800 hover:bg-gray-100 px-8 py-4 rounded-md font-medium shadow-lg transition-colors duration-200 flex items-center justify-center"
                            >
                                Order Now{" "}
                                <ArrowRight className="ml-2 h-5 w-5" />
                            </a>
                            <a
                                href="#services"
                                className="border border-white text-white hover:bg-white hover:text-blue-800 px-8 py-4 rounded-md font-medium transition-colors duration-200"
                            >
                                Browse Services
                            </a>
                        </motion.div>
                    </motion.div>
                </div>
            </section>

            {/* Services Grid Section */}
            <section
                id="services"
                className="py-20 bg-gray-50 dark:bg-gray-900"
            >
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center mb-16"
                    >
                        <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                            Our Academic Writing Services
                        </h2>
                        <div className="h-1 w-24 bg-blue-600 mx-auto rounded-full mb-6"></div>
                        <p className="text-xl text-gray-600 dark:text-gray-300 max-w-3xl mx-auto">
                            Choose from our comprehensive range of academic
                            writing services, each delivered by subject-matter
                            experts with advanced degrees.
                        </p>
                    </motion.div>

                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={staggerContainer}
                        className="grid md:grid-cols-2 lg:grid-cols-3 gap-8"
                    >
                        {services.map((service, index) => {
                            const IconComponent =
                                iconMap[service.icon] || FileText;

                            return (
                                <motion.div
                                    key={index}
                                    variants={fadeIn}
                                    className="bg-white dark:bg-gray-800 rounded-lg shadow-lg hover:shadow-xl transition-all duration-300 hover:transform hover:-translate-y-2 overflow-hidden"
                                >
                                    <div className="p-8">
                                        <div className="flex items-center justify-center h-16 w-16 rounded-full bg-blue-600 text-white mb-6 mx-auto">
                                            <IconComponent className="h-8 w-8" />
                                        </div>

                                        <h3 className="text-xl font-bold text-gray-900 dark:text-white mb-4 text-center">
                                            {service.title}
                                        </h3>

                                        <p className="text-gray-600 dark:text-gray-300 mb-6 text-center">
                                            {service.description}
                                        </p>

                                        <div className="space-y-2 mb-6">
                                            {service.features.map(
                                                (feature, featureIndex) => (
                                                    <div
                                                        key={featureIndex}
                                                        className="flex items-center text-sm text-gray-600 dark:text-gray-300"
                                                    >
                                                        <CheckCircle className="h-4 w-4 text-green-500 mr-2 flex-shrink-0" />
                                                        {feature}
                                                    </div>
                                                )
                                            )}
                                        </div>

                                        <a
                                            href={service.url}
                                            className="block w-full bg-blue-600 hover:bg-blue-700 text-white text-center py-3 px-4 rounded-md font-medium transition-colors duration-200"
                                        >
                                            Learn More
                                        </a>
                                    </div>
                                </motion.div>
                            );
                        })}
                    </motion.div>
                </div>
            </section>

            {/* Why Choose Us Section */}
            <section className="py-20 bg-white dark:bg-gray-800">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center mb-16"
                    >
                        <h2 className="text-3xl font-bold text-gray-900 dark:text-white mb-4">
                            Why Students Choose Academic Scribe
                        </h2>
                        <div className="h-1 w-24 bg-blue-600 mx-auto rounded-full mb-6"></div>
                        <p className="text-xl text-gray-600 dark:text-gray-300 max-w-3xl mx-auto">
                            We've helped thousands of students achieve academic
                            success with our professional writing services.
                        </p>
                    </motion.div>

                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={staggerContainer}
                        className="grid md:grid-cols-2 lg:grid-cols-4 gap-8"
                    >
                        <motion.div variants={fadeIn} className="text-center">
                            <div className="flex items-center justify-center h-16 w-16 rounded-full bg-blue-600 text-white mb-4 mx-auto">
                                <Award className="h-8 w-8" />
                            </div>
                            <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-2">
                                Expert Writers
                            </h3>
                            <p className="text-gray-600 dark:text-gray-300">
                                PhD-qualified writers with expertise in your
                                subject area
                            </p>
                        </motion.div>

                        <motion.div variants={fadeIn} className="text-center">
                            <div className="flex items-center justify-center h-16 w-16 rounded-full bg-green-600 text-white mb-4 mx-auto">
                                <Shield className="h-8 w-8" />
                            </div>
                            <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-2">
                                Plagiarism-Free
                            </h3>
                            <p className="text-gray-600 dark:text-gray-300">
                                100% original content with comprehensive
                                plagiarism checks
                            </p>
                        </motion.div>

                        <motion.div variants={fadeIn} className="text-center">
                            <div className="flex items-center justify-center h-16 w-16 rounded-full bg-orange-600 text-white mb-4 mx-auto">
                                <Clock className="h-8 w-8" />
                            </div>
                            <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-2">
                                On-Time Delivery
                            </h3>
                            <p className="text-gray-600 dark:text-gray-300">
                                Guaranteed delivery before your deadline, every
                                time
                            </p>
                        </motion.div>

                        <motion.div variants={fadeIn} className="text-center">
                            <div className="flex items-center justify-center h-16 w-16 rounded-full bg-purple-600 text-white mb-4 mx-auto">
                                <CheckCircle className="h-8 w-8" />
                            </div>
                            <h3 className="text-lg font-bold text-gray-900 dark:text-white mb-2">
                                24/7 Support
                            </h3>
                            <p className="text-gray-600 dark:text-gray-300">
                                Round-the-clock customer support and assistance
                            </p>
                        </motion.div>
                    </motion.div>
                </div>
            </section>

            {/* CTA Section */}
            <section className="py-20 bg-gradient-to-r from-blue-600 to-indigo-700">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <motion.div
                        initial="hidden"
                        whileInView="visible"
                        viewport={{ once: true }}
                        variants={fadeIn}
                        className="text-center text-white"
                    >
                        <h2 className="text-3xl md:text-4xl font-bold mb-6">
                            Ready to Excel in Your Academic Journey?
                        </h2>
                        <p className="text-xl mb-8 max-w-3xl mx-auto opacity-90">
                            Join thousands of successful students who have
                            improved their grades with our professional academic
                            writing services.
                        </p>

                        <div className="flex flex-col sm:flex-row justify-center items-center space-y-4 sm:space-y-0 sm:space-x-6">
                            <a
                                href="/place-order"
                                className="px-8 py-4 bg-white text-blue-600 hover:bg-gray-100 rounded-md font-bold text-lg shadow-xl transition-all duration-200 transform hover:scale-105 flex items-center"
                            >
                                Start Your Order{" "}
                                <ArrowRight className="ml-2 h-5 w-5" />
                            </a>
                            <a
                                href="/contact"
                                className="px-8 py-4 border-2 border-white text-white hover:bg-white hover:text-blue-600 rounded-md font-bold text-lg transition-colors duration-200"
                            >
                                Get Free Quote
                            </a>
                        </div>

                        <div className="mt-8 text-sm opacity-75">
                            <p>
                                ✓ No obligation • ✓ 100% confidential • ✓
                                Money-back guarantee
                            </p>
                        </div>
                    </motion.div>
                </div>
            </section>
        </MainLayout>
    );
}

ServicesIndex.propTypes = {
    services: PropTypes.arrayOf(
        PropTypes.shape({
            title: PropTypes.string.isRequired,
            description: PropTypes.string.isRequired,
            url: PropTypes.string.isRequired,
            icon: PropTypes.string.isRequired,
            features: PropTypes.arrayOf(PropTypes.string).isRequired,
        })
    ).isRequired,
};
