import { useState } from "react";
import { Head } from "@inertiajs/react";
import ClientLayout from "@/Layouts/ClientLayout";
import { Bell, CreditCard, Settings as SettingsIcon, Shield, CheckCircle } from "lucide-react";
import NotificationSettings from "./NotificationSettings";
import BillingSettings from "./BillingSettings";
import PreferencesSettings from "./PreferencesSettings";
import PrivacySettings from "./PrivacySettings";

export default function Settings({
    clientPreferences,
    notificationPreferences,
    user,
    academicLevels,
    paperTypes,
    subjects,
    languages,
    countries,
}) {
    const [activeTab, setActiveTab] = useState("notifications");
    const [successMessage, setSuccessMessage] = useState(null);

    const tabs = [
        { id: "notifications", label: "Notifications", icon: Bell },
        { id: "billing", label: "Billing & Payment", icon: CreditCard },
        { id: "preferences", label: "Order Preferences", icon: SettingsIcon },
        { id: "privacy", label: "Privacy & Communication", icon: Shield },
    ];

    return (
        <ClientLayout>
            <Head title="Settings" />

            <div className="py-8">
                <div className="max-w-7xl mx-auto sm:px-6 lg:px-8">
                    {/* Header */}
                    <div className="mb-6">
                        <h1 className="text-3xl font-bold text-gray-900 dark:text-white">
                            Settings
                        </h1>
                        <p className="mt-2 text-sm text-gray-600 dark:text-gray-400">
                            Manage your account settings and preferences
                        </p>
                    </div>

                    {/* Success Message */}
                    {successMessage && (
                        <div className="mb-6 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg p-4 flex items-center gap-2">
                            <CheckCircle className="h-5 w-5 text-green-600 dark:text-green-400" />
                            <p className="text-sm text-green-800 dark:text-green-200">
                                {successMessage}
                            </p>
                        </div>
                    )}

                    <div className="bg-white dark:bg-gray-800 shadow rounded-lg overflow-hidden">
                        {/* Tabs */}
                        <div className="border-b border-gray-200 dark:border-gray-700">
                            <nav className="flex -mb-px">
                                {tabs.map((tab) => {
                                    const Icon = tab.icon;
                                    return (
                                        <button
                                            key={tab.id}
                                            onClick={() => {
                                                setActiveTab(tab.id);
                                                setSuccessMessage(null);
                                            }}
                                            className={`
                                                px-6 py-4 text-sm font-medium border-b-2 transition-colors
                                                ${
                                                    activeTab === tab.id
                                                        ? "border-indigo-500 text-indigo-600 dark:text-indigo-400"
                                                        : "border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300 dark:text-gray-400 dark:hover:text-gray-300"
                                                }
                                            `}
                                        >
                                            <div className="flex items-center gap-2">
                                                <Icon className="h-5 w-5" />
                                                {tab.label}
                                            </div>
                                        </button>
                                    );
                                })}
                            </nav>
                        </div>

                        {/* Tab Content */}
                        <div className="p-6">
                            {activeTab === "notifications" && (
                                <NotificationSettings
                                    notificationPreferences={notificationPreferences}
                                    onSuccess={(message) => setSuccessMessage(message)}
                                />
                            )}

                            {activeTab === "billing" && (
                                <BillingSettings
                                    user={user}
                                    countries={countries}
                                    onSuccess={(message) => setSuccessMessage(message)}
                                />
                            )}

                            {activeTab === "preferences" && (
                                <PreferencesSettings
                                    clientPreferences={clientPreferences}
                                    academicLevels={academicLevels}
                                    paperTypes={paperTypes}
                                    subjects={subjects}
                                    languages={languages}
                                    onSuccess={(message) => setSuccessMessage(message)}
                                />
                            )}

                            {activeTab === "privacy" && (
                                <PrivacySettings
                                    clientPreferences={clientPreferences}
                                    onSuccess={(message) => setSuccessMessage(message)}
                                />
                            )}
                        </div>
                    </div>
                </div>
            </div>
        </ClientLayout>
    );
}







