import { useState } from 'react';
import { Head, Link, router } from '@inertiajs/react';
import WriterLayout from '@/Layouts/WriterLayout';
import { 
    Clock, 
    FileText, 
    AlertTriangle, 
    CheckCircle, 
    XCircle, 
    Edit3,
    Download,
    Eye,
    Calendar,
    User,
    BookOpen,
    AlertCircle,
    RefreshCw
} from 'lucide-react';
import { formatCurrency, formatDeadline } from '@/lib/formatters';

export default function Assignments({ assignedOrders, expiredOrders, completedOrders, flash }) {
    const [activeTab, setActiveTab] = useState('active');
    const [showInfractions, setShowInfractions] = useState({});
    const [processingOrders, setProcessingOrders] = useState(new Set());
    const [successOrders, setSuccessOrders] = useState(new Set());

    const getStatusDisplayName = (status) => {
        const statusMap = {
            'assigned': 'Assigned',
            'in_progress': 'In Progress',
            'submitted': 'Submitted',
            'under_review': 'Under Review',
            'revision_requested': 'Revision Requested'
        };
        return statusMap[status] || status;
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'assigned':
                return <BookOpen size={16} className="text-blue-500" />;
            case 'in_progress':
                return <Edit3 size={16} className="text-yellow-500" />;
            case 'submitted':
                return <CheckCircle size={16} className="text-green-500" />;
            case 'under_review':
                return <Eye size={16} className="text-purple-500" />;
            case 'revision_requested':
                return <AlertCircle size={16} className="text-orange-500" />;
            default:
                return <FileText size={16} className="text-gray-500" />;
        }
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'assigned':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'in_progress':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'submitted':
                return 'bg-green-100 text-green-800 border-green-200';
            case 'under_review':
                return 'bg-purple-100 text-purple-800 border-purple-200';
            case 'revision_requested':
                return 'bg-orange-100 text-orange-800 border-orange-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const hasInfractions = (order) => {
        return order.writer_infractions && order.writer_infractions.length > 0;
    };

    const getInfractionSeverityColor = (severity) => {
        switch (severity) {
            case 'warning':
                return 'bg-yellow-100 text-yellow-800 border-yellow-200';
            case 'minor':
                return 'bg-blue-100 text-blue-800 border-blue-200';
            case 'major':
                return 'bg-orange-100 text-orange-800 border-orange-200';
            case 'critical':
                return 'bg-red-100 text-red-800 border-red-200';
            default:
                return 'bg-gray-100 text-gray-800 border-gray-200';
        }
    };

    const getInfractionTypeLabel = (type) => {
        const typeMap = {
            'deadline_violation': 'Deadline Violation',
            'quality_issues': 'Quality Issues',
            'plagiarism': 'Plagiarism',
            'communication_failure': 'Communication Failure',
            'policy_violation': 'Policy Violation',
            'unprofessional_behavior': 'Unprofessional Behavior',
            'client_complaint': 'Client Complaint',
            'other': 'Other'
        };
        return typeMap[type] || type;
    };


    const renderOrderCard = (order, isExpired = false) => {
        const hasInfraction = hasInfractions(order);
        
        return (
            <div key={order.id} className={`bg-white rounded-lg shadow-md border-l-4 ${
                isExpired ? 'border-red-500 opacity-75' : 'border-green-500'
            }`}>
                <div className="p-6">
                    <div className="flex items-start justify-between mb-4">
                        <div className="flex-1">
                            <div className="flex items-center gap-3 mb-2">
                                <h3 className="text-lg font-semibold text-gray-900">
                                    {order.title}
                                </h3>
                                <span className={`inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border ${getStatusColor(order.order_status)}`}>
                                    {getStatusIcon(order.order_status)}
                                    <span className="ml-1">{getStatusDisplayName(order.order_status)}</span>
                                </span>
                                {hasInfraction && (
                                    <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-orange-100 text-orange-800 border-orange-200">
                                        <AlertTriangle size={12} className="mr-1" />
                                        Has Infractions
                                    </span>
                                )}
                            </div>
                            
                            <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 text-sm text-gray-600">
                                <div className="flex items-center gap-2">
                                    <BookOpen size={16} />
                                    <span>{order.type_of_paper}</span>
                                </div>
                                <div className="flex items-center gap-2">
                                    <User size={16} />
                                    <span>{order.academic_level}</span>
                                </div>
                                <div className="flex items-center gap-2">
                                    <FileText size={16} />
                                    <span>{order.pages} pages</span>
                                </div>
                                <div className="flex items-center gap-2">
                                    <span className="font-medium">{formatCurrency(order.writer_amount)}</span>
                                </div>
                            </div>
                        </div>
                        
                        {isExpired && (
                            <div className="flex items-center gap-2 text-red-600">
                                <AlertTriangle size={20} />
                                <span className="text-sm font-medium">Expired</span>
                            </div>
                        )}
                    </div>

                    <div className="border-t pt-4">
                        <div className="flex items-center justify-between text-sm">
                            <div className="flex items-center gap-4">
                                <div className="flex items-center gap-2">
                                    <Calendar size={16} className="text-gray-500" />
                                    <span className="text-gray-600">
                                        Assigned: {order.writer_assigned_at ? new Date(order.writer_assigned_at).toLocaleDateString() : 'N/A'}
                                    </span>
                                </div>
                                <div className="flex items-center gap-2">
                                    <Clock size={16} className="text-gray-500" />
                                    <span className={`font-medium ${
                                        isExpired ? 'text-red-600' : 'text-gray-900'
                                    }`}>
                                        Deadline: {formatDeadline(order.writer_deadline)}
                                    </span>
                                </div>
                            </div>
                            
                            <div className="flex items-center gap-2">
                                {!isExpired && (
                                    <>
                                        {order.order_status === 'assigned' && (
                                            <div className="flex items-center gap-2">
                                                <button
                                                    onClick={() => handleStartWorking(order.id)}
                                                    disabled={processingOrders.has(order.id) || successOrders.has(order.id)}
                                                    className={`inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-white focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500 ${
                                                        processingOrders.has(order.id)
                                                            ? 'bg-gray-400 cursor-not-allowed'
                                                            : successOrders.has(order.id)
                                                            ? 'bg-green-500 cursor-not-allowed'
                                                            : 'bg-green-600 hover:bg-green-700'
                                                    }`}
                                                >
                                                    {processingOrders.has(order.id) ? (
                                                        <>
                                                            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white mr-2"></div>
                                                            Starting...
                                                        </>
                                                    ) : successOrders.has(order.id) ? (
                                                        <>
                                                            <CheckCircle className="w-4 h-4 mr-2" />
                                                            Started!
                                                        </>
                                                    ) : (
                                                        'Start Working'
                                                    )}
                                                </button>
                                                
                                                {successOrders.has(order.id) && (
                                                    <button
                                                        onClick={() => window.location.reload()}
                                                        className="inline-flex items-center px-2 py-2 border border-gray-300 text-sm leading-4 font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                                        title="Refresh to see updated status"
                                                    >
                                                        <RefreshCw className="w-4 h-4" />
                                                    </button>
                                                )}
                                            </div>
                                        )}
                                        
                                        {order.order_status === 'in_progress' && (
                                            <Link
                                                href={route('writer.orders.submit-work', order.id)}
                                                className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-white bg-blue-600 hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-blue-500"
                                            >
                                                Submit Work
                                            </Link>
                                        )}
                                        
                                        <Link
                                            href={route('writer.orders.show', order.id)}
                                            className="inline-flex items-center px-3 py-2 border border-gray-300 text-sm leading-4 font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-indigo-500"
                                        >
                                            View Details
                                        </Link>
                                    </>
                                )}
                                
                                {isExpired && (
                                    <span className="text-sm text-red-600 font-medium">
                                        Contact support for assistance
                                    </span>
                                )}
                                
                                {/* Allow submission for expired orders that haven't been submitted */}
                                {isExpired && order.order_status !== 'submitted' && order.order_status !== 'under_review' && (
                                    <Link
                                        href={route('writer.orders.submit-work', order.id)}
                                        className="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-white bg-orange-600 hover:bg-orange-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-orange-500"
                                    >
                                        Submit Work (Late)
                                    </Link>
                                )}
                            </div>
                        </div>

                        {/* Infractions Section */}
                        {hasInfraction && (
                            <div className="mt-4 p-4 bg-orange-50 border border-orange-200 rounded-md">
                                <div className="flex items-center justify-between mb-3">
                                    <h4 className="text-sm font-medium text-orange-800 flex items-center gap-2">
                                        <AlertTriangle size={16} />
                                        Infractions ({order.writer_infractions.length})
                                    </h4>
                                    <button
                                        onClick={() => setShowInfractions(prev => ({ ...prev, [order.id]: !prev[order.id] }))}
                                        className="text-sm text-orange-600 hover:text-orange-800 underline"
                                    >
                                        {showInfractions[order.id] ? 'Hide Details' : 'Show Details'}
                                    </button>
                                </div>
                                {showInfractions[order.id] && (
                                    <div className="space-y-3">
                                        {order.writer_infractions.map((infraction, index) => (
                                            <div key={index} className="bg-white p-3 rounded border border-orange-200">
                                                <div className="flex items-center justify-between mb-2">
                                                    <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium border ${getInfractionSeverityColor(infraction.severity_level)}`}>
                                                        {infraction.severity_level.charAt(0).toUpperCase() + infraction.severity_level.slice(1)}
                                                    </span>
                                                    <span className="text-xs text-gray-500">
                                                        {new Date(infraction.reported_at).toLocaleDateString()}
                                                    </span>
                                                </div>
                                                <div className="text-sm font-medium text-gray-900 mb-1">
                                                    {getInfractionTypeLabel(infraction.infraction_type)}
                                                </div>
                                                <div className="text-sm text-gray-600 mb-2">
                                                    {infraction.description}
                                                </div>
                                                <div className="text-xs text-gray-500">
                                                    Status: {infraction.status.replace('_', ' ').replace(/\b\w/g, l => l.toUpperCase())}
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </div>
                        )}
                    </div>
                </div>
            </div>
        );
    };

    const handleStartWorking = async (orderId) => {
        // Add order to processing set to show loading state
        setProcessingOrders(prev => new Set(prev).add(orderId));

        try {
            // Make POST request to start working
            await router.post(route('writer.orders.start-working', orderId), {}, {
                onSuccess: (response) => {
                    // Add order to success set to show success state
                    setSuccessOrders(prev => new Set(prev).add(orderId));
                    
                    // Remove success state after 3 seconds
                    setTimeout(() => {
                        setSuccessOrders(prev => {
                            const newSet = new Set(prev);
                            newSet.delete(orderId);
                            return newSet;
                        });
                    }, 3000);
                    
                    // Show success feedback
                    console.log('Work started successfully!');
                    
                    // Refresh the page to show updated data
                    window.location.reload();
                },
                onError: (errors) => {
                    console.error('Failed to start working:', errors);
                },
                onFinish: () => {
                    // Remove order from processing set
                    setProcessingOrders(prev => {
                        const newSet = new Set(prev);
                        newSet.delete(orderId);
                        return newSet;
                    });
                }
            });
        } catch (error) {
            console.error('Error starting work:', error);
            // Remove order from processing set
            setProcessingOrders(prev => {
                const newSet = new Set(prev);
                newSet.delete(orderId);
                return newSet;
            });
        }
    };

    return (
        <WriterLayout>
            <Head title="My Assignments" />
            
            <div className="py-6">
                <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                    <div className="mb-8">
                        <h1 className="text-3xl font-bold text-gray-900">My Assignments</h1>
                        <p className="mt-2 text-gray-600">
                            Manage your currently assigned orders and track their progress
                        </p>
                    </div>

                    {/* Flash Messages */}
                    {flash?.success && (
                        <div className="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                            <div className="flex items-center">
                                <CheckCircle className="w-5 h-5 text-green-500 mr-3" />
                                <p className="text-sm text-green-700">
                                    {flash.success}
                                </p>
                            </div>
                        </div>
                    )}

                    {flash?.error && (
                        <div className="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                            <div className="flex items-center">
                                <AlertCircle className="w-5 h-5 text-red-500 mr-3" />
                                <p className="text-sm text-red-700">
                                    {flash.error}
                                </p>
                            </div>
                        </div>
                    )}

                    {/* Tab Navigation */}
                    <div className="border-b border-gray-200 mb-6">
                        <nav className="-mb-px flex space-x-8">
                            <button
                                onClick={() => setActiveTab('active')}
                                className={`py-2 px-1 border-b-2 font-medium text-sm ${
                                    activeTab === 'active'
                                        ? 'border-green-500 text-green-600'
                                        : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                }`}
                            >
                                Active Assignments ({assignedOrders.data.length})
                            </button>
                            <button
                                onClick={() => setActiveTab('expired')}
                                className={`py-2 px-1 border-b-2 font-medium text-sm ${
                                    activeTab === 'expired'
                                        ? 'border-red-500 text-red-600'
                                        : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                }`}
                            >
                                Expired Orders ({expiredOrders.length})
                            </button>
                            <button
                                onClick={() => setActiveTab('completed')}
                                className={`py-2 px-1 border-b-2 font-medium text-sm ${
                                    activeTab === 'completed'
                                        ? 'border-green-500 text-green-600'
                                        : 'border-transparent text-gray-500 hover:text-gray-700 hover:border-gray-300'
                                }`}
                            >
                                Completed Orders ({completedOrders.length})
                            </button>
                        </nav>
                    </div>

                    {/* Active Assignments Tab */}
                    {activeTab === 'active' && (
                        <div>
                            {assignedOrders.data.length === 0 ? (
                                <div className="text-center py-12">
                                    <FileText size={48} className="mx-auto text-gray-400 mb-4" />
                                    <h3 className="text-lg font-medium text-gray-900 mb-2">No Active Assignments</h3>
                                    <p className="text-gray-600 mb-6">
                                        You don't have any active assignments at the moment.
                                    </p>
                                    <Link
                                        href={route('writer.orders')}
                                        className="inline-flex items-center px-4 py-2 border border-transparent text-sm font-medium rounded-md text-white bg-green-600 hover:bg-green-700 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-green-500"
                                    >
                                        Browse Available Orders
                                    </Link>
                                </div>
                            ) : (
                                <div className="space-y-6">
                                    {assignedOrders.data.map(order => renderOrderCard(order, false))}
                                    
                                    {/* Pagination */}
                                    {assignedOrders.links && assignedOrders.links.length > 3 && (
                                        <div className="mt-8">
                                            <nav className="flex items-center justify-between">
                                                <div className="flex-1 flex justify-between sm:hidden">
                                                    {assignedOrders.prev_page_url && (
                                                        <Link
                                                            href={assignedOrders.prev_page_url}
                                                            className="relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                                                        >
                                                            Previous
                                                        </Link>
                                                    )}
                                                    {assignedOrders.next_page_url && (
                                                        <Link
                                                            href={assignedOrders.next_page_url}
                                                            className="ml-3 relative inline-flex items-center px-4 py-2 border border-gray-300 text-sm font-medium rounded-md text-gray-700 bg-white hover:bg-gray-50"
                                                        >
                                                            Next
                                                        </Link>
                                                    )}
                                                </div>
                                                <div className="hidden sm:flex-1 sm:flex sm:items-center sm:justify-between">
                                                    <div>
                                                        <p className="text-sm text-gray-700">
                                                            Showing{' '}
                                                            <span className="font-medium">{assignedOrders.from}</span>
                                                            {' '}to{' '}
                                                            <span className="font-medium">{assignedOrders.to}</span>
                                                            {' '}of{' '}
                                                            <span className="font-medium">{assignedOrders.total}</span>
                                                            {' '}results
                                                        </p>
                                                    </div>
                                                    <div>
                                                        <nav className="relative z-0 inline-flex rounded-md shadow-sm -space-x-px">
                                                            {assignedOrders.links.map((link, index) => (
                                                                <Link
                                                                    key={index}
                                                                    href={link.url}
                                                                    className={`relative inline-flex items-center px-4 py-2 text-sm font-medium ${
                                                                        link.url === null
                                                                            ? 'text-gray-400 cursor-not-allowed'
                                                                            : link.active
                                                                            ? 'z-10 bg-green-50 border-green-500 text-green-600'
                                                                            : 'bg-white border-gray-300 text-gray-500 hover:bg-gray-50'
                                                                    } border`}
                                                                    dangerouslySetInnerHTML={{ __html: link.label }}
                                                                />
                                                            ))}
                                                        </nav>
                                                    </div>
                                                </div>
                                            </nav>
                                        </div>
                                    )}
                                </div>
                            )}
                        </div>
                    )}

                    {/* Expired Orders Tab */}
                    {activeTab === 'expired' && (
                        <div>
                            {expiredOrders.length === 0 ? (
                                <div className="text-center py-12">
                                    <CheckCircle size={48} className="mx-auto text-green-400 mb-4" />
                                    <h3 className="text-lg font-medium text-gray-900 mb-2">No Expired Orders</h3>
                                    <p className="text-gray-600">
                                        Great! All your assignments are up to date.
                                    </p>
                                </div>
                            ) : (
                                <div className="space-y-6">
                                    <div className="bg-yellow-50 border border-yellow-200 rounded-md p-4 mb-6">
                                        <div className="flex">
                                            <AlertTriangle size={20} className="text-yellow-400" />
                                            <div className="ml-3">
                                                <h3 className="text-sm font-medium text-yellow-800">
                                                    Expired Orders Notice
                                                </h3>
                                                <div className="mt-2 text-sm text-yellow-700">
                                                    <p>
                                                        The following orders have passed their deadline. Please contact support for assistance with expired orders.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    {expiredOrders.map(order => renderOrderCard(order, true))}
                                </div>
                            )}
                        </div>
                    )}

                    {/* Completed Orders Tab */}
                    {activeTab === 'completed' && (
                        <div>
                            {completedOrders.length === 0 ? (
                                <div className="text-center py-12">
                                    <CheckCircle size={48} className="mx-auto text-green-400 mb-4" />
                                    <h3 className="text-lg font-medium text-gray-900 mb-2">No Completed Orders</h3>
                                    <p className="text-gray-600">
                                        You haven't completed any orders yet. Keep up the great work!
                                    </p>
                                </div>
                            ) : (
                                <div className="space-y-6">
                                    <div className="bg-green-50 border border-green-200 rounded-md p-4 mb-6">
                                        <div className="flex">
                                            <CheckCircle size={20} className="text-green-400" />
                                            <div className="ml-3">
                                                <h3 className="text-sm font-medium text-green-800">
                                                    Completed Orders
                                                </h3>
                                                <div className="mt-2 text-sm text-green-700">
                                                    <p>
                                                        These are your successfully completed orders. You can view details and track any infractions or issues.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    {completedOrders.map(order => renderOrderCard(order, false))}
                                </div>
                            )}
                        </div>
                    )}
                </div>
            </div>
        </WriterLayout>
    );
} 