import React from "react";
import { Head, Link, usePage, router } from "@inertiajs/react";
import { useEffect, useState } from "react";
import WriterLayout from "@/Layouts/WriterLayout";
import { Eye, Trash2, BadgeCheck, Clock, XCircle } from "lucide-react";

export default function MyBids({ bids }) {
    const { flash } = usePage().props;
    const [flashMsg, setFlashMsg] = useState(flash.message);
    useEffect(() => {
        if (flashMsg) {
            const timer = setTimeout(() => {
                setFlashMsg("");
            }, 3000);
            return () => clearTimeout(timer);
        }
    }, [flashMsg]);

    useEffect(() => {
        setFlashMsg(flash.message);
    }, [flash.message]);

    const getStatusStyle = (status) => {
        switch (status) {
            case "pending":
                return {
                    text: "Pending",
                    icon: <Clock size={16} className="text-yellow-600" />,
                    className:
                        "bg-yellow-100 text-yellow-800 dark:bg-yellow-800 dark:text-yellow-100",
                };
            case "accepted":
                return {
                    text: "Accepted",
                    icon: <BadgeCheck size={16} className="text-green-600" />,
                    className:
                        "bg-green-100 text-green-800 dark:bg-green-800 dark:text-green-100",
                };
            case "rejected":
            default:
                return {
                    text: "Rejected",
                    icon: <XCircle size={16} className="text-red-600" />,
                    className:
                        "bg-red-100 text-red-800 dark:bg-red-800 dark:text-red-100",
                };
        }
    };

    return (
        <WriterLayout>
            <Head title="My Bids" />
            {bids.length === 0 ? (
                <div className="text-center py-12">
                    <p className="text-gray-700 dark:text-gray-300 text-lg font-medium">
                        You have not placed any bids yet.
                    </p>
                </div>
            ) : (
                <>
                    {/* Desktop Table */}
                    <div className="hidden lg:block">
                        <div className="px-4 py-4 sm:px-6 lg:px-8">
                            <h1 className="text-2xl font-semibold text-gray-900 dark:text-white">
                                My Bids
                            </h1>
                        </div>

                        {flashMsg && (
                            <div
                                className="bg-red-900 border border-green-400 text-gray-100 px-4 py-3 rounded-lg relative"
                                role="alert"
                            >
                                <span className="block sm:inline">
                                    {flashMsg}
                                </span>
                            </div>
                        )}
                        <div className="rounded-xl overflow-hidden shadow-lg bg-gradient-to-br from-green-700 via-green-600 to-green-700 dark:bg-none dark:bg-gray-900">
                            <table className="w-full table-auto border-collapse">
                                <thead className="bg-green-800 dark:bg-gray-800">
                                    <tr>
                                        <th className="px-6 py-4 text-left text-sm font-semibold tracking-wide text-gray-100 dark:text-gray-300 uppercase">
                                            Order #
                                        </th>
                                        <th className="px-6 py-4 text-left text-sm font-semibold tracking-wide text-gray-100 dark:text-gray-300 uppercase">
                                            Bid Amount
                                        </th>
                                        <th className="px-6 py-4 text-left text-sm font-semibold tracking-wide text-gray-100 dark:text-gray-300 uppercase">
                                            Completion Time
                                        </th>
                                        <th className="px-6 py-4 text-left text-sm font-semibold tracking-wide text-gray-100 dark:text-gray-300 uppercase">
                                            Status
                                        </th>
                                        <th className="px-6 py-4 text-left text-sm font-semibold tracking-wide text-gray-100 dark:text-gray-300 uppercase">
                                            Actions
                                        </th>
                                    </tr>
                                </thead>
                                <tbody>
                                    {bids.map((bid) => {
                                        const status = getStatusStyle(
                                            bid.status
                                        );
                                        return (
                                            <tr
                                                key={bid.id}
                                                className="border-t border-gray-200 dark:border-gray-700 hover:bg-green-800 dark:hover:bg-gray-800 transition"
                                            >
                                                <td className="px-6 py-4 text-gray-100 dark:text-gray-200 font-medium">
                                                    #{bid.order.order_number}
                                                </td>
                                                <td className="px-6 py-4 text-gray-100 dark:text-gray-200">
                                                    ${bid.bid_amount}
                                                </td>
                                                <td className="px-6 py-4 text-gray-100 dark:text-gray-200">
                                                    {new Date(
                                                        bid.completion_time
                                                    ).toLocaleString()}
                                                </td>
                                                <td className="px-6 py-4">
                                                    <span
                                                        className={`inline-flex items-center px-2 py-1 text-xs font-semibold rounded-full ${status.className}`}
                                                    >
                                                        {status.icon}
                                                        <span className="ml-1">
                                                            {status.text}
                                                        </span>
                                                    </span>
                                                </td>
                                                <td className="px-6 py-4">
                                                    <div className="flex space-x-2">
                                                        <Link
                                                            href={`/writer/bids/${bid.id}`}
                                                            className="flex items-center gap-1 px-3 py-1 bg-blue-600 text-white text-sm rounded hover:bg-blue-700 dark:bg-blue-800 dark:hover:bg-blue-900"
                                                        >
                                                            <Eye size={16} />{" "}
                                                            View
                                                        </Link>

                                                        <div>
                                                            <button
                                                                onClick={() => {
                                                                    if (
                                                                        confirm(
                                                                            "Are you sure you want to delete this bid?"
                                                                        )
                                                                    ) {
                                                                        router.delete(
                                                                            route(
                                                                                "writer.bids.destroy",
                                                                                bid.id
                                                                            )
                                                                        );
                                                                    }
                                                                }}
                                                                className="flex items-center gap-2 px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-md transition-colors"
                                                            >
                                                                <Trash2 className="h-5 w-5" />
                                                                Delete
                                                            </button>
                                                        </div>
                                                    </div>
                                                </td>
                                            </tr>
                                        );
                                    })}
                                </tbody>
                            </table>
                        </div>
                    </div>

                    {/* Mobile Cards */}
                    <div className="lg:hidden space-y-4">
                        <div className="px-4 py-2">
                            <h1 className="text-2xl font-semibold text-gray-900 dark:text-white">
                                My Bids
                            </h1>
                        </div>

                        {flashMsg && (
                            <div
                                className="bg-red-900 border border-green-400 text-gray-100 px-4 py-3 rounded-lg relative"
                                role="alert"
                            >
                                <span className="block sm:inline">
                                    {flashMsg}
                                </span>
                            </div>
                        )}
                        {bids.map((bid) => {
                            const status = getStatusStyle(bid.status);
                            return (
                                <div
                                    key={bid.id}
                                    className="bg-gradient-to-br from-green-700 via-green-600 to-green-700 dark:bg-none  dark:bg-gray-800 rounded-xl shadow-md p-4 border border-gray-200 dark:border-gray-700"
                                >
                                    <div className="flex justify-between items-center mb-2">
                                        <h2 className="text-lg font-semibold text-gray-800 dark:text-white">
                                            Order #{bid.order.order_number}
                                        </h2>
                                        <span
                                            className={`inline-flex items-center px-2 py-1 text-xs font-semibold rounded-full ${status.className}`}
                                        >
                                            {status.icon}
                                            <span className="ml-1">
                                                {status.text}
                                            </span>
                                        </span>
                                    </div>
                                    <div className="text-sm space-y-1 text-gray-100 dark:text-gray-300">
                                        <p>
                                            <strong>Bid:</strong> $
                                            {bid.bid_amount}
                                        </p>
                                        <p>
                                            <strong>Due:</strong>{" "}
                                            {new Date(
                                                bid.completion_time
                                            ).toLocaleString()}
                                        </p>
                                    </div>
                                    <div className="flex justify-end space-x-2 mt-4">
                                        <Link
                                            href={`/writer/bids/${bid.id}`}
                                            className="flex items-center gap-1 px-3 py-1 bg-blue-600 text-white rounded-md hover:bg-blue-700 dark:bg-blue-800 dark:hover:bg-blue-900"
                                        >
                                            <Eye size={16} /> View More
                                        </Link>

                                        <button
                                            onClick={() => {
                                                if (
                                                    confirm(
                                                        "Are you sure you want to delete this bid?"
                                                    )
                                                ) {
                                                    router.delete(
                                                        route(
                                                            "writer.bids.destroy",
                                                            bid.id
                                                        )
                                                    );
                                                }
                                            }}
                                            className="flex items-center gap-2 px-4 py-2 bg-red-600 hover:bg-red-700 text-white rounded-md transition-colors"
                                        >
                                            <Trash2 className="h-5 w-5" />
                                            Delete
                                        </button>
                                    </div>
                                </div>
                            );
                        })}
                    </div>
                </>
            )}

            <div className="w-full mx-auto px-6 py-4 space-y-6 dark:bg-gray-900">
                {/* Desktop Table */}
                <div className="hidden lg:block">
                    <div className="rounded-xl overflow-hidden shadow-lg bg-gradient-to-br from-green-700 via-green-600 to-green-700 dark:bg-none dark:bg-gray-900"></div>
                </div>

                {/* Mobile Cards */}
                <div className="lg:hidden space-y-4"></div>
            </div>
        </WriterLayout>
    );
}
