import React, { useState, useEffect } from 'react';
import { Head } from '@inertiajs/react';
import WriterLayout from '@/Layouts/WriterLayout';
import { 
    DollarSign, 
    TrendingUp, 
    Calendar,
    Award,
    BarChart3,
    Clock,
    TrendingDown,
    Target
} from 'lucide-react';

export default function Earnings({ earningsData }) {
    const [data, setData] = useState(earningsData || {
        lifetime_earnings: 0,
        current_month_earnings: 0,
        previous_month_earnings: 0,
        year_to_date_earnings: 0,
        average_earnings_per_order: 0,
        highest_single_order_payment: 0,
        total_orders_completed: 0,
        current_month_orders: 0,
        previous_month_orders: 0,
        year_to_date_orders: 0
    });

    useEffect(() => {
        if (earningsData) {
            setData(earningsData);
        }
    }, [earningsData]);

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    const calculateGrowth = (current, previous) => {
        if (previous === 0) return 100;
        return ((current - previous) / previous * 100).toFixed(1);
    };

    const getGrowthIcon = (current, previous) => {
        const growth = calculateGrowth(current, previous);
        if (growth > 0) {
            return <TrendingUp size={16} className="text-green-500" />;
        } else if (growth < 0) {
            return <TrendingDown size={16} className="text-red-500" />;
        }
        return <BarChart3 size={16} className="text-gray-500" />;
    };

    const getGrowthColor = (current, previous) => {
        const growth = calculateGrowth(current, previous);
        if (growth > 0) return 'text-green-600';
        if (growth < 0) return 'text-red-600';
        return 'text-gray-600';
    };



    return (
        <WriterLayout>
            <Head title="Earnings Overview" />
            
            <div className="p-6 max-w-7xl mx-auto">
                {/* Header */}
                <div className="mb-8">
                    <h1 className="text-3xl font-bold text-gray-900 dark:text-white mb-2">
                        Earnings Overview
                    </h1>
                    <p className="text-gray-600 dark:text-gray-400">
                        Track your financial performance and earnings growth
                    </p>
                </div>

                {/* Financial Summary Cards */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                    {/* Lifetime Earnings */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 shadow-sm p-6">
                        <div className="flex items-center justify-between mb-4">
                            <div className="p-3 bg-blue-100 dark:bg-blue-900/30 rounded-full">
                                <DollarSign size={24} className="text-blue-600 dark:text-blue-400" />
                            </div>
                            <div className="text-right">
                                <span className="text-xs text-gray-500 dark:text-gray-400">Total Orders</span>
                                <p className="text-lg font-semibold text-gray-900 dark:text-white">
                                    {data.total_orders_completed}
                                </p>
                            </div>
                        </div>
                        <div>
                            <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">Lifetime Earnings</p>
                            <p className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                                {formatCurrency(data.lifetime_earnings)}
                            </p>
                        </div>
                    </div>

                    {/* Current Month Earnings */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 shadow-sm p-6">
                        <div className="flex items-center justify-between mb-4">
                            <div className="p-3 bg-green-100 dark:bg-green-900/30 rounded-full">
                                <Calendar size={24} className="text-green-600 dark:text-green-400" />
                            </div>
                            <div className="flex items-center gap-1">
                                {getGrowthIcon(data.current_month_earnings, data.previous_month_earnings)}
                                <span className={`text-xs font-medium ${getGrowthColor(data.current_month_earnings, data.previous_month_earnings)}`}>
                                    {calculateGrowth(data.current_month_earnings, data.previous_month_earnings)}%
                                </span>
                            </div>
                        </div>
                        <div>
                            <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">Current Month</p>
                            <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                                {formatCurrency(data.current_month_earnings)}
                            </p>
                            <p className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                                {data.current_month_orders} orders
                            </p>
                        </div>
                    </div>

                    {/* Year-to-Date Earnings */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 shadow-sm p-6">
                        <div className="flex items-center justify-between mb-4">
                            <div className="p-3 bg-purple-100 dark:bg-purple-900/30 rounded-full">
                                <TrendingUp size={24} className="text-purple-600 dark:text-purple-400" />
                            </div>
                            <div className="text-right">
                                <span className="text-xs text-gray-500 dark:text-gray-400">YTD Orders</span>
                                <p className="text-lg font-semibold text-gray-900 dark:text-white">
                                    {data.year_to_date_orders}
                                </p>
                            </div>
                        </div>
                        <div>
                            <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">Year to Date</p>
                            <p className="text-2xl font-bold text-purple-600 dark:text-purple-400">
                                {formatCurrency(data.year_to_date_earnings)}
                            </p>
                        </div>
                    </div>

                    {/* Average Earnings Per Order */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 shadow-sm p-6">
                        <div className="flex items-center justify-between mb-4">
                            <div className="p-3 bg-orange-100 dark:bg-orange-900/30 rounded-full">
                                <BarChart3 size={24} className="text-orange-600 dark:text-orange-400" />
                            </div>
                            <div className="text-right">
                                <span className="text-xs text-gray-500 dark:text-gray-400">Best Order</span>
                                <p className="text-lg font-semibold text-gray-900 dark:text-white">
                                    {formatCurrency(data.highest_single_order_payment)}
                                </p>
                            </div>
                        </div>
                        <div>
                            <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">Average Per Order</p>
                            <p className="text-2xl font-bold text-orange-600 dark:text-orange-400">
                                {formatCurrency(data.average_earnings_per_order)}
                            </p>
                        </div>
                    </div>
                </div>

                {/* Performance Metrics */}
                <div className="grid grid-cols-1 lg:grid-cols-2 gap-6 mb-8">
                    {/* Monthly Comparison */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 shadow-sm p-6">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                            <Calendar size={20} className="text-gray-600 dark:text-gray-400" />
                            Monthly Performance
                        </h3>
                        <div className="space-y-4">
                            <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                                <div>
                                    <p className="text-sm font-medium text-gray-900 dark:text-white">Current Month</p>
                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                        {new Date().toLocaleDateString('en-US', { month: 'long', year: 'numeric' })}
                                    </p>
                                </div>
                                <div className="text-right">
                                    <p className="text-lg font-semibold text-green-600 dark:text-green-400">
                                        {formatCurrency(data.current_month_earnings)}
                                    </p>
                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                        {data.current_month_orders} orders
                                    </p>
                                </div>
                            </div>
                            <div className="flex items-center justify-between p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                                <div>
                                    <p className="text-sm font-medium text-gray-900 dark:text-white">Previous Month</p>
                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                        {new Date(new Date().getFullYear(), new Date().getMonth() - 1).toLocaleDateString('en-US', { month: 'long', year: 'numeric' })}
                                    </p>
                                </div>
                                <div className="text-right">
                                    <p className="text-lg font-semibold text-gray-600 dark:text-gray-400">
                                        {formatCurrency(data.previous_month_earnings)}
                                    </p>
                                    <p className="text-xs text-gray-500 dark:text-gray-400">
                                        {data.previous_month_orders} orders
                                    </p>
                                </div>
                            </div>
                        </div>
                    </div>

                    {/* Key Achievements */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 shadow-sm p-6">
                        <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                            <Award size={20} className="text-yellow-500" />
                            Key Achievements
                        </h3>
                        <div className="space-y-4">
                            <div className="flex items-center justify-between p-3 bg-yellow-50 dark:bg-yellow-900/20 rounded-lg border border-yellow-200 dark:border-yellow-800">
                                <div className="flex items-center gap-3">
                                    <Award size={20} className="text-yellow-600 dark:text-yellow-400" />
                                    <div>
                                        <p className="text-sm font-medium text-gray-900 dark:text-white">Highest Single Order</p>
                                        <p className="text-xs text-gray-500 dark:text-gray-400">Best performing order</p>
                                    </div>
                                </div>
                                <p className="text-lg font-bold text-yellow-600 dark:text-yellow-400">
                                    {formatCurrency(data.highest_single_order_payment)}
                                </p>
                            </div>
                            <div className="flex items-center justify-between p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-800">
                                <div className="flex items-center gap-3">
                                    <Target size={20} className="text-blue-600 dark:text-blue-400" />
                                    <div>
                                        <p className="text-sm font-medium text-gray-900 dark:text-white">Average Per Order</p>
                                        <p className="text-xs text-gray-500 dark:text-gray-400">Consistent performance</p>
                                    </div>
                                </div>
                                <p className="text-lg font-bold text-blue-600 dark:text-blue-400">
                                    {formatCurrency(data.average_earnings_per_order)}
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                {/* Quick Stats */}
                <div className="bg-white dark:bg-gray-800 rounded-lg border dark:border-gray-700 shadow-sm p-6">
                    <h3 className="text-lg font-semibold text-gray-900 dark:text-white mb-4 flex items-center gap-2">
                        <BarChart3 size={20} className="text-gray-600 dark:text-gray-400" />
                        Quick Statistics
                    </h3>
                    <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
                        <div className="text-center p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                            <p className="text-2xl font-bold text-gray-900 dark:text-white">
                                {data.total_orders_completed}
                            </p>
                            <p className="text-xs text-gray-500 dark:text-gray-400">Total Orders</p>
                        </div>
                        <div className="text-center p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                            <p className="text-2xl font-bold text-green-600 dark:text-green-400">
                                {data.current_month_orders}
                            </p>
                            <p className="text-xs text-gray-500 dark:text-gray-400">This Month</p>
                        </div>
                        <div className="text-center p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                            <p className="text-2xl font-bold text-blue-600 dark:text-blue-400">
                                {data.year_to_date_orders}
                            </p>
                            <p className="text-xs text-gray-500 dark:text-gray-400">Year to Date</p>
                        </div>
                        <div className="text-center p-3 bg-gray-50 dark:bg-gray-700 rounded-lg">
                            <p className="text-2xl font-bold text-purple-600 dark:text-purple-400">
                                {data.lifetime_earnings > 0 ? (data.lifetime_earnings / data.total_orders_completed).toFixed(2) : 0}
                            </p>
                            <p className="text-xs text-gray-500 dark:text-gray-400">Avg/Order</p>
                        </div>
                    </div>
                </div>
            </div>
        </WriterLayout>
    );
} 