import { useState } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import WriterLayout from "@/Layouts/WriterLayout";
import axios from "axios";

import {
    Calendar,
    Clock,
    FileText,
    CheckCircle,
    XCircle,
    AlertTriangle,
    ArrowRight,
    BookOpen,
    ListChecks,
    MessageSquare,
    Upload,
    FileType,
    HardDrive,
} from "lucide-react";

export default function Instructions({
    canTakeEssay,
    waitingPeriod,
    daysRemaining,
    retryDate,
    essayTopic,
    previousAttempt,
    maxFileSize,
    testDuration,
}) {
    console.log("canTakeEssay:", canTakeEssay);
    console.log("essayTopic:", essayTopic);
    console.log("waitingPeriod:", waitingPeriod);
    const [showFullRequirements, setShowFullRequirements] = useState(false);
    const [isLoading, setIsLoading] = useState(false);

    const startEssay = () => {
        const token = document
            .querySelector('meta[name="csrf-token"]')
            ?.getAttribute("content");

        // Show loading state
        setIsLoading(true);

        axios
            .post(
                route("writer.essay.start"),
                {
                    topic_id: essayTopic?.id,
                },
                {
                    headers: {
                        "X-CSRF-TOKEN": token,
                        "Content-Type": "application/json",
                    },
                }
            )
            .then((response) => {
                console.log("Essay start response:", response);
                if (response.data.redirect) {
                    window.location.href = response.data.redirect;
                } else {
                    window.location.href = route("writer.essay.upload");
                }
            })
            .catch((error) => {
                console.error("Error starting essay:", error);
                alert(
                    error.response?.data?.message ||
                        "Failed to start essay test"
                );
            })
            .finally(() => {
                setIsLoading(false);
            });
    };

    return (
        <WriterLayout>
            <Head title="Writer Essay Test - Instructions" />

            <div className="py-12">
                <div className="max-w-4xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                            <h1 className="text-2xl font-bold text-gray-800 dark:text-gray-100 mb-6">
                                Writer Essay Test - Instructions
                            </h1>

                            {waitingPeriod && (
                                <div className="mb-6 p-4 bg-yellow-50 dark:bg-yellow-900/20 border-l-4 border-yellow-400 dark:border-yellow-500 text-yellow-700 dark:text-yellow-300">
                                    <div className="flex">
                                        <div className="flex-shrink-0">
                                            <Calendar
                                                className="h-5 w-5 text-yellow-400 dark:text-yellow-300"
                                                aria-hidden="true"
                                            />
                                        </div>
                                        <div className="ml-3">
                                            <p className="text-sm">
                                                <strong>Waiting Period:</strong>{" "}
                                                You must wait {daysRemaining}{" "}
                                                more days before retrying the
                                                essay test.
                                            </p>
                                            <p className="text-sm mt-1">
                                                You can retry on{" "}
                                                <strong>{retryDate}</strong>.
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            )}

                            {previousAttempt && (
                                <div className="mb-6 p-4 bg-gray-50 dark:bg-gray-700/50 border border-gray-200 dark:border-gray-600 rounded-md">
                                    <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 mb-2">
                                        Previous Attempt
                                    </h2>
                                    <div className="mb-2">
                                        <span className="text-sm text-gray-500 dark:text-gray-400">
                                            Date:
                                        </span>
                                        <span className="ml-2 text-gray-900 dark:text-gray-100">
                                            {previousAttempt.date}
                                        </span>
                                    </div>
                                    {previousAttempt.feedback && (
                                        <div className="mt-2">
                                            <h3 className="text-sm font-medium text-gray-700 dark:text-gray-300">
                                                Feedback:
                                            </h3>
                                            <p className="mt-1 text-sm text-gray-600 dark:text-gray-400 bg-gray-100 dark:bg-gray-800 p-3 rounded">
                                                {previousAttempt.feedback}
                                            </p>
                                        </div>
                                    )}
                                </div>
                            )}

                            <div className="space-y-6">
                                <div>
                                    <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 flex items-center">
                                        <BookOpen className="h-5 w-5 mr-2 text-blue-500 dark:text-blue-400" />
                                        Essay Test Overview
                                    </h2>
                                    <p className="mt-1 text-gray-600 dark:text-gray-400">
                                        This essay test evaluates your writing
                                        skills, critical thinking, and ability
                                        to develop coherent arguments. It's the
                                        final step in our qualification process.
                                    </p>
                                </div>

                                <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                                    <div className="p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg">
                                        <div className="flex items-center mb-2">
                                            <FileText className="h-5 w-5 text-blue-500 dark:text-blue-400 mr-2" />
                                            <h3 className="font-medium text-blue-700 dark:text-blue-300">
                                                Essay Format
                                            </h3>
                                        </div>
                                        <p className="text-sm text-blue-600 dark:text-blue-400">
                                            Upload a well-structured essay with
                                            an introduction, body paragraphs,
                                            and conclusion.
                                        </p>
                                    </div>

                                    <div className="p-4 bg-purple-50 dark:bg-purple-900/20 rounded-lg">
                                        <div className="flex items-center mb-2">
                                            <Clock className="h-5 w-5 text-purple-500 dark:text-purple-400 mr-2" />
                                            <h3 className="font-medium text-purple-700 dark:text-purple-300">
                                                Time Limit
                                            </h3>
                                        </div>
                                        <p className="text-sm text-purple-600 dark:text-purple-400">
                                            {testDuration} minutes to prepare
                                            and upload your essay. Your session
                                            will expire when time is up.
                                        </p>
                                    </div>

                                    <div className="p-4 bg-green-50 dark:bg-green-900/20 rounded-lg">
                                        <div className="flex items-center mb-2">
                                            <MessageSquare className="h-5 w-5 text-green-500 dark:text-green-400 mr-2" />
                                            <h3 className="font-medium text-green-700 dark:text-green-300">
                                                Word Count
                                            </h3>
                                        </div>
                                        <p className="text-sm text-green-600 dark:text-green-400">
                                            Minimum{" "}
                                            {essayTopic?.word_count || 500}{" "}
                                            words. Quality matters more than
                                            quantity, but ensure sufficient
                                            depth.
                                        </p>
                                    </div>
                                </div>

                                <div className="p-4 bg-amber-50 dark:bg-amber-900/20 rounded-lg border border-amber-200 dark:border-amber-800">
                                    <div className="flex items-center mb-2">
                                        <Upload className="h-5 w-5 text-amber-500 dark:text-amber-400 mr-2" />
                                        <h3 className="font-medium text-amber-700 dark:text-amber-300">
                                            File Upload Requirements
                                        </h3>
                                    </div>
                                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mt-3">
                                        <div className="flex items-start">
                                            <FileType className="h-5 w-5 text-amber-500 dark:text-amber-400 mr-2 mt-0.5" />
                                            <div>
                                                <p className="text-sm font-medium text-amber-700 dark:text-amber-300">
                                                    File Types
                                                </p>
                                                <p className="text-xs text-amber-600 dark:text-amber-400">
                                                    Only DOCX or PDF files are
                                                    accepted
                                                </p>
                                            </div>
                                        </div>
                                        <div className="flex items-start">
                                            <HardDrive className="h-5 w-5 text-amber-500 dark:text-amber-400 mr-2 mt-0.5" />
                                            <div>
                                                <p className="text-sm font-medium text-amber-700 dark:text-amber-300">
                                                    File Size
                                                </p>
                                                <p className="text-xs text-amber-600 dark:text-amber-400">
                                                    Maximum {maxFileSize || 10}{" "}
                                                    MB
                                                </p>
                                            </div>
                                        </div>
                                        <div className="flex items-start">
                                            <AlertTriangle className="h-5 w-5 text-amber-500 dark:text-amber-400 mr-2 mt-0.5" />
                                            <div>
                                                <p className="text-sm font-medium text-amber-700 dark:text-amber-300">
                                                    Important
                                                </p>
                                                <p className="text-xs text-amber-600 dark:text-amber-400">
                                                    Ensure your file is properly
                                                    formatted
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                {essayTopic && (
                                    <div className="bg-gray-50 dark:bg-gray-700/50 p-5 rounded-lg border border-gray-200 dark:border-gray-600">
                                        <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 mb-3">
                                            Essay Topic
                                        </h2>
                                        <div className="mb-4">
                                            <h3 className="text-md font-medium text-gray-800 dark:text-gray-100">
                                                {essayTopic.title}
                                            </h3>
                                            <p className="mt-2 text-gray-600 dark:text-gray-400">
                                                {essayTopic.description}
                                            </p>
                                        </div>

                                        <div className="mt-4">
                                            <div
                                                className="flex items-center cursor-pointer text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300"
                                                onClick={() =>
                                                    setShowFullRequirements(
                                                        !showFullRequirements
                                                    )
                                                }
                                            >
                                                <ListChecks className="h-4 w-4 mr-1" />
                                                <span className="text-sm font-medium">
                                                    {showFullRequirements
                                                        ? "Hide Detailed Requirements"
                                                        : "Show Detailed Requirements"}
                                                </span>
                                            </div>

                                            {showFullRequirements && (
                                                <div className="mt-3 p-3 bg-white dark:bg-gray-800 rounded border border-gray-200 dark:border-gray-700">
                                                    <div
                                                        className="prose dark:prose-invert prose-sm max-w-none"
                                                        dangerouslySetInnerHTML={{
                                                            __html: essayTopic.requirements,
                                                        }}
                                                    />
                                                </div>
                                            )}
                                        </div>
                                    </div>
                                )}

                                <div>
                                    <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 flex items-center">
                                        <AlertTriangle className="h-5 w-5 mr-2 text-amber-500 dark:text-amber-400" />
                                        Important Notes
                                    </h2>
                                    <ul className="mt-2 space-y-2 text-gray-600 dark:text-gray-400 list-disc list-inside">
                                        <li>
                                            Once you start the test, you cannot
                                            pause or restart it.
                                        </li>
                                        <li>
                                            You have {testDuration} minutes to
                                            prepare and upload your essay.
                                        </li>
                                        <li>
                                            Your essay must be uploaded as a
                                            DOCX or PDF file.
                                        </li>
                                        <li>
                                            Maximum file size is{" "}
                                            {maxFileSize || 10} MB.
                                        </li>
                                        <li>
                                            Plagiarism will result in automatic
                                            disqualification.
                                        </li>
                                        <li>
                                            If you fail the test, you must wait
                                            3 months before retrying.
                                        </li>
                                    </ul>
                                </div>

                                <div>
                                    <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 flex items-center">
                                        <CheckCircle className="h-5 w-5 mr-2 text-green-500 dark:text-green-400" />
                                        Evaluation Criteria
                                    </h2>
                                    <ul className="mt-2 space-y-1 text-gray-600 dark:text-gray-400 list-disc list-inside">
                                        <li>
                                            Content and argument development
                                        </li>
                                        <li>
                                            Organization and structural
                                            coherence
                                        </li>
                                        <li>
                                            Grammar, spelling, and punctuation
                                        </li>
                                        <li>
                                            Style, clarity, and academic tone
                                        </li>
                                        <li>
                                            Critical thinking and analytical
                                            skills
                                        </li>
                                    </ul>
                                </div>

                                <div className="pt-4 border-t border-gray-200 dark:border-gray-700">
                                    <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200">
                                        Ready to Begin?
                                    </h2>
                                    <p className="mt-1 text-gray-600 dark:text-gray-400">
                                        Make sure you have a stable internet
                                        connection and {testDuration} minutes of
                                        uninterrupted time to prepare and upload
                                        your essay.
                                    </p>

                                    <div className="mt-6 flex justify-end">
                                        <Link href={route("writer.dashboard")}>
                                            <span className="inline-flex items-center px-4 py-2 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md font-semibold text-xs text-gray-700 dark:text-gray-200 uppercase tracking-widest shadow-sm hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-indigo-500 dark:focus:ring-indigo-600 focus:ring-offset-2 dark:focus:ring-offset-gray-800 disabled:opacity-25 transition ease-in-out duration-150 mr-3">
                                                Return to Dashboard
                                            </span>
                                        </Link>

                                        {canTakeEssay && essayTopic ? (
                                            <button
                                                type="button"
                                                onClick={startEssay}
                                                disabled={isLoading}
                                                className="flex items-center px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700 transition disabled:opacity-75 disabled:bg-blue-500"
                                            >
                                                {isLoading ? (
                                                    <>
                                                        <svg
                                                            className="animate-spin -ml-1 mr-2 h-4 w-4 text-white"
                                                            xmlns="http://www.w3.org/2000/svg"
                                                            fill="none"
                                                            viewBox="0 0 24 24"
                                                        >
                                                            <circle
                                                                className="opacity-25"
                                                                cx="12"
                                                                cy="12"
                                                                r="10"
                                                                stroke="currentColor"
                                                                strokeWidth="4"
                                                            ></circle>
                                                            <path
                                                                className="opacity-75"
                                                                fill="currentColor"
                                                                d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                            ></path>
                                                        </svg>
                                                        Starting...
                                                    </>
                                                ) : (
                                                    <>
                                                        Start Essay Test
                                                        <ArrowRight className="ml-2 h-4 w-4" />
                                                    </>
                                                )}
                                            </button>
                                        ) : (
                                            <button
                                                disabled
                                                className="flex items-center px-4 py-2 bg-gray-400 text-white rounded cursor-not-allowed"
                                            >
                                                {waitingPeriod
                                                    ? "Waiting Period Active"
                                                    : "Test Unavailable"}
                                            </button>
                                        )}
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </WriterLayout>
    );
}
