import { useState, useEffect, useRef } from "react";
import { Head, Link, useForm } from "@inertiajs/react";
import WriterLayout from "@/Layouts/WriterLayout";
import axios from "axios";

import {
    Clock,
    FileText,
    Upload as UploadIcon,
    CheckCircle,
    XCircle,
    AlertTriangle,
    ArrowRight,
    FileType,
    HardDrive,
    File,
    Trash2,
    Send,
} from "lucide-react";

export default function Upload({
    writerProfile,
    essayTest,
    remainingTime,
    maxFileSize,
    allowedFileTypes,
    essayTopic,
}) {
    const [timeLeft, setTimeLeft] = useState(remainingTime || 0);
    const [isUploading, setIsUploading] = useState(false);
    const [uploadProgress, setUploadProgress] = useState(0);
    const [uploadedFile, setUploadedFile] = useState(null);
    const [uploadError, setUploadError] = useState(null);
    const [isSubmitting, setIsSubmitting] = useState(false);
    const [showWarning, setShowWarning] = useState(false);
    const [warningLevel, setWarningLevel] = useState(null); // null, 'low', 'medium', 'high'
    const [isAutoSubmitting, setIsAutoSubmitting] = useState(false);
    const fileInputRef = useRef(null);
    const timerRef = useRef(null);
    const pingRef = useRef(null);
    const autoSubmitTimeoutRef = useRef(null);

    // Format time in seconds to HH:MM:SS
    const formatTime = (seconds) => {
        const hours = Math.floor(seconds / 3600);
        const minutes = Math.floor((seconds % 3600) / 60);
        const secs = seconds % 60;
        return `${hours.toString().padStart(2, "0")}:${minutes
            .toString()
            .padStart(2, "0")}:${secs.toString().padStart(2, "0")}`;
    };

    // Format file size to human-readable format
    const formatFileSize = (bytes) => {
        if (bytes < 1024) return bytes + " bytes";
        else if (bytes < 1048576) return (bytes / 1024).toFixed(1) + " KB";
        else return (bytes / 1048576).toFixed(1) + " MB";
    };

    // Get file type display name
    const getFileTypeName = (mimeType) => {
        switch (mimeType) {
            case "application/pdf":
                return "PDF Document";
            case "application/vnd.openxmlformats-officedocument.wordprocessingml.document":
                return "Word Document (DOCX)";
            default:
                return mimeType;
        }
    };

    // Check remaining time from server
    const checkRemainingTime = async () => {
        try {
            const response = await axios.get(route("writer.essay.check-time"));

            // If server indicates the test is expired, handle it
            if (response.data.isExpired) {
                if (response.data.redirect) {
                    window.location.href = response.data.redirect;
                    return;
                }
            }

            if (response.data.remainingTime !== undefined) {
                setTimeLeft(response.data.remainingTime);
            }
        } catch (error) {
            console.error("Error checking time:", error);
            // If we get a 400 error with redirect, follow it
            if (
                error.response?.status === 400 &&
                error.response?.data?.redirect
            ) {
                window.location.href = error.response.data.redirect;
            }
        }
    };

    // Record activity to prevent session timeout
    const recordActivity = async () => {
        try {
            await axios.post(route("writer.essay.check-time"));
        } catch (error) {
            console.error("Error recording activity:", error);
            // If we get a redirect instruction, follow it
            if (error.response?.data?.redirect) {
                window.location.href = error.response.data.redirect;
            }
        }
    };

    // Auto-submit the essay when time expires
    const handleAutoSubmit = async () => {
        // Prevent multiple auto-submissions
        if (isAutoSubmitting) return;

        setIsAutoSubmitting(true);

        try {
            const token = document
                .querySelector('meta[name="csrf-token"]')
                ?.getAttribute("content");

            const response = await axios.post(
                route("writer.essay.auto-submit"),
                {},
                {
                    headers: {
                        "X-CSRF-TOKEN": token,
                        "Content-Type": "application/json",
                    },
                }
            );

            if (response.data.redirect) {
                window.location.href = response.data.redirect;
            }
        } catch (error) {
            console.error("Auto-submission error:", error);
            // Even if auto-submission fails, redirect to instructions
            window.location.href = route("writer.essay.instructions");
        }
    };

    // Handle file upload
    const handleFileUpload = async (event) => {
        const file = event.target.files[0];
        if (!file) return;

        // Validate file size
        if (file.size > maxFileSize * 1024 * 1024) {
            setUploadError(
                `File is too large. Maximum size is ${maxFileSize} MB.`
            );
            return;
        }

        // Validate file type
        const validTypes = [
            "application/pdf",
            "application/vnd.openxmlformats-officedocument.wordprocessingml.document",
        ];
        if (!validTypes.includes(file.type)) {
            setUploadError(
                "Invalid file type. Please upload a PDF or DOCX file."
            );
            return;
        }

        setIsUploading(true);
        setUploadProgress(0);
        setUploadError(null);

        const formData = new FormData();
        formData.append("essay_file", file);

        try {
            const token = document
                .querySelector('meta[name="csrf-token"]')
                ?.getAttribute("content");

            const response = await axios.post(
                route("writer.essay.upload-file"),
                formData,
                {
                    headers: {
                        "Content-Type": "multipart/form-data",
                        "X-CSRF-TOKEN": token,
                    },
                    onUploadProgress: (progressEvent) => {
                        const percentCompleted = Math.round(
                            (progressEvent.loaded * 100) / progressEvent.total
                        );
                        setUploadProgress(percentCompleted);
                    },
                }
            );

            if (response.data.success) {
                setUploadedFile({
                    name: response.data.filename,
                    type: response.data.fileType,
                    size: response.data.fileSize,
                });
                setUploadError(null);
            } else {
                setUploadError(response.data.error || "Failed to upload file.");
            }
        } catch (error) {
            console.error("Upload error:", error);
            setUploadError(
                error.response?.data?.error ||
                    "Failed to upload file. Please try again."
            );

            // If we get a redirect instruction, follow it
            if (error.response?.data?.redirect) {
                window.location.href = error.response.data.redirect;
            }
        } finally {
            setIsUploading(false);
        }
    };

    // Handle file removal
    const handleRemoveFile = () => {
        setUploadedFile(null);
        if (fileInputRef.current) {
            fileInputRef.current.value = "";
        }
    };

    // Handle test submission
    const handleSubmit = async () => {
        if (!uploadedFile) {
            setUploadError("Please upload your essay file before submitting.");
            return;
        }

        setIsSubmitting(true);

        try {
            const token = document
                .querySelector('meta[name="csrf-token"]')
                ?.getAttribute("content");

            const response = await axios.post(
                route("writer.essay.submit"),
                {},
                {
                    headers: {
                        "X-CSRF-TOKEN": token,
                    },
                }
            );

            if (response.data.redirect) {
                window.location.href = response.data.redirect;
            }
        } catch (error) {
            console.error("Submission error:", error);
            setUploadError(
                error.response?.data?.error ||
                    "Failed to submit test. Please try again."
            );

            // If we get a redirect instruction, follow it
            if (error.response?.data?.redirect) {
                window.location.href = error.response.data.redirect;
            }

            setIsSubmitting(false);
        }
    };

    // Update warning level based on remaining time
    const updateWarningLevel = (seconds) => {
        if (seconds <= 60) {
            // 1 minute or less
            setWarningLevel("high");
            setShowWarning(true);
        } else if (seconds <= 300) {
            // 5 minutes or less
            setWarningLevel("medium");
            setShowWarning(true);
        } else if (seconds <= 600) {
            // 10 minutes or less
            setWarningLevel("low");
            setShowWarning(true);
        } else {
            setWarningLevel(null);
            setShowWarning(false);
        }
    };

    // Initialize timers and cleanup on unmount
    useEffect(() => {
        // Initial time check
        checkRemainingTime();

        // Set up timer to update countdown
        timerRef.current = setInterval(() => {
            setTimeLeft((prevTime) => {
                if (prevTime <= 1) {
                    // Time's up, trigger auto-submission
                    clearInterval(timerRef.current);
                    handleAutoSubmit();
                    return 0;
                }

                // Update warning level
                updateWarningLevel(prevTime - 1);

                return prevTime - 1;
            });
        }, 1000);

        // Set up periodic time check with server
        const timeCheckInterval = setInterval(checkRemainingTime, 30000); // Every 30 seconds

        // Set up activity recording
        pingRef.current = setInterval(recordActivity, 60000); // Every minute

        // Set up auto-submit timeout as a backup
        // This ensures auto-submit happens even if the interval is delayed
        if (remainingTime) {
            autoSubmitTimeoutRef.current = setTimeout(() => {
                handleAutoSubmit();
            }, remainingTime * 1000 + 1000); // Add 1 second buffer
        }

        // Cleanup on unmount
        return () => {
            clearInterval(timerRef.current);
            clearInterval(timeCheckInterval);
            clearInterval(pingRef.current);
            clearTimeout(autoSubmitTimeoutRef.current);
        };
    }, []);

    // Set up beforeunload event to warn user before leaving page
    useEffect(() => {
        const handleBeforeUnload = (e) => {
            if (timeLeft > 0 && !isSubmitting && !isAutoSubmitting) {
                const message =
                    "You are in the middle of your essay test. If you leave, your progress may not be saved.";
                e.returnValue = message;
                return message;
            }
        };

        window.addEventListener("beforeunload", handleBeforeUnload);

        return () => {
            window.removeEventListener("beforeunload", handleBeforeUnload);
        };
    }, [timeLeft, isSubmitting, isAutoSubmitting]);

    return (
        <WriterLayout>
            <Head title="Writer Essay Test - Upload" />

            <div className="py-12">
                <div className="max-w-4xl mx-auto sm:px-6 lg:px-8">
                    <div className="bg-white dark:bg-gray-800 overflow-hidden shadow-sm sm:rounded-lg">
                        <div className="p-6 bg-white dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
                            <div className="flex justify-between items-center mb-6">
                                <h1 className="text-2xl font-bold text-gray-800 dark:text-gray-100">
                                    Upload Your Essay
                                </h1>
                                <div
                                    className={`flex items-center px-4 py-2 rounded-lg ${
                                        warningLevel === "high"
                                            ? "bg-red-50 dark:bg-red-900/30 text-red-700 dark:text-red-300 animate-pulse"
                                            : warningLevel === "medium"
                                            ? "bg-orange-50 dark:bg-orange-900/30 text-orange-700 dark:text-orange-300"
                                            : "bg-blue-50 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300"
                                    }`}
                                >
                                    <Clock
                                        className={`h-5 w-5 mr-2 ${
                                            warningLevel === "high"
                                                ? "text-red-500 dark:text-red-400"
                                                : warningLevel === "medium"
                                                ? "text-orange-500 dark:text-orange-400"
                                                : "text-blue-500 dark:text-blue-400"
                                        }`}
                                    />
                                    <span className="font-mono">
                                        {formatTime(timeLeft)}
                                    </span>
                                </div>
                            </div>

                            {/* Time warning */}
                            {showWarning && (
                                <div
                                    className={`mb-6 p-4 border-l-4 ${
                                        warningLevel === "high"
                                            ? "bg-red-50 dark:bg-red-900/20 border-red-500 text-red-700 dark:text-red-300"
                                            : warningLevel === "medium"
                                            ? "bg-orange-50 dark:bg-orange-900/20 border-orange-500 text-orange-700 dark:text-orange-300"
                                            : "bg-yellow-50 dark:bg-yellow-900/20 border-yellow-400 text-yellow-700 dark:text-yellow-300"
                                    }`}
                                >
                                    <div className="flex">
                                        <div className="flex-shrink-0">
                                            <AlertTriangle
                                                className={`h-5 w-5 ${
                                                    warningLevel === "high"
                                                        ? "text-red-500"
                                                        : warningLevel ===
                                                          "medium"
                                                        ? "text-orange-500"
                                                        : "text-yellow-500"
                                                }`}
                                                aria-hidden="true"
                                            />
                                        </div>
                                        <div className="ml-3">
                                            <p className="text-sm font-medium">
                                                {warningLevel === "high"
                                                    ? "Time is almost up! Your test will be automatically submitted in less than a minute."
                                                    : warningLevel === "medium"
                                                    ? "Warning: Less than 5 minutes remaining. Please finish and submit your essay soon."
                                                    : "Notice: Less than 10 minutes remaining. Consider finalizing your submission."}
                                            </p>
                                        </div>
                                    </div>
                                </div>
                            )}

                            {/* Essay topic reminder */}
                            {essayTopic && (
                                <div className="mb-6 p-4 bg-indigo-50 dark:bg-indigo-900/20 border border-indigo-200 dark:border-indigo-800 rounded-md">
                                    <h2 className="text-md font-semibold text-indigo-700 dark:text-indigo-300 mb-1">
                                        Essay Topic Reminder
                                    </h2>
                                    <p className="text-sm text-indigo-600 dark:text-indigo-400">
                                        {essayTopic.title}
                                    </p>
                                </div>
                            )}

                            {/* Upload instructions */}
                            <div className="mb-6 p-5 bg-gray-50 dark:bg-gray-700/50 rounded-lg border border-gray-200 dark:border-gray-600">
                                <h2 className="text-lg font-semibold text-gray-700 dark:text-gray-200 mb-3 flex items-center">
                                    <FileText className="h-5 w-5 mr-2 text-blue-500 dark:text-blue-400" />
                                    Upload Instructions
                                </h2>

                                <ul className="space-y-2 text-gray-600 dark:text-gray-400 list-disc list-inside">
                                    <li>
                                        Upload your completed essay as a PDF or
                                        DOCX file.
                                    </li>
                                    <li>
                                        Maximum file size:{" "}
                                        <strong>{maxFileSize} MB</strong>
                                    </li>
                                    <li>
                                        Make sure your essay follows all the
                                        requirements from the instructions.
                                    </li>
                                    <li>
                                        You can replace your file by uploading a
                                        new one before submitting.
                                    </li>
                                    <li>
                                        <strong>Important:</strong> Your test
                                        will be automatically submitted when
                                        time expires.
                                    </li>
                                </ul>
                            </div>

                            {/* File upload area */}
                            <div className="mb-6">
                                {!uploadedFile ? (
                                    <div className="border-2 border-dashed border-gray-300 dark:border-gray-600 rounded-lg p-6 text-center">
                                        <input
                                            type="file"
                                            id="essay_file"
                                            ref={fileInputRef}
                                            className="hidden"
                                            onChange={handleFileUpload}
                                            accept=".pdf,.docx,application/pdf,application/vnd.openxmlformats-officedocument.wordprocessingml.document"
                                            disabled={isUploading}
                                        />

                                        <label
                                            htmlFor="essay_file"
                                            className="flex flex-col items-center justify-center cursor-pointer"
                                        >
                                            <UploadIcon className="h-12 w-12 text-gray-400 dark:text-gray-500 mb-3" />
                                            <span className="text-lg font-medium text-gray-700 dark:text-gray-300 mb-1">
                                                {isUploading
                                                    ? "Uploading..."
                                                    : "Upload your essay file"}
                                            </span>
                                            <span className="text-sm text-gray-500 dark:text-gray-400 mb-3">
                                                PDF or DOCX, max {maxFileSize}{" "}
                                                MB
                                            </span>

                                            <span className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition">
                                                {isUploading
                                                    ? `Uploading (${uploadProgress}%)`
                                                    : "Select File"}
                                            </span>
                                        </label>

                                        {isUploading && (
                                            <div className="mt-4">
                                                <div className="w-full bg-gray-200 dark:bg-gray-700 rounded-full h-2.5">
                                                    <div
                                                        className="bg-blue-600 h-2.5 rounded-full"
                                                        style={{
                                                            width: `${uploadProgress}%`,
                                                        }}
                                                    ></div>
                                                </div>
                                            </div>
                                        )}
                                    </div>
                                ) : (
                                    <div className="border border-gray-200 dark:border-gray-700 rounded-lg p-4">
                                        <div className="flex items-start justify-between">
                                            <div className="flex items-start">
                                                <div className="flex-shrink-0">
                                                    <File className="h-10 w-10 text-blue-500 dark:text-blue-400" />
                                                </div>
                                                <div className="ml-3">
                                                    <h3 className="text-lg font-medium text-gray-800 dark:text-gray-200">
                                                        {uploadedFile.name}
                                                    </h3>
                                                    <div className="mt-1 flex items-center text-sm text-gray-500 dark:text-gray-400">
                                                        <span className="mr-3">
                                                            {getFileTypeName(
                                                                uploadedFile.type
                                                            )}
                                                        </span>
                                                        <span>
                                                            {formatFileSize(
                                                                uploadedFile.size
                                                            )}
                                                        </span>
                                                    </div>
                                                    <div className="mt-2">
                                                        <span className="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800 dark:bg-green-900/30 dark:text-green-300">
                                                            <CheckCircle className="h-3 w-3 mr-1" />
                                                            Ready to submit
                                                        </span>
                                                    </div>
                                                </div>
                                            </div>
                                            <button
                                                type="button"
                                                onClick={handleRemoveFile}
                                                className="text-red-500 hover:text-red-700 dark:text-red-400 dark:hover:text-red-300"
                                                disabled={
                                                    isSubmitting ||
                                                    isAutoSubmitting
                                                }
                                            >
                                                <Trash2 className="h-5 w-5" />
                                            </button>
                                        </div>
                                    </div>
                                )}

                                {uploadError && (
                                    <div className="mt-2 text-sm text-red-600 dark:text-red-400 flex items-center">
                                        <XCircle className="h-4 w-4 mr-1" />
                                        {uploadError}
                                    </div>
                                )}
                            </div>

                            {/* Auto-submission warning for low time */}
                            {warningLevel === "high" && (
                                <div className="mb-6 p-4 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-md">
                                    <h3 className="text-md font-semibold text-red-700 dark:text-red-300 mb-1 flex items-center">
                                        <AlertTriangle className="h-4 w-4 mr-1" />
                                        Auto-Submission Imminent
                                    </h3>
                                    <p className="text-sm text-red-600 dark:text-red-400">
                                        Your test will be automatically
                                        submitted when the timer reaches zero.
                                        {uploadedFile
                                            ? " Your current file will be submitted."
                                            : " Please upload a file immediately if you haven't already."}
                                    </p>
                                </div>
                            )}

                            {/* Submit button */}
                            <div className="mt-8 flex justify-end">
                                <Link href={route("writer.essay.instructions")}>
                                    <span className="inline-flex items-center px-4 py-2 bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-md font-semibold text-xs text-gray-700 dark:text-gray-200 uppercase tracking-widest shadow-sm hover:bg-gray-50 dark:hover:bg-gray-600 focus:outline-none focus:ring-2 focus:ring-indigo-500 dark:focus:ring-indigo-600 focus:ring-offset-2 dark:focus:ring-offset-gray-800 disabled:opacity-25 transition ease-in-out duration-150 mr-3">
                                        Back to Instructions
                                    </span>
                                </Link>

                                <button
                                    type="button"
                                    onClick={handleSubmit}
                                    disabled={
                                        !uploadedFile ||
                                        isSubmitting ||
                                        isAutoSubmitting
                                    }
                                    className="flex items-center px-4 py-2 bg-green-600 text-white rounded hover:bg-green-700 transition disabled:opacity-75 disabled:bg-green-500 disabled:cursor-not-allowed"
                                >
                                    {isSubmitting || isAutoSubmitting ? (
                                        <>
                                            <svg
                                                className="animate-spin -ml-1 mr-2 h-4 w-4 text-white"
                                                xmlns="http://www.w3.org/2000/svg"
                                                fill="none"
                                                viewBox="0 0 24 24"
                                            >
                                                <circle
                                                    className="opacity-25"
                                                    cx="12"
                                                    cy="12"
                                                    r="10"
                                                    stroke="currentColor"
                                                    strokeWidth="4"
                                                ></circle>
                                                <path
                                                    className="opacity-75"
                                                    fill="currentColor"
                                                    d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"
                                                ></path>
                                            </svg>
                                            {isAutoSubmitting
                                                ? "Auto-Submitting..."
                                                : "Submitting..."}
                                        </>
                                    ) : (
                                        <>
                                            Submit Essay
                                            <Send className="ml-2 h-4 w-4" />
                                        </>
                                    )}
                                </button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </WriterLayout>
    );
}
