import WriterLayout from "@/Layouts/WriterLayout";
import ExpiresIn from "@/Components/ExpiresIn";
import { Package2, Clock, MoveLeft, Eye } from "lucide-react";
import { useState, useEffect } from "react";
import { Head, Link } from "@inertiajs/react";
import CleanPagination from "@/Components/CleanPagination";
import OrderPostedTime from "@/Components/OrderPostedTime";

export default function UserOrders({ user, orders }) {
    const [isMobile, setIsMobile] = useState(window.innerWidth < 768);

    // Handle responsive layout
    useEffect(() => {
        const handleResize = () => {
            setIsMobile(window.innerWidth < 768);
        };
        window.addEventListener("resize", handleResize);
        return () => window.removeEventListener("resize", handleResize);
    }, []);

    // Format date to more readable format
    const formatDate = (dateString) => {
        if (!dateString) return "N/A";
        return new Date(dateString).toLocaleString();
    };

    return (
        <WriterLayout>
            <Head title=" Available Orders" />

            {/* Main page container with professional gradient background */}
            <div className=" min-h-screen transition-colors duration-200 ">
                <div className="flex flex-col md:flex-row md:justify-between md:items-center gap-4 mt-6 p-4">
                    <h1 className="text-2xl font-bold text-green-800 dark:text-green-400">
                        Available Orders to Bid
                    </h1>
                </div>

                <div className="flex justify-end items-center mb-4 px-4"></div>

                {!orders?.data?.length ? (
                    <div className="flex flex-col items-center justify-center p-8 bg-white dark:bg-gray-800 rounded-lg shadow-md dark:shadow-gray-700 transition-colors mx-4">
                        <Package2 className="w-16 h-16 text-green-400 dark:text-green-500 mb-4" />
                        <p className="text-gray-600 dark:text-gray-400 text-lg">
                            No orders available for bidding.
                        </p>
                    </div>
                ) : (
                    <>
                        <div className="px-4 mt-4">
                            {orders.data.length > 0 ? (
                                <>
                                    <div className="hidden lg:flex bg-gradient-to-r from-green-800 to-green-700 dark:from-gray-800 dark:to-gray-900 rounded-t-lg p-6 font-bold text-white dark:text-gray-200">
                                        <div className="w-1/6">
                                            Order Number
                                        </div>
                                        <div className="w-1/6">Pages</div>
                                        <div className="w-1/6">
                                            Subject Category
                                        </div>
                                        <div className="w-1/6">Paper Type</div>
                                        <div className="w-1/6">Posted </div>
                                        <div className="w-1/6">Time Left</div>
                                        <div className="w-1/6">Amount</div>
                                        <div className="w-1/6 text-center">
                                            Action
                                        </div>
                                    </div>

                                    <div className="space-y-4 lg:space-y-2 mb-6">
                                        {orders.data.map((order) => (
                                            <div
                                                key={order.id}
                                                className="bg-gradient-to-b from-green-800 via-green-700 to-green-800 dark:from-gray-800 dark:via-gray-700 dark:to-gray-800 text-white p-4 lg:p-0 shadow-md rounded-lg border border-green-300 dark:border-green-900 overflow-hidden transition-all duration-200 hover:shadow-lg"
                                            >
                                                <div className="hidden lg:flex items-center p-4 hover:bg-green-400/30 dark:hover:bg-green-800/40 transition-colors">
                                                    <div className="w-1/6 font-medium text-white dark:text-green-200 hover:text-green-100 dark:hover:text-white">
                                                        <Link
                                                            href={`/writer/orders/${order.id}`}
                                                        >
                                                            {order.order_number}
                                                        </Link>
                                                    </div>

                                                    <div className="w-1/6 font-medium">
                                                        {order.pages}
                                                    </div>

                                                    <div className="w-1/6 font-medium">
                                                        {order.subject}
                                                    </div>

                                                    <div className="w-1/6 font-medium">
                                                        {order.type_of_paper}
                                                    </div>
                                                    <div className="w-1/6 font-mono text-sm text-white dark:text-green-200">
                                                        <OrderPostedTime
                                                            postedDate={
                                                                order.dateposted
                                                            }
                                                        />
                                                    </div>

                                                    <div className="w-1/6">
                                                        <ExpiresIn
                                                            deadline={
                                                                order.writer_deadline
                                                            }
                                                        />
                                                    </div>

                                                    <div className="w-1/6">
                                                        $ {order.writer_amount}
                                                    </div>

                                                    <div className="w-1/6">
                                                        <Link
                                                            href={`/writer/orders/${order.id}`}
                                                        >
                                                            <button className="w-full flex items-center justify-center gap-2 bg-white text-green-700 dark:bg-green-200 dark:text-green-900 py-2 rounded-md hover:bg-green-50 dark:hover:bg-green-300 transition-colors duration-200 font-medium shadow-sm">
                                                                <Eye
                                                                    size={16}
                                                                />
                                                                View
                                                            </button>
                                                        </Link>
                                                    </div>
                                                </div>

                                                {/* small screen view */}
                                                <div className="lg:hidden p-4">
                                                    <div className="flex justify-between items-center mb-4">
                                                        <h2 className="text-lg font-semibold text-white dark:text-green-200">
                                                            <Link
                                                                href={`/orders/${order.id}`}
                                                            >
                                                                Order #
                                                                {
                                                                    order.order_number
                                                                }
                                                            </Link>
                                                        </h2>
                                                        <span
                                                            className={`px-3 py-1 rounded-full text-sm font-medium ${
                                                                order.order_status ===
                                                                "completed"
                                                                    ? "bg-green-100 dark:bg-green-800 text-green-800 dark:text-green-200"
                                                                    : order.order_status ===
                                                                      "assigned"
                                                                    ? "bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200"
                                                                    : "bg-yellow-100 dark:bg-yellow-800 text-yellow-800 dark:text-yellow-200"
                                                            }`}
                                                        >
                                                            {order.order_status}
                                                        </span>
                                                    </div>
                                                    <div className="space-y-2 text-sm">
                                                        <div className="flex justify-between mb-2">
                                                            <span className="text-white dark:text-green-200">
                                                                Posted:
                                                            </span>
                                                            <span className="text-white dark:text-green-100 font-medium">
                                                                <OrderPostedTime
                                                                    postedDate={
                                                                        order.dateposted
                                                                    }
                                                                />
                                                            </span>
                                                        </div>
                                                        <div className="flex justify-between mb-2">
                                                            <span className="text-white dark:text-green-200">
                                                                Payment Status:
                                                            </span>
                                                            <span
                                                                className={`px-3 py-1 rounded-full text-sm font-medium ${
                                                                    order.payment_status ===
                                                                    "paid"
                                                                        ? "bg-green-100 dark:bg-green-800 text-green-800 dark:text-green-200"
                                                                        : "bg-red-100 dark:bg-red-800 text-red-800 dark:text-red-200"
                                                                }`}
                                                            >
                                                                {
                                                                    order.payment_status
                                                                }
                                                            </span>
                                                        </div>
                                                        <div className="flex justify-between">
                                                            <span className="text-white dark:text-green-200">
                                                                Expires On:
                                                            </span>
                                                            <span className="font-medium text-white dark:text-green-100">
                                                                {new Date(
                                                                    order.deadline
                                                                ).toLocaleString(
                                                                    "en-US",
                                                                    {
                                                                        year: "numeric",
                                                                        month: "short",
                                                                        day: "numeric",
                                                                        hour: "2-digit",
                                                                        minute: "2-digit",
                                                                    }
                                                                )}
                                                            </span>
                                                        </div>
                                                        <div className="flex justify-between">
                                                            <span className="text-white dark:text-green-200">
                                                                Amount:
                                                            </span>
                                                            <span className="font-medium text-white dark:text-green-100">
                                                                $
                                                                {(
                                                                    parseFloat(
                                                                        order.net_amount
                                                                    ) * 0.4
                                                                ).toFixed(2)}
                                                            </span>
                                                        </div>
                                                    </div>
                                                    <div className="mt-4">
                                                        <Link
                                                            href={`/writer/orders/${order.id}`}
                                                        >
                                                            <button className="w-full bg-white text-green-700 dark:bg-green-200 dark:text-green-900 py-2 rounded-md hover:bg-green-50 dark:hover:bg-green-300 transition-colors duration-200 font-medium shadow-sm">
                                                                View
                                                            </button>
                                                        </Link>
                                                    </div>
                                                </div>
                                            </div>
                                        ))}
                                    </div>
                                </>
                            ) : (
                                <div className="text-center p-6 bg-white dark:bg-gray-800 rounded-lg shadow-md mx-4">
                                    <p className="text-gray-500 dark:text-gray-300">
                                        You have no orders yet.
                                    </p>
                                    <Link
                                        href={route("order.create")}
                                        className="mt-4 inline-block bg-green-600 dark:bg-green-700 text-white px-4 py-2 rounded-md hover:bg-green-700 dark:hover:bg-green-600 transition-colors duration-200"
                                    >
                                        Create an Order
                                    </Link>
                                </div>
                            )}

                            {/* Pagination */}
                            <div className="mt-6">
                                <CleanPagination
                                    links={orders.links}
                                    total={orders.total}
                                    perPage={orders.per_page}
                                    currentPage={orders.current_page}
                                />
                            </div>
                        </div>
                    </>
                )}
            </div>
        </WriterLayout>
    );
}
