import React, { useState, useEffect } from 'react';
import { Head, Link, usePage } from '@inertiajs/react';
import WriterLayout from '@/Layouts/WriterLayout';
import WithdrawalRequestForm from '@/Components/Writer/WithdrawalRequestForm';
import { 
    DollarSign, 
    Clock, 
    Wallet, 
    TrendingUp, 
    Calendar,
    Download,
    AlertCircle,
    CheckCircle,
    XCircle
} from 'lucide-react';

export default function PaymentDashboard({ summary, eligibility, recentWithdrawals }) {
    const [showWithdrawalForm, setShowWithdrawalForm] = useState(false);
    const { flash } = usePage().props;

    const formatCurrency = (amount) => {
        return new Intl.NumberFormat('en-US', {
            style: 'currency',
            currency: 'USD'
        }).format(amount);
    };

    const formatDate = (dateString) => {
        return new Date(dateString).toLocaleDateString('en-US', {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        });
    };

    const getStatusIcon = (status) => {
        switch (status) {
            case 'approved':
            case 'paid':
                return <CheckCircle size={16} className="text-green-500" />;
            case 'rejected':
                return <XCircle size={16} className="text-red-500" />;
            default:
                return <Clock size={16} className="text-yellow-500" />;
        }
    };

    const getStatusColor = (status) => {
        switch (status) {
            case 'approved':
            case 'paid':
                return 'bg-green-100 text-green-800';
            case 'rejected':
                return 'bg-red-100 text-red-800';
            case 'processing':
                return 'bg-blue-100 text-blue-800';
            default:
                return 'bg-yellow-100 text-yellow-800';
        }
    };

    return (
        <WriterLayout>
            <Head title="Payment Dashboard" />
            
            <div className="p-6 max-w-7xl mx-auto">
                {/* Flash Messages */}
                {flash.success && (
                    <div className="mb-6 p-4 bg-green-100 border border-green-400 text-green-700 rounded-md flex items-center">
                        <CheckCircle size={20} className="mr-2" />
                        {flash.success}
                    </div>
                )}
                
                {flash.error && (
                    <div className="mb-6 p-4 bg-red-100 border border-red-400 text-red-700 rounded-md flex items-center">
                        <AlertCircle size={20} className="mr-2" />
                        {flash.error}
                    </div>
                )}

                {/* Header */}
                <div className="mb-8">
                    <h1 className="text-3xl font-bold text-gray-900 mb-2">Payment Dashboard</h1>
                    <p className="text-gray-600">
                        Track your earnings, pending payments, and withdrawal history
                    </p>
                </div>

                {/* Summary Cards */}
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
                    <div className="bg-white rounded-lg border shadow-sm p-6">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 mb-1">Pending</p>
                                <p className="text-2xl font-bold text-yellow-600">
                                    {formatCurrency(summary.pending_amount)}
                                </p>
                                <p className="text-xs text-gray-500 mt-1">
                                    {summary.pending_orders_count} orders
                                </p>
                            </div>
                            <div className="p-3 bg-yellow-100 rounded-full">
                                <Clock size={24} className="text-yellow-600" />
                            </div>
                        </div>
                        {summary.pending_orders_count > 0 && (
                            <Link
                                href="/writer/payment/pending"
                                className="inline-flex items-center text-sm text-yellow-600 hover:text-yellow-800 mt-3"
                            >
                                View pending orders →
                            </Link>
                        )}
                    </div>

                    <div className="bg-white rounded-lg border shadow-sm p-6">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 mb-1">Available</p>
                                <p className="text-2xl font-bold text-green-600">
                                    {formatCurrency(summary.available_amount)}
                                </p>
                                <p className="text-xs text-gray-500 mt-1">
                                    {summary.available_orders_count} orders
                                </p>
                            </div>
                            <div className="p-3 bg-green-100 rounded-full">
                                <Wallet size={24} className="text-green-600" />
                            </div>
                        </div>
                        {summary.available_orders_count > 0 && (
                            <Link
                                href="/writer/payment/available"
                                className="inline-flex items-center text-sm text-green-600 hover:text-green-800 mt-3"
                            >
                                View available orders →
                            </Link>
                        )}
                    </div>

                    <div className="bg-white rounded-lg border shadow-sm p-6">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 mb-1">Withdrawn</p>
                                <p className="text-2xl font-bold text-blue-600">
                                    {formatCurrency(summary.withdrawn_amount)}
                                </p>
                                <p className="text-xs text-gray-500 mt-1">
                                    Total withdrawn
                                </p>
                            </div>
                            <div className="p-3 bg-blue-100 rounded-full">
                                <Download size={24} className="text-blue-600" />
                            </div>
                        </div>
                    </div>

                    <div className="bg-white rounded-lg border shadow-sm p-6">
                        <div className="flex items-center justify-between">
                            <div>
                                <p className="text-sm text-gray-600 mb-1">Total Earnings</p>
                                <p className="text-2xl font-bold text-purple-600">
                                    {formatCurrency(summary.total_earnings)}
                                </p>
                                <p className="text-xs text-gray-500 mt-1">
                                    All time
                                </p>
                            </div>
                            <div className="p-3 bg-purple-100 rounded-full">
                                <TrendingUp size={24} className="text-purple-600" />
                            </div>
                        </div>
                    </div>
                </div>

                <div className="grid grid-cols-1 lg:grid-cols-3 gap-8">
                    {/* Withdrawal Section */}
                    <div className="lg:col-span-2">
                        <div className="bg-white rounded-lg border shadow-sm">
                            <div className="px-6 py-4 border-b">
                                <h2 className="text-xl font-semibold text-gray-900">Request Withdrawal</h2>
                            </div>
                            <div className="p-6">
                                {eligibility.can_withdraw ? (
                                    <div className="text-center">
                                        <div className="p-4 bg-green-100 rounded-full w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                                            <CheckCircle size={32} className="text-green-600" />
                                        </div>
                                        <h3 className="text-lg font-medium text-gray-900 mb-2">
                                            Ready for Withdrawal
                                        </h3>
                                        <p className="text-gray-600 mb-6">
                                            You have {formatCurrency(eligibility.available_amount)} available for withdrawal.
                                            {eligibility.minimum_withdrawal && (
                                                <span className="block text-sm text-gray-500 mt-1">
                                                    (Minimum: {formatCurrency(eligibility.minimum_withdrawal)})
                                                </span>
                                            )}
                                        </p>
                                        <button
                                            className="bg-green-600 text-white px-6 py-3 rounded-lg hover:bg-green-700 transition-colors font-medium"
                                            onClick={() => setShowWithdrawalForm(true)}
                                        >
                                            Request Withdrawal
                                        </button>
                                    </div>
                                ) : (
                                    <div className="text-center">
                                        <div className="p-4 bg-yellow-100 rounded-full w-16 h-16 mx-auto mb-4 flex items-center justify-center">
                                            <AlertCircle size={32} className="text-yellow-600" />
                                        </div>
                                        <h3 className="text-lg font-medium text-gray-900 mb-2">
                                            Withdrawal Not Available
                                        </h3>
                                        <div className="space-y-2 mb-6">
                                            {eligibility.reasons.map((reason, index) => (
                                                <p key={index} className="text-sm text-gray-600">
                                                    • {reason}
                                                </p>
                                            ))}
                                        </div>
                                        {eligibility.next_withdrawal_date && (
                                            <div className="flex items-center justify-center gap-2 text-sm text-gray-600">
                                                <Calendar size={16} />
                                                <span>
                                                    Next withdrawal date: {formatDate(eligibility.next_withdrawal_date)}
                                                </span>
                                            </div>
                                        )}
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>

                    {/* Recent Withdrawals */}
                    <div>
                        <div className="bg-white rounded-lg border shadow-sm">
                            <div className="px-6 py-4 border-b">
                                <h2 className="text-lg font-semibold text-gray-900">Recent Withdrawals</h2>
                            </div>
                            <div className="p-6">
                                {recentWithdrawals.length === 0 ? (
                                    <p className="text-gray-600 text-center">No withdrawal history</p>
                                ) : (
                                    <div className="space-y-4">
                                        {recentWithdrawals.map((withdrawal) => (
                                            <div key={withdrawal.id} className="flex items-center justify-between py-2">
                                                <div>
                                                    <div className="flex items-center gap-2">
                                                        {getStatusIcon(withdrawal.status)}
                                                        <span className="font-medium text-gray-900">
                                                            {formatCurrency(withdrawal.requested_amount)}
                                                        </span>
                                                    </div>
                                                    <p className="text-sm text-gray-600">
                                                        {formatDate(withdrawal.created_at)}
                                                    </p>
                                                </div>
                                                <span className={`px-2 py-1 text-xs font-medium rounded-full ${getStatusColor(withdrawal.status)}`}>
                                                    {withdrawal.status}
                                                </span>
                                            </div>
                                        ))}
                                    </div>
                                )}
                            </div>
                        </div>
                    </div>
                </div>

                {/* Quick Actions */}
                <div className="mt-8 bg-gray-50 rounded-lg p-6">
                    <h3 className="text-lg font-medium text-gray-900 mb-4">Quick Actions</h3>
                    <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
                        <Link
                            href="/writer/payment/pending"
                            className="flex items-center gap-3 p-4 bg-white rounded-lg border hover:border-yellow-300 transition-colors"
                        >
                            <Clock size={20} className="text-yellow-600" />
                            <div>
                                <p className="font-medium text-gray-900">View Pending</p>
                                <p className="text-sm text-gray-600">{summary.pending_orders_count} orders</p>
                            </div>
                        </Link>
                        
                        <Link
                            href="/writer/payment/available"
                            className="flex items-center gap-3 p-4 bg-white rounded-lg border hover:border-green-300 transition-colors"
                        >
                            <Wallet size={20} className="text-green-600" />
                            <div>
                                <p className="font-medium text-gray-900">View Available</p>
                                <p className="text-sm text-gray-600">{summary.available_orders_count} orders</p>
                            </div>
                        </Link>
                        
                        <Link
                            href="/writer/orders"
                            className="flex items-center gap-3 p-4 bg-white rounded-lg border hover:border-blue-300 transition-colors"
                        >
                            <TrendingUp size={20} className="text-blue-600" />
                            <div>
                                <p className="font-medium text-gray-900">All Orders</p>
                                <p className="text-sm text-gray-600">View all orders</p>
                            </div>
                        </Link>
                    </div>
                </div>
            </div>

            {/* Withdrawal Request Form Modal */}
            <WithdrawalRequestForm
                isOpen={showWithdrawalForm}
                onClose={() => setShowWithdrawalForm(false)}
                eligibility={eligibility}
                availableAmount={summary.available_amount}
            />
        </WriterLayout>
    );
}